/*****************************************************************************
 * vlms.c: simple MPEG stream server
 *****************************************************************************
 * Copyright (C) 1998, 1999, 2000 VideoLAN
 *
 * Authors:
 * Regis Duchesne,     VIA, ECP, France <regis@via.ecp.fr>,   27/01/97
 * Michel Lespinasse,  VIA, ECP, France <walken@via.ecp.fr>,  02/02/97
 * Jean-Marc Dressler, VIA, ECP, France <polux@via.ecp.fr>,   17/05/99
 * Samuel Hocevar,     VIA, ECP, France <sam@via.ecp.fr>,     17/12/99
 * Damien Lucas,       VIA, ECP, France <nitrox@via.ecp.fr>,  30/05/01
 * Marc Ariberti,      VIA, ECP, France <marcari@via.ecp.fr>, 29/04/02
 * Christophe Massiot, VIA, ECP, France <massiot@via.ecp.fr>, 30/05/02
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 * Principle:
 *   Read a Transport Stream (TS) file from the hard disk, and emit it at the
 *   same speed that the encoder clock on the network using UDP/IP. This is an
 *   implementation reference of a simple Video On Demand server. 
 *****************************************************************************/

/*
 * TODO LIST:
 * - Use real-time threads, because the scheduling delays our usleep too
 *   much (>0.3 s) on a machine with numerous processes.
 *
 * Remarks:
 *   Implementing it by mmap()ing the TS file would be a bad idea, as the TS
 *   file is usually big enough to make to process to be swapped, which
 *   delays our read from the disk.
 */ 
 
/* common stuff */
#include "vlms.h"

#ifdef LINUX
#  include <net/if.h>
#endif

/* TS parsing functions */
#include "ts.h"
/* PS/PES parsing functions */
#include "ps.h"

#define COPYRIGHT \
"VideoLAN Mini Server - version 0.2.2 Onatopp - (c)1996-2000 VideoLAN\n"

#define VERSION \
COPYRIGHT \
"This program comes with NO WARRANTY, to the extent permitted by law.\n" \
"You may redistribute it under the terms of the GNU General Public " \
"License;\n" \
"see the file named COPYING for details.\n" \
"Written by the VideoLAN team at Ecole Centrale, Paris.\n"

#define HELP \
"Usage: %s [OPTION] [FILE]...\n" \
COPYRIGHT \
"\n" \
"Options:\n" \
"  -d host[:port]           target address\n" \
"  -t ttl                   ttl value\n" \
"  -p pid                   optional PCR PID\n" \
"\n" \
"  -a [ac3|lpcm|mpeg|off]   choose audio type\n" \
"  -i dev                   choose to device to emit (LINUX)\n" \
"  -c [0-15]                choose audio channel\n" \
"  -s [0-31]                choose subtitle channel\n" \
"\n" \
"  -l                       loops for ever\n" \
"  -n nbloops               loops the playlist nbloops times\n" \
"\n" \
"  -h, --help               display this help and exit\n" \
"  -v, --version            output version information and exit\n" \
"\n" \
"With no FILE, or when FILE is -, read standard input.\n"

/* Long options */
static const struct option longopts[] =
{
    /*  name,               has_arg,    flag,   val */
    {   "version",          0,          0,      'v' },
    {   "help",             0,          0,      'h' },
    {   0,                  0,          0,      0 }
};

/* Short options */
static const char *shortopts = "d:p:t:a:c:s:i:n:lhv";

/******************************************************************************
 * GetTime : returns the current time in microseconds
 ******************************************************************************/

s64 GetTime(void)
{
    struct timeval tmp;

    gettimeofday(&tmp, NULL);
    return( (s64)tmp.tv_sec * 1000000 + (s64)tmp.tv_usec );
}

/******************************************************************************
 * ConvertPCRTime : extracts and converts the PCR time in microseconds
 ******************************************************************************/

s64 ConvertPCRTime(file_ts_packet *_pcr_buff)
{
    u8 * pcr_buff = (u8 *)_pcr_buff;
    return((((s64)(pcr_buff[6]) << 25) |
            ((s64)(pcr_buff[7]) << 17) |
            ((s64)(pcr_buff[8]) << 9) |
            ((s64)(pcr_buff[9]) << 1) |
            ((s64)(pcr_buff[10]) >> 7)) * 300 / 27 );
}

/******************************************************************************
 * msleep : sleeps 'delay' microseconds
 ******************************************************************************/

void msleep( s64 delay )
{
    struct timeval tv_delay;

    tv_delay.tv_sec = delay / 1000000;
    tv_delay.tv_usec = delay % 1000000;
    /* select() return value should be tested, since several possible errors
     * can occur. However, they should only happen in very particular occasions
     * (i.e. when a signal is sent to the thread, or when memory is full), and
     * can be ignored. */
    select( 0, NULL, NULL, NULL, &tv_delay );
}

/******************************************************************************
 * adjust : Adjust the encoder clock & remove the PCR from own_pcr
 ******************************************************************************/

static void adjust(file_ts_packet *ts)
{
    file_ts_packet *next_pcr;
    int no_discontinuity = 1;
    
    if( ((u8*)ts)[5] & 0x80 )
    {
        /* There is a discontinuity - I recalculate the delta */
        synchro.delta_clock = GetTime() - ConvertPCRTime(ts);
        fprintf( stderr, "Clock Discontinuity\n" );
        no_discontinuity = 0;
    }
    else
    {
        synchro.last_pcr = ts;
        synchro.last_pcr_time = ConvertPCRTime( ts );
    }
        
    pthread_mutex_lock(&own_pcr.lock);
    own_pcr.start++;
    own_pcr.start %= (TS_BUFFER_SIZE+1)*TS_IN_UDP+1;
    
    /* If we have 2 consecutiv PCR, we can reevaluate slope */
    if( (own_pcr.start != own_pcr.end) &&
        no_discontinuity &&
        !((((u8*) next_pcr = own_pcr.buf[own_pcr.start]))[5] & 0x80))
    {
        s64 current_pcr_time = ConvertPCRTime(ts);
        s64 next_pcr_time =    ConvertPCRTime(next_pcr);
        
        if( (next_pcr_time - current_pcr_time < 0) || (next_pcr_time - current_pcr_time > 700000))
        {
            fprintf( stderr, "Warning: possible discontinuity\n" );
            synchro.delta_clock = GetTime() - next_pcr_time;
        }
        else
        {
                //fprintf(stderr,"next - current : %Ld\n", next_pcr_time - current_pcr_time);
            synchro.slope = (next_pcr_time - current_pcr_time) /
                ((next_pcr - ts + (TS_BUFFER_SIZE+1)*TS_IN_UDP) % ((TS_BUFFER_SIZE+1)*TS_IN_UDP));
                //fprintf(stderr,"slope : %Ld\n", synchro.slope);
        }
    }
    
    pthread_mutex_unlock(&own_pcr.lock);
}


/******************************************************************************
 * wait_a_moment : Compute how long we must wait before sending a TS packet
 ******************************************************************************/

static void wait_a_moment(file_ts_packet *ts)
{
    static int retard_count = 0;
    static s64 wait_max = 0;
    s64 sendtime; /* the date at which the TS packet should be sent */
    s64 wait;
    
    sendtime = synchro.last_pcr_time + synchro.delta_clock +
        synchro.slope * ((ts - synchro.last_pcr + (TS_BUFFER_SIZE+1)*TS_IN_UDP) % ((TS_BUFFER_SIZE+1)*TS_IN_UDP)); 
    wait = sendtime - GetTime();
    //fprintf(stderr,"last  PCR Time : %Ld\n", synchro.last_pcr_time );
    if( wait > 0 )
    { 
        retard_count = 0;
        if(wait > 100000)
        {
            fprintf( stderr, "Warning : wait time may be too long : %Ld\n", wait );
            return;
        }
        msleep( wait );
    }
    else
    {
        if( wait < wait_max )
        {
            wait_max = wait;
        }
        retard_count++;
        if( retard_count == 64 )
        {
            retard_count = 0;
            fprintf( stderr, "Delay : %Ldms, max delay : %Ldms\n", -wait/1000, -wait_max/1000 );
        fflush(stdout);
        }
    }
}


/******************************************************************************
 * network_thread : empties the data buffer
 ******************************************************************************/

void *network_thread(void *arg)
{
#define s ((struct s_netthread*)arg)

    int i, howmany, sent_bytes;
    int pcr_count = 0;
    file_ts_packet *ts;

    howmany = TS_IN_UDP;
    synchro.slope = 0;
    
    /*
     * Initialisation of the synchro mecanism : wait for 1 PCR
     * to evaluate delta_clock
     */
    fprintf( stderr, "Synchro Initialisation : wait for PCR ...\n" );
    while(s->left)
    {
        if(s->left < TS_IN_UDP)
        {
            howmany = s->left;
        }
        
        pthread_mutex_lock(&in_data.lock);
        while( in_data.end == in_data.start )
        {
            pthread_cond_wait(&in_data.notempty, &in_data.lock);
        }
        pthread_mutex_unlock(&in_data.lock);
        
        ts = (file_ts_packet*)(in_data.buf + in_data.start);
        for( i=0 ; i < howmany ; i++, ts++ )
        {
            if( ts  == own_pcr.buf[own_pcr.start] && !(((u8*)ts)[5] & 0x80) )
            {
                synchro.last_pcr = ts;
                synchro.last_pcr_time = ConvertPCRTime( ts );
                synchro.delta_clock = GetTime() - ConvertPCRTime(ts);
                adjust( ts );
                pcr_count++;
            }
        }
        
        pthread_mutex_lock(&in_data.lock);
        in_data.start++;
        in_data.start %= TS_BUFFER_SIZE + 1;
        pthread_cond_signal(&in_data.notfull);
        pthread_mutex_unlock(&in_data.lock);
        s->left -= howmany;
        
        if(pcr_count)
            break; 
    }
    
    /*
     * Main network loop
     */
    fprintf( stderr, "Network Loop started\n" );
    while(s->left)
    {
        if(s->left < TS_IN_UDP)
        {
            howmany = s->left;
        }
        
        pthread_mutex_lock(&in_data.lock);
        while( in_data.end == in_data.start )
        {
            if( in_data.b_die )
            {
                exit( 0 );
            }
            pthread_cond_wait(&in_data.notempty, &in_data.lock);
        }
        pthread_mutex_unlock(&in_data.lock);
        
        ts = (file_ts_packet*)(in_data.buf + in_data.start);
        for( i=0 ; i < howmany ; i++, ts++ )
        {             
            if( synchro.slope && (i == howmany-1) )
            {
                wait_a_moment( ts );
            }
            if( ts  == own_pcr.buf[own_pcr.start] )
            {
                /* the TS packet has a PCR, just try to adjust the clock */
                adjust( ts );
                pcr_count++;
            }
        }
        
        sent_bytes = 0;
        /* Send the UDP packet over the network */
        sent_bytes += sendto( s->out, (char*)(ts - howmany),
                              howmany * TS_PACKET_SIZE, 0,
               (struct sockaddr *) &(s->remote_addr), sizeof(s->remote_addr) );

        pthread_mutex_lock(&in_data.lock);
        in_data.start++;
        in_data.start %= TS_BUFFER_SIZE + 1;
        pthread_cond_signal(&in_data.notfull);
        pthread_mutex_unlock(&in_data.lock);
        s->left -= howmany;
    }
    
    return NULL;    
#undef s
}

/******************************************************************************
 * keep_pcr : Put a TS packet in the fifo if it owns a PCR
 ******************************************************************************/

int keep_pcr(int pcr_pid, file_ts_packet *ts)
{
#define p ((u8 *)ts)
    if ((p[3] & 0x20) && p[4] && (p[5] & 0x10)
        && ((((p[1]<<8)+p[2]) & 0x1fff) == pcr_pid))
    {
        /* adaptation_field_control is set, adaptation_field_lenght is not 0,
         * PCR_flag is set, pid == pcr_pid */
        pthread_mutex_lock(&own_pcr.lock);
        own_pcr.buf[own_pcr.end++] = ts;
        own_pcr.end %= (TS_BUFFER_SIZE+1)*TS_IN_UDP+1;
        pthread_mutex_unlock(&own_pcr.lock);
        return 1;
    } 
    else
        return 0;
#undef p
}

/******************************************************************************
 * file_next : Opens the next available file
 ******************************************************************************/

int file_next( options_t *options )
{
    /* the check for index == 0 should be done _before_ */
    options->i_list_index--;

    if( options->in != -1 )
    {
        close( options->in );
    }

    if( !strcmp( options->playlist[options->i_list_index], "-" ) )
    {
        /* read stdin */
        return ( options->in = 0 );
    }
    else
    {
        /* read the actual file */
        fprintf( stderr, "Playing file %s\n",
                 options->playlist[options->i_list_index] );
        return ( options->in = open( options->playlist[options->i_list_index],
                                     O_RDONLY | O_NDELAY ) );
    }
}

/******************************************************************************
 * safe_read : Buffered reading method
 ******************************************************************************/

ssize_t safe_read( options_t *options, unsigned char *buf, int count )
{
    int ret, cnt=0;

    while( cnt < count )
    {
        ret = read( options->in, buf + cnt, count - cnt );

        if( ret < 0 )
            return ret;

        if( ret == 0 )
        {
            /* zero means end of file */
            if( options->i_list_index )
            {
                file_next( options );
            }
            else
            {
                /* test if we reached the number of loops */
                /* a negative value means looping infinitely */
                if( options->loop > 0 )
                {
                    options->loop--;
                }
                if( options->loop == 0)
                {
                    break;
                }
                else 
                {
                    options->i_list_index = options->i_list_size;
                    file_next( options );
                }
            }
        }

        cnt += ret;
    }

    return cnt;
}

/******************************************************************************
 * open_socket
 ******************************************************************************/

int open_socket( struct s_options *options,
                 struct sockaddr_in *remote_addr )
{
    int socket_fd, sockopt;
    struct sockaddr_in local_addr;
    
    /* open a socket for UDP/IP */
    if( (socket_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0 )
    {
        err_sys( "socket() error" );
    }

    BuildInetAddr( &local_addr, NULL );
    if( options->host != NULL )
    {
        BuildInetAddr( remote_addr, options->host );
    }
#ifdef LINUX
    else
    {
        int i_if_nb;
        net_descr_t net_descr;

        /* Get informations about the net configuration (interface ...)
         * to retrieve the broadcast address we need */
        ReadNetConf( socket_fd, &net_descr );
        i_if_nb = net_descr.i_if_number;
        if( i_if_nb == 0 )
        {
            fprintf( stderr, "Fatal error: no interface detected\n" );
            close( socket_fd );
            return( -1 );
        }
        fprintf( stderr, "Broadcast on interface %s\n", net_descr.a_if[i_if_nb-1].psz_ifname );
        *remote_addr = net_descr.a_if[i_if_nb-1].sa_bcast_addr;
        free( net_descr.a_if );
    }
#endif

    if( options->ttl != 0 ) 
    {
        sockopt = options->ttl;
        if( IN_CLASSD( ntohl( remote_addr->sin_addr.s_addr ) ) )
        {
            if( setsockopt( socket_fd, IPPROTO_IP, IP_MULTICAST_TTL, 
                            (void *)&sockopt, sizeof(sockopt) ) < 0 )
            {
                err_sys( "IP_MULTICAST_TTL setsockopt TTL error" );
            }
        }
        else 
        {
            if( setsockopt( socket_fd, IPPROTO_IP, IP_TTL, 
                            (void *)&sockopt, sizeof(sockopt) ) < 0 )
            {
                err_sys( "IP_TTL setsockopt TTL error" );
            }
        }
    }
    
    sockopt = 1;
    if( setsockopt( socket_fd, SOL_SOCKET, SO_BROADCAST,
                    (void *)&sockopt, sizeof(sockopt) ) < 0 )
    {
        err_sys( "SO_BROADCAST setsockopt error" );
    }
#ifdef LINUX
    if(options->dev != NULL)
    {
      struct ifreq sInterface;
      strncpy(sInterface.ifr_ifrn.ifrn_name,options->dev, IFNAMSIZ);
      if(setsockopt( socket_fd, SOL_SOCKET, SO_BINDTODEVICE, (char *)&sInterface, sizeof(sInterface)) < 0 )
      {
        err_sys( "SO_BINDTODEVICE setsockopt error" );
      }
    }
#endif
    
    if( bind(socket_fd, (struct sockaddr *)&local_addr, sizeof(local_addr)) < 0 )
    {
        err_sys( "bind() error" );
    }
    
    return socket_fd;
}

/******************************************************************************
 * usage : Show an help message and exit
 ******************************************************************************/

static void usage( char *prog )
{
    fprintf( stderr, HELP, prog );
}

/******************************************************************************
 * version : Show a version message and exit
 ******************************************************************************/

static void version( char *prog )
{
    fprintf( stderr, VERSION, prog );
}

/******************************************************************************
 * main : Main mini-server program
 ******************************************************************************/

int main(int argc, char **argv)
{
    options_t options;
    unsigned char first_byte;
    int opt;

    options.pcr_pid = 0;
    options.ttl = 0;
    options.loop = 1;
    options.host = NULL;
    options.dev = NULL;
    options.audio_type = REQUESTED_AC3;
    options.audio_channel = 0;
    options.subtitles_channel = NO_SUBTITLES;

    /* Argument checking */
    /* optional */
    opterr = 0; /* force getopt not to generate an error message for '?' */

    /* while((opt = getopt(argc, argv, shortopts)) != EOF) */
    while((opt = getopt_long( argc, argv, shortopts, longopts, 0)) != EOF)
    {
        switch(opt)
        {
        case 'd':
            options.host = optarg;
            break;
        case 'p':
            options.pcr_pid = atoi(optarg);
            break;
        case 't':
            options.ttl = atoi(optarg);
            break;
        case 'l':
            options.loop = -1;
            break;
        case 'n':
            options.loop = atoi(optarg);
            break;
        case 'a' :
            if ( ! strcmp(optarg, "mpeg") )
                options.audio_type = REQUESTED_MPEG;
            else if ( ! strcmp(optarg, "lpcm") )
                options.audio_type = REQUESTED_LPCM;
            else if ( ! strcmp(optarg, "off") )
                options.audio_type = REQUESTED_NOAUDIO;
            else if ( strcmp(optarg, "ac3") )
                fprintf( stderr, "Unknown audio type ! %s. Assuming ac3.\n" );
            break;
        case 'c' :
            options.audio_channel = atoi(optarg);
            if ( options.audio_channel > MAX_AUDIO_CHANNEL)
                fprintf( stderr, "Invalid audio channel. Assuming 0.\n" );
            break;
        case 's' :
            options.subtitles_channel = atoi(optarg);
            if ( options.subtitles_channel > MAX_SUBTITLES_CHANNEL )
                fprintf( stderr,"Invalid subtitles channel. Assuming 0.\n" );
            break;
        case 'i':
              options.dev = optarg;
              fprintf(stderr, "Interface set to %s\n",optarg);
            break;
        case 'h' :
            usage( *argv );
            exit( 0 );
            break;
        case 'v' :
            version( *argv );
            exit( 0 );
            break;
        case '?':
            fprintf( stderr, "%s: invalid option or missing parameter -- %c\n"
                             "Try `%s --help' for more information.\n",
                     *argv, optopt, *argv );
            exit( 1 );
            break;
        }
    }

    fprintf( stderr, COPYRIGHT );
    
    options.i_list_index = 0;

    if( optind < argc )
    {
        int i_index = 0;
        options.playlist = malloc( (options.i_list_index = argc - optind)
                                        * sizeof(int) );

        while( argc - i_index > optind )
        {
            options.playlist[ i_index ] = argv[ argc - i_index - 1];
            i_index++;
        }
    }
    else
    {
        options.playlist = malloc( sizeof(int) );
        options.playlist[ 0 ] = "-";
        options.i_list_index = 1;
    }

    /* takes note of the length of the playlist for looping */
    options.i_list_size = options.i_list_index;

    /* Don't use output buffer to have a real-time display */
    setvbuf( stdout, NULL, _IONBF ,0 );

    /* assume no filedescriptor is open yet */
    options.in = -1;

    if( file_next( &options ) < 0 )
    {
        err_sys( "open() error" );
    }

    safe_read( &options, &first_byte, 1 );

    switch( first_byte )
    {
        case 0x00:
            fprintf( stderr, "First byte is 0x00 - assuming PS/PES stream\n" );
            if( !options.pcr_pid ) options.pcr_pid = 0x20;
                                   /* first video stream */
            ps_thread( &options );
            break;
        case 0x47:
            fprintf( stderr, "First byte is 0x47 - assuming TS stream\n" );
            if( !options.pcr_pid ) options.pcr_pid = 120;
                                   /* goldeneye specific */
            ts_thread( &options );
            break;
        default:
            fprintf( stderr, "Error : cannot determine stream type\n" );
            return 2;
    }
    
    return 0;
}

