/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.jrobin.core.RrdDb;
import org.jrobin.core.RrdDbPool;
import org.jrobin.core.RrdDef;
import org.jrobin.core.RrdException;
import org.jrobin.core.Sample;
import org.jrobin.core.Util;
import org.jrobin.graph.RrdGraph;
import org.jrobin.graph.RrdGraphDef;
import org.videolan.vlma.dao.VLMaDao;
import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.Satellite;
import org.videolan.vlma.model.Server;
import org.videolan.vlma.monitor.OrderMonitor;
import org.videolan.vlma.monitor.ServerStateMonitor;
import org.videolan.vlma.notifier.Notifier;
import org.videolan.vlma.retriever.Data;
import org.videolan.vlma.retriever.DataRetriever;

/**
 * This is an implementation of the Data interface. This class is used to
 * contain the data and make them available through remoting.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 * @author Fabien Vinas
 */
public class VLMaServiceImpl implements VLMaService {

    public static final String DATA_FILE = "data.xml";
    public static final int SOCKET_TIMEOUT = 5000;

    private static final Logger logger = Logger.getLogger(VLMaServiceImpl.class);

    private Set<Order> orders = new HashSet<Order>();
    private Set<Order> ordersToCancel = new HashSet<Order>();

    private Configuration configuration;
    private DataRetriever dataRetriever;
    private OrderGiver orderGiver;
    private OrderMonitor orderMonitor;
    private ServerStateMonitor serverStateMonitor;
    private Set<Notifier> notifiers;
    private VLMaDao vlmaDao;

    public List<Satellite> getSatellites() {
        return vlmaDao.getSatellites();
    }

    public List<Server> getServers() {
        return vlmaDao.getServers();
    }

    public List<Media> getMedias() {
        return vlmaDao.getMedias();
    }

    public Set<Order> getOrders() {
        synchronized (orders) {
            return orders;
        }
    }

    public Set<Order> getOrdersToCancel() {
        synchronized (ordersToCancel) {
            return ordersToCancel;
        }
    }

    public void giveOrders() {
        orderGiver.giveOrders();
    }

    public void cancelOrder(Order order) {
        synchronized (ordersToCancel) {
            ordersToCancel.add(order);
        }
    }

    public void sendNotification(String message) {
        for (Notifier notifier : notifiers) {
            synchronized (notifier) {
                notifier.sendNotification(message);
            }
        }
    }

    /* Starts the punctual monitoring controls */
    public void startOrderMonitoring() {
        orderMonitor.startOrderMonitoringThread();
    }

    public void startCheckAllVLCs() {
        serverStateMonitor.startCheckVLCThread();
    }

    /**
     * Gets the directory where RRD data is saved.
     *
     * @return the RRD folder
     */
    private File getRrdDir() throws IOException {
        String basePath = configuration.getString("vlma.data");
        File rrdDir = new File(basePath, "rrd");
        if (!rrdDir.exists())
            rrdDir.mkdir();
        return rrdDir;
    }

    /**
     * Creates the RRD file if needed and returns its path.
     *
     * @return the RRD file path
     */
    private String createRrdFileIfNecessary(Server server) throws IOException {
        File rrdR = new File(getRrdDir(), server.getName() + ".rrd");
        if (!rrdR.exists()) {
            logger.info("Creating RRD file of " + server.getName());
            try {
                RrdDef rrdDef = new RrdDef(rrdR.getPath());
                rrdDef.setStartTime(Util.getTime());
                rrdDef.addDatasource(Data.CPU_LOAD.name().toLowerCase(),    "GAUGE",   300, 0, Double.NaN);
                rrdDef.addDatasource(Data.TRAFFIC_IN.name().toLowerCase(),  "COUNTER", 300, 0, Double.NaN);
                rrdDef.addDatasource(Data.TRAFFIC_OUT.name().toLowerCase(), "COUNTER", 300, 0, Double.NaN);
                rrdDef.addDatasource(Data.VLC_CPU.name().toLowerCase(),     "GAUGE",   300, 0, Double.NaN);
                rrdDef.addDatasource(Data.VLC_MEM.name().toLowerCase(),     "GAUGE",   300, 0, Double.NaN);
                rrdDef.addArchive("AVERAGE", 0.5,   1, 4000);
                rrdDef.addArchive("AVERAGE", 0.5,   6, 4000);
                rrdDef.addArchive("AVERAGE", 0.5,  24, 4000);
                rrdDef.addArchive("AVERAGE", 0.5, 288, 4000);
                RrdDbPool rrdPool = RrdDbPool.getInstance();
                RrdDb rrdDb = rrdPool.requestRrdDb(rrdDef);
                rrdPool.release(rrdDb);
            } catch (RrdException e) {
                logger.error("Error while creating RRD file of " + rrdR.getPath(), e);
            } catch (IOException e) {
                logger.error("Error while creating RRD file of " + rrdR.getPath(), e);
            }
        }
        return rrdR.getPath();
    }

    public void updateSnmpData(Server server) {
        String rrdFile;
        try {
            rrdFile = createRrdFileIfNecessary(server);
        } catch (IOException e) {
            logger.error("Error while creating RRD file of " + server.getName(), e);
            return;
        }

        RrdDbPool rrdPool = RrdDbPool.getInstance();
        RrdDb rrdDb = null;
        Sample sample = null;
        try {
            rrdDb = rrdPool.requestRrdDb(rrdFile);
            sample = rrdDb.createSample();
        } catch (Exception e) {
            logger.error("Cannot instanciate the RRD database for server" + server.getName(), e);
        }

        for(Data data : Data.values()) {
            String value = dataRetriever.retrieve(server, data);
            if(value == null) {
                // the dataRetriever failed to retrieve data
                value = "0";
            }
            try {
                if(Data.TRAFFIC_IN.equals(data) || Data.TRAFFIC_OUT.equals(data)) {
                    sample.setValue(data.name().toLowerCase(), Long.parseLong(value.trim()));
                } else {
                    sample.setValue(data.name().toLowerCase(), Double.parseDouble(value.trim()));
                }
            } catch (Exception e) {
                if (e instanceof NumberFormatException)
                    logger.error("Cannot convert value to the expected format (Server " + server.getName() + ")", e);
                else
                    logger.error("Error while adding value to the RRD database", e);
                // Even if there was an error, update the sample so that every data in the sample has the same timestamp
                try {
                    if(Data.TRAFFIC_IN.equals(data) || Data.TRAFFIC_OUT.equals(data)) {
                        sample.setValue(data.name().toLowerCase(), 0L);
                    } else {
                        sample.setValue(data.name().toLowerCase(), 0d);
                    }
                } catch (RrdException f) {
                    // Should not happen
                    logger.error("Unexpected error!", f);
                }
            }
        }

        try {
            sample.update();
        } catch (Exception e) {
            logger.error("Cannot update sample of server " + server.getName(), e);
        }
        try {
            rrdPool.release(rrdDb);
        } catch (Exception e) {
            logger.error("Cannot release RRD database for server " + server.getName(), e);
        }
    }

    public void updateRrdGraph(Server server) {
        RrdGraph rrdGraph;
        long startTime, endTime = Util.getTime();
        startTime = endTime - 3600 * 24;

        try {
            File rrdDir = getRrdDir();

            RrdGraphDef graphDef = new RrdGraphDef();
            graphDef.datasource("cpu_load", createRrdFileIfNecessary(server), "cpu_load", "AVERAGE");
            graphDef.area("cpu_load", Color.RED, "CPU load@L");
            graphDef.line("cpu_load", Color.BLUE, "CPU load@L", 3);
            graphDef.gprint("cpu_load", "AVERAGE", "Average CPU load: @3@r");
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Load");
            rrdGraph = new RrdGraph(graphDef);
            File rrdGraphFile = new File(rrdDir, server.getName() + "-cpu_load.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());

            graphDef = new RrdGraphDef();
            graphDef.datasource("traffic_out", createRrdFileIfNecessary(server), "traffic_out", "AVERAGE");
            graphDef.line("traffic_out", Color.BLUE, "Outgoing traffic@L", 3);
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Outgoing traffic");
            rrdGraph = new RrdGraph(graphDef);
            rrdGraphFile = new File(rrdDir, server.getName() + "-traffic_out.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());

            graphDef = new RrdGraphDef();
            graphDef.datasource("vlc_cpu", createRrdFileIfNecessary(server), "vlc_cpu", "AVERAGE");
            graphDef.datasource("vlc_mem", createRrdFileIfNecessary(server), "vlc_mem", "AVERAGE");
            graphDef.line("vlc_cpu", Color.RED, "CPU usage for VLC");
            graphDef.line("vlc_mem", Color.BLUE, "Memory usage for VLC");
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Resources used for VLC");
            rrdGraph = new RrdGraph(graphDef);
            rrdGraphFile = new File(rrdDir, server.getName() + "-vlc.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());
        } catch (RrdException e) {
            logger.error("Error while creating RRD file of " + server.getName(), e);
        } catch (IOException e) {
            logger.error("Error while creating RRD file of " + server.getName(), e);
        }
    }

    synchronized public boolean checkVLC(Server server) {
        boolean formerState = server.isUp();
        try {
            Socket socket = new Socket();
            socket.connect(new InetSocketAddress(server.getIp(), configuration.getInt("vlc.telnet.port")), SOCKET_TIMEOUT);
            socket.close();
            server.setUp(true);
            logger.debug("VLC of " + server.getName() + " is reachable");
        } catch (IOException e) {
            logger.info("Unable to contact VLC server of " + server.getName() + " through telnet interface");
            server.setUp(false);
        }
        boolean newState = server.isUp();
        if (newState != formerState)
            sendNotification(server.getName() + " is " + (newState ? "up" : "down"));
        return server.isUp();
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    /**
     * Sets the data retriever.
     *
     * @param dataRetriever the dataRetriever to set
     */
    public void setDataRetriever(DataRetriever dataRetriever) {
        this.dataRetriever = dataRetriever;
    }

    /**
     * Sets the OrderGiver implementation to use.
     *
     * @param orderGiver the order giver to set
     */
    public void setOrderGiver(OrderGiver orderGiver) {
        this.orderGiver = orderGiver;
    }

    /**
     * Sets the order monitor.
     *
     * @param orderMonitor the order monitor to set
     */
    public void setOrderMonitor(OrderMonitor orderMonitor) {
        this.orderMonitor = orderMonitor;
    }

    /**
     * Sets the server state monitor.
     *
     * @param serverStateMonitor the server state monitor to set
     */
    public void setServerStateMonitor(ServerStateMonitor serverStateMonitor) {
        this.serverStateMonitor = serverStateMonitor;
    }

    /**
     * Sets the notifiers.
     *
     * @param notifiers the notifiers to set
     */
    public void setNotifiers(Set<Notifier> notifiers) {
        this.notifiers = notifiers;
    }

    /**
     * Sets the VLMa DAO.
     *
     * @param vlmaDao the vlmaDao to set
     */
    public void setVlmaDao(VLMaDao vlmaDao) {
        this.vlmaDao = vlmaDao;
    }

}
