/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.model;

import java.io.Serializable;

/**
 * Abstract adapter. Defines methods common to several adapters.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public abstract class Adapter implements Serializable {

    public static final String DTT = "DVB-T adapter";
    public static final String SAT = "DVB-S adapter";
    public static final String FILE = "File adapter";
    public static final String STREAM = "Stream adapter";

    private boolean isUp;

    private String name;

    private Server server;

    public Adapter() {
        name = "";
        setUp(true);
    }

    /**
     * Indicates whether a media is readable.
     *
     * @param media the media to test
     * @return the readability of the media
     */
    public abstract boolean canRead(Media media);

    /**
     * Check whether this adapter can read the provided media group.
     *
     * @param group the group to test
     * @return true if and only if the adapter can read the media group
     */
    public boolean canRead(MediaGroup group) {
        if (group == null || group.size() == 0)
            return false;
        else
            return canRead(group.get(0));
    }

    /**
     * Returns the adapter name.
     *
     * @return the adapter name
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the server of this adapter.
     *
     * @return VlServer the server
     */
    public Server getServer() {
        return server;
    }

    /**
     * Returns the adapter type.
     *
     * @return the adapter type
     */
    public abstract String getType();

    /**
     * Returns the state of the adapter.
     *
     * @return the state of the adapter (true if up, false otherwise)
     */
    public boolean isUp() {
        return isUp;
    }

    /**
     * Sets the adapter name.
     *
     * @param name the new adapter name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Sets the server of this adapter.
     *
     */
    public void setServer(Server server) {
        this.server = server;
    }

    /**
     * Sets the state of the adater.
     *
     * @param isUp the new adapter state
     */
    public void setUp(boolean isUp) {
        this.isUp = isUp;
    }

    @Override
    public boolean equals(Object other) {
        if (other == null || !(other instanceof Adapter)) {
            return false;
        }
        Adapter adapter = (Adapter) other;
        return adapter.getName().equals(getName()) &&
            adapter.getServer().equals(getServer());
    }

    @Override
    public int hashCode() {
        return name.hashCode() + server.getIp().hashCode();
    }

}
