/*****************************************************************************
 * logger.c
 * Print a log to a file or to screen
 *****************************************************************************
 * Copyright (C) 1998, 1999, 2000, 2001 VideoLAN
 * $Id: logger.c,v 1.17 2001/04/29 03:41:49 nitrox Exp $
 *
 * Authors: Damien Lucas <nitrox@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <pthread.h>                                   /* pthread_mutex_lock */
#include <stdarg.h>                                                /* va_arg */
#include <time.h>                                                    /* time */
#include <stdio.h>                                           /* fflush , ... */

#include "types.h"
#include "logger.h"


/* Local Functions prototypes */
static FILE * fdLogFile;
static unsigned char method;

static pthread_mutex_t ptmLogFile;
static pthread_mutex_t ptmLogScreen;




                   
/*****************************************************************************
 * VS_log_File : Write a log inside fdLogfile
 *****************************************************************************/
static void VS_log_File(LOG_LEVELS level,\
                        LOG_MODULES module,\
                        char* format,\
                        va_list ap)
{
  char MsgBuff[LOGBUFF_MAXSIZE+20];
  time_t MsgTime;
  struct tm * ptmTime;

#ifndef DEBUG
  if(level!=LOGDEBUG)
  {
#endif
  /* Record the time */
  time(&MsgTime);
  ptmTime=localtime(&MsgTime);
  sprintf(MsgBuff, "%02i/%02i %02i:%02i:%02i ", ptmTime->tm_mday,
          ptmTime->tm_mon+1, ptmTime->tm_hour, 
          ptmTime->tm_min, ptmTime->tm_sec);



  /* Determinate in which module it happened */
  switch (module)
  {
    case SNMP:
      sprintf(MsgBuff+strlen(MsgBuff), "[SNMP  : ");
      break;
    
    case SERVER:
      sprintf(MsgBuff+strlen(MsgBuff), "[SERVER: ");
      break;
      
    case DB:
      sprintf(MsgBuff+strlen(MsgBuff), "[DB    : ");
      break;

    case CFG:
      sprintf(MsgBuff+strlen(MsgBuff), "[CFG   : ");
      break;
      
    case IF:
      sprintf(MsgBuff+strlen(MsgBuff), "[IF    : ");
      break;
    
    case TELNET:
      sprintf(MsgBuff+strlen(MsgBuff), "[TELNET: ");
      break;

    case LOGGER:
      sprintf(MsgBuff+strlen(MsgBuff), "[LOGGER: ");
      break;
      
  }

  /* Determinate the error level */
  switch (level)
  {
    case LOGDEBUG:
      sprintf(MsgBuff+strlen(MsgBuff), "debug   ] ");
      break;
    case LOGINFO:
      sprintf(MsgBuff+strlen(MsgBuff), "info    ] ");
      break;
    case LOGWARNING:
      sprintf(MsgBuff+strlen(MsgBuff), "Warning ] ");
      break;
    case LOGFATAL:
        sprintf(MsgBuff+strlen(MsgBuff), "FATAL   ] ");
        break;
    case LOGERROR:
        sprintf(MsgBuff+strlen(MsgBuff), "ERROR   ] ");
        break;
  }
 
  /* XXX No error checking ! What length is the message ? */
  vsnprintf(MsgBuff+strlen(MsgBuff), LOGBUFF_MAXSIZE, format, ap);
  sprintf(MsgBuff+strlen(MsgBuff), "\n");

  pthread_mutex_lock(&ptmLogFile);
  fprintf(fdLogFile, MsgBuff);
  fflush(fdLogFile);
  pthread_mutex_unlock(&ptmLogFile);

#ifndef DEBUG
  }
#endif
}



/*****************************************************************************
 * Write a log on stderr
 *****************************************************************************/
static void VS_log_Screen(LOG_LEVELS level,\
                          LOG_MODULES module,\
                          char* format,\
                          va_list ap)\
{
  char MsgBuff[LOGBUFF_MAXSIZE+20];
  time_t MsgTime;
  struct tm * ptmTime;

#ifndef DEBUG
  if(level!=LOGDEBUG)
  {
#endif
  /* Record the time */
  time(&MsgTime);
  ptmTime=localtime(&MsgTime);
  sprintf(MsgBuff, "%02i/%02i %02i:%02i:%02i ", ptmTime->tm_mday,
          ptmTime->tm_mon+1, ptmTime->tm_hour, 
          ptmTime->tm_min, ptmTime->tm_sec);



  /* Determinate in which module it happened */
  switch (module)
  {
    case SNMP:
      sprintf(MsgBuff+strlen(MsgBuff), "[SNMP  : ");
      break;
    
    case SERVER:
      sprintf(MsgBuff+strlen(MsgBuff), "[SERVER: ");
      break;
      
    case DB:
      sprintf(MsgBuff+strlen(MsgBuff), "[DB    : ");
      break;

    case CFG:
      sprintf(MsgBuff+strlen(MsgBuff), "[CFG   : ");
      break;
      
    case IF:
      sprintf(MsgBuff+strlen(MsgBuff), "[IF    : ");
      break;
    
    case TELNET:
      sprintf(MsgBuff+strlen(MsgBuff), "[TELNET: ");
      break;
    case LOGGER:
      sprintf(MsgBuff+strlen(MsgBuff), "[LOGGER: ");
      break;
  }

  /* Determinate the error level */
  switch (level)
  {
    case LOGDEBUG:
      sprintf(MsgBuff+strlen(MsgBuff), "debug   ] ");
      break;
    case LOGINFO:
      sprintf(MsgBuff+strlen(MsgBuff), "info    ] ");
      break;
    case LOGWARNING:
      sprintf(MsgBuff+strlen(MsgBuff), "Warning ] ");
      break;
    case LOGFATAL:
      {
#ifdef LINUX_COLOR
        int i=strlen(MsgBuff);
        MsgBuff[i]=27;
        MsgBuff[i+1]='\0';
        sprintf(MsgBuff+strlen(MsgBuff), "[40;31;1m");
#endif        
        sprintf(MsgBuff+strlen(MsgBuff), "FATAL   ] ");
        break;
      }
    case LOGERROR:
      {
#ifdef LINUX_COLOR
        int i=strlen(MsgBuff);
        MsgBuff[i]=27;
        MsgBuff[i+1]='\0';
        sprintf(MsgBuff+strlen(MsgBuff), "[40;31;1m");
#endif 
        sprintf(MsgBuff+strlen(MsgBuff), "ERROR   ] ");
        break;
      }
  }
 
  /* XXX No error checking ! What length is the message ? */
  vsnprintf(MsgBuff+strlen(MsgBuff), LOGBUFF_MAXSIZE, format, ap);
  {
    int i=strlen(MsgBuff);
    MsgBuff[i]=27;
    MsgBuff[i+1]=0;
    sprintf(MsgBuff+strlen(MsgBuff), "[0m\n");
  }
        
  pthread_mutex_lock(&ptmLogScreen);
  fprintf(stderr, "%s", MsgBuff);
  pthread_mutex_unlock(&ptmLogScreen);
#ifndef DEBUG
  }
#endif
}




/*****************************************************************************
 * Main log function 
 *****************************************************************************/
void VS_log(LOG_LEVELS level, LOG_MODULES module, char* format, ...)
{
  va_list ap;
  
  va_start(ap,format);
  if (method==LOG_FILE_ONLY)
  {
    VS_log_File(level, module,format,ap);
  }
  else if (method==LOG_SCREEN_ONLY)
  {
    VS_log_Screen(level, module,format,ap);
  }
  else if (method==LOG_BOTH)
  {
    VS_log_File(level, module,format,ap);
    VS_log_Screen(level, module,format,ap);
  }
  va_end(ap);
  
}



/*****************************************************************************
 * Open the log file 
 *****************************************************************************/
int OpenLogFile (char * sLogName)
{
  int iRc;
  iRc=0;
  
  if( !(fdLogFile = fopen (sLogName, "w+")))
  {
    iRc=1;
    VS_log(LOGFATAL, LOGGER, "Unable to open %s log file ", sLogName);
  }
  else
  {
    /* Init the mutex */
    iRc = pthread_mutex_init (&ptmLogFile, NULL);
  }
  return iRc;
}




/*****************************************************************************
 * Close the LogFile
 *****************************************************************************/
int CloseLogFile ()
{
  int iRc;
  iRc=0;

  if(fdLogFile)
  {
    iRc=fclose(fdLogFile);
    if(iRc)
    {
      VS_log(LOGERROR, LOGGER, "Unable to close file");
    }
    iRc=pthread_mutex_destroy(&ptmLogFile);
  }
  else
  {
    iRc=1;
    VS_log(LOGERROR, LOGGER, "Unable to close log file: file not open");
  }
  return iRc;
        
}




/*****************************************************************************
 * LOG_init() : Initialize the log process
 *****************************************************************************
 * TODO Gestion d'erreurs
 *****************************************************************************/
int LOG_init(char * sLogName, unsigned char ucM)
{
  switch (ucM)
  {
    case LOG_SCREEN_ONLY:
      pthread_mutex_init(&ptmLogScreen, NULL);
      method=LOG_SCREEN_ONLY;
      VS_log(LOGINFO,LOGGER,"Logs inited on screen");
      break;

    case LOG_FILE_ONLY:
      OpenLogFile(sLogName);
      method=LOG_FILE_ONLY;
      VS_log(LOGINFO,LOGGER,"Logs inited with file: %s",sLogName);
      break;

    case LOG_BOTH:
      pthread_mutex_init(&ptmLogScreen, NULL);
      OpenLogFile(sLogName);
      method=LOG_BOTH;
      VS_log(LOGINFO,LOGGER,"Logs inited on screen and with file: %s",\
                                                                     sLogName);
      break;
  }
  return 0;
}


/*****************************************************************************
 * LOG_change() : Change the log method
 *****************************************************************************
 * Note that if no method change is ask, LogFile could have change ...
 * TODO Error Checking
 *****************************************************************************/
int LOG_change(char * sLogName, unsigned char ucM)
{
    switch (ucM)
    {
      case LOG_NONE:
        switch (method)
        {
          case LOG_FILE_ONLY:
            CloseLogFile();
            break;
          
          case LOG_SCREEN_ONLY:
            pthread_mutex_destroy(&ptmLogScreen);
            break;
            
          case LOG_BOTH:
            CloseLogFile();
            pthread_mutex_destroy(&ptmLogScreen);
            break;
        }
        method=LOG_NONE;
        break;
        
      case LOG_SCREEN_ONLY:
        switch (method)
        {
          case LOG_BOTH:
            CloseLogFile();
            break;
            
          case LOG_FILE_ONLY:
            CloseLogFile();

          case LOG_NONE:
            pthread_mutex_init(&ptmLogScreen, NULL);
            break;
        }
        method=LOG_SCREEN_ONLY;
        VS_log(LOGINFO,LOGGER,"Logs inited on screen");
        break;
      
      
      case LOG_FILE_ONLY:
        switch (method)
        {
          case LOG_BOTH:
            pthread_mutex_destroy(&ptmLogScreen);
            CloseLogFile();
            OpenLogFile(sLogName);
            break;

          case LOG_SCREEN_ONLY:
            pthread_mutex_destroy(&ptmLogScreen);

          case LOG_NONE:
            OpenLogFile(sLogName);
            break;

          case LOG_FILE_ONLY:
            CloseLogFile();
            OpenLogFile(sLogName);
            break;
        }
        VS_log(LOGINFO,LOGGER,"Logs inited with file: %s",sLogName);
        method=LOG_FILE_ONLY;
        break;

      case LOG_BOTH:
        switch (method)
        {
          case LOG_FILE_ONLY:
            CloseLogFile();

          case LOG_NONE:
            OpenLogFile(sLogName);
            pthread_mutex_init(&ptmLogScreen, NULL);
            break;

          case LOG_SCREEN_ONLY:
            OpenLogFile(sLogName);
            break;

          case LOG_BOTH:
            CloseLogFile();
            OpenLogFile(sLogName);
            break;
        }
        method=LOG_BOTH;
        VS_log(LOGINFO,LOGGER,"Logs inited on screen and with file: %s",\
                                                                     sLogName);
        break;
    }
  return 0;
}



/****************************************************************************
 * LOG_close() : Close the log process
 ****************************************************************************
 * TODO Write it !
 ****************************************************************************/






/****************************************************************************
 * VS_log_errstr() : Returns a string corresponding to an ERR_CODE
 ****************************************************************************/
char * VS_log_errstr(ERR_CODE err)
{
  static char * str[]=
  {
    "no error",
    "object not found",
    "trying to do a walk on a switch is currently walked by an other thread",
    "unable to bind a socket",
    "unable to connect to remote host",
    "unable to resolve the dns",
    "initiatization of the switch failed",
    "unable to open, read or write a file",
    "unable to allocate memory",
    "parse error",
    "pipe creation error",
    "unable to write to a pipe",
    "unable to spawn a thread",
    "unable to receive data from remote host",
    "unable to send data to remote host",
    "unable to execute a select() operation",
    "snmp error",
    "unable to open a socket",
    "time error",
    "timeout"
  };
  return str[err];
}
