/*****************************************************************************
 * aout_beos.cpp: beos interface
 *****************************************************************************
 * Copyright (C) 1999, 2000 VideoLAN
 *
 * Authors:
 * Samuel Hocevar <sam@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include "defs.h"

#include <stdio.h>
#include <stdlib.h>                                      /* malloc(), free() */
#include <sys/types.h>                        /* on BSD, uio.h needs types.h */
#include <sys/uio.h>                                            /* "input.h" */
#include <kernel/OS.h>
#include <View.h>
#include <Application.h>
#include <Message.h>
#include <Locker.h>
#include <media/MediaDefs.h>
#include <game/PushGameSound.h>
#include <malloc.h>
#include <string.h>

extern "C"
{
#include "config.h"
#include "common.h"
#include "threads.h"
#include "mtime.h"
#include "plugins.h"

#include "audio_output.h"

#include "intf_msg.h"

#include "main.h"
}

/*****************************************************************************
 * aout_sys_t: esd audio output method descriptor
 *****************************************************************************
 * This structure is part of the audio output thread descriptor.
 * It describes some esd specific variables.
 *****************************************************************************/
typedef struct aout_sys_s
{
    BPushGameSound * p_sound;
    gs_audio_format * p_format;
    void * p_buffer;
    size_t i_buffer_size;
    size_t i_buffer_pos;

} aout_sys_t;

extern "C"
{

/*****************************************************************************
 * aout_SysOpen: opens an esd socket
 *****************************************************************************/
int aout_SysOpen( aout_thread_t *p_aout )
{
    /* Allocate structure */
    p_aout->p_sys = (aout_sys_t*) malloc( sizeof( aout_sys_t ) );
    if( p_aout->p_sys == NULL )
    {
        intf_ErrMsg("error: %s\n", strerror(ENOMEM) );
        return( 1 );
    }

    /* Allocate gs_audio_format */
    p_aout->p_sys->p_format = (gs_audio_format *) malloc( sizeof( gs_audio_format ) );
    if( p_aout->p_sys->p_format == NULL )
    {
        free( p_aout->p_sys );
        intf_ErrMsg("error: cannot allocate memory for gs_audio_format\n" );
        return( 1 );
    }

    /* Initialize some variables */
    p_aout->i_format = AOUT_DEFAULT_FORMAT;
    p_aout->i_channels = 1 + main_GetIntVariable( AOUT_STEREO_VAR,
                                                  AOUT_STEREO_DEFAULT );
    p_aout->l_rate = main_GetIntVariable( AOUT_RATE_VAR, AOUT_RATE_DEFAULT );

    p_aout->p_sys->p_format->frame_rate = 48000;
    p_aout->p_sys->p_format->channel_count = p_aout->i_channels;
    p_aout->p_sys->p_format->format = /*B_GS_U8*/ 0x2;
    p_aout->p_sys->p_format->byte_order = B_MEDIA_LITTLE_ENDIAN;
    p_aout->p_sys->p_format->buffer_size = 131072;

    /* Allocate BPushGameSound */
    p_aout->p_sys->p_sound = new BPushGameSound( 1024,
                                                 p_aout->p_sys->p_format,
                                                 128, NULL );
    if( p_aout->p_sys->p_sound == NULL )
    {
        free( p_aout->p_sys->p_format );
        free( p_aout->p_sys );
        intf_ErrMsg("error: cannot allocate memory for BPushGameSound\n" );
        return( 1 );
    }

    if( p_aout->p_sys->p_sound->InitCheck() == B_OK )
    {
        fprintf( stderr, "InitCheck OK sa race\n" );
    }

    p_aout->p_sys->p_sound->LockForCyclic( &p_aout->p_sys->p_buffer,
                            (size_t *)&p_aout->p_sys->i_buffer_size );

    p_aout->p_sys->p_sound->StartPlaying( );

	if( p_aout->p_sys->p_sound->IsPlaying() == true )
	{
        fprintf( stderr, "IsPlaying() == true\n" );
	}
 	
    return( 0 );
}
/*****************************************************************************
 * aout_SysReset: resets the dsp
 *****************************************************************************/
int aout_SysReset( aout_thread_t *p_aout )
{
    return( 0 );
}

/*****************************************************************************
 * aout_SysSetFormat: sets the dsp output format
 *****************************************************************************/
int aout_SysSetFormat( aout_thread_t *p_aout )
{
    return( 0 );
}

/*****************************************************************************
 * aout_SysSetChannels: sets the dsp's stereo or mono mode
 *****************************************************************************/
int aout_SysSetChannels( aout_thread_t *p_aout )
{
    return( 0 );
}

/*****************************************************************************
 * aout_SysSetRate: sets the dsp's audio output rate
 *****************************************************************************/
int aout_SysSetRate( aout_thread_t *p_aout )
{
    return( 0 );
}

/*****************************************************************************
 * aout_SysGetBufInfo: buffer status query
 *****************************************************************************/
long aout_SysGetBufInfo( aout_thread_t *p_aout, long l_buffer_limit )
{

    unsigned long int i_hard_pos = p_aout->p_sys->p_sound->CurrentPosition();

    if( i_hard_pos < p_aout->p_sys->i_buffer_pos )
    {
        i_hard_pos += p_aout->p_sys->i_buffer_size;
    }

    fprintf( stderr, "hard(sb) - buffer(us) = %li\n",
             i_hard_pos - p_aout->p_sys->i_buffer_pos );

	return( l_buffer_limit );

	fprintf( stderr, "waiting %li\n", i_hard_pos - p_aout->p_sys->i_buffer_pos );
    return( 4 * (i_hard_pos - p_aout->p_sys->i_buffer_pos) );
}

/*****************************************************************************
 * aout_SysPlaySamples: plays a sound samples buffer
 *****************************************************************************
 * This function writes a buffer of i_length bytes in the dsp
 *****************************************************************************/
void aout_SysPlaySamples( aout_thread_t *p_aout, byte_t *buffer, int i_size )
{
    unsigned long int i_newbuf_pos;

//    fprintf( stderr, "Locking... " );

    if( (i_newbuf_pos = p_aout->p_sys->i_buffer_pos + i_size)
              > p_aout->p_sys->i_buffer_size )
    {
        memcpy( (void *)((int)p_aout->p_sys->p_buffer
                        + p_aout->p_sys->i_buffer_pos),
                buffer,
                p_aout->p_sys->i_buffer_size - p_aout->p_sys->i_buffer_pos );

        memcpy( (void *)((int)p_aout->p_sys->p_buffer),
                buffer,
                i_size - ( p_aout->p_sys->i_buffer_size
                             - p_aout->p_sys->i_buffer_pos ) );

        p_aout->p_sys->i_buffer_pos = i_newbuf_pos - p_aout->p_sys->i_buffer_size;
    }
    else
    {
        memcpy( (void *)((int)p_aout->p_sys->p_buffer + p_aout->p_sys->i_buffer_pos),
                buffer, i_size );
        p_aout->p_sys->i_buffer_pos = i_newbuf_pos;
    }
}

/*****************************************************************************
 * aout_SysClose: closes the dsp audio device
 *****************************************************************************/
void aout_SysClose( aout_thread_t *p_aout )
{
    p_aout->p_sys->p_sound->UnlockCyclic();
    p_aout->p_sys->p_sound->StopPlaying( );
    delete p_aout->p_sys->p_sound;
    free( p_aout->p_sys->p_format );
    free( p_aout->p_sys );
}

} /* extern "C" */

