/*******************************************************************************
* Copyright 2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

#ifndef __HELPERS_HPP__
#define __HELPERS_HPP__

//
// Add all esimd helper functions
//
#include "EsimdHelpers.hpp"


namespace { // anonymous

//
// Add all other hlper functions
//

// helper arithmetic functions
template<typename T, typename std::enable_if<std::is_integral<T>::value>::type* = nullptr>
T ceil_div(const T num, const T denom)
{
#ifndef __SYCL_DEVICE_ONLY__
    assert(denom > 0);
#endif
    return (num + denom - 1) / denom;
}

template<typename T, typename std::enable_if<std::is_integral<T>::value>::type* = nullptr>
T floor_div(const T num, const T denom)
{
#ifndef __SYCL_DEVICE_ONLY__
    assert(denom > 0);
#endif
    return num / denom;
}

// stays the same or rounds up to the next multiple of pow_2
// where pow_2 should be a power of two
template<typename T, typename std::enable_if<std::is_integral<T>::value>::type* = nullptr>
T round_up_next_multiple(const T count, const T pow_2_mult)
{
    assert(pow_2_mult != 0 && ((pow_2_mult & pow_2_mult - 1) == 0)); // check pow_2_mult is a power of two
    return (count + pow_2_mult - 1) & ~(pow_2_mult - 1);
}

// stays the same or rounds down to the previous multiple of pow_2
// where pow_2 should be a power of two
template<typename T, typename std::enable_if<std::is_integral<T>::value>::type* = nullptr>
T round_down_next_multiple(const T count, const T pow_2_mult)
{
    assert(pow_2_mult != 0 && ((pow_2_mult & pow_2_mult - 1) == 0)); // check pow_2_mult is a power of two
    return (count) & ~(pow_2_mult - 1);
}

//
// Return smallest power of 2 that is equal to or greater than input x_in
//
// using 64 bit integers
//
static inline
size_t next_power_of_two(size_t x)
{

    if (x == 0) {
        return 0;
    }
    // algorithm:
    //  a) move x to x-1 to handle the input is exact power of 2 already case
    //  b) fill x-1 with all 1's below leading bit using log2(64) |= steps
    //  c) subtract off all but leading bit of x-1 so we have a power of two
    //  d) shift left to get original power of two, or next higher power of two
    //
    x--;
    x |= (x >> 1);
    x |= (x >> 2);
    x |= (x >> 4);
    x |= (x >> 8);
    x |= (x >> 16);
    x |= (x >> 32);
    return (x - (x >> 1)) << 1;
}


} // anonymous namespace


#endif // #ifndef __HELPERS_HPP__
