"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToExcel = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
var _excelBuilder = require("./excelBuilder");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  let sorting = report._source.sorting;

  // We expect a list of [field, order] pairs for sorting. In some migration paths, though it's not
  // clear why, this list can get unnested in the case of one sort, [["field", "asc"]] becomes
  // ["field", "asc"]. The true root cause remains a mystery, so we work around it.
  // See: https://github.com/opensearch-project/dashboards-reporting/issues/371
  if (sorting.length > 0 && typeof sorting[0] === "string") {
    sorting = [sorting];
  }
  if (sorting.length > 0) {
    const sorts = sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sorts);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields[dateField];
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if (dateValue.length !== 0 && dateValue instanceof Array) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if (dateValue.length !== 0 && dateValue instanceof Array) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

// Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(hits, result = {}, prefix = '') {
  Object.entries(hits).forEach(([key, value]) => {
    if (value !== null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, `${prefix}${key}.`);
    } else {
      result[`${prefix.replace(/^_source\./, '')}${key}`] = value;
    }
  });
  return result;
}
function flattenObject(obj = {}, parentKey = '', result = {}) {
  for (const [key, value] of Object.entries(obj)) {
    const newKey = parentKey ? `${parentKey}.${key}` : key;
    if (typeof value == 'object' && value !== null && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenObject(value, newKey, result);
    } else if (Array.isArray(value)) {
      result[newKey] = JSON.stringify(value);
    } else {
      result[newKey] = value;
    }
  }
  return result;
}
function flattenArray(array = []) {
  return array.map(item => flattenObject(item));
}
const convertToExcel = async dataset => {
  const flatDataset = flattenArray(dataset[0]);
  const excelBuilder = new _excelBuilder.ExcelBuilder();
  const base64 = await excelBuilder.addHeaders(flatDataset).addRows(flatDataset).updateColumnWidths().applyHeaderStyles().applyAutoFilter().applyFreeze().getWorkbookAsBase64();
  return 'data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,' + base64;
};

//Return only the selected fields
exports.convertToExcel = convertToExcel;
function traverse(data, keys, result = {}) {
  // Flatten the data if necessary (ensure all nested fields are at the top level)
  data = flattenHits(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);
    if (value !== undefined) {
      result[key] = value;
    } else {
      const flattenedValues = {};
      Object.keys(data).forEach(dataKey => {
        if (dataKey.startsWith(key + '.')) {
          result[dataKey] = data[dataKey];
        }
        const arrayValue = data[dataKey];
        if (Array.isArray(arrayValue)) {
          arrayValue.forEach(item => {
            if (typeof item === 'object' && item !== null) {
              Object.keys(item).forEach(subKey => {
                const newKey = `${dataKey}.${subKey}`;
                if (!flattenedValues[newKey]) {
                  flattenedValues[newKey] = [];
                }
                flattenedValues[newKey].push(item[subKey]);
              });
            }
          });
        }
      });
      Object.keys(flattenedValues).forEach(newKey => {
        result[newKey] = flattenedValues[newKey];
      });
    }
  });
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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