#!/bin/sh

# Copyright (c) 2023 Intel Corporation                                          
# SPDX-License-Identifier: MIT                                                  
# shellcheck shell=sh
# shellcheck source=/dev/null
# shellcheck disable=SC1091

# FPGA environment setup for oneAPI

# ############################################################################

# Detect host environment (fail fast)
ARCH=$(uname -m)
arch_type=
case $ARCH in
x86_64) # Linux x86-64
  arch_type=linux64
  ;;
armv7l) # Linux SoC
  arch_type=arm32
  ;;
*)
  echo "Platform '$ARCH' is not supported by the Intel(R) FPGA SDK for OpenCL(TM)"
  exit 1
  ;;
esac

# FPGA_VARS_DIR is an exported variable from the compiler/<version>/env/vars.sh script
# It is the directory where this script exists
SCRIPTDIR="${FPGA_VARS_DIR}"

# Prepend an item to a list of `:`separated items,
# only if the item isn't in the list yet.
# prepend <list_name> <item>
prepend() {
  list=$1
  item=$2

  # Get content of current path through indirect reference
  eval list_items=\$"$list"

  # Add item only if not already in list
  case ":$list_items:" in
  "::") # Path variable empty or unset
    eval "export $list=$item"
    ;;
  *":$item:"*) # item already present in list
    ;;
  *) # Prepend new item to list
    eval "export $list=$item:$list_items"
    ;;
  esac
}

# Set up the Installable Client Driver (ICD) file for the sim/HW flow
prepend OCL_ICD_FILENAMES "libalteracl.so"
# Set up the ICD file for the emulator
prepend OCL_ICD_FILENAMES "libintelocl_emu.so"

# The following line is meant for the runtime:
# Applications need to know where the FPGA Client Drivers (FCD)
# for the FPGA boards are
export ACL_BOARD_VENDOR_PATH=/opt/Intel/OpenCLFPGA/oneAPI/Boards

# Set this so VTune can find aocl (it uses this variable to search)
export INTELFPGAOCLSDKROOT="${SCRIPTDIR}"

# Update PATH variables so the compiler can find information:
# PATH           : executables  (aoc, aocl)
# LD_LIBRARY_PATH: libraries    (libOpenCL.so, ...)
# CPATH          : header files (acl_types.h, ...)
# and headers on the respective paths so icpx can find them
prepend PATH "${INTELFPGAOCLSDKROOT}/bin"
prepend LD_LIBRARY_PATH "${INTELFPGAOCLSDKROOT}/host/$arch_type/lib:${LD_LIBRARY_PATH}"
prepend CPATH "$INTELFPGAOCLSDKROOT/include"
