/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.routing.allocation.decider;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.Version;
import org.opensearch.cluster.ClusterInfo;
import org.opensearch.cluster.routing.RoutingNode;
import org.opensearch.cluster.routing.RoutingPool;
import org.opensearch.cluster.routing.ShardRouting;
import org.opensearch.cluster.routing.allocation.DiskThresholdSettings;
import org.opensearch.cluster.routing.allocation.RoutingAllocation;
import org.opensearch.cluster.routing.allocation.decider.AllocationDecider;
import org.opensearch.cluster.routing.allocation.decider.Decision;
import org.opensearch.cluster.routing.allocation.decider.DiskThresholdDecider;
import org.opensearch.common.settings.ClusterSettings;
import org.opensearch.common.settings.Settings;
import org.opensearch.core.common.unit.ByteSizeValue;
import org.opensearch.index.store.remote.filecache.AggregateFileCacheStats;
import org.opensearch.index.store.remote.filecache.FileCacheSettings;

public class WarmDiskThresholdDecider
extends AllocationDecider {
    private static final Logger logger = LogManager.getLogger(WarmDiskThresholdDecider.class);
    public static final String NAME = "warm_disk_threshold";
    private final FileCacheSettings fileCacheSettings;
    private final DiskThresholdSettings diskThresholdSettings;
    private final boolean enableForSingleDataNode;

    public WarmDiskThresholdDecider(Settings settings, ClusterSettings clusterSettings) {
        this.fileCacheSettings = new FileCacheSettings(settings, clusterSettings);
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
        assert (Version.CURRENT.major < 9) : "remove enable_for_single_data_node in 9";
        this.enableForSingleDataNode = DiskThresholdSettings.ENABLE_FOR_SINGLE_DATA_NODE.get(settings);
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        RoutingPool nodeRoutingPool = RoutingPool.getNodePool(node);
        RoutingPool shardRoutingPool = RoutingPool.getShardPool(shardRouting, allocation);
        if (nodeRoutingPool != RoutingPool.REMOTE_CAPABLE || shardRoutingPool != RoutingPool.REMOTE_CAPABLE) {
            return Decision.ALWAYS;
        }
        Decision decision = this.earlyTerminate(node, allocation);
        if (decision != null) {
            return decision;
        }
        long shardSize = DiskThresholdDecider.getExpectedShardSize(shardRouting, 0L, allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable());
        long totalAddressableSpace = this.calculateTotalAddressableSpace(node, allocation);
        long currentNodeRemoteShardSize = this.calculateCurrentNodeRemoteShardSize(node, allocation, false);
        long freeSpace = Math.max(totalAddressableSpace - currentNodeRemoteShardSize, 0L);
        long freeSpaceAfterAllocation = Math.max(freeSpace - shardSize, 0L);
        long freeSpaceLowThreshold = this.calculateFreeSpaceLowThreshold(this.diskThresholdSettings, totalAddressableSpace);
        ByteSizeValue freeSpaceLowThresholdInByteSize = new ByteSizeValue(freeSpaceLowThreshold);
        ByteSizeValue freeSpaceInByteSize = new ByteSizeValue(freeSpace);
        ByteSizeValue freeSpaceAfterAllocationInByteSize = new ByteSizeValue(freeSpaceAfterAllocation);
        ByteSizeValue shardSizeInByteSize = new ByteSizeValue(shardSize);
        if (freeSpaceAfterAllocation < freeSpaceLowThreshold) {
            logger.warn("after allocating [{}] node [{}] would have less than the required threshold of {} free (currently {} free, estimated shard size is {}), preventing allocation", (Object)shardRouting, (Object)node.nodeId(), (Object)freeSpaceLowThresholdInByteSize, (Object)freeSpaceInByteSize, (Object)shardSizeInByteSize);
            return allocation.decision(Decision.NO, NAME, "allocating the shard to this node will bring the node above the low watermark cluster setting [%s] and cause it to have less than the minimum required [%s] of addressable remote free space (free: [%s], estimated remote shard size: [%s])", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK_SETTING.getKey(), freeSpaceLowThresholdInByteSize, freeSpaceInByteSize, shardSizeInByteSize);
        }
        return allocation.decision(Decision.YES, NAME, "enough available remote addressable space for shard on node, free: [%s], shard size: [%s], free after allocating shard: [%s]", freeSpaceInByteSize, shardSizeInByteSize, freeSpaceAfterAllocationInByteSize);
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!shardRouting.currentNodeId().equals(node.nodeId())) {
            throw new IllegalArgumentException("Shard [" + String.valueOf(shardRouting) + "] is not allocated on node: [" + node.nodeId() + "]");
        }
        RoutingPool nodeRoutingPool = RoutingPool.getNodePool(node);
        RoutingPool shardRoutingPool = RoutingPool.getShardPool(shardRouting, allocation);
        if (nodeRoutingPool != RoutingPool.REMOTE_CAPABLE || shardRoutingPool != RoutingPool.REMOTE_CAPABLE) {
            return Decision.ALWAYS;
        }
        Decision decision = this.earlyTerminate(node, allocation);
        if (decision != null) {
            return decision;
        }
        long totalAddressableSpace = this.calculateTotalAddressableSpace(node, allocation);
        long currentNodeRemoteShardSize = this.calculateCurrentNodeRemoteShardSize(node, allocation, true);
        long freeSpace = Math.max(totalAddressableSpace - currentNodeRemoteShardSize, 0L);
        long freeSpaceHighThreshold = this.calculateFreeSpaceHighThreshold(this.diskThresholdSettings, totalAddressableSpace);
        ByteSizeValue freeSpaceHighThresholdInByteSize = new ByteSizeValue(freeSpaceHighThreshold);
        ByteSizeValue freeSpaceInByteSize = new ByteSizeValue(freeSpace);
        if (freeSpace < freeSpaceHighThreshold) {
            logger.warn("less than the required {} of free remote addressable space threshold left ({} free) on node [{}], shard cannot remain", (Object)freeSpaceHighThresholdInByteSize, (Object)freeSpaceInByteSize, (Object)node.nodeId());
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because it is above the high watermark cluster setting [%s] and there is less than the required [%s] free remote addressable space on node, actual free: [%s]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), freeSpaceHighThresholdInByteSize, freeSpaceInByteSize);
        }
        return allocation.decision(Decision.YES, NAME, "there is enough remote addressable space on this node for the shard to remain, free: [%s]", freeSpaceInByteSize);
    }

    private long calculateFreeSpaceLowThreshold(DiskThresholdSettings diskThresholdSettings, long totalAddressableSpace) {
        double percentageThreshold = diskThresholdSettings.getFreeDiskThresholdLow();
        if (percentageThreshold > 0.0) {
            return (long)((double)totalAddressableSpace * percentageThreshold / 100.0);
        }
        double dataToFileCacheSizeRatio = this.fileCacheSettings.getRemoteDataRatio();
        ByteSizeValue bytesThreshold = diskThresholdSettings.getFreeBytesThresholdLow();
        if (bytesThreshold != null && bytesThreshold.getBytes() > 0L) {
            return bytesThreshold.getBytes() * (long)dataToFileCacheSizeRatio;
        }
        return 0L;
    }

    private long calculateFreeSpaceHighThreshold(DiskThresholdSettings diskThresholdSettings, long totalAddressableSpace) {
        double percentageThreshold = diskThresholdSettings.getFreeDiskThresholdHigh();
        if (percentageThreshold > 0.0) {
            return (long)((double)totalAddressableSpace * percentageThreshold / 100.0);
        }
        double dataToFileCacheSizeRatio = this.fileCacheSettings.getRemoteDataRatio();
        ByteSizeValue bytesThreshold = diskThresholdSettings.getFreeBytesThresholdHigh();
        if (bytesThreshold != null && bytesThreshold.getBytes() > 0L) {
            return bytesThreshold.getBytes() * (long)dataToFileCacheSizeRatio;
        }
        return 0L;
    }

    private long calculateCurrentNodeRemoteShardSize(RoutingNode node, RoutingAllocation allocation, boolean subtractLeavingShards) {
        List remoteShardsOnNode = StreamSupport.stream(node.spliterator(), false).filter(shard -> shard.primary() && RoutingPool.REMOTE_CAPABLE.equals((Object)RoutingPool.getShardPool(shard, allocation)) && (!subtractLeavingShards || !shard.relocating())).collect(Collectors.toList());
        long remoteShardSize = 0L;
        for (ShardRouting shard2 : remoteShardsOnNode) {
            remoteShardSize += DiskThresholdDecider.getExpectedShardSize(shard2, 0L, allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable());
        }
        return remoteShardSize;
    }

    private long calculateTotalAddressableSpace(RoutingNode node, RoutingAllocation allocation) {
        ClusterInfo clusterInfo = allocation.clusterInfo();
        double dataToFileCacheSizeRatio = this.fileCacheSettings.getRemoteDataRatio();
        AggregateFileCacheStats fileCacheStats = clusterInfo.getNodeFileCacheStats().getOrDefault(node.nodeId(), null);
        long nodeCacheSize = fileCacheStats != null ? fileCacheStats.getTotal().getBytes() : 0L;
        return (long)dataToFileCacheSizeRatio * nodeCacheSize;
    }

    private Decision earlyTerminate(RoutingNode node, RoutingAllocation allocation) {
        if (!this.diskThresholdSettings.isWarmThresholdEnabled()) {
            return allocation.decision(Decision.YES, NAME, "the warm disk threshold decider is disabled", new Object[0]);
        }
        if (!this.enableForSingleDataNode && allocation.nodes().getDataNodes().size() <= 1) {
            if (logger.isTraceEnabled()) {
                logger.trace("only a single data node is present, allowing allocation");
            }
            return allocation.decision(Decision.YES, NAME, "there is only a single data node present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (logger.isTraceEnabled()) {
                logger.trace("cluster info unavailable for file cache threshold decider, allowing allocation.");
            }
            return allocation.decision(Decision.YES, NAME, "the cluster info is unavailable", new Object[0]);
        }
        AggregateFileCacheStats fileCacheStats = clusterInfo.getNodeFileCacheStats().getOrDefault(node.nodeId(), null);
        if (fileCacheStats == null) {
            if (logger.isTraceEnabled()) {
                logger.trace("unable to get file cache stats for node [{}], allowing allocation", (Object)node.nodeId());
            }
            return allocation.decision(Decision.YES, NAME, "File Cache Stat is unavailable", new Object[0]);
        }
        double remoteDataRatio = this.fileCacheSettings.getRemoteDataRatio();
        if (remoteDataRatio == 0.0) {
            return allocation.decision(Decision.YES, NAME, "Remote data ratio is set to 0, no limit on allocation", new Object[0]);
        }
        return null;
    }
}

