/*
 * kconfig.c - provide (compressed) kernel config via /proc/config.gz
 *
 * Copyright (c) 2001 Andreas Ferber <af@devcon.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/init.h>

#ifndef CONFIG_PROC_FS
#error /proc filesystem not enabled
#endif

#include "confdata.h"

#ifdef CONFIG_PROC_CONFIG_GZ
# define KCONF_PROCNAME "config.gz"
#else
# ifdef CONFIG_PROC_CONFIG_BZ2
#  define KCONF_PROCNAME "config.bz2"
# else
#  define KCONF_PROCNAME "config"
# endif
#endif

static struct proc_dir_entry *proc_kconfig;

#if 0
#define DEBUGP(format, args...) printk(KERN_DEBUG format, ## args)
#else
#define DEBUGP(format, args...)	/* empty */
#endif

static int
proc_kconfig_read(char *buf, char **start, off_t offset,
		  int len, int *eof, void *unused)
{
	DEBUGP("proc_kconfig_read: offset = %i, len = %i\n", offset, len);

	if (offset + len > sizeof(kconfig_data)) {
		len = sizeof(kconfig_data) - offset;
		*eof = 1;
		DEBUGP
		    ("proc_kconfig_read: eof reached (returning len = %i)\n",
		     len);
	}

	memcpy(buf, kconfig_data + offset, len);
	*start = buf;
	return len;
}

static int __init kconfig_init(void)
{
	DEBUGP("kconfig: init module\n");

#ifdef CONFIG_GRKERNSEC_PROC_ADD
#ifdef CONFIG_GRKERNSEC_PROC_USER
	proc_kconfig = create_proc_entry(KCONF_PROCNAME, S_IRUSR, NULL);
#elif CONFIG_GRKERNSEC_PROC_USERGROUP
	proc_kconfig = create_proc_entry(KCONF_PROCNAME, S_IRUSR | S_IRGRP, NULL);
#endif
#else
	proc_kconfig = create_proc_entry(KCONF_PROCNAME, 0, NULL);
#endif

	if (proc_kconfig == NULL) {
		printk(KERN_ERR "kconfig: unable to register /proc/"
		       KCONF_PROCNAME "\n");
		return -1;
	}

	proc_kconfig->size = sizeof(kconfig_data);
	proc_kconfig->read_proc = proc_kconfig_read;

	return 0;
}

/* for 2.2.x compatibility */
#if LINUX_KERNEL_VERSION < KERNEL_VERSION(2,3,0)
#define __exit			/* empty */
#endif

static void __exit kconfig_exit(void)
{
	DEBUGP("kconfig: removing module\n");

	remove_proc_entry(KCONF_PROCNAME, NULL);
}

module_init(kconfig_init);
module_exit(kconfig_exit);

MODULE_AUTHOR("Andreas Ferber");
MODULE_DESCRIPTION
    ("Provides /proc/config.gz with the (compressed) kernel config");

/*
 * vi:ts=8 sw=8 expandtab
 */
