/*
 * High performance packet classification - user/kernel mode compatibility
 *
 * Authors: Michael Bellion and Thomas Heinz
 * (c) 2002-2003 by the hipac core team <nf@hipac.org>:
 *      +-----------------------+----------------------+
 *      |   Michael Bellion     |     Thomas Heinz     |
 *      | <mbellion@hipac.org>  |  <creatix@hipac.org> |
 *      +-----------------------+----------------------+
 * Licenced under the GNU General Public Licence, version >= 2.
 */


#ifndef _MODE_H
#define _MODE_H

#include <linux/stddef.h> // offsetof
#include <asm/page.h>

/* maximal number of bytes allocatable by mini_alloc */
#define MINI_ALLOC_MAX 131072



/*
 * NEVER use big_alloc and big_free. Use hp_alloc and hp_free instead.
 * The only exceptions to this rule is the implementation of hp_alloc,
 * hp_realloc and hp_free.
 *
 * mini_alloc and mini_free can be used for small (<= MINI_ALLOC_MAX bytes)
 * data structures if one wants to avoid the overhead of hp_alloc and hp_free
 */

static inline unsigned
mini_alloc_size(unsigned size)
{
	if (size > 0 && size <= MINI_ALLOC_MAX) {
		/* round size up to the closest power of 2 */
		size--;
		size |= size >> 1;
		size |= size >> 2;
		size |= size >> 4;
		size |= size >> 8;
		size |= size >> 16;
		size++;
#if PAGE_SIZE == 4096
		return size <= 32 ? 32 : size;
#else
		return size <= 64 ? 64 : size;
#endif
	}
	return 0;
}

static inline unsigned
big_alloc_size(unsigned size)
{
	return size == 0 ? 0 : (((size - 1) + PAGE_SIZE) & ~(PAGE_SIZE - 1));
}


#ifdef __KERNEL__
/*
 * Kernel space
 */
#include <linux/config.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>   // ULONG_MAX
#include <linux/compiler.h> // likely, unlikely
#include <linux/smp.h>      // smp_num_cpus, cpu_number_map, smp_processor_id
#include <linux/brlock.h>   // BR_HIPAC_LOCK, br_lock
#include <linux/spinlock.h> // spin_lock
#include <linux/cache.h>    // __cacheline_aligned
#include <linux/highmem.h>
#include <linux/vmalloc.h>
#include <linux/types.h>
#include <linux/list.h>

#define assert(as)           do {} while (0)
#define printf(str, args...) printk(str , ## args)
#define CURRENT_CPU          (cpu_number_map(smp_processor_id()))
#define SMP_CACHE_ALIGN(x)   ((((x) - 1) + SMP_CACHE_BYTES) & \
			      ~(SMP_CACHE_BYTES - 1))

/* with the linux pre-emptible patch applied we don't have to care about
   scheduling issues */
#ifdef CONFIG_PREEMPT
#  define SCHEDULE do {} while (0)
#else
#  define SCHEDULE schedule()
#endif

/* for small amounts of memory only (up to 128 KB) */
static inline void *
mini_alloc(unsigned size)
{
	if (size > 0 && size <= MINI_ALLOC_MAX) {
		return kmalloc(size, GFP_KERNEL);
	}
	return NULL;
}

static inline void
mini_free(void *p)
{
	kfree(p);
}

/* memory is allocated in amounts of multiples of PAGE_SIZE */
static inline void *
big_alloc(unsigned size)
{
	return vmalloc(size);
}

static inline void
big_free(void *p)
{
	vfree(p);
}

/* dirty hack to make stuff work with uml (otherwise high_physmem and end_vm
   are not defined) */
#ifdef  CONFIG_UML_NET
#  undef  TOP
#  ifdef  CONFIG_HOST_2G_2G
#    define TOP 0x80000000
#  else
#    define TOP 0xc0000000
#  endif
#  undef  SIZE
#  define SIZE  ((CONFIG_NEST_LEVEL + CONFIG_KERNEL_HALF_GIGS) * 0x20000000)
#  undef  START
#  define START (TOP - SIZE)
#  undef  VMALLOC_OFFSET
#  define VMALLOC_OFFSET (8 * 1024 * 1024)
#  undef  VMALLOC_START
#  define VMALLOC_START (((unsigned long) (START + 32 * 1024 * 1024) + \
 	VMALLOC_OFFSET) & ~(VMALLOC_OFFSET - 1))
static unsigned long high_physmem = START + 32 * 1024 * 1024;
static unsigned long end_vm       = VMALLOC_START + 32 * 1024 * 1024;
#endif  /* CONFIG_UML_NET */




#else /* __KERNEL__ */
/*
 * User space
 */
#include <features.h>
#if defined(__GLIBC__) && __GLIBC__ == 2
#  include <asm/types.h>
#else /* libc5 */
#  include <linux/types.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>  // ULONG_MAX
#include <assert.h>
#ifndef WITH_MPATROL
#  include <malloc.h>
#endif
#ifdef INCLUDE_LIST_H
#  include "list.h"
#endif

/* no assertions if not debugging */
#ifndef DEBUG
#  undef  assert
#  define assert(as) do {} while (0)
#endif

/* single processor in user space */
#define smp_num_cpus        1
#define CURRENT_CPU         0

/* locking unnecessary in user space */
typedef struct {} spinlock_t;
#define spin_lock_init(x)     do {} while (0)
#define spin_lock(x)          do {} while (0)
#define spin_unlock(x)        do {} while (0)
#define spin_lock_bh(x)       do {} while (0)
#define spin_unlock_bh(x)     do {} while (0)
#define BR_HIPAC_LOCK         0
#define br_read_lock(x)       do {} while (0)
#define br_read_unlock(x)     do {} while (0)
#define br_write_lock_bh(x)   do {} while (0)
#define br_write_unlock_bh(x) do {} while (0)

/* not needed in user space */
#undef  __cacheline_aligned
#define __cacheline_aligned
#undef  SMP_CACHE_ALIGN
#define SMP_CACHE_ALIGN(x)    (x)

/* printk compatibility */
#define KERN_EMERG    "KERN_EMERG: "
#define KERN_ALERT    "KERN_ALERT: "
#define KERN_CRIT     "KERN_CRIT: "
#define KERN_ERR      "KERN_ERR: "
#define KERN_WARNING  "KERN_WARNING: "
#define KERN_NOTICE   "KERN_NOTICE: "
#define KERN_INFO     "KERN_INFO: "
#define KERN_DEBUG    "KERN_DEBUG: "
#define printk(str, args...) printf(str , ## args)

/* scheduling unnecessary in user space */
#define SCHEDULE do {} while (0)

/* macro to annotate likely branch directions which results in the
   blocks being reordered appropriately */
#if __GNUC__ == 2 && __GNUC_MINOR__ < 96
#  define __builtin_expect(x, expected_value) (x)
#endif
#define likely(x)   __builtin_expect((x), 1)
#define unlikely(x) __builtin_expect((x), 0)

/* for small amounts of memory only (up to 128 KB) */
static inline void *
mini_alloc(unsigned size)
{
	return malloc(mini_alloc_size(size));
}

static inline void
mini_free(void *p)
{
	free(p);
}

/* memory is allocated in amounts of multiples of PAGE_SIZE */
static inline void *
big_alloc(unsigned size)
{
	return malloc(big_alloc_size(size));
}

static inline void
big_free(void *p)
{
	free(p);
}

#endif /* __KERNEL__ */

#endif
