/*
 *  linux/fs/stat.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/config.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/smp_lock.h>
#include <linux/highuid.h>

#include <asm/uaccess.h>

/* RSBAC */
#ifdef CONFIG_RSBAC
#include <rsbac/adf.h>
#endif

/*
 * Revalidate the inode. This is required for proper NFS attribute caching.
 */
static __inline__ int
do_revalidate(struct dentry *dentry)
{
	struct inode * inode = dentry->d_inode;
	if (inode->i_op && inode->i_op->revalidate)
		return inode->i_op->revalidate(dentry);
	return 0;
}


#if !defined(__alpha__) && !defined(__sparc__) && !defined(__ia64__) && !defined(CONFIG_ARCH_S390) && !defined(__hppa__) && !defined(__x86_64__)

/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
static int cp_old_stat(struct inode * inode, struct __old_kernel_stat * statbuf)
{
	static int warncount = 5;
	struct __old_kernel_stat tmp;
	loff_t i_size;

	memset(&tmp, 0, sizeof(struct __old_kernel_stat));
	
	if (warncount > 0) {
		warncount--;
		printk(KERN_WARNING "VFS: Warning: %s using old stat() call. Recompile your binary.\n",
			current->comm);
	} else if (warncount < 0) {
		/* it's laughable, but... */
		warncount = 0;
	}

	tmp.st_dev = kdev_t_to_nr(inode->i_dev);
	tmp.st_ino = inode->i_ino;
	tmp.st_mode = inode->i_mode;
	tmp.st_nlink = inode->i_nlink;
	SET_OLDSTAT_UID(tmp, inode->i_uid);
	SET_OLDSTAT_GID(tmp, inode->i_gid);
	tmp.st_rdev = kdev_t_to_nr(inode->i_rdev);

	i_size = i_size_read(inode);
#if BITS_PER_LONG == 32
	if (i_size > MAX_NON_LFS)
		return -EOVERFLOW;
#endif	
	tmp.st_size = i_size;
	tmp.st_atime = inode->i_atime;
	tmp.st_mtime = inode->i_mtime;
	tmp.st_ctime = inode->i_ctime;
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}

#endif

static int cp_new_stat(struct inode * inode, struct stat * statbuf)
{
	struct stat tmp;
	unsigned int blocks, indirect;
	loff_t i_size;

	memset(&tmp, 0, sizeof(tmp));
	tmp.st_dev = kdev_t_to_nr(inode->i_dev);
	tmp.st_ino = inode->i_ino;
	tmp.st_mode = inode->i_mode;
	tmp.st_nlink = inode->i_nlink;
	SET_STAT_UID(tmp, inode->i_uid);
	SET_STAT_GID(tmp, inode->i_gid);
	tmp.st_rdev = kdev_t_to_nr(inode->i_rdev);

	i_size = i_size_read(inode);
#if BITS_PER_LONG == 32
	if (i_size > MAX_NON_LFS)
		return -EOVERFLOW;
#endif	
	tmp.st_size = i_size;
	tmp.st_atime = inode->i_atime;
	tmp.st_mtime = inode->i_mtime;
	tmp.st_ctime = inode->i_ctime;
/*
 * st_blocks and st_blksize are approximated with a simple algorithm if
 * they aren't supported directly by the filesystem. The minix and msdos
 * filesystems don't keep track of blocks, so they would either have to
 * be counted explicitly (by delving into the file itself), or by using
 * this simple algorithm to get a reasonable (although not 100% accurate)
 * value.
 */

/*
 * Use minix fs values for the number of direct and indirect blocks.  The
 * count is now exact for the minix fs except that it counts zero blocks.
 * Everything is in units of BLOCK_SIZE until the assignment to
 * tmp.st_blksize.
 */
#define D_B   7
#define I_B   (BLOCK_SIZE / sizeof(unsigned short))

	if (!inode->i_blksize) {
		blocks = (tmp.st_size + BLOCK_SIZE - 1) / BLOCK_SIZE;
		if (blocks > D_B) {
			indirect = (blocks - D_B + I_B - 1) / I_B;
			blocks += indirect;
			if (indirect > 1) {
				indirect = (indirect - 1 + I_B - 1) / I_B;
				blocks += indirect;
				if (indirect > 1)
					blocks++;
			}
		}
		tmp.st_blocks = (BLOCK_SIZE / 512) * blocks;
		tmp.st_blksize = BLOCK_SIZE;
	} else {
		tmp.st_blocks = inode->i_blocks;
		tmp.st_blksize = inode->i_blksize;
	}
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}


#if !defined(__alpha__) && !defined(__sparc__) && !defined(__ia64__) && !defined(CONFIG_ARCH_S390) && !defined(__hppa__) && !defined(__x86_64__)
/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
asmlinkage long sys_stat(char * filename, struct __old_kernel_stat * statbuf)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_stat(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                  }
                #endif

		if (!error)
			error = cp_old_stat(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}
#endif

asmlinkage long sys_newstat(char * filename, struct stat * statbuf)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_newstat(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                  }
                #endif

		if (!error)
			error = cp_new_stat(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}

#if !defined(__alpha__) && !defined(__sparc__) && !defined(__ia64__) && !defined(CONFIG_ARCH_S390) && !defined(__hppa__) && !defined(__x86_64__)

/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
asmlinkage long sys_lstat(char * filename, struct __old_kernel_stat * statbuf)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk_link(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_lstat(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                  }
                #endif

		if (!error)
			error = cp_old_stat(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}

#endif

asmlinkage long sys_newlstat(char * filename, struct stat * statbuf)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk_link(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_newlstat(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                       rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                  }
                #endif

		if (!error)
			error = cp_new_stat(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}

#if !defined(__alpha__) && !defined(__sparc__) && !defined(__ia64__) && !defined(CONFIG_ARCH_S390) && !defined(__hppa__) && !defined(__x86_64__)

/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
asmlinkage long sys_fstat(unsigned int fd, struct __old_kernel_stat * statbuf)
{
	struct file * f;
	int err = -EBADF;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	f = fget(fd);
	if (f) {
		struct dentry * dentry = f->f_dentry;

		err = do_revalidate(dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!err)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_fstat(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(dentry->d_inode->i_mode))
                      rsbac_target = T_FIFO;
                    else if (S_ISLNK(dentry->d_inode->i_mode))
                       rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        err = -EPERM;
                      }
                  }
                #endif

		if (!err)
			err = cp_old_stat(dentry->d_inode, statbuf);
		fput(f);
	}
	return err;
}

#endif

asmlinkage long sys_newfstat(unsigned int fd, struct stat * statbuf)
{
	struct file * f;
	int err = -EBADF;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	f = fget(fd);
	if (f) {
		struct dentry * dentry = f->f_dentry;

		err = do_revalidate(dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!err)
                  {
                    rsbac_target = T_NONE;
                    if (S_ISDIR(dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else
                    if (S_ISFIFO(dentry->d_inode->i_mode))
                      rsbac_target = T_FIFO;
                    else
                    if (S_ISLNK(dentry->d_inode->i_mode))
                      rsbac_target = T_SYMLINK;
                    else
                    if (   S_ISREG(dentry->d_inode->i_mode)
                        || S_ISCHR(dentry->d_inode->i_mode)
                        || S_ISBLK(dentry->d_inode->i_mode)
                       )
                      rsbac_target = T_FILE;
                    if(rsbac_target != T_NONE)
                      {
#ifdef CONFIG_RSBAC_DEBUG
                        if (rsbac_debug_aef)
                          printk(KERN_DEBUG
                                 "sys_newfstat(): calling ADF for mode %o target\n",
                                 dentry->d_inode->i_mode);
#endif
                        rsbac_target_id.file.device = dentry->d_inode->i_dev;
                        rsbac_target_id.file.inode  = dentry->d_inode->i_ino;
                        rsbac_target_id.file.dentry_p = dentry;
                        rsbac_attribute_value.dummy = 0;
                        if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                               current->pid,
                                               rsbac_target,
                                               rsbac_target_id,
                                               A_none,
                                               rsbac_attribute_value))
                          {
                            err = -EPERM;
                          }
                      }
                  }
                #endif

		if (!err)
			err = cp_new_stat(dentry->d_inode, statbuf);
		fput(f);
	}
	return err;
}
/* this function should be called vfs_readlink & vfs_readlink
   generic_readlink */

int vfs_readlink_real(struct dentry * dentry, char * buf, int bufsiz)
{
	struct inode * inode = dentry->d_inode;
	int error = -EINVAL;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	if (inode->i_op && inode->i_op->readlink &&
	    !(error = do_revalidate(dentry))) {
                        /* RSBAC */
                        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
                        if (rsbac_debug_aef)
                          printk(KERN_DEBUG "sys_readlink(): calling ADF\n");
#endif
                        rsbac_target_id.file.device = inode->i_dev;
                        rsbac_target_id.file.inode  = inode->i_ino;
                        rsbac_target_id.file.dentry_p = dentry;
                        rsbac_attribute_value.dummy = 0;
                        if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                               current->pid,
                                               T_SYMLINK,
                                               rsbac_target_id,
                                               A_none,
                                               rsbac_attribute_value))
                          {
/*                            path_release(&nd);	*/
                            return -EPERM;
                          }
                        #endif /* CONFIG_RSBAC */

		UPDATE_ATIME(inode);
		error = inode->i_op->readlink(dentry, buf, bufsiz);
	}
	return error;
}

asmlinkage long sys_readlink(const char * path, char * buf, int bufsiz)
{
	struct nameidata nd;
	int error;

        /* RSBAC */

	if (bufsiz <= 0)
		return -EINVAL;

	error = user_path_walk_link(path, &nd);
	if (!error) {
		error = vfs_readlink_real(nd.dentry, buf, bufsiz);
		path_release(&nd);
	}
	return error;
}


/* ---------- LFS-64 ----------- */
#if !defined(__alpha__) && !defined(__ia64__) && !defined(__mips64) && !defined(__x86_64__) && !defined(CONFIG_ARCH_S390X)

static long cp_new_stat64(struct inode * inode, struct stat64 * statbuf)
{
	struct stat64 tmp;
	unsigned int blocks, indirect;

	memset(&tmp, 0, sizeof(tmp));
	tmp.st_dev = kdev_t_to_nr(inode->i_dev);
	tmp.st_ino = inode->i_ino;
#ifdef STAT64_HAS_BROKEN_ST_INO
	tmp.__st_ino = inode->i_ino;
#endif
	tmp.st_mode = inode->i_mode;
	tmp.st_nlink = inode->i_nlink;
	tmp.st_uid = inode->i_uid;
	tmp.st_gid = inode->i_gid;
	tmp.st_rdev = kdev_t_to_nr(inode->i_rdev);
	tmp.st_atime = inode->i_atime;
	tmp.st_mtime = inode->i_mtime;
	tmp.st_ctime = inode->i_ctime;
	tmp.st_size = i_size_read(inode);
/*
 * st_blocks and st_blksize are approximated with a simple algorithm if
 * they aren't supported directly by the filesystem. The minix and msdos
 * filesystems don't keep track of blocks, so they would either have to
 * be counted explicitly (by delving into the file itself), or by using
 * this simple algorithm to get a reasonable (although not 100% accurate)
 * value.
 */

/*
 * Use minix fs values for the number of direct and indirect blocks.  The
 * count is now exact for the minix fs except that it counts zero blocks.
 * Everything is in units of BLOCK_SIZE until the assignment to
 * tmp.st_blksize.
 */
#define D_B   7
#define I_B   (BLOCK_SIZE / sizeof(unsigned short))

	if (!inode->i_blksize) {
		blocks = (tmp.st_size + BLOCK_SIZE - 1) >> BLOCK_SIZE_BITS;
		if (blocks > D_B) {
			indirect = (blocks - D_B + I_B - 1) / I_B;
			blocks += indirect;
			if (indirect > 1) {
				indirect = (indirect - 1 + I_B - 1) / I_B;
				blocks += indirect;
				if (indirect > 1)
					blocks++;
			}
		}
		tmp.st_blocks = (BLOCK_SIZE / 512) * blocks;
		tmp.st_blksize = BLOCK_SIZE;
	} else {
		tmp.st_blocks = inode->i_blocks;
		tmp.st_blksize = inode->i_blksize;
	}
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}

asmlinkage long sys_stat64(char * filename, struct stat64 * statbuf, long flags)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_stat64(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                   }
                #endif

		if (!error)
			error = cp_new_stat64(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}

asmlinkage long sys_lstat64(char * filename, struct stat64 * statbuf, long flags)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk_link(filename, &nd);
	if (!error) {
		error = do_revalidate(nd.dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!error)
                  {
#ifdef CONFIG_RSBAC_DEBUG
                    if (rsbac_debug_aef) printk(KERN_DEBUG "sys_lstat64(): calling ADF\n");
#endif
                    rsbac_target = T_FILE;
                    if (S_ISDIR(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_FIFO;
                    else if (S_ISLNK(nd.dentry->d_inode->i_mode))
                      rsbac_target = T_SYMLINK;
                    rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
                    rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
                    rsbac_target_id.file.dentry_p = nd.dentry;
                    rsbac_attribute_value.dummy = 0;
                    if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                           current->pid,
                                           rsbac_target,
                                           rsbac_target_id,
                                           A_none,
                                           rsbac_attribute_value))
                      {
                        error = -EPERM;
                      }
                  }
                #endif

		if (!error)
			error = cp_new_stat64(nd.dentry->d_inode, statbuf);
		path_release(&nd);
	}
	return error;
}

asmlinkage long sys_fstat64(unsigned long fd, struct stat64 * statbuf, long flags)
{
	struct file * f;
	int err = -EBADF;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	f = fget(fd);
	if (f) {
		struct dentry * dentry = f->f_dentry;

		err = do_revalidate(dentry);

                /* RSBAC */
                #ifdef CONFIG_RSBAC
                if(!err)
                  {
                    rsbac_target = T_NONE;
                    if (S_ISDIR(dentry->d_inode->i_mode))
                      rsbac_target = T_DIR;
                    else
                    if (S_ISFIFO(dentry->d_inode->i_mode))
                      rsbac_target = T_FIFO;
                    else
                    if (S_ISLNK(dentry->d_inode->i_mode))
                      rsbac_target = T_SYMLINK;
                    else
                    if (   S_ISREG(dentry->d_inode->i_mode)
                        || S_ISCHR(dentry->d_inode->i_mode)
                        || S_ISBLK(dentry->d_inode->i_mode)
                       )
                      rsbac_target = T_FILE;
                    if(rsbac_target != T_NONE)
                      {
#ifdef CONFIG_RSBAC_DEBUG
                        if (rsbac_debug_aef)
                          printk(KERN_DEBUG
                                 "sys_fstat64(): calling ADF for mode %o target\n",
                                 dentry->d_inode->i_mode);
#endif
                        rsbac_target_id.file.device = dentry->d_inode->i_dev;
                        rsbac_target_id.file.inode  = dentry->d_inode->i_ino;
                        rsbac_target_id.file.dentry_p = dentry;
                        rsbac_attribute_value.dummy = 0;
                        if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                               current->pid,
                                               rsbac_target,
                                               rsbac_target_id,
                                               A_none,
                                               rsbac_attribute_value))
                          {
                            err = -EPERM;
                          }
                      }
                  }
                #endif

		if (!err)
			err = cp_new_stat64(dentry->d_inode, statbuf);
		fput(f);
	}
	return err;
}

#endif /* LFS-64 */
