/*
 * load.c
 *
 * Copyright 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, version 2.
 */

#include "mconfig.h"
#include "parser.tab.h"



/*
 * Process a statement to load an answer from a def_line.
 * FIXME: need some input validation and coercion here.
 */

static void 
scb_load(const statement_type * statement, void *p_UNUSED)
{
	switch (statement->verb) {
		/* default: */
	case verb_NULL:
	case verb_IF:
		error_enum_bogus();
		break;

	case verb_MENU:
	case verb_mainmenu_name:
	case verb_comment:
	case verb_text:
		break;

	case verb_unset:
		{
			symbol_type    *symbol;

			for (symbol = statement->su_unset_list->symbol_first;
			     symbol != NULL;
			     symbol = symbol->next) {
				symbol->value = piece_empty;
				symbol->origin = origin_unset;
			}
		}
		break;

	case verb_ask_bool:
	case verb_ask_hex:
	case verb_ask_int:
	case verb_ask_string:
	case verb_ask_tristate:
		if (statement->sb_symbol->def_line != NULL) {
			statement->sb_symbol->value = statement->sb_symbol->def_line->value;
			statement->sb_symbol->origin = origin_defconfig;
		}
		break;

	case verb_def_bool:
	case verb_def_hex:
	case verb_def_int:
	case verb_def_string:
	case verb_def_tristate:
		statement->sb_symbol->value = word_eval(statement->sb_def_value, 1);
		statement->sb_symbol->origin = origin_statement;
		break;

	case verb_dep_bool:
	case verb_dep_mbool:
	case verb_dep_hex:
	case verb_dep_int:
	case verb_dep_string:
	case verb_dep_tristate:
		if (statement->sb_symbol->def_line != NULL) {
			statement->sb_symbol->value = statement->sb_symbol->def_line->value;
			statement->sb_symbol->origin = origin_defconfig;
		}
		break;

	case verb_nchoice:
		{
			const symbol_type *symbol;
			int             index;

			for (symbol = statement->sn_choice_list->symbol_first, index = 0;
			     symbol != NULL;
			     symbol = symbol->next, ++index) {
				if (symbol->def_line != NULL) {
					symbol->value = symbol->def_line->value;
					symbol->origin = origin_defconfig;
					if (symbol->value.len == 1 && symbol->value.ptr[0] == 'y')
						statement->sn_choice_index = index;
				}
			}
		}
		break;
	}
}



/*
 * Load a .config or arch/$ARCH/defconfig file.
 *
 * These files are written in a very small language which has some nonterminals
 * in common with Config Language, but actually has a different grammar.  So I
 * fiddle with the parser to tell it which language to parse.
 *
 * After parsing, I take the fresh values and do *not* put them in the symbol
 * table!  Instead, I run through the script and answer questions with them.
 * Thanks to Linus Torvalds for suggesting this.  In his words: "there are
 * no default values; there are only default answers to questions."
 *
 * FIXME: return errors instead of just error-exiting.
 */

void 
load_defconfig(const block_type * block, const char *filename)
{
	const char     *error_string;
	def_line_type  *def_line;

	/*
         * Grab the file.
         */

	if (input_push_file(filename, &error_string) < 0)
		error_exit(error_string);

	/*
         * Crank the parser.
         */

	parser_magic_cookie = COOKIE_DEF_CONFIG;
	if (yyparse() != 0 || parser_error_count > 0) {
		if (argument.mode != mode_syntax) {
			char           *buf = grab_memory(strlen(filename) + 32);
			sprintf(buf, "%s: failed to parse", filename);
			error_exit(buf);
		}
	}
	/*
         * Clear the symbol table and create new def_line pointers.  I don't
         * want to do this in the parser, because the parse might not finish
         * properly.
         */

	symbol_table_reset();

	for (def_line = parser_def_line_list->first;
	     def_line != NULL;
	     def_line = def_line->next) {
		def_line->symbol->def_line = def_line;
	}

	/*
         * Walk the tree and fill in values.
         */

	block_walk(block, 1, scb_load, NULL);
}
