/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <openssl/crypto.h>
#include <stdlib.h>
#include <pthread.h>
#include <ctype.h>

/* multi-threading support */
#define USE_RWLOCK

#ifdef USE_RWLOCK
#  define LOCK_T pthread_rwlock_t
#  define LOCK_INIT(l) pthread_rwlock_init(l, NULL)
#  define LOCK_DESTROY(l) pthread_rwlock_destroy(l)
#  define READ_LOCK(l) pthread_rwlock_rdlock(l)
#  define READ_UNLOCK(l) pthread_rwlock_unlock(l)
#  define WRITE_LOCK(l) pthread_rwlock_wrlock(l)
#  define WRITE_UNLOCK(l) pthread_rwlock_unlock(l)
#else /* ! USE_RWLOCK */
#  define LOCK_T pthread_mutex_t
#  define LOCK_INIT(l) pthread_mutex_init(l, NULL)
#  define LOCK_DESTROY(l) pthread_mutex_destroy(l)
#  define READ_LOCK(l) pthread_mutex_lock(l)
#  define READ_UNLOCK(l) pthread_mutex_unlock(l)
#  define WRITE_LOCK(l) pthread_mutex_lock(l)
#  define WRITE_UNLOCK(l) pthread_mutex_unlock(l)
#endif /* ! USE_RWLOCK */

static LOCK_T *locks;

static void locking_callback(int mode, int type, const char *file,
	int line)
	{
	DPRINTF(stderr, "thread=%4d mode=%s lock=%s %s:%d\n",
		CRYPTO_thread_id(),
		(mode & CRYPTO_LOCK) ? "l" : "u",
		(mode & CRYPTO_READ) ? "r" : "w", file, line);

	if (mode & CRYPTO_LOCK)
		if (mode & CRYPTO_READ)
			READ_LOCK(locks + type);
		else
			WRITE_LOCK(locks + type);
	else
		if (mode & CRYPTO_READ)
			READ_UNLOCK(locks + type);
		else
			WRITE_UNLOCK(locks + type);
	}

static unsigned long locking_thread_id(void)
	{
	return (unsigned long)pthread_self();
	}


static void locking_setup(void)
	{
	int i;

	locks = OPENSSL_malloc(CRYPTO_num_locks() * sizeof(*locks));
	for (i = 0; i < CRYPTO_num_locks(); i++)
		{
		LOCK_INIT(locks + i);
		}

	CRYPTO_set_id_callback(locking_thread_id);
	CRYPTO_set_locking_callback(locking_callback);
	}

static void locking_cleanup(void)
	{
	int i;

	CRYPTO_set_locking_callback(NULL);
	for (i = 0; i < CRYPTO_num_locks(); i++)
		{
		LOCK_DESTROY(locks + i);
		}
	OPENSSL_free(locks);
	}

/* utilities to parse command line arguments */
static int nibble(const int c)
	{
	if (isdigit(c))
		return c - '0';
	else if (islower(c))
		return c - 'a' + 10;
	else
		return c - 'A' + 10;
	}

static int parse_hex(unsigned char *out, int max_out, const char *in)
	{
	int i;
	for (i = 0; in[0] && i < max_out; i++)
		{
		if (!in[1])
			return 0; /* odd length */
		*out = (nibble(in[0]) << 4) | (nibble(in[1]));
		in += 2;
		out++;
		}
	return !in[0];
	}

static int parse_size(long long *size, const char *str)
	{
	long long v;
	char *end;

	v = strtoll(str, &end, 0);
	if (strcmp("", end) == 0)
		; /* do nothing */

	else if (strcmp("K", end) == 0)
		v <<= 10;
	else if (strcmp("M", end) == 0)
		v <<= 20;
	else if (strcmp("G", end) == 0)
		v <<= 30;
	else if (strcmp("T", end) == 0)
		v <<= 40;

	else if (strcmp("k", end) == 0)
		v *= 1000;
	else if (strcmp("m", end) == 0)
		v *= 1000000;
	else if (strcmp("g", end) == 0)
		v *= 1000000000LL;
	else if (strcmp("t", end) == 0)
		v *= 1000000000000LL;

	else
		return 0;

	*size = v;
	return 1;
	}
