# 1 "src/xml_lexer.mll"
 (*
 * Xml Light, an small Xml parser/printer with DTD support.
 * Copyright (C) 2003 Nicolas Cannasse (ncannasse@motion-twin.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library has the special exception on linking described in file
 * README.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *)

open Lexing
open Xml_parser
open Xml_light_types

type error =
	| EUnterminatedComment
	| EUnterminatedString
	| EIdentExpected
	| ECloseExpected
	| ENodeExpected
	| EAttributeNameExpected
	| EAttributeValueExpected
	| EUnterminatedEntity

type dtd_error =
	| EInvalidDTDDecl
	| EInvalidDTDTag
	| EDTDItemExpected
	| EInvalidDTDElement
	| EInvalidDTDAttribute

exception Error of error
exception DTDError of dtd_error

type pos = int * int * int * int

type dtd_decl =
	| DTDFile of string
	| DTDData of dtd

type dtd_item_type =
	| TElement
	| TAttribute

type token =
	| Tag of string * (string * string) list * bool
	| PCData of string
	| Endtag of string
	| DocType of (string * dtd_decl)
	| Eof

let last_pos = ref 0
and current_line = ref 0
and current_line_start = ref 0

let tmp = Buffer.create 200


let entities = [
  "gt"  ,62, ">";
  "lt"  ,60, "<";
  "amp" ,38, "&";
  "apos",39, "'";
  "quot",34, "\"";
]

let idents = Hashtbl.create 0
let _ = begin
  List.iter (fun (str,code,res) ->
      Hashtbl.add idents (str^";") res;
      if code > 0
      then Hashtbl.add idents ("#" ^ string_of_int code ^ ";") res
    ) entities
end

let init lexbuf =
	current_line := 1;
	current_line_start := lexeme_start lexbuf;
	last_pos := !current_line_start

let close lexbuf =
	Buffer.reset tmp

let pos lexbuf =
	!current_line ,	!current_line_start ,
	!last_pos ,
	lexeme_start lexbuf

let restore (cl,cls,lp,_) =
	current_line := cl;
	current_line_start := cls;
	last_pos := lp

let newline lexbuf =
	incr current_line;
	last_pos := lexeme_end lexbuf;
	current_line_start := !last_pos

let error lexbuf e =
	last_pos := lexeme_start lexbuf;
	raise (Error e)

let dtd_error lexbuf e =
	last_pos := lexeme_start lexbuf;
	raise (DTDError e)

let error_pos_of_pos (line, lstart, min, max) =
	{
		Xml_light_errors.eline = line;
		Xml_light_errors.eline_start = lstart;
		Xml_light_errors.emin = min;
		Xml_light_errors.emax = max;
	}

let error_pos lexbuf = error_pos_of_pos (pos lexbuf)

# 132 "src/xml_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\244\255\245\255\001\000\247\255\003\000\008\000\009\000\
    \255\255\011\000\017\000\250\255\000\000\001\000\000\000\000\000\
    \003\000\000\000\000\000\000\000\003\000\253\255\005\000\000\000\
    \001\000\009\000\001\000\252\255\251\255\020\000\044\000\255\255\
    \006\000\252\255\253\255\003\000\255\255\013\000\254\255\014\000\
    \252\255\253\255\016\000\255\255\254\255\005\000\251\255\000\000\
    \254\255\015\000\019\000\253\255\073\000\254\255\077\000\075\000\
    \253\255\152\000\237\000\255\255\056\001\162\000\176\000\131\001\
    \254\255\169\001\254\255\247\001\020\000\254\255\255\255\041\000\
    \032\000\255\255\254\255\069\002\254\255\147\002\190\000\253\255\
    \005\003\006\003\008\003\009\003\254\255\255\255\003\000\252\255\
    \253\255\008\000\255\255\254\255\074\000\252\255\253\255\015\000\
    \255\255\254\255\021\000\252\255\253\255\007\000\017\000\043\000\
    \038\000\043\000\051\000\255\255\036\000\036\000\052\000\045\000\
    \254\255\200\000\253\255\254\255\255\255\041\000\255\255\011\003\
    \251\000\254\255\255\255\074\000\254\255\255\255\078\000\253\255\
    \001\000\010\000\011\000\255\255\105\000\255\255\171\000\253\255\
    \084\000\093\000\101\000\094\000\129\000\142\000\139\000\255\255\
    \151\000\161\000\165\000\164\000\164\000\254\255\215\000\252\255\
    \253\255\172\000\172\000\163\000\255\255\173\000\170\000\169\000\
    \254\255\201\000\254\255\255\255\204\000\254\255\255\255\036\003\
    \243\255\114\003\245\255\246\255\247\255\187\000\249\255\250\255\
    \251\255\252\255\253\255\180\001\255\255\201\000\201\000\214\000\
    \197\000\217\000\248\255\239\002\250\255\251\255\227\000\220\000\
    \230\000\234\000\216\000\236\000\255\255\244\000\252\000\001\001\
    \045\001\038\001\254\255\035\001\049\001\049\001\252\255\203\003\
    \254\255\025\004\079\001\253\255\254\255\255\255\113\001\249\255\
    \134\001\254\255\255\255\080\001\077\001\074\001\122\001\111\001\
    \109\001\122\001\123\001\137\001\140\001\253\255\129\001\134\001\
    \138\001\144\001\147\001\252\255\172\001\192\001\194\001\251\255\
    \193\001\200\001\208\001\218\001\222\001\250\255\048\002\253\255\
    \254\255\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\009\000\255\255\007\000\001\000\001\000\
    \255\255\006\000\007\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\002\000\001\000\255\255\
    \255\255\255\255\255\255\003\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\003\000\255\255\255\255\255\255\255\255\003\000\
    \255\255\000\000\255\255\255\255\002\000\255\255\000\000\255\255\
    \255\255\002\000\002\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\002\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \002\000\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\003\000\255\255\255\255\255\255\255\255\255\255\003\000\
    \255\255\255\255\255\255\255\255\255\255\003\000\003\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\001\000\255\255\003\000\
    \002\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \002\000\001\000\255\255\255\255\001\000\255\255\255\255\255\255\
    \002\000\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\003\000\003\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\011\000\255\255\255\255\255\255\012\000\255\255\255\255\
    \255\255\255\255\255\255\001\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\005\000\005\000\
    \005\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\002\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \006\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_default =
   "\003\000\000\000\000\000\003\000\000\000\255\255\255\255\003\000\
    \000\000\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\255\255\255\255\000\000\
    \033\000\000\000\000\000\255\255\000\000\255\255\000\000\040\000\
    \000\000\000\000\255\255\000\000\000\000\049\000\000\000\255\255\
    \000\000\049\000\255\255\000\000\054\000\000\000\054\000\056\000\
    \000\000\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \000\000\066\000\000\000\255\255\069\000\000\000\000\000\255\255\
    \255\255\000\000\000\000\076\000\000\000\255\255\079\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\000\000\087\000\000\000\
    \000\000\255\255\000\000\000\000\093\000\000\000\000\000\255\255\
    \000\000\000\000\099\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \000\000\114\000\000\000\000\000\000\000\255\255\000\000\255\255\
    \255\255\000\000\000\000\124\000\000\000\000\000\127\000\000\000\
    \255\255\255\255\255\255\000\000\255\255\000\000\135\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\151\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \000\000\162\000\000\000\000\000\165\000\000\000\000\000\168\000\
    \000\000\255\255\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\000\000\188\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\000\000\208\000\
    \000\000\255\255\211\000\000\000\000\000\000\000\215\000\000\000\
    \255\255\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\247\000\000\000\
    \000\000\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\007\000\008\000\255\255\010\000\006\000\255\255\048\000\
    \036\000\006\000\007\000\255\255\009\000\006\000\006\000\000\000\
    \043\000\255\255\010\000\000\000\000\000\030\000\031\000\000\000\
    \007\000\030\000\129\000\010\000\012\000\090\000\004\000\255\255\
    \006\000\007\000\091\000\009\000\000\000\013\000\028\000\255\255\
    \037\000\010\000\009\000\035\000\030\000\030\000\097\000\130\000\
    \131\000\030\000\000\000\000\000\005\000\255\255\001\000\255\255\
    \009\000\024\000\011\000\022\000\015\000\255\255\017\000\255\255\
    \021\000\023\000\026\000\038\000\030\000\042\000\044\000\016\000\
    \020\000\051\000\070\000\255\255\018\000\025\000\255\255\255\255\
    \072\000\019\000\255\255\014\000\027\000\050\000\074\000\089\000\
    \108\000\024\000\047\000\022\000\091\000\102\000\103\000\073\000\
    \101\000\023\000\026\000\097\000\255\255\104\000\057\000\053\000\
    \100\000\096\000\105\000\255\255\106\000\025\000\107\000\109\000\
    \110\000\111\000\112\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\255\255\118\000\255\255\
    \125\000\255\255\128\000\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\095\000\133\000\
    \144\000\138\000\139\000\140\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\141\000\081\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\142\000\064\000\081\000\143\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\116\000\145\000\136\000\146\000\147\000\115\000\
    \137\000\063\000\063\000\063\000\063\000\063\000\063\000\148\000\
    \149\000\157\000\155\000\080\000\156\000\158\000\159\000\152\000\
    \002\000\255\255\160\000\088\000\120\000\046\000\034\000\163\000\
    \120\000\255\255\166\000\181\000\182\000\183\000\041\000\255\255\
    \062\000\063\000\063\000\063\000\063\000\063\000\063\000\184\000\
    \154\000\185\000\186\000\120\000\153\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\203\000\
    \059\000\197\000\193\000\194\000\195\000\196\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \198\000\255\255\094\000\199\000\200\000\255\255\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\201\000\059\000\202\000\204\000\205\000\206\000\
    \213\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\218\000\216\000\240\000\236\000\230\000\
    \217\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\179\000\064\000\223\000\
    \224\000\179\000\225\000\226\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\219\000\212\000\220\000\227\000\228\000\221\000\
    \229\000\231\000\232\000\233\000\179\000\234\000\067\000\235\000\
    \222\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\237\000\238\000\239\000\241\000\
    \067\000\242\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\243\000\244\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\245\000\000\000\000\000\000\000\000\000\000\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\249\000\000\000\000\000\000\000\067\000\248\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\077\000\000\000\000\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \000\000\000\000\000\000\000\000\077\000\000\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\000\000\000\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\000\000\077\000\000\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\082\000\083\000\
    \000\000\082\000\083\000\000\000\120\000\121\000\000\000\189\000\
    \120\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\082\000\083\000\085\000\
    \082\000\083\000\085\000\120\000\084\000\179\000\180\000\084\000\
    \000\000\179\000\192\000\000\000\000\000\000\000\000\000\000\000\
    \190\000\000\000\000\000\000\000\000\000\191\000\000\000\000\000\
    \000\000\000\000\000\000\082\000\179\000\000\000\082\000\173\000\
    \000\000\000\000\000\000\000\000\178\000\177\000\172\000\171\000\
    \176\000\169\000\000\000\000\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\000\000\
    \000\000\000\000\175\000\170\000\000\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\000\000\
    \000\000\000\000\000\000\169\000\000\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \174\000\000\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\000\000\000\000\000\000\
    \000\000\169\000\000\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \209\000\000\000\000\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\000\000\000\000\
    \000\000\000\000\000\000\122\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\000\000\000\000\
    \000\000\000\000\209\000\000\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\000\000\
    \000\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\000\000\000\000\000\000\000\000\
    \209\000\000\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\003\000\005\000\000\000\003\000\045\000\
    \032\000\006\000\007\000\007\000\009\000\006\000\007\000\255\255\
    \039\000\049\000\010\000\255\255\255\255\029\000\029\000\255\255\
    \000\000\029\000\128\000\005\000\005\000\086\000\000\000\003\000\
    \006\000\007\000\089\000\009\000\255\255\012\000\013\000\007\000\
    \035\000\010\000\005\000\032\000\029\000\030\000\095\000\129\000\
    \130\000\030\000\255\255\255\255\000\000\003\000\000\000\003\000\
    \010\000\023\000\005\000\014\000\012\000\007\000\016\000\007\000\
    \020\000\022\000\025\000\037\000\030\000\039\000\042\000\015\000\
    \019\000\050\000\068\000\052\000\017\000\024\000\052\000\054\000\
    \071\000\018\000\054\000\012\000\026\000\047\000\072\000\086\000\
    \101\000\023\000\045\000\014\000\089\000\098\000\102\000\071\000\
    \098\000\022\000\025\000\095\000\049\000\103\000\055\000\052\000\
    \098\000\092\000\104\000\054\000\105\000\024\000\106\000\108\000\
    \109\000\110\000\111\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\052\000\117\000\052\000\
    \123\000\054\000\126\000\054\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\092\000\132\000\
    \136\000\137\000\138\000\139\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\140\000\078\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\141\000\061\000\078\000\142\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\113\000\144\000\134\000\145\000\146\000\113\000\
    \134\000\062\000\062\000\062\000\062\000\062\000\062\000\147\000\
    \148\000\153\000\154\000\078\000\155\000\157\000\158\000\150\000\
    \000\000\003\000\159\000\086\000\120\000\045\000\032\000\161\000\
    \120\000\007\000\164\000\173\000\181\000\182\000\039\000\049\000\
    \057\000\062\000\062\000\062\000\062\000\062\000\062\000\183\000\
    \150\000\184\000\185\000\120\000\150\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\190\000\
    \058\000\191\000\192\000\193\000\194\000\195\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \197\000\052\000\092\000\198\000\199\000\054\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\200\000\060\000\201\000\203\000\204\000\205\000\
    \210\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\214\000\214\000\219\000\220\000\221\000\
    \214\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\179\000\063\000\222\000\
    \223\000\179\000\224\000\225\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\216\000\210\000\216\000\226\000\227\000\216\000\
    \228\000\230\000\231\000\232\000\179\000\233\000\065\000\234\000\
    \216\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\236\000\237\000\238\000\240\000\
    \065\000\241\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\067\000\242\000\243\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\244\000\255\255\255\255\255\255\255\255\255\255\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\246\000\255\255\255\255\255\255\067\000\246\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\075\000\255\255\255\255\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\255\255\255\255\255\255\075\000\255\255\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \077\000\255\255\255\255\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\255\255\255\255\
    \255\255\255\255\077\000\255\255\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\080\000\081\000\
    \255\255\082\000\083\000\255\255\119\000\119\000\255\255\187\000\
    \119\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\080\000\081\000\080\000\
    \082\000\083\000\082\000\119\000\080\000\167\000\167\000\082\000\
    \255\255\167\000\187\000\255\255\255\255\255\255\255\255\255\255\
    \187\000\255\255\255\255\255\255\255\255\187\000\255\255\255\255\
    \255\255\255\255\255\255\081\000\167\000\255\255\083\000\167\000\
    \255\255\255\255\255\255\255\255\167\000\167\000\167\000\167\000\
    \167\000\167\000\255\255\255\255\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\255\255\
    \255\255\255\255\167\000\167\000\255\255\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\255\255\
    \255\255\255\255\255\255\167\000\255\255\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\169\000\
    \167\000\255\255\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\255\255\255\255\255\255\
    \255\255\169\000\255\255\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \207\000\255\255\255\255\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\255\255\119\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\207\000\255\255\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\209\000\255\255\
    \255\255\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\255\255\255\255\255\255\255\255\
    \209\000\255\255\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 142 "src/xml_lexer.mll"
  (
			newline lexbuf;
			token lexbuf
		)
# 587 "src/xml_lexer.ml"

  | 1 ->
# 147 "src/xml_lexer.mll"
  (
			last_pos := lexeme_end lexbuf;
			token lexbuf
		)
# 595 "src/xml_lexer.ml"

  | 2 ->
# 152 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			ignore_spaces lexbuf;
			let root = ident_name lexbuf in
			ignore_spaces lexbuf;
			let data = dtd_data lexbuf in
			DocType (root, data)
		)
# 607 "src/xml_lexer.ml"

  | 3 ->
# 161 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			Buffer.reset tmp;
			PCData (cdata lexbuf)
		)
# 616 "src/xml_lexer.ml"

  | 4 ->
# 167 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			comment lexbuf;
			token lexbuf
		)
# 625 "src/xml_lexer.ml"

  | 5 ->
# 173 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			header lexbuf;
			token lexbuf;
		)
# 634 "src/xml_lexer.ml"

  | 6 ->
# 179 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			let tag = ident_name lexbuf in
			ignore_spaces lexbuf;
			close_tag lexbuf;
			Endtag tag
		)
# 645 "src/xml_lexer.ml"

  | 7 ->
# 187 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			let tag = ident_name lexbuf in
			ignore_spaces lexbuf;
			let attribs, closed = attributes lexbuf in
			Tag(tag, attribs, closed)
		)
# 656 "src/xml_lexer.ml"

  | 8 ->
# 195 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			Buffer.reset tmp;
			Buffer.add_string tmp (entity lexbuf);
			PCData (pcdata lexbuf)
		)
# 666 "src/xml_lexer.ml"

  | 9 ->
# 202 "src/xml_lexer.mll"
  (
			last_pos := lexeme_start lexbuf;
			Buffer.reset tmp;
			Buffer.add_string tmp (lexeme lexbuf);
			PCData (pcdata lexbuf)
		)
# 676 "src/xml_lexer.ml"

  | 10 ->
# 208 "src/xml_lexer.mll"
       ( Eof )
# 681 "src/xml_lexer.ml"

  | 11 ->
# 210 "src/xml_lexer.mll"
  ( error lexbuf ENodeExpected )
# 686 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and ignore_spaces lexbuf =
   __ocaml_lex_ignore_spaces_rec lexbuf 29
and __ocaml_lex_ignore_spaces_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 214 "src/xml_lexer.mll"
  (
			newline lexbuf;
			ignore_spaces lexbuf
		)
# 701 "src/xml_lexer.ml"

  | 1 ->
# 219 "src/xml_lexer.mll"
  ( ignore_spaces lexbuf )
# 706 "src/xml_lexer.ml"

  | 2 ->
# 221 "src/xml_lexer.mll"
  ( () )
# 711 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_ignore_spaces_rec lexbuf __ocaml_lex_state

and comment lexbuf =
   __ocaml_lex_comment_rec lexbuf 32
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 225 "src/xml_lexer.mll"
  (
			newline lexbuf;
			comment lexbuf
		)
# 726 "src/xml_lexer.ml"

  | 1 ->
# 230 "src/xml_lexer.mll"
  ( () )
# 731 "src/xml_lexer.ml"

  | 2 ->
# 232 "src/xml_lexer.mll"
  ( raise (Error EUnterminatedComment) )
# 736 "src/xml_lexer.ml"

  | 3 ->
# 234 "src/xml_lexer.mll"
  ( comment lexbuf )
# 741 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and header lexbuf =
   __ocaml_lex_header_rec lexbuf 39
and __ocaml_lex_header_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 238 "src/xml_lexer.mll"
  (
			newline lexbuf;
			header lexbuf
		)
# 756 "src/xml_lexer.ml"

  | 1 ->
# 243 "src/xml_lexer.mll"
  ( () )
# 761 "src/xml_lexer.ml"

  | 2 ->
# 245 "src/xml_lexer.mll"
  ( error lexbuf ECloseExpected )
# 766 "src/xml_lexer.ml"

  | 3 ->
# 247 "src/xml_lexer.mll"
  ( header lexbuf )
# 771 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_header_rec lexbuf __ocaml_lex_state

and cdata lexbuf =
   __ocaml_lex_cdata_rec lexbuf 45
and __ocaml_lex_cdata_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 251 "src/xml_lexer.mll"
  (
			Buffer.add_string tmp (lexeme lexbuf);
			cdata lexbuf
		)
# 786 "src/xml_lexer.ml"

  | 1 ->
# 256 "src/xml_lexer.mll"
  (
			newline lexbuf;
			Buffer.add_string tmp (lexeme lexbuf);
			cdata lexbuf
		)
# 795 "src/xml_lexer.ml"

  | 2 ->
# 262 "src/xml_lexer.mll"
  ( Buffer.contents tmp )
# 800 "src/xml_lexer.ml"

  | 3 ->
# 264 "src/xml_lexer.mll"
  (
			Buffer.add_string tmp (lexeme lexbuf);
			cdata lexbuf
		)
# 808 "src/xml_lexer.ml"

  | 4 ->
# 269 "src/xml_lexer.mll"
  ( error lexbuf ECloseExpected )
# 813 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_cdata_rec lexbuf __ocaml_lex_state

and pcdata lexbuf =
   __ocaml_lex_pcdata_rec lexbuf 52
and __ocaml_lex_pcdata_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 273 "src/xml_lexer.mll"
  (
			Buffer.add_string tmp (lexeme lexbuf);
			pcdata lexbuf
		)
# 828 "src/xml_lexer.ml"

  | 1 ->
# 278 "src/xml_lexer.mll"
  (
			Buffer.add_string tmp (entity lexbuf);
			pcdata lexbuf
		)
# 836 "src/xml_lexer.ml"

  | 2 ->
# 283 "src/xml_lexer.mll"
  ( Buffer.contents tmp )
# 841 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_pcdata_rec lexbuf __ocaml_lex_state

and entity lexbuf =
   __ocaml_lex_entity_rec lexbuf 55
and __ocaml_lex_entity_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 287 "src/xml_lexer.mll"
  (
			let ident = lexeme lexbuf in
			try
				Hashtbl.find idents (String.lowercase_ascii ident)
			with
				Not_found -> "&" ^ ident
		)
# 859 "src/xml_lexer.ml"

  | 1 ->
# 296 "src/xml_lexer.mll"
  (
			let ident = lexeme lexbuf in
			try
				Hashtbl.find idents (String.lowercase_ascii ident)
			with
				Not_found -> "&" ^ ident
		)
# 870 "src/xml_lexer.ml"

  | 2 ->
# 304 "src/xml_lexer.mll"
  ( raise (Error EUnterminatedEntity) )
# 875 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_entity_rec lexbuf __ocaml_lex_state

and ident_name lexbuf =
   __ocaml_lex_ident_name_rec lexbuf 65
and __ocaml_lex_ident_name_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 308 "src/xml_lexer.mll"
  ( lexeme lexbuf )
# 887 "src/xml_lexer.ml"

  | 1 ->
# 310 "src/xml_lexer.mll"
  ( error lexbuf EIdentExpected )
# 892 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_ident_name_rec lexbuf __ocaml_lex_state

and close_tag lexbuf =
   __ocaml_lex_close_tag_rec lexbuf 68
and __ocaml_lex_close_tag_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 314 "src/xml_lexer.mll"
  ( () )
# 904 "src/xml_lexer.ml"

  | 1 ->
# 316 "src/xml_lexer.mll"
  ( error lexbuf ECloseExpected )
# 909 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_close_tag_rec lexbuf __ocaml_lex_state

and attributes lexbuf =
   __ocaml_lex_attributes_rec lexbuf 71
and __ocaml_lex_attributes_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 320 "src/xml_lexer.mll"
  ( [], false )
# 921 "src/xml_lexer.ml"

  | 1 ->
# 322 "src/xml_lexer.mll"
  ( [], true )
# 926 "src/xml_lexer.ml"

  | 2 ->
# 324 "src/xml_lexer.mll"
  (
			let key = attribute lexbuf in
			let data = attribute_data lexbuf in
			ignore_spaces lexbuf;
			let others, closed = attributes lexbuf in
			(key, data) :: others, closed
		)
# 937 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attributes_rec lexbuf __ocaml_lex_state

and attribute lexbuf =
   __ocaml_lex_attribute_rec lexbuf 75
and __ocaml_lex_attribute_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 334 "src/xml_lexer.mll"
  ( lexeme lexbuf )
# 949 "src/xml_lexer.ml"

  | 1 ->
# 336 "src/xml_lexer.mll"
  ( error lexbuf EAttributeNameExpected )
# 954 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attribute_rec lexbuf __ocaml_lex_state

and attribute_data lexbuf =
   __ocaml_lex_attribute_data_rec lexbuf 78
and __ocaml_lex_attribute_data_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 340 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			last_pos := lexeme_end lexbuf;
			dq_string lexbuf
		)
# 970 "src/xml_lexer.ml"

  | 1 ->
# 346 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			last_pos := lexeme_end lexbuf;
			q_string lexbuf
		)
# 979 "src/xml_lexer.ml"

  | 2 ->
# 352 "src/xml_lexer.mll"
  ( error lexbuf EAttributeValueExpected )
# 984 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attribute_data_rec lexbuf __ocaml_lex_state

and dq_string lexbuf =
   __ocaml_lex_dq_string_rec lexbuf 86
and __ocaml_lex_dq_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 356 "src/xml_lexer.mll"
  ( Buffer.contents tmp )
# 996 "src/xml_lexer.ml"

  | 1 ->
# 358 "src/xml_lexer.mll"
  (
			Buffer.add_char tmp (lexeme_char lexbuf 1);
			dq_string lexbuf
		)
# 1004 "src/xml_lexer.ml"

  | 2 ->
# 363 "src/xml_lexer.mll"
  ( raise (Error EUnterminatedString) )
# 1009 "src/xml_lexer.ml"

  | 3 ->
# 365 "src/xml_lexer.mll"
  (
			Buffer.add_char tmp (lexeme_char lexbuf 0);
			dq_string lexbuf
		)
# 1017 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dq_string_rec lexbuf __ocaml_lex_state

and q_string lexbuf =
   __ocaml_lex_q_string_rec lexbuf 92
and __ocaml_lex_q_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 372 "src/xml_lexer.mll"
  ( Buffer.contents tmp )
# 1029 "src/xml_lexer.ml"

  | 1 ->
# 374 "src/xml_lexer.mll"
  (
			Buffer.add_char tmp (lexeme_char lexbuf 1);
			q_string lexbuf
		)
# 1037 "src/xml_lexer.ml"

  | 2 ->
# 379 "src/xml_lexer.mll"
  ( raise (Error EUnterminatedString) )
# 1042 "src/xml_lexer.ml"

  | 3 ->
# 381 "src/xml_lexer.mll"
  (
			Buffer.add_char tmp (lexeme_char lexbuf 0);
			q_string lexbuf
		)
# 1050 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_q_string_rec lexbuf __ocaml_lex_state

and dtd_data lexbuf =
   __ocaml_lex_dtd_data_rec lexbuf 98
and __ocaml_lex_dtd_data_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 388 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			(* skipping Public ID *)
			let _ = dtd_file lexbuf in
			let file = dtd_file lexbuf in
			dtd_end_decl lexbuf;
			DTDFile file
		)
# 1069 "src/xml_lexer.ml"

  | 1 ->
# 397 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			let file = dtd_file lexbuf in
			dtd_end_decl lexbuf;
			DTDFile file
		)
# 1079 "src/xml_lexer.ml"

  | 2 ->
# 404 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			let data = dtd_intern lexbuf in
			dtd_end_decl lexbuf;
			DTDData data
		)
# 1089 "src/xml_lexer.ml"

  | 3 ->
# 411 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDDecl )
# 1094 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_data_rec lexbuf __ocaml_lex_state

and dtd_file lexbuf =
   __ocaml_lex_dtd_file_rec lexbuf 113
and __ocaml_lex_dtd_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 415 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let s = dq_string lexbuf in
			ignore_spaces lexbuf;
			s
		)
# 1111 "src/xml_lexer.ml"

  | 1 ->
# 422 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let s = q_string lexbuf in
			ignore_spaces lexbuf;
			s
		)
# 1121 "src/xml_lexer.ml"

  | 2 ->
# 429 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDDecl )
# 1126 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_file_rec lexbuf __ocaml_lex_state

and dtd_intern lexbuf =
   __ocaml_lex_dtd_intern_rec lexbuf 117
and __ocaml_lex_dtd_intern_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 433 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			[]
		)
# 1141 "src/xml_lexer.ml"

  | 1 ->
# 438 "src/xml_lexer.mll"
  (
			let l = dtd_item lexbuf in
			l @ (dtd_intern lexbuf)
		)
# 1149 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_intern_rec lexbuf __ocaml_lex_state

and dtd lexbuf =
   __ocaml_lex_dtd_rec lexbuf 119
and __ocaml_lex_dtd_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 445 "src/xml_lexer.mll"
  ( [] )
# 1161 "src/xml_lexer.ml"

  | 1 ->
# 447 "src/xml_lexer.mll"
  (
			newline lexbuf;
			dtd lexbuf
		)
# 1169 "src/xml_lexer.ml"

  | 2 ->
# 452 "src/xml_lexer.mll"
  ( dtd lexbuf )
# 1174 "src/xml_lexer.ml"

  | 3 ->
# 454 "src/xml_lexer.mll"
  (
			let l = dtd_item lexbuf in
			l @ (dtd lexbuf)
		)
# 1182 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_rec lexbuf __ocaml_lex_state

and dtd_end_decl lexbuf =
   __ocaml_lex_dtd_end_decl_rec lexbuf 123
and __ocaml_lex_dtd_end_decl_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 461 "src/xml_lexer.mll"
  ( ignore_spaces lexbuf )
# 1194 "src/xml_lexer.ml"

  | 1 ->
# 463 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDDecl )
# 1199 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_end_decl_rec lexbuf __ocaml_lex_state

and dtd_item lexbuf =
   __ocaml_lex_dtd_item_rec lexbuf 126
and __ocaml_lex_dtd_item_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 467 "src/xml_lexer.mll"
  (
			comment lexbuf;
			[];
		)
# 1214 "src/xml_lexer.ml"

  | 1 ->
# 472 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			let t = dtd_item_type lexbuf in
			let name = (try ident_name lexbuf with Error EIdentExpected -> raise (DTDError EInvalidDTDDecl)) in
			ignore_spaces lexbuf;
			match t with
			| TElement -> [ DTDElement (name , (dtd_element_type lexbuf)) ]
			| TAttribute -> List.map (fun (attrname,atype,adef) -> DTDAttribute (name, attrname, atype, adef)) (dtd_attributes lexbuf)
		)
# 1227 "src/xml_lexer.ml"

  | 2 ->
# 482 "src/xml_lexer.mll"
  (  dtd_error lexbuf EDTDItemExpected )
# 1232 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_item_rec lexbuf __ocaml_lex_state

and dtd_attributes lexbuf =
   __ocaml_lex_dtd_attributes_rec lexbuf 132
and __ocaml_lex_dtd_attributes_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 486 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			[]
		)
# 1247 "src/xml_lexer.ml"

  | 1 ->
# 491 "src/xml_lexer.mll"
  (
			let attrname = (try ident_name lexbuf with Error EIdentExpected -> raise (DTDError EInvalidDTDAttribute)) in
			ignore_spaces lexbuf;
			let atype = dtd_attr_type lexbuf in
			let adef = dtd_attr_default lexbuf in
			let a = (attrname, atype, adef) in
			a :: (dtd_attributes lexbuf)
		)
# 1259 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attributes_rec lexbuf __ocaml_lex_state

and dtd_item_type lexbuf =
   __ocaml_lex_dtd_item_type_rec lexbuf 134
and __ocaml_lex_dtd_item_type_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 502 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			TElement
		)
# 1274 "src/xml_lexer.ml"

  | 1 ->
# 507 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			TAttribute
		)
# 1282 "src/xml_lexer.ml"

  | 2 ->
# 512 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDTag )
# 1287 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_item_type_rec lexbuf __ocaml_lex_state

and dtd_element_type lexbuf =
   __ocaml_lex_dtd_element_type_rec lexbuf 150
and __ocaml_lex_dtd_element_type_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 516 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			dtd_end_element lexbuf;
			DTDAny
		)
# 1303 "src/xml_lexer.ml"

  | 1 ->
# 522 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			dtd_end_element lexbuf;
			DTDEmpty
		)
# 1312 "src/xml_lexer.ml"

  | 2 ->
# 528 "src/xml_lexer.mll"
  (
			try
				let item = Xml_parser.dtd_element dtd_element_token lexbuf in
				ignore_spaces lexbuf;
				DTDChild item
			with
				Parsing.Parse_error -> dtd_error lexbuf EInvalidDTDElement
		)
# 1324 "src/xml_lexer.ml"

  | 3 ->
# 537 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDElement )
# 1329 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_element_type_rec lexbuf __ocaml_lex_state

and dtd_end_element lexbuf =
   __ocaml_lex_dtd_end_element_rec lexbuf 161
and __ocaml_lex_dtd_end_element_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 541 "src/xml_lexer.mll"
  ( ignore_spaces lexbuf )
# 1341 "src/xml_lexer.ml"

  | 1 ->
# 543 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDElement )
# 1346 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_end_element_rec lexbuf __ocaml_lex_state

and dtd_end_attribute lexbuf =
   __ocaml_lex_dtd_end_attribute_rec lexbuf 164
and __ocaml_lex_dtd_end_attribute_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 547 "src/xml_lexer.mll"
  ( ignore_spaces lexbuf )
# 1358 "src/xml_lexer.ml"

  | 1 ->
# 549 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1363 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_end_attribute_rec lexbuf __ocaml_lex_state

and dtd_element_token lexbuf =
   __ocaml_lex_dtd_element_token_rec lexbuf 167
and __ocaml_lex_dtd_element_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 553 "src/xml_lexer.mll"
  (
			newline lexbuf;
			dtd_element_token lexbuf
		)
# 1378 "src/xml_lexer.ml"

  | 1 ->
# 558 "src/xml_lexer.mll"
  ( dtd_element_token lexbuf )
# 1383 "src/xml_lexer.ml"

  | 2 ->
# 560 "src/xml_lexer.mll"
  ( OPEN )
# 1388 "src/xml_lexer.ml"

  | 3 ->
# 562 "src/xml_lexer.mll"
  ( CLOSE )
# 1393 "src/xml_lexer.ml"

  | 4 ->
# 564 "src/xml_lexer.mll"
  ( NEXT )
# 1398 "src/xml_lexer.ml"

  | 5 ->
# 566 "src/xml_lexer.mll"
  ( END )
# 1403 "src/xml_lexer.ml"

  | 6 ->
# 568 "src/xml_lexer.mll"
  ( OR )
# 1408 "src/xml_lexer.ml"

  | 7 ->
# 570 "src/xml_lexer.mll"
  ( PCDATA )
# 1413 "src/xml_lexer.ml"

  | 8 ->
# 572 "src/xml_lexer.mll"
  ( STAR )
# 1418 "src/xml_lexer.ml"

  | 9 ->
# 574 "src/xml_lexer.mll"
  ( PLUS )
# 1423 "src/xml_lexer.ml"

  | 10 ->
# 576 "src/xml_lexer.mll"
  ( QUESTION )
# 1428 "src/xml_lexer.ml"

  | 11 ->
# 578 "src/xml_lexer.mll"
  ( IDENT (lexeme lexbuf) )
# 1433 "src/xml_lexer.ml"

  | 12 ->
# 580 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDElement )
# 1438 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_element_token_rec lexbuf __ocaml_lex_state

and dtd_attr_type lexbuf =
   __ocaml_lex_dtd_attr_type_rec lexbuf 187
and __ocaml_lex_dtd_attr_type_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 584 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDCData
		)
# 1453 "src/xml_lexer.ml"

  | 1 ->
# 589 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDNMToken
		)
# 1461 "src/xml_lexer.ml"

  | 2 ->
# 594 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
		 	DTDID
		)
# 1469 "src/xml_lexer.ml"

  | 3 ->
# 599 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDIDRef
		)
# 1477 "src/xml_lexer.ml"

  | 4 ->
# 604 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDEnum (dtd_attr_enum lexbuf)
		)
# 1485 "src/xml_lexer.ml"

  | 5 ->
# 609 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1490 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attr_type_rec lexbuf __ocaml_lex_state

and dtd_attr_enum lexbuf =
   __ocaml_lex_dtd_attr_enum_rec lexbuf 207
and __ocaml_lex_dtd_attr_enum_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 613 "src/xml_lexer.mll"
  (
			let v = lexeme lexbuf in
			ignore_spaces lexbuf;
			v :: (dtd_attr_enum_next lexbuf)
		)
# 1506 "src/xml_lexer.ml"

  | 1 ->
# 619 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1511 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attr_enum_rec lexbuf __ocaml_lex_state

and dtd_attr_enum_next lexbuf =
   __ocaml_lex_dtd_attr_enum_next_rec lexbuf 210
and __ocaml_lex_dtd_attr_enum_next_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 623 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			[]
		)
# 1526 "src/xml_lexer.ml"

  | 1 ->
# 628 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			dtd_attr_enum lexbuf
		)
# 1534 "src/xml_lexer.ml"

  | 2 ->
# 633 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1539 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attr_enum_next_rec lexbuf __ocaml_lex_state

and dtd_attr_default lexbuf =
   __ocaml_lex_dtd_attr_default_rec lexbuf 214
and __ocaml_lex_dtd_attr_default_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 637 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let v = (try dq_string lexbuf with Error EUnterminatedString -> raise (DTDError EInvalidDTDAttribute)) in
			ignore_spaces lexbuf;
			DTDDefault v
		)
# 1556 "src/xml_lexer.ml"

  | 1 ->
# 644 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let v = (try q_string lexbuf with Error EUnterminatedString -> raise (DTDError EInvalidDTDAttribute)) in
			ignore_spaces lexbuf;
			DTDDefault v
		)
# 1566 "src/xml_lexer.ml"

  | 2 ->
# 651 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDRequired
		)
# 1574 "src/xml_lexer.ml"

  | 3 ->
# 656 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDImplied
		)
# 1582 "src/xml_lexer.ml"

  | 4 ->
# 661 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDFixed (dtd_attr_string lexbuf)
		)
# 1590 "src/xml_lexer.ml"

  | 5 ->
# 666 "src/xml_lexer.mll"
  (
			ignore_spaces lexbuf;
			DTDDefault (dtd_attr_string lexbuf)
		)
# 1598 "src/xml_lexer.ml"

  | 6 ->
# 671 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1603 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attr_default_rec lexbuf __ocaml_lex_state

and dtd_attr_string lexbuf =
   __ocaml_lex_dtd_attr_string_rec lexbuf 246
and __ocaml_lex_dtd_attr_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 675 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let v = (try dq_string lexbuf with Error EUnterminatedString -> raise (DTDError EInvalidDTDAttribute)) in
			ignore_spaces lexbuf;
			v
		)
# 1620 "src/xml_lexer.ml"

  | 1 ->
# 682 "src/xml_lexer.mll"
  (
			Buffer.reset tmp;
			let v = (try q_string lexbuf with Error EUnterminatedString -> raise (DTDError EInvalidDTDAttribute)) in
			ignore_spaces lexbuf;
			v
		)
# 1630 "src/xml_lexer.ml"

  | 2 ->
# 689 "src/xml_lexer.mll"
  ( dtd_error lexbuf EInvalidDTDAttribute )
# 1635 "src/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dtd_attr_string_rec lexbuf __ocaml_lex_state

;;

