/* Copyright (C) 2012 and 2013 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

// to suppress deprecated warnings with GStaticRecMutex
#define GLIB_VERSION_MIN_REQUIRED GLIB_VERSION_2_30

#include <unistd.h>
#include <glib.h>
#include <errno.h>
#include <pthread.h>
#include <utility>    // for std::move

#include <c++-gtk-utils/thread.h>
#include <c++-gtk-utils/mutex.h>
#include <c++-gtk-utils/rw_lock.h>
#include <c++-gtk-utils/cgu_config.h>

using namespace Cgu;

Thread::Cond cond;
Thread::Mutex mutex;
Thread::RWLock rw_lock;
GStaticRecMutex grm = G_STATIC_REC_MUTEX_INIT;
int counter = 0;

#ifdef CGU_HAVE_RECURSIVE_MUTEX
Thread::RecMutex rec_mutex;
#endif

extern "C" {
static void test_thread_cond() {

  struct ThreadFunc {
    static void func() {
      Thread::Mutex::TrackLock lock{mutex};
      while (!counter) cond.wait(lock);
      g_assert_cmpint(counter, ==, 1);
      counter = 0;
      lock.unlock();
    }
  };

  std::unique_ptr<Thread::Thread> thread;
  {
    thread = Thread::Thread::start(Callback::make(&ThreadFunc::func), true);
    g_assert(thread.get() != 0);
    Thread::Mutex::Lock lock{mutex};
    counter = 1;
    cond.signal();
  }
  thread->join();
  Thread::Mutex::Lock lock{mutex};
  g_assert_cmpint(counter, ==, 0);

  timespec ts;
  Cgu::Thread::Cond::get_abs_time(ts, 50);
  g_assert_cmpint(cond.timed_wait(lock, ts), ==, ETIMEDOUT);
}

static void test_thread_mutex() {

  {
    g_assert_cmpint(mutex.lock(), ==, 0);
    Thread::Mutex::Lock l{mutex, Thread::locked};
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.lock(), ==, 0);
  }

  {
    Thread::Mutex::Lock l{mutex};
    g_assert_cmpint(l.trylock(), ==, EBUSY);
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }

  {
    g_assert_cmpint(mutex.lock(), ==, 0);
    Thread::Mutex::TrackLock l{mutex, Thread::locked};
    g_assert(l.is_owner());
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert(!l.is_owner());
  }

  {
    Thread::Mutex::TrackLock l{mutex, Thread::defer};
    g_assert(!l.is_owner());
    g_assert_cmpint(l.lock(), ==, 0);
    g_assert(l.is_owner());
  }

  {
    Thread::Mutex::TrackLock l{mutex};
    g_assert_cmpint(l.trylock(), ==, EBUSY);
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }

  {
    Thread::GrecmutexLock grec_mutex1{grm};
    g_static_rec_mutex_lock(grec_mutex1.get());
    Thread::GrecmutexLock grec_mutex2{grm, Thread::locked};
  }

#ifdef CGU_HAVE_RECURSIVE_MUTEX
  {
    Thread::RecMutex::Lock l1{rec_mutex};
    g_assert_cmpint(l1.lock(), ==, 0);
    Thread::RecMutex::Lock l2{rec_mutex, Thread::locked};
  }

  {
    Thread::RecMutex::Lock l{rec_mutex};
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }

  {
    g_assert_cmpint(rec_mutex.lock(), ==, 0);
    Thread::RecMutex::TrackLock l{rec_mutex, Thread::locked};
    g_assert(l.is_owner());
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert(!l.is_owner());
  }

  {
    Thread::RecMutex::TrackLock l{rec_mutex, Thread::defer};
    g_assert(!l.is_owner());
    g_assert_cmpint(l.lock(), ==, 0);
    g_assert(l.is_owner());
  }

  {
    Thread::RecMutex::TrackLock l{rec_mutex};
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }
#endif
}

static void test_thread_rw_lock() {
  {
    Thread::RWLock::ReaderLock l1{rw_lock};
    g_assert_cmpint(rw_lock.reader_lock(), ==, 0);
    Thread::RWLock::ReaderLock l2{rw_lock, Thread::locked};
    g_assert_cmpint(l1.unlock(), ==, 0);
    g_assert_cmpint(l1.lock(), ==, 0);
  }

  {
    g_assert_cmpint(rw_lock.writer_lock(), ==, 0);
    Thread::RWLock::WriterLock l{rw_lock, Thread::locked};
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.lock(), ==, 0);
  }

  {
    Thread::RWLock::ReaderLock l{rw_lock};
    g_assert_cmpint(rw_lock.writer_trylock(), ==, EBUSY);
    g_assert_cmpint(rw_lock.reader_trylock(), ==, 0);
    g_assert_cmpint(rw_lock.unlock(), ==, 0);
  }

  {
    Thread::RWLock::WriterLock l{rw_lock};
    g_assert_cmpint(rw_lock.reader_trylock(), ==, EBUSY);
    g_assert_cmpint(l.trylock(), ==, EBUSY);
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }

  {
    Thread::RWLock::ReaderLock l1{rw_lock};
    g_assert_cmpint(rw_lock.reader_lock(), ==, 0);
    Thread::RWLock::ReaderTrackLock l2{rw_lock, Thread::locked};
    g_assert(l2.is_owner());
    g_assert_cmpint(l2.unlock(), ==, 0);
    g_assert(!l2.is_owner());
  }

  {
    g_assert_cmpint(rw_lock.writer_lock(), ==, 0);
    Thread::RWLock::WriterTrackLock l{rw_lock, Thread::locked};
    g_assert(l.is_owner());
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert(!l.is_owner());
  }

  {
    Thread::RWLock::ReaderTrackLock l{rw_lock, Thread::defer};
    g_assert(!l.is_owner());
    g_assert_cmpint(l.lock(), ==, 0);
    g_assert(l.is_owner());
  }

  {
    Thread::RWLock::WriterTrackLock l{rw_lock, Thread::defer};
    g_assert(!l.is_owner());
    g_assert_cmpint(l.lock(), ==, 0);
    g_assert(l.is_owner());
  }

  {
    Thread::RWLock::ReaderTrackLock l{rw_lock};
    g_assert_cmpint(rw_lock.writer_trylock(), ==, EBUSY);
    g_assert_cmpint(rw_lock.reader_trylock(), ==, 0);
    g_assert_cmpint(rw_lock.unlock(), ==, 0);
    g_assert(l.is_owner());
  }

  {
    Thread::RWLock::WriterTrackLock l{rw_lock};
    g_assert_cmpint(rw_lock.reader_trylock(), ==, EBUSY);
    g_assert_cmpint(l.trylock(), ==, EBUSY);
    g_assert_cmpint(l.unlock(), ==, 0);
    g_assert_cmpint(l.trylock(), ==, 0);
  }
}

static void test_thread_exit() {
  auto thread_func = [] () {
    counter = 1;
    g_usleep(10000);
    throw Thread::Exit();
  };

  counter = 0;
  Thread::JoinableHandle t{
    Thread::Thread::start(thread_func, true),
    Thread::JoinableHandle::detach_on_exit};
  g_assert(t.is_managing());
  t.join(); // synchronises memory
  g_assert_cmpint(counter, ==, 1);
}

static void test_thread_cancel() {
  auto thread_func = [] () {
    {
      Thread::CancelBlock b;
      g_usleep(200000);
      pthread_testcancel();
      counter = 1;
    }
    sleep(10); // SUS, System Interfaces, Section 2.9.5
               // specifies sleep() as a cancellation point
    g_assert_not_reached();
  };

  counter = 0;
  Thread::JoinableHandle t{
    Thread::Thread::start(std::move(thread_func), true),
    Thread::JoinableHandle::detach_on_exit
  };
  g_assert(t.is_managing());
  t.cancel();
  t.join(); // synchronises memory
  g_assert_cmpint(counter, ==, 1);
}

static void test_thread_joinable_move() {
  struct ThreadFunc {
    static void func() {
      counter = 1;
    }
  };

  {
    counter = 0;
    Thread::JoinableHandle t1{
      Thread::Thread::start([] () {counter = 1;}, true),
      Thread::JoinableHandle::join_on_exit
    };
    Thread::JoinableHandle t2{std::move(t1)};
    g_assert(!t1.is_managing());
    g_assert(t2.is_managing());
    t2.join(); // synchronises memory
    g_assert(!t2.is_managing());
    g_assert_cmpint(counter, ==, 1);
  }
  {
    counter = 0;
    Thread::JoinableHandle t1{
      Thread::Thread::start(Callback::make(&ThreadFunc::func), true),
      Thread::JoinableHandle::join_on_exit
    };
    Thread::JoinableHandle t2;
    t2 = std::move(t1);
    g_assert(!t1.is_managing());
    g_assert(t2.is_managing());
    t2.join(); // synchronises memory
    g_assert(!t2.is_managing());
    g_assert_cmpint(counter, ==, 1);
  }
}

} // extern "C"


int main (int argc, char* argv[]) {
#if !(GLIB_CHECK_VERSION(2,32,0))
  g_thread_init(0);
#endif
  g_test_init(&argc, &argv, static_cast<void*>(0));

  g_test_add_func("/thread/cond", test_thread_cond); 
  g_test_add_func("/thread/mutex", test_thread_mutex); 
  g_test_add_func("/thread/rw_lock", test_thread_rw_lock); 
  g_test_add_func("/thread/exit", test_thread_exit); 
  g_test_add_func("/thread/cancel", test_thread_cancel); 
  g_test_add_func("/thread/joinable_move", test_thread_joinable_move);

  return g_test_run();
}
