#!/usr/local/bin/perl -w

use strict;
use File::Basename "basename";
use Getopt::Long "GetOptions";

my $LINES = 0;

my %options;
die "GetOptions error" unless (GetOptions(\%options,
    "dump=s", "force", "help", "list", "map=s"));

Help() if ($options{help});
Help() unless ($options{dump} && $options{map});

CheckARGV();

my %FILES;

unless ($options{list}) {
    RestoreNFilter();
} else {
}

sub RestoreNFilter {
    ProcessFileNames();

    my %keys;
    my @lines;

    my $n = 0;
    open(my $ncbi_in, $options{ncbi_in})
        or die "cannot open '$options{ncbi_in}'";
    # read alignments from sam-dump into (@lines, %keys)
    while (<$ncbi_in>) {
    #    print;
        last if ($LINES && ++$n > $LINES);
        if (/^@/) {
            Print('ncbi_out', $_);
            next;
        }
        my %fld = ParseLine($_);
        my $key = Key(%fld);
        push @lines, $_;
        unless ($keys{$key}) { # mmap
            $keys{$key} = [ $#lines ];
        } else {
            push @{ $keys{$key} }, $#lines;
        }
    }
    close $ncbi_in;

    $n = 0;
    open(my $cg_in, $options{cg_in}) or die "cannot open '$options{cg_in}'";
    # read alignments from map2sam
    # if (line is found in sam-dump): write to *.filtered
    # else write to <map2sam>.filtered.nomatch
    while (<$cg_in>) { #    print;
        last if ($LINES && ++$n > $LINES);
        if (/^@/) {
            Print('cg_out', $_);
            next;
        }
        my %fld = ParseLine($_);
        my @line = GenerateRaw($_, %fld);
        my $key = Key(%fld);
        if (defined $keys{$key}) {
            die $key if ($#{ $keys{$key} } < 0); # mmap
            for (my $i = 0; $i <= $#line; ++$i) {
                Print('cg_out', "\t") if ($i);
                Print('cg_out', $line[$i]);
            }
            Print('cg_out', "\n");
            Print('ncbi_out', $lines[$_]) foreach (@{ $keys{$key} }); { # mmap
            # TODO: print to draft and merge with '.filtered' at the end?
            delete $keys{$key}; # = []; # mmap
        } else {
            for (my $i = 0; $i <= $#line; ++$i) {
                Print('cg_out_bad', "\t") if ($i);
                Print('cg_out_bad', $line[$i]);
            }
            Print('cg_out_bad', "\n");
        }
    }
    close $cg_in;

    # write sam-dump lines that are not found in map2sam
    # into <sam-dump>.filtered.nomatch
    my %lines;
    foreach my $key (keys %keys) {
        foreach (@{ $keys{$key} }) { # mmap
            $lines{$_} = $key;
        }
    }
    foreach (sort keys %lines) {
        my $key = $lines{$_};
        die "$_ $key" unless (defined $keys{$key});
        die "$_ $key" if ($#{ $keys{$key} } < 0); # mmap
        foreach my $line (@{ $keys{$key} }) { # mmap
            die "$_ $key" if ($line < 0 || $line > $#lines); 
            Print('ncbi_out_bad', $lines[$line]);
        }
    }

    Close();
}

sub Key {
    my (%fld) = (@_);
    my $key = "$fld{RNAME}_$fld{POS}";
    return $key;
    if ($fld{CIGAR} eq '*') {
        $key .= '_*';
    } elsif ($fld{FLAG} & 0x100) {  # secondary alignment
        $key .= '_s';
    }
    return $key;
}

sub ParseLine {
    ($_) = @_;
    my %fld;
    my @optional;
    ($fld{QNAME}, $fld{FLAG}, $fld{RNAME}, $fld{POS}, $fld{MAPQ}, $fld{CIGAR},
        $fld{RNEXT}, $fld{PNEXT}, $fld{TLEN}, $fld{SEQ}, $fld{QUAL}, @optional)
      = split /\t/;
    foreach (@optional) {
        die "Bad optinal alignment line field: '$_'"
            unless (/^([A-Za-z][A-Za-z0-9]):([AifZHB]):(.+)/);
        my ($TAG, $TYPE, $VALUE) = ($1, $2, $3);
        if ($TAG eq 'RG' || $TAG eq 'GS' || $TAG eq 'GC' || $TAG eq 'GQ') {
            die "'$_': Unexpected type '$TYPE' of optional field tag '$TAG'"
                unless ($TYPE eq 'Z');
            $fld{$TAG} = $VALUE;
        } elsif ($TAG eq 'NM') {
            die "'$_': Unexpected type '$TYPE' of optional field tag '$TAG'"
                unless ($TYPE eq 'i');
            $fld{$TAG} = $VALUE;
        } else {
            chomp;
            die "'$_': Unexpected optinal alignment line field tag '$TAG'";
        }
    }
    return %fld;
}

################################################################################
=begin COMMENT
SAM Format Specification 0.1.2-draft (20090820)
http://www.projet-plume.org/files/SAM1.pdf
SAM Format Specification 0.1.2-draft (20090820)
2. SAM Format Specification
2.2. Alignment Section
2.2.4. Format of optional fields
Note 6: Some bases from reads generated by Complete Genomics
may come from the same nucleotide.
The SEQ and QUAL fields always store the flattened sequence and quality
in that bases and qualities from the same nucleotide are collapsed to one.
The three optional tags GS/GQ/GC describes how to generate the raw read.
For example, given a raw read AAACGCGAAAA,
?CG? starting from 4th and 6th position come from the same oligonucleotide.
Suppose this read is mapped without gaps.
In SAM, the read alignment is stored as:
SEQ=AAACGAAAA, CIGAR=9M, GS:Z:CGCG, and GC:Z:3S2G4S,
where GS keeps the bases in the overlap
and GC says that to get the raw read sequence, we need to copy 3 bases from SEQ,
copy 2+2 bases from GS and then copy 4 bases from the SEQ field again.
#end COMMENT
=cut

sub GenerateRaw {
    my ($line, %fld) = @_;
    my ($SEQ, $QUAL) = ('', '');
    if ($fld{GC} || $fld{GS} || $fld{GQ}) {
        die "Bad alignment line '$line'"
            unless ($fld{SEQ} && $fld{QUAL}
                  && $fld{GC} && $fld{GS} && $fld{GQ});
        die "Bad alignment line:\nSEQ ='$fld{SEQ}'\nQUAL=$fld{QUAL}"
            unless (length($fld{SEQ}) == length($fld{QUAL}));
        die "Bad tags:\nGS='$fld{GS}'\nGQ='fld{GQ}'"
            unless (length($fld{GS}) == length($fld{GQ}));

        my ($n, $used_SEQ, $used_G) = (0, 0, 0);
        for (my $i = 0; $i < length($fld{GC}); ++$i) {
            my $c = substr($fld{GC}, $i, 1);
            if ($c =~ /\d/) {
                $n = $n * 10 + $c;
            } elsif ($c eq 'S') {
                $SEQ .= substr($fld{SEQ} , $used_SEQ, $n);
                $QUAL.= substr($fld{QUAL}, $used_SEQ, $n);
                $used_SEQ += $n;
                $n = 0;
            } elsif ($c eq 'G') {
                $SEQ .= substr($fld{GS}, $used_G, $n * 2);
                $QUAL.= substr($fld{GQ}, $used_G, $n * 2);
                $used_G += 2 * $n;
                $used_SEQ += $n;
                $n = 0;
            } else {
                die "'$fld{GC}': unexpected characted '$c' in GC";
            }
        }

        die "Bad GenerateRaw: $used_G: '$fld{GS}'; $used_SEQ: '$fld{SEQ}'"
            unless (length($fld{SEQ}) == $used_SEQ
                  && length($fld{GS}) == $used_G);
    } else {
        ($SEQ, $QUAL) = ($fld{SEQ}, $fld{QUAL});
    }

    my @line = ($fld{QNAME}, $fld{FLAG}, $fld{RNAME}, $fld{POS}, $fld{MAPQ},
        $fld{CIGAR}, $fld{RNEXT}, $fld{PNEXT}, $fld{TLEN}, $SEQ, $QUAL);
    push @line, "RG:Z:$fld{RG}" if ($fld{RG});

    return @line;
}

sub CheckARGV {
    Help() unless ($options{dump} && $options{map});
}

sub ProcessFileNames {
    die "bad options" if ($options{ncbi_in} || $options{ncbi_out}
        || $options{cg_in} || $options{cg_out});
    die "bad in options" unless ($options{dump} && $options{map});
    $options{ncbi_in} = $options{dump};
    $options{cg_in} = $options{map};
    $options{ncbi_out} = "$options{dump}.filtered";
    $options{cg_out}   =  "$options{map}.filtered";
    $options{ncbi_out_bad} = "$options{dump}.filtered.nomatch";
    $options{cg_out_bad}   =  "$options{map}.filtered.nomatch";
    my $found;
    foreach ($options{ncbi_in}, $options{cg_in}) {
        unless (-e $_) {
            ++$found;
            print "$_: not found\n";
        }
    }
    if ($found) {
        print "\n";
        Help();
    }
    foreach ($options{ncbi_out}, $options{cg_out},
        $options{ncbi_out_bad}, $options{cg_out_bad})
    {
        if (-e $_) {
            unless ($options{force}) {
                ++$found;
                print "$_ exists\n";
            } else {
                unlink or die "cannot unlink '$_'";
            }
        }
    }
    if ($found) {
        print "\n";
        Help();
    }
}

sub Close {
    foreach (keys %FILES) {
        close $FILES{$_};
    }
}

sub Print {
    my ($key, $line) = @_;
    die $key unless ($options{$key});
    my $name = $options{$key};
    unless ($FILES{$key}) {
        open(my $fh, ">$name") or die "cannot open $name";
        $FILES{$key} = $fh;
    }
    print { $FILES{$key} } $line or die "cannot print to $name";
}

sub Help {
    my $name = basename($0);
    print STDERR <<EndText;
Usage:
    $name --dump <sam-dump-file> --map <map2sam-dump-file> [--list] [--force]
        where
            <sam-dump-file>     is the result of "sam-dump SRR"
            <map2sam-dump-file> is the result of cgatools map2sam
        Use --force to overwrite existing output files.
        When --list is present:
            <sam-dump-file> and <map2sam-dump-file>
            present suffixes for list of file couples. See example below.
        Input files: <sam-dump-file>
                     <map2sam-dump-file>
        Output files:
            <sam-dump-file>.filtered :
                <sam-dump-file> filtered and reordered
                that its lines match to <map2sam-dump-file>.filtered
            <map2sam-dump-file>.filtered : <map2sam-dump-file> filtered
                that its lines match to <sam-dump-file>.filtered
            <sam-dump-file>.filtered.nomatch :
                contains the part of lines of <sam-dump-file>
                that do not match to any line of <map2sam-dump-file>
            <map2sam-dump-file>.filtered.nomatch :
                contains the part of lines of <map2sam-dump-file>
                that do not match to any line of <sam-dump-file>
    
    $name -h
    $name --helo
                    print this help

Examples:

to convert a couple of files:
\$ $name --dump SRZ-tiny.sam.chr17 --map 100.sam.chr17

to convert a list of files:
\$ $name --dump SRZ-tiny.sam. --map 100.sam. --list
The previous command will convert
all couples of SRZ-tiny.sam.* and 100.sam.* matched by suffix
except the ones having suffix '.filtered' or '.filtered.nomatch'
EndText

    exit 1;
}
