#!/usr/local/bin/perl -w

use strict;
use Getopt::Long "GetOptions";

my %options;
die "GetOptions error" unless (GetOptions(\%options, "help"));

if ($options{help}) {
    print STDERR <<EndText;
Convert map2sam output with flattened sequence and quality
(i.e. with removed overlaps) to the raw reads.
The result goes to stdout.

Usage:
    cgatools map2sam <options> | $0
        or
    $0 <map2sam output file>
    
    $0 -h
    $0 -h
        print this help
EndText

    exit 0;
}

while (<>) {
#   print "$_\n";
    next if (/$@/);
    my @f = split /\t/;
    my %fld;
    my @optional;
    ($fld{QNAME}, $fld{FLAG}, $fld{RNAME}, $fld{POS}, $fld{MAPQ}, $fld{CIGAR},
        $fld{RNEXT}, $fld{PNEXT}, $fld{TLEN}, $fld{SEQ}, $fld{QUAL}, @optional)
      = @f;
    foreach (@optional) {
        die "Bad optinal alignment line field: '$_'"
            unless (/^([A-Za-z][A-Za-z0-9]):([AifZHB]):(.+)/);
        my ($TAG, $TYPE, $VALUE) = ($1, $2, $3);
        if ($TAG eq 'RG' || $TAG eq 'GS' || $TAG eq 'GC' || $TAG eq 'GQ') {
            die "'$_': Unexpected type '$TYPE' of optional field tag '$TAG'"
                unless ($TYPE eq 'Z');
            $fld{$TAG} = $VALUE;
        } else { die "'$_': Unexpected optinal alignment line field tag"; }
    }
#print "$fld{SEQ}\n$fld{QUAL}\nGC='$fld{GC}'\nGS='$fld{GS}'\nGQ = '$fld{GQ}'\n";
    my @line = GenerateRaw(%fld);
    for (my $i = 0; $i <= $#line; ++$i) {
        print "\t" if ($i);
        print $line[$i];
    }
    print "\n";
}

################################################################################
=begin COMMENT
SAM Format Specification 0.1.2-draft (20090820)
http://www.projet-plume.org/files/SAM1.pdf
SAM Format Specification 0.1.2-draft (20090820)
2. SAM Format Specification
2.2. Alignment Section
2.2.4. Format of optional fields
Note 6: Some bases from reads generated by Complete Genomics
may come from the same nucleotide.
The SEQ and QUAL fields always store the flattened sequence and quality
in that bases and qualities from the same nucleotide are collapsed to one.
The three optional tags GS/GQ/GC describes how to generate the raw read.
For example, given a raw read AAACGCGAAAA,
?CG? starting from 4th and 6th position come from the same oligonucleotide.
Suppose this read is mapped without gaps.
In SAM, the read alignment is stored as:
SEQ=AAACGAAAA, CIGAR=9M, GS:Z:CGCG, and GC:Z:3S2G4S,
where GS keeps the bases in the overlap
and GC says that to get the raw read sequence, we need to copy 3 bases from SEQ,
copy 2+2 bases from GS and then copy 4 bases from the SEQ field again.
#end COMMENT
=cut

sub GenerateRaw {
    my (%fld) = @_;
    die "Bad alignment line"
        unless ($fld{SEQ} && $fld{QUAL} && $fld{GC} && $fld{GS} && $fld{GQ});
    die "Bad alignment line:\nSEQ ='$fld{SEQ}'\nQUAL=$fld{QUAL}"
        unless (length($fld{SEQ}) == length($fld{QUAL}));
    die "Bad tags:\nGS='$fld{GS}'\nGQ='fld{GQ}'"
        unless (length($fld{GS}) == length($fld{GQ}));

    my ($SEQ, $QUAL) = ('', '');
    my ($n, $used_SEQ, $used_G) = (0, 0, 0);
    for (my $i = 0; $i < length($fld{GC}); ++$i) {
        my $c = substr($fld{GC}, $i, 1);
        if ($c =~ /\d/) {
            $n = $n * 10 + $c;
        } elsif ($c eq 'S') {
            $SEQ .= substr($fld{SEQ} , $used_SEQ, $n);
            $QUAL.= substr($fld{QUAL}, $used_SEQ, $n);
            $used_SEQ += $n;
            $n = 0;
        } elsif ($c eq 'G') {
            $SEQ .= substr($fld{GS}, $used_G, $n * 2);
            $QUAL.= substr($fld{GQ}, $used_G, $n * 2);
            $used_G += 2 * $n;
            $used_SEQ += $n;
            $n = 0;
        } else {
            die "'$fld{GC}': unexpected characted '$c' in GC";
        }
    }

    die "Bad GenerateRaw: $used_G: '$fld{GS}'; $used_SEQ: '$fld{SEQ}'"
        unless (length($fld{SEQ}) == $used_SEQ && length($fld{GS}) == $used_G);

    my @line = ($fld{QNAME}, $fld{FLAG}, $fld{RNAME}, $fld{POS}, $fld{MAPQ},
        $fld{CIGAR}, $fld{RNEXT}, $fld{PNEXT}, $fld{TLEN}, $SEQ, $QUAL);
    push @line, "RG:Z:$fld{RG}" if ($fld{RG});

    return @line;
}
