/* 
pwavecat Jan Panteltje's wave concatenation program copyright 2004-always Jan panteltje
email: panteltje@yahoo.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pwavecat.h"


struct part *lookup_part(char *name)
{
struct part *pa;

/* pa points to next entry */
for(pa = parttab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(strcmp(pa -> name, name) == 0) return pa;
	}

return 0; /* not found */
} /* end function lookup_part */


struct part *install_part_at_end_of_list(char *name)
{
struct part *plast, *pnew;
struct part *lookup_part();

if(debug_flag) fprintf(stderr, "install_part(): arg name=%s\n", name);

/* create new structure */
pnew = (struct part *) calloc(1, sizeof(*pnew) );
if(! pnew) return 0;
pnew -> name = strsave(name);
if(! pnew -> name) return 0;

/* get previous structure */
plast = parttab[1]; /* end list */

/* set new structure pointers */
pnew -> nxtentr = 0; /* new points top zero (is end) */
pnew -> prventr = plast; /* point to previous entry, or 0 if first entry */

/* set previuos structure pointers */
if( !parttab[0] ) parttab[0] = pnew; /* first element in list */
else plast -> nxtentr = pnew;

/* set array end pointer */
parttab[1] = pnew;

return pnew; /* pointer to new structure */
} /* end function install_part_at_end_of_list */


struct part *install_part_in_position(char *name, int position)
{
struct part *pa, *pprev, *plastmatch, *plast, *pnew, *pnext;

if(debug_flag)
	{
	fprintf(stderr, "install_part_in_position(): arg name=%s position=%d\n", name, position);
	}

if(! name) return 0;
if(position < 0) return 0;

plastmatch = 0; /* for gcc -Wall */
for(pa = parttab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(position == pa -> browser_line)
		{
		plastmatch = pa;
		break;
		}
	}

if(debug_flag)
	{
	fprintf(stderr, "install_part_in_position(): plastmatch=%p\n", plastmatch);
	}

/* plastmatch now points to insert point, or zero if not found, */

/* create new structure */
pnew = (struct part *) calloc(1, sizeof(*pnew) );
if(! pnew) return 0;

pnew -> name = strsave(name);
if(! pnew -> name) return 0;

/* if list was empty, set parttab[0] to point to new entry. */
if(! plastmatch)
	{
	if(parttab[0] == 0)
		{
		/* list was empty */
		parttab[0] = pnew;
		pnew -> prventr = 0;
		pnew -> nxtentr = 0;

		return pnew;
		}
	else /* add to end of list */
		{
		/* get previous structure */
		plast = parttab[1]; /* end list */

		/* set new structure pointers */
		pnew -> nxtentr = 0; /* new points top zero (is end) */
		pnew -> prventr = plast; /* point to previous entry, or 0 if first entry */

		/* set previuos structure pointers */
		if( !parttab[0] ) parttab[0] = pnew; /* first element in list */
		else plast -> nxtentr = pnew;

		/* set array end pointer */
		parttab[1] = pnew;

		return pnew;
		}
	}

/* plastmatch now holds insert position, insert article */
/* insert the structure AFTER plastmatch */
		
/* get previous and next structure */
pnext = plastmatch -> nxtentr;
pprev = plastmatch;

/* set pointers for new struture */
pnew -> nxtentr = pnext;
pnew -> prventr = pprev;

/* set pointers for next structure */
if(pnext == 0) parttab[1] = pnew;
else pnext -> prventr = pnew;		

/* set pointer for previous structure */
if(pprev == 0) parttab[0] = pnew;
else pprev -> nxtentr = pnew;
		
/* return pointer to new structure */
return pnew;
} /* end function install_part_in_position */


int clear_parts_list()/* delete all entries from table */
{
struct part *pa;

if(debug_flag) 
	{
	fprintf(stderr, "clear_parts_list() arg none\n");
	}

while(1)
	{	
	pa = parttab[0];
	if(! pa) break;
	parttab[0] = pa -> nxtentr;
	free(pa -> name);
	free(pa); /* free structure */
	} /* end while all structures */
parttab[1] = 0;
return 1;
} /* end function clear_parts_list */


int remove_from_part_list(int part)
{
struct part *pa, *pdel, *pnext, *pprev;

if(debug_flag)
	{
	fprintf(stderr, "remove_from_part_list(): arg part=%d\n", part);
	}

/* argument check */
if(part < 0) return 0;

pa = parttab[0];
while(1)
	{
	/* if end list, return not found */
	if(! pa) return 0;

	/* test for match in name */
	if(pa -> browser_line != part)
		{
		/* point to next element in list */
		pa = pa -> nxtentr;

		/* loop for next element in list */
		continue;
		}

	/* we now know which struture to delete */
	pdel = pa;

	/* get previous and next structure */
	pnext = pa -> nxtentr;
	pprev = pa -> prventr;

	/* set pointers for previous structure */
	/* if first one, modify articletab[0] */
	if(pprev == 0) parttab[0] = pnext;
	else pprev -> nxtentr = pnext;

	/* set pointers for next structure */
	/* if last one, modify articletab[1] */
	if(pnext == 0) parttab[1] = pprev;
	else pnext -> prventr = pprev;
	
	/* delete structure */	
	free(pdel -> name);
	free(pdel); /* free structure */

	/* return OK deleted */
	return 1;
	}/* end for all structures */

return 1;
} /* end function remove_from_part_list */


int show_part_list()
{
struct part *pa;
int browser_line;

if(debug_flag)
	{
	fprintf(stderr, "show_part_list(): arg none\n");	
	}

browser_line = 0;
for(pa = parttab[0]; pa != 0; pa = pa -> nxtentr)
	{
	pa -> browser_line = browser_line;

	fprintf(stderr, "%d length=%lu file=%s\n",\
	pa -> browser_line, (unsigned long) pa -> length, pa -> name);
	browser_line++;
	}

return 1;
} /* end function show_part_list */


int write_all(char *filename)
{
int a;
struct part *pa;
int header_size;
wave_header *file_header;
FILE *ifptr, *ofptr;
size_t length, ta, bytes_read_in_part, bytes_written_in_part, bytes_written;
double da;
unsigned char *ucptr;
int display_counter;
 
if(debug_flag)
	{
	fprintf(stderr, "write_all(): arg filename=%s\n", filename);
	}

if(! filename) return 0;

/* get total length */
length = 0;
for(pa = parttab[0]; pa != 0; pa = pa -> nxtentr)
	{
	length += pa -> length;

	if(debug_flag)
		{
		fprintf(stderr, "file %s data length=%lu\n", pa -> name, (unsigned long)pa -> length); 
		}
	}

if(debug_flag)
	{
	fprintf(stderr, "total length=%lu\n", (unsigned long) length);
	}

if(! have_format_flag)
	{
	fprintf(stderr, "write_all(): no input files, no format known\n");
	
	return 0;
	}

ucptr = calloc(byte_p_spl, BUFFER_SIZE); //sizeof(char));
if(! ucptr)
	{
	fprintf(stderr, "write_all(): could not calloc space for ucptr\n");

	return 0;
	}

/* try to open the output file for write */
ofptr = fopen(filename, "w");
if(! ofptr)
	{
	fprintf(stderr, "write_all(): ERROR could not open output file %s for write.\n", filename);

	return 0;
	}
	
/* write the wave header */
/* get size of wave header */
header_size = sizeof(wave_header);

if(debug_flag)
	{
	fprintf(stderr, "write_file(): wave header_size=%d\n", header_size);
	}

/* allocate space for wave header */
file_header = malloc( sizeof(wave_header) );
if(! file_header)
	{
	fprintf(stderr, "pwavecat: write_file(): ERROR could not allocate space for file_header.\n");

	return 0;
	}

strcpy(file_header -> main_chunk, "RIFF");
file_header -> length = length + header_size - 8;
strcpy(file_header -> chunk_type, "WAVE"); 
strcpy(file_header -> sub_chunk, "fmt ");	// watch out the zero termination overwrites next var
file_header -> length_chunk = 16;							//always 16
file_header -> format = 1;									//PCM
file_header -> modus = modus;								// stereo
file_header -> sample_fq = sample_fq;						// 44100, 48000, etc...
file_header -> byte_p_sec = byte_p_sec;						// little endian
file_header -> byte_p_spl = byte_p_spl;		 				// 4 stereo
file_header -> bit_p_spl = bit_p_spl;						// 16 bits
strcpy(file_header -> data_chunk, "data");	// watch out the zero termination overwrites next var
file_header -> data_length = (unsigned long)length;

if(debug_flag)
	{
	fprintf(stderr,\
	"write_all(): file_header -> data_length=%lu\n", (unsigned long)file_header -> data_length);

	fprintf(stderr,\
	"write_all(): file_header -> length %lu\n", (unsigned long)file_header -> length);
	}

/* write out wave header */
ta = fwrite(file_header, sizeof(char), header_size, ofptr);
if(ta != header_size)
	{
	fprintf(stderr,\
	"pwavecat: write_all(): could only write %lu of %d bytes of out header, aborting.\n",\
	(unsigned long) ta, header_size);   

	return 0;
	}

/* write the data */
bytes_written = 0;
display_counter = 0;

if(debug_flag) fprintf(stderr, "writing wave format to %s\n", filename);

for(pa = parttab[0]; pa != 0; pa = pa -> nxtentr)
	{
	/* open input file */
	ifptr = fopen(pa -> name, "r");
	if(! ifptr)
		{
		fprintf(stderr, "write_all(): could not open file %s for read\n", pa -> name);

		return 0;
		}

	/* seek 44 bytes in source */
	a = fseek(ifptr, 44, SEEK_SET);
	if(a < 0)
		{
		fprintf(stderr, "seek in file %s failed\n", pa -> name);

		fclose(ifptr);
		fclose(ofptr);

		free(file_header);

		return 0;
		}

	bytes_written_in_part = 0;
	while(1)
		{
		/* copy length from source */

		/* read data */
		bytes_read_in_part = fread(ucptr, sizeof(char), BUFFER_SIZE, ifptr);
		if(ferror(ifptr) )
			{
			fprintf(stderr, "write_all(): read error in file %s at byte %lu\n",\
			pa -> name, (unsigned long) bytes_written_in_part); 

			fclose(ifptr);
			fclose(ofptr);

			free(file_header);

			return 0;
			}

		/* write the data, as many bytes as read */
		ta = fwrite(ucptr, sizeof(char), bytes_read_in_part, ofptr);
		if(ferror(ofptr) )
			{
			fprintf(stderr, "write_all(): write error in file %s at byte %lu\n",\
			filename, (unsigned long) bytes_written_in_part); 

			fclose(ifptr);
			fclose(ofptr);

			free(file_header);

			return 0;
			}

		bytes_written_in_part += ta;		

		if(verbose_flag)
			{
			bytes_written += ta;

			display_counter += ta;

			/* report progress every now end then... */
			if( ( display_counter > 1024) || (bytes_written == length) || feof(ifptr) )
				{
				da = 100.0 * ((double)bytes_written / (double)length);

				fprintf(stderr,\
				"writing output file %s %ld (%.2f%%)\r", filename, (unsigned long)bytes_written, da);
				display_counter = 0;
				}
			}

		if(feof(ifptr) )
			{
			if(bytes_written_in_part != pa -> length)
				{
				fprintf(stderr,\
				"write_all(): ERROR file %s bytes read is %lu, this differs from length %lu indicated in header.\n",\
				pa -> name, (unsigned long) bytes_written_in_part, (unsigned long) pa -> length);

				fclose(ifptr);
				fclose(ofptr);

				free(file_header);

				return 0;
				}
			
			else
				{
				/* OK */
				break;
				} 
			}

		} /* end while write data */

	fclose(ifptr);

	} /* end for all in list */

fclose(ofptr);
free(file_header);

if(verbose_flag)
	{
	fprintf(stderr, "\n");
	}

return 1;
} /* end function write_all */

