/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  admameli.c

  Advanced MAME List format files functions
*/

#include "common.h"
#include <stdio.h>
#include <mamory/readinfo.h>
#include <mamory/modeconv.h>
#include <mamory/admameli.h>

#include <sortbox/sortbox.h>


#define ROMONLY 0
#define TOTALGAME 1

#define NBR_TOKENS 92
#define BUFSIZx2 (BUFSIZ*4)

FILE *GamesSourceFile=NULL;
unsigned char GamesSourceFileLine[BUFSIZx2]="\0";
unsigned int GamesSourceLinePtr=0;
unsigned int NbrUnknownTokens=0;
s_SortBox **SamplesSortBox=NULL;

s_SortBox *RomContentsSortBox=NULL;
s_SortBox *GamesSortBox=NULL;

s_GameInfo **TempGames=NULL;
unsigned int NbrTempGames=0;

typedef enum e_Token 
  {
    tk_unknown=0,
    tk_mamory,tk_source,tk_target,tk_targetver,tk_comment,tk_manager,
    tk_forcestoragemode,
    tk_file, tk_exec, tk_directory, tk_admameli,
    tk_clrmamepro, tk_category, tk_version, tk_author,tk_storagemode,
    tk_forcemerging,tk_forcestorageformat, tk_forcezipping,
    tk_game,tk_resource,tk_rom, tk_history,tk_disk,tk_name,tk_merge,
    tk_compsize,tk_size,tk_crc,tk_md5,tk_sample,tk_clone,tk_chip,tk_type,
    tk_flags,tk_clock,tk_video,tk_screen,tk_orientation,tk_vertical,
    tk_horizontal,tk_x,tk_y,tk_colors,tk_freq,tk_sound,
    tk_channels,tk_input,tk_players,tk_control,tk_buttons,
    tk_coins,tk_service,tk_tilt,tk_dipswitch,tk_entry,
    tk_default,tk_driver,tk_status,tk_color,tk_palettesize,
    tk_description,tk_year,tk_manufacturer,tk_cloneof,
    tk_romof,tk_sampleof,tk_opened_par,tk_closed_par,
    tk_region,tk_offs,tk_aspectx,tk_aspecty,tk_blit,tk_index,
    tk_baddump,tk_nodump,tk_sha1,tk_soundonly,tk_biosset,tk_bios,
    tk_sourcefile,tk_display, tk_rotate, tk_flipx, tk_0,
    tk_90, tk_180, tk_270,tk_yes,tk_no
  } e_Token;

unsigned char *Tokens[NBR_TOKENS]=
  {
    NULL,
    "mamory","source","target","targetver","comment","manager",
    "forcestoragemode",
    "file","exec","directory","admameli",
    "clrmamepro","category","version","author","storagemode",
    "forcemerging","forcestorageformat","forcezipping",
    "game","resource","rom","history","disk","name","merge","compsize",
    "size","crc","md5","sample","clone","chip","type","flags","clock",
    "video","screen","orientation","vertical","horizontal",
    "x","y","colors","freq","sound","channels",
    "input","players","control","buttons","coins",
    "service","tilt","dipswitch","entry","default",
    "driver","status","color","palettesize","description",
    "year","manufacturer","cloneof","romof","sampleof",
    "(",")","region","offs","aspectx","aspecty","blit","index",
    "baddump","nodump","sha1","soundonly","biosset","bios",
    "sourcefile", "display", "rotate","flipx", "0", 
    "90","180","270","yes","no"
  };

int ReadNextLine(void)
{
  memset(GamesSourceFileLine,0,BUFSIZx2);
  if (!fgets(GamesSourceFileLine,BUFSIZx2,GamesSourceFile))
    return(0);
  
  GamesSourceFileLine[strlen(GamesSourceFileLine)-1]=0;
  GamesSourceLinePtr=0;
  return 1;
};

e_Token ReadNextToken(unsigned char **Token)
{
  unsigned int i;
  unsigned int TokenMemSize=0;
  unsigned int TokenStart;

  XFREE(*Token);
  
  if (GamesSourceLinePtr>=strlen(GamesSourceFileLine))
    if(!ReadNextLine())
      return tk_unknown;
      

  while(GamesSourceFileLine[GamesSourceLinePtr]==' ' || 
	GamesSourceFileLine[GamesSourceLinePtr]=='\t' ||
	GamesSourceFileLine[GamesSourceLinePtr]=='\0')
    {
     if (GamesSourceLinePtr>=strlen(GamesSourceFileLine) ||
	 GamesSourceFileLine[GamesSourceLinePtr]=='\0')
       {
	 if(!ReadNextLine())
	   return tk_unknown;
       }
     else
       GamesSourceLinePtr++;
    };

  if (GamesSourceFileLine[GamesSourceLinePtr]!='"')
    {
      TokenStart=GamesSourceLinePtr;
      while(GamesSourceFileLine[GamesSourceLinePtr]!=' ' &&
	    GamesSourceFileLine[GamesSourceLinePtr]!='\t' &&
	    GamesSourceLinePtr<strlen(GamesSourceFileLine))
	{    
	  GamesSourceLinePtr++;
	  TokenMemSize++;
	};
    }
  else
    {
      GamesSourceLinePtr++;
      TokenStart=GamesSourceLinePtr;
      while(GamesSourceFileLine[GamesSourceLinePtr]!='"' &&
	    GamesSourceLinePtr<strlen(GamesSourceFileLine))
	{
	  if(GamesSourceFileLine[GamesSourceLinePtr]=='\\')
	    {
	      GamesSourceLinePtr++;
	      TokenMemSize++;
	    };
	  GamesSourceLinePtr++;
	  TokenMemSize++;
	};
      GamesSourceLinePtr++;
    };

  *Token=XCALLOC(unsigned char,TokenMemSize+1);

  strncpy(*Token,&GamesSourceFileLine[TokenStart],TokenMemSize);
  
  if((*Token)[strlen(*Token)-1]=='\r')
    (*Token)[strlen(*Token)-1]=0;

  for (i=1;i<NBR_TOKENS;i++)
    if (strcmp(*Token,Tokens[i])==0)
      break;

  if (i==NBR_TOKENS)
    return(tk_unknown);
  else
    return(i);
};

s_RomInfo *ReadRomInfo(s_GamesList *GamesList,unsigned char Type)
{
  s_RomInfo *Rom=NULL;
  s_RomContent *Content=NULL;
  s_RomContent **Contents=NULL;

  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndRomInfo=0;
  unsigned char nbr=0,i;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;
  
  Rom=XMALLOC(s_RomInfo,1);

  Rom->Source=GamesList->Sources[0];
//  Rom->StorageFormat=STFMT_UNSET;
  Rom->Name=NULL;
  Rom->GameName=NULL;
  Rom->Merge=NULL;
  Rom->Status=UNKNOWN_DUMP;
  Rom->Content=NULL;
  Rom->NbrShared=0;
  Rom->Shared=NULL;
  if (Type==TOTALGAME)
    {
      Rom->Hardware=XCALLOC(s_RomHardwareInfo,1);

      Rom->Hardware->Region=NULL;
      Rom->Hardware->Flags=NULL;
      Rom->Hardware->Offset=0;
    }
  else
    Rom->Hardware=NULL;

  Content=XCALLOC(s_RomContent,1);

  
  
  while(!EndRomInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  Rom->Name=XSTRDUP(Token);
	  break;
	case tk_bios:
	  TokenID=ReadNextToken(&Token);
	  break;
	case tk_merge:
	  TokenID=ReadNextToken(&Token);
	  Rom->Merge=XSTRDUP(Token);
	  break;
	case tk_md5:
	  TokenID=ReadNextToken(&Token);
	  i=strlen(Token)+1;
	  // Some TOSEC files uses the minus character if they don't have the
	  // information
	  if(strcmp(Token,"-")==0)
	    break;
	  // the following correct a bug in certain version of xmame where
	  // the md5 value is filled with garbage if mameinfo.dat and/or
	  // history.dat were used
	  while(i<32)
	    {
	      TokenID=ReadNextToken(&Token);
	      i+=strlen(Token)+1;
	    };
	  break;
	case tk_region:
	  if (Type==TOTALGAME)
	    {
	      TokenID=ReadNextToken(&Token);
	      Rom->Hardware->Region=XSTRDUP(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_flags:
	  TokenID=ReadNextToken(&Token);
	  switch(TokenID)
	    {
	    case tk_baddump:
	      Rom->Status=NO_GOOD_DUMP_KNOWN;
	      break;
	    case tk_nodump:
	      Content->CRC=0;
	      Rom->Status=NEED_REDUMP;
	      break;  
	    default:
	      if (Type==TOTALGAME)
		Rom->Hardware->Flags=XSTRDUP(Token);
	    };
	  break;
	case tk_offs:
	  if (Type==TOTALGAME)
	    {
	      TokenID=ReadNextToken(&Token);
	      Rom->Hardware->Offset=atoi(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_size:
	  TokenID=ReadNextToken(&Token);
	  Content->Size=atoi(Token);
	  break;
	case tk_compsize:
	  TokenID=ReadNextToken(&Token);
	  Content->CompSize=atoi(Token);
	  break; 
	case tk_crc:
	  TokenID=ReadNextToken(&Token);
	  Content->CRC=strtoul(Token,NULL,16);
	  if(Content->CRC==0)
	    Rom->Status=NEED_REDUMP;
	  break;
	case tk_sha1:
	  TokenID=ReadNextToken(&Token);
	  break;
    // for compatibility with xmame 0.68
	case tk_baddump:
	  Rom->Status=NO_GOOD_DUMP_KNOWN;
	  break;
    // for compatibility with xmame 0.68
	case tk_nodump:
	  Content->CRC=0;
	  Rom->Status=NEED_REDUMP;
	  break;
	case tk_clone:
	  TokenID=ReadNextToken(&Token);
	  Rom->GameName=XSTRDUP(Token);
	  break;
	case tk_source:
	  TokenID=ReadNextToken(&Token);
	  i=atoi(Token);
	  Rom->Source=GamesList->Sources[i];
	  break;
/*
	case tk_storageformat:
	  TokenID=ReadNextToken(&Token);
	  if(strcmp(Token,"raw")==0)
	    Rom->StorageFormat=STFMT_RAW;
	  if(strcmp(Token,"zip")==0)
	    Rom->StorageFormat=STFMT_ZIP;
	  break;	  
*/
	case tk_closed_par:
	  EndRomInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadRomInfo",Token);
	  NbrUnknownTokens++;
	};
    };

  XFREE(Token);
  
  if (Content->CRC!=0)
    Contents=(s_RomContent**)SortBox_FindTheUint(RomContentsSortBox,
						 Content->CRC,
						 (int*)&nbr);

  if(Contents!=NULL)
    for(i=0;i<nbr;i++)
      if (Contents[i]->Size==Content->Size)
	break;

  if(Contents==NULL || (Contents!=NULL && i==nbr))
    {
      if(Content->CRC!=0)
	SortBox_AddOneUint(RomContentsSortBox,Content->CRC,Content); 
      Rom->Content=Content;
      for(i=0;i<nbr;i++)
	Contents[i]=NULL;
      XFREE(Contents);
      Contents=NULL;
      if (Content->CRC!=0)
	{
	  GamesList->RomContents=XREALLOC(GamesList->RomContents,
					  s_RomContent*,GamesList->NbrRomContents+1);

	  GamesList->RomContents[GamesList->NbrRomContents]=Rom->Content;
	  GamesList->NbrRomContents++;
	};

      return Rom;
    };

  Rom->Content=Contents[i];
  for(i=0;i<nbr;i++)
    Contents[i]=NULL;
  XFREE(Contents);
  XFREE(Content);

  return Rom;
};

s_DiskInfo *ReadDiskInfo(unsigned char Type)
{
  s_DiskInfo *Disk;
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndDiskInfo=0;

  Disk=XMALLOC(s_DiskInfo,1);

  Disk->Name=NULL;
  Disk->md5=NULL;

  if(Type==TOTALGAME)
    {
      Disk->Hardware=XMALLOC(s_DiskHardwareInfo,1);

      Disk->Hardware->Region=NULL;
      Disk->Hardware->Index=0;
    }
  else
    Disk->Hardware=NULL;
  
  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  while(!EndDiskInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)  
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  Disk->Name=XSTRDUP(Token);
	  break;
	case tk_merge: 
	  TokenID=ReadNextToken(&Token);
	  break;
	case tk_md5:
	  TokenID=ReadNextToken(&Token);
	  Disk->md5=XSTRDUP(Token);
	  break;
	case tk_sha1:
	  TokenID=ReadNextToken(&Token);
	  break;
	case tk_region:
	  if(Type==TOTALGAME)
	    {
	      TokenID=ReadNextToken(&Token);
	      Disk->Hardware->Region=XSTRDUP(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_index:
	  if(Type==TOTALGAME)
	    {
	      TokenID=ReadNextToken(&Token);
	      Disk->Hardware->Index=atoi(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_flags:
	  ReadNextToken(&Token);
	  break;
	case tk_closed_par:
	  EndDiskInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadDiskInfo",Token);
	  NbrUnknownTokens++;
	};
    };

  XFREE(Token);
  return Disk;
};

s_ChipInfo *ReadChipInfo(void)
{
  s_ChipInfo *Chip=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndChipInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  Chip=XMALLOC(s_ChipInfo,1);

  Chip->Type=NULL;
  Chip->Flags=NULL;
  Chip->Name=NULL;
  Chip->Clock=0;

  while(!EndChipInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_type:
	  TokenID=ReadNextToken(&Token);
	  Chip->Type=XSTRDUP(Token);
	  break;
	case tk_flags:
	  TokenID=ReadNextToken(&Token);
	  Chip->Flags=XSTRDUP(Token);
	  break;
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  Chip->Name=XSTRDUP(Token);
	  break;
	case tk_clock:
	  TokenID=ReadNextToken(&Token);
	  Chip->Clock=atoi(Token);
	  break;
	case tk_closed_par:
	  EndChipInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadChipInfo",Token);
	  NbrUnknownTokens++;

	};
    };
  XFREE(Token);
  return Chip;
};

/* was ReadVideoInfo before mame 0.107 */
s_DisplayInfo *ReadDisplayInfo(void) 
{
  s_DisplayInfo *Display=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndDisplayInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  Display=XMALLOC(s_DisplayInfo,1);

  Display->Type=NULL;
  Display->Rotate=0;
  Display->X=0;
  Display->Y=0;
  Display->AspectX=0;
  Display->AspectY=0;
  Display->Flipx=0;
  Display->Colors=0;
  Display->Freq=0;

  while(!EndDisplayInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_type:
	  TokenID=ReadNextToken(&Token);
	  Display->Type=XSTRDUP(Token);
	  break;
	case tk_orientation:
	  TokenID=ReadNextToken(&Token);
	  switch(TokenID)
	    {
	    case tk_horizontal:
	      Display->Rotate=HORIZONTAL;
	      break;
	    case tk_vertical:
	      Display->Rotate=VERTICAL;
	      break;
	    case tk_0:
	      Display->Rotate=ROTATE_0;
	      break;
	    case tk_90:
	      Display->Rotate=ROTATE_90;
	      break;
	    case tk_180:
	      Display->Rotate=ROTATE_180;
	      break;
	    case tk_270:
	      Display->Rotate=ROTATE_270;
	      break;
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadDisplayInfo - orientation",Token);
	      XFREE(Token);
	      FreeDisplayInfo(Display);
	      return NULL;
	    };
	  break;
	case tk_x:
	  TokenID=ReadNextToken(&Token);
	  Display->X=atoi(Token);
	  break;
	case tk_y:
	  TokenID=ReadNextToken(&Token);
	  Display->Y=atoi(Token);
	  break;
	case tk_aspectx:
	  TokenID=ReadNextToken(&Token);
	  Display->AspectX=(unsigned char)atoi(Token);
	  break;
	case tk_aspecty:
	  TokenID=ReadNextToken(&Token);
	  Display->AspectY=(unsigned char)atoi(Token);
	  break;
	case tk_flipx:
	  TokenID=ReadNextToken(&Token);
	  switch(TokenID)
	    {
	    case tk_yes:
	      Display->Flipx=TRUE;
	      break;
	    case tk_vertical:
	      Display->Flipx=FALSE;
	      break;
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadDisplayInfo - orientation",Token);
	      XFREE(Token);
	      FreeDisplayInfo(Display);
	      return NULL;
	    };
	  break;
	case tk_colors:
	  TokenID=ReadNextToken(&Token);
	  Display->Colors=atoi(Token);
	  break;
	case tk_freq:
	  TokenID=ReadNextToken(&Token);
	  Display->Freq=atof(Token);
	  break;
	case tk_closed_par:
	  EndDisplayInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in  ReadDisplayInfo",Token);
	  NbrUnknownTokens++;
	};
    };
  XFREE(Token);
  return Display;
};

s_SoundInfo *ReadSoundInfo(void)
{
  s_SoundInfo *Sound=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndSoundInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  Sound=XMALLOC(s_SoundInfo,1);

  Sound->Channels=0;

   while(!EndSoundInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_channels:
	  TokenID=ReadNextToken(&Token);
	  Sound->Channels=(unsigned char)atoi(Token);
	  break;
	case tk_closed_par:
	  EndSoundInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadSoundInfo",Token);
	  NbrUnknownTokens++;
	};
    };
   XFREE(Token);
   return Sound;
};

s_InputInfo *ReadInputInfo(void)
{
  /* TODO: deal with control as defined in mame 0.107 */

  s_InputInfo *Input=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndInputInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  Input=XMALLOC(s_InputInfo,1);

  Input->Players=0;
  Input->Controls=NULL;
  Input->Buttons=0;
  Input->Coins=0;
  Input->Tilt=NULL;
  Input->Service=NULL;

  while(!EndInputInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_players:
	  TokenID=ReadNextToken(&Token);
	  Input->Players=(unsigned char)atoi(Token);
	  break;
	case tk_control:
	  TokenID=ReadNextToken(&Token);
	  /* TODO: change to reflect modification made in mame 0.107 
	  Input->Control=XSTRDUP(Token);
	  */
	  break;
	case tk_buttons:
	  TokenID=ReadNextToken(&Token);
	  Input->Buttons=(unsigned char)atoi(Token);
	  break;
	case tk_coins:
	  TokenID=ReadNextToken(&Token);
	  Input->Coins=(unsigned char)atoi(Token);
	  break;
	case tk_tilt:
	  TokenID=ReadNextToken(&Token);
	  Input->Tilt=XSTRDUP(Token);
	  break;
	case tk_service:
	  TokenID=ReadNextToken(&Token);
	  Input->Service=XSTRDUP(Token);
	  break;
	case tk_closed_par:
	  EndInputInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadInputInfo",Token);
	  NbrUnknownTokens++;
	};
    };
   XFREE(Token);
   return Input;
};  

s_DipSwitchInfo *ReadDipSwitchInfo(void)
{
  s_DipSwitchInfo *DS=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndDipSwitchInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  DS=XMALLOC(s_DipSwitchInfo,1);

  DS->Name=NULL;
  DS->NbrEntries=0;
  DS->Entries=NULL;
  DS->Default=NULL;

  while(!EndDipSwitchInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  DS->Name=XSTRDUP(Token);
	  break;  
	case tk_entry:
	  TokenID=ReadNextToken(&Token);
	  DS->Entries=XREALLOC(DS->Entries,
			      unsigned char*,DS->NbrEntries+1);

	  DS->Entries[DS->NbrEntries]=XSTRDUP(Token);
	  DS->NbrEntries++;
	  break;
	case tk_default:
	  TokenID=ReadNextToken(&Token);
	  DS->Default=XSTRDUP(Token);
	  break;  
	case tk_closed_par:
	  EndDipSwitchInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadDipSwitchInfo",Token);
	  NbrUnknownTokens++;
	};
    };
   XFREE(Token);
   return DS;  
};

s_DriverInfo *ReadDriverInfo(void)
{
  s_DriverInfo *Driver=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndDriverInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;
 
  Driver=XMALLOC(s_DriverInfo,1);

  Driver->Status=NULL;
  Driver->Color=NULL;
  Driver->Sound=NULL;
  Driver->Blit=NULL;
  Driver->PaletteSize=0;

  while(!EndDriverInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_status:
	  TokenID=ReadNextToken(&Token);
	  Driver->Status=XSTRDUP(Token);
	  break;
	case tk_color:
	  TokenID=ReadNextToken(&Token);
	  Driver->Color=XSTRDUP(Token);
	  break;
	case tk_sound:
	  TokenID=ReadNextToken(&Token);
	  Driver->Sound=XSTRDUP(Token);
	  break;
	case tk_blit:
	  TokenID=ReadNextToken(&Token);
	  Driver->Blit=XSTRDUP(Token);
	  break;
	case tk_palettesize:
	  TokenID=ReadNextToken(&Token);
	  Driver->PaletteSize=(unsigned char)atoi(Token);
	  break;
	case tk_closed_par:
	  EndDriverInfo=1;
	  break;
	default:
	  LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadDriverInfo",Token);
	  NbrUnknownTokens++;
	};
    };
   XFREE(Token);
   return Driver;  
};

s_CloneInfo *ReadClone(unsigned char Type)
{
  s_CloneInfo *Clone=NULL;
  s_ChipInfo *Chip=NULL;
  s_DipSwitchInfo *DipSwitch=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndCloneInfo=0;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;
 
  Clone=XMALLOC(s_CloneInfo,1);

  Clone->Name=NULL;
  Clone->Description=NULL;

  if(Type==TOTALGAME)
    {
      Clone->Misc=XMALLOC(s_MiscGameInfo,1);

      Clone->Misc->Year=0;
      Clone->Misc->Manufacturer=NULL;
      Clone->Misc->NbrChips=0;
      Clone->Misc->Chips=NULL;
      Clone->Misc->Displays=NULL;
      Clone->Misc->NbrDisplays=0;
      Clone->Misc->Sound=NULL;
      Clone->Misc->Input=NULL;
      Clone->Misc->NbrDipSwitches=0;
      Clone->Misc->DipSwitches=NULL;
      Clone->Misc->Driver=NULL;
    }
  else
    Clone->Misc=NULL;

  while(!EndCloneInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  Clone->Name=XSTRDUP(Token);
	  break;
	case tk_description:
	  TokenID=ReadNextToken(&Token);
	  Clone->Description=XSTRDUP(Token);
	  break;
	case tk_year:
	  if (Type!=ROMONLY)
	    {
	      TokenID=ReadNextToken(&Token);
	      Clone->Misc->Year=(short)atoi(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_manufacturer:
	  if (Type!=ROMONLY)
	    {
	      TokenID=ReadNextToken(&Token);
	      Clone->Misc->Manufacturer=XSTRDUP(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_chip:
	  if(Type!=ROMONLY)
	    {
	      Chip=ReadChipInfo();
	      if(Chip!=NULL)
		{
		  Clone->Misc->Chips=XREALLOC(Clone->Misc->Chips,
					      s_ChipInfo *,
					      Clone->Misc->NbrChips+1);

		  Clone->Misc->Chips[Clone->Misc->NbrChips]=Chip;
		  Clone->Misc->NbrChips++;
		};
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token); 
	  break;
	case tk_video:
	case tk_display:
	  if(Type!=ROMONLY)
	    {
	      Clone->Misc->Displays=XREALLOC(Clone->Misc->Displays,
					     s_DisplayInfo*,
					     Clone->Misc->NbrDisplays+1);
	      Clone->Misc->Displays[Clone->Misc->NbrDisplays]=
		ReadDisplayInfo();
	      Clone->Misc->NbrDisplays++;
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_sound:
	  if(Type!=ROMONLY)
	    Clone->Misc->Sound=ReadSoundInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_input:
	  if(Type!=ROMONLY)
	    Clone->Misc->Input=ReadInputInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_dipswitch:
	  if(Type!=ROMONLY)
	    {
	      DipSwitch=ReadDipSwitchInfo();
	      if(DipSwitch!=NULL)
		{
		  Clone->Misc->DipSwitches=XREALLOC(Clone->Misc->DipSwitches,
						    s_DipSwitchInfo*,
						    Clone->Misc->NbrDipSwitches+1);

		  Clone->Misc->DipSwitches[Clone->Misc->NbrDipSwitches]=DipSwitch;
		  Clone->Misc->NbrDipSwitches++;
		};
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_driver:
	  if(Type!=ROMONLY)
	    Clone->Misc->Driver=ReadDriverInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_closed_par:
	  EndCloneInfo=1;
	  break;
	default:
	  if(strlen(Token)!=0)
	    {
	      NbrUnknownTokens++;
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadClone",Token);
	    };
	};
    };
   XFREE(Token);
   return Clone;  
}
unsigned char *AddSampleToMameSamples(unsigned char *Sample,
				      s_GamesList *GamesList)
{
  unsigned int i;

 for(i=0;i<GamesList->NbrSoundSamples;i++)
  if(strcmp(GamesList->SoundSamples[i],Sample)==0)
      {
	XFREE(Sample);
	return GamesList->SoundSamples[i];
      };

  GamesList->SoundSamples=XREALLOC(GamesList->SoundSamples,
				  unsigned char*,
				  GamesList->NbrSoundSamples+1);

  GamesList->SoundSamples[GamesList->NbrSoundSamples]=Sample;
  GamesList->NbrSoundSamples++;
  return Sample;
};  

void CheckGamesListResources(s_GamesList *GamesList,
			     s_GameInfo *GameInfo)
{
  unsigned int i;

  for(i=0;i<GamesList->NbrResources;i++)
    {
      if(strcmp(GamesList->Resources[i]->Name,GameInfo->More->Resource)==0)
	{
	  GamesList->Resources[i]->AssociatedGames=
	    XREALLOC(GamesList->Resources[i]->AssociatedGames,
		    s_GameInfo*,
		    GamesList->Resources[i]->NbrAssociatedGames+1);

	  GamesList->Resources[i]->AssociatedGames[GamesList->Resources[i]->NbrAssociatedGames]=GameInfo;
	  GamesList->Resources[i]->NbrAssociatedGames++;
	  break;
	};
    };
  
  if(i==GamesList->NbrResources)
    {
      GamesList->Resources=XREALLOC(GamesList->Resources,
				    s_ResourceInfo*,
				    GamesList->NbrResources+1);

      GamesList->Resources[GamesList->NbrResources]=XCALLOC(s_ResourceInfo,1);

      GamesList->Resources[GamesList->NbrResources]->Name=XSTRDUP(GameInfo->More->Resource);
      GamesList->Resources[GamesList->NbrResources]->AssociatedGames=
	XREALLOC(GamesList->Resources[GamesList->NbrResources]->AssociatedGames,
		 s_GameInfo*,GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames+1);

      GamesList->Resources[GamesList->NbrResources]->AssociatedGames[GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames]=GameInfo;
      GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames++;
      GamesList->NbrResources++;
    };
};


s_GameInfo *ReadGameInfo(s_GamesList *GamesList,
			 unsigned char Type)
{
  s_GameInfo *GameInfo=NULL;
  s_GameInfo **Games=NULL;
  unsigned int nbrdata=0;
  s_CloneInfo *Clone=NULL;
  s_RomInfo *Rom=NULL;
  s_DiskInfo *Disk=NULL;
  unsigned char *Sample=NULL;
  s_ChipInfo *Chip=NULL;
  s_DipSwitchInfo *DipSwitch=NULL;
  
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned char EndGameInfo=0;
  unsigned int i;

  unsigned char *GameName=NULL;
  unsigned char *GameDescription=NULL;

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    {
      XFREE(Token);
      return NULL;
    };

  GameInfo=XCALLOC(s_GameInfo,1);
  GameInfo->More=XCALLOC(s_MoreGameInfo,1);

  if(Type==TOTALGAME)
    GameInfo->More->Misc=XCALLOC(s_MiscGameInfo,1);

  while(!EndGameInfo)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  GameInfo->Name=XSTRDUP(Token);
	  break;
	case tk_sourcefile:
	  TokenID=ReadNextToken(&Token);	   
	  break;
	   /*
	case tk_storagemode:
	  TokenID=ReadNextToken(&Token);
	  if(strcmp(Token,"merge")==0)
	    GameInfo->StorageMode=STMODE_MERGE;
	  if(strcmp(Token,"split")==0)
	    GameInfo->StorageMode=STMODE_SPLIT;
	  if(strcmp(Token,"full")==0)
	      GameInfo->StorageMode=STMODE_FULL;
	  break;	  
	case tk_storageformat:
	  TokenID=ReadNextToken(&Token);
	  if(strcmp(Token,"raw")==0)
	    GameInfo->StorageFormat|=STFMT_RAW;
	  if(strcmp(Token,"zip")==0)
	    GameInfo->StorageFormat|=STFMT_ZIP;
	  break;
*/
	case tk_description:
	  TokenID=ReadNextToken(&Token);
	  GameInfo->More->Description=XSTRDUP(Token);
	  break;
	case tk_year:
	  if (Type!=ROMONLY)
	    {
	      TokenID=ReadNextToken(&Token);
	      GameInfo->More->Misc->Year=(short)atoi(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_manufacturer:
	  if (Type!=ROMONLY)
	    {
	      TokenID=ReadNextToken(&Token);
	      GameInfo->More->Misc->Manufacturer=XSTRDUP(Token);
	    }
	  else
	    ReadNextToken(&Token);
	  break;
	case tk_clone:
	  Clone=ReadClone(Type);
	  if(Clone!=NULL)
	    {
	      GameInfo->More->Clones=XREALLOC(GameInfo->More->Clones,
					      s_CloneInfo*,
					      GameInfo->More->NbrClones+1);

	      GameInfo->More->Clones[GameInfo->More->NbrClones]=Clone;
	      GameInfo->More->NbrClones++;
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token); 	  
	  break;
	case tk_resource:
	  TokenID=ReadNextToken(&Token);
	  XFREE(GameInfo->More->Resource);
	  GameInfo->More->Resource=XSTRDUP(Token);
	  CheckGamesListResources(GamesList,GameInfo);
	  break;
	case tk_cloneof:
	  TokenID=ReadNextToken(&Token);
	  GameInfo->More->CloneOf=XSTRDUP(Token);
	  break;
	case tk_romof:
	  TokenID=ReadNextToken(&Token);
	  GameInfo->More->RomOf=XSTRDUP(Token);
	  if(GameInfo->More->Resource==NULL)
	    {
	      if(GameInfo->More->CloneOf==NULL)
		{
		  GameInfo->More->Resource=XSTRDUP(Token); 
		  CheckGamesListResources(GamesList,GameInfo);	  
		}
	      else
		{
		  nbrdata=1;
		  Games=SortBox_FindTheWord(GamesSortBox,
					    GameInfo->More->CloneOf,
					    (int*)&nbrdata);
		  if(nbrdata==1)
		    {
		      if(Games[0]->More->Resource!=NULL)
			{
			  GameInfo->More->Resource=XSTRDUP(Games[0]->More->Resource);
			  CheckGamesListResources(GamesList,GameInfo);
			};

		    }
		  else
		    {
		      // stocker les games dans liste temporaire
		      TempGames=XREALLOC(TempGames,s_GameInfo*,NbrTempGames+1);

		      TempGames[NbrTempGames]=GameInfo;
		      NbrTempGames++;
		    }
		  for(i=0;i<nbrdata;i++)
		    Games[i]=NULL;
		  XFREE(Games);
		}
	    };
	  break;
	case tk_biosset:
	  while(TokenID!=tk_closed_par)
	    TokenID=ReadNextToken(&Token);
	  break;	  
	case tk_history:
	  TokenID=ReadNextToken(&Token);
	  break;
	case tk_rom:
	  Rom=ReadRomInfo(GamesList,ROMONLY);

	  if (Rom!=NULL)
	    {
//	      if(Rom->StorageFormat==STFMT_UNSET)
//		Rom->StorageFormat=GameInfo->StorageFormat;

	      if(Rom->Merge!=NULL || Rom->GameName!=NULL)
		{
		  if(Rom->GameName==NULL)
		    {
		      if(GameInfo->More->CloneOf!=NULL)
			Rom->GameName=XSTRDUP(GameInfo->More->CloneOf);
		      else
			Rom->GameName=XSTRDUP(GameInfo->Name);
		    };
//		  if(Rom->Merge==NULL &&
//		     strcmp(Rom->GameName,GameInfo->Name)!=0)
//		    Rom->Merge=XSTRDUP(Rom->Name);
		  switch(Rom->Source->StorageMode)
		    {
		    case STMODE_SPLIT: 
		      if(Rom->Merge==NULL &&
			 strcmp(Rom->GameName,GameInfo->Name)!=0)
			Rom->Merge=XSTRDUP(Rom->Name);
		      AddRomToGameMergedRoms(GameInfo,Rom);
		      break;
		    case STMODE_MERGE: 
		      if(GameInfo->More->Resource!=NULL &&
			 strcmp(Rom->GameName,GameInfo->More->Resource)==0)
			AddRomToGameMergedRoms(GameInfo,Rom);
		      else
			{
			  Rom=AddRomToRomSharedRoms(GameInfo,Rom,NULL);
			  if(Rom!=NULL)
			    AddRomToGameRoms(GameInfo,Rom);
			}
		      break;
		    case STMODE_FULL: 
		      if(Rom->Merge==NULL &&
			 strcmp(Rom->GameName,GameInfo->Name)!=0)
			Rom->Merge=XSTRDUP(Rom->Name);
		      AddRomToGameRoms(GameInfo,Rom);
		      break;
		    };
		}
	      else
		{
		  if(Rom->GameName==NULL)
		    Rom->GameName=XSTRDUP(GameInfo->Name);
		  switch(Rom->Source->StorageMode)
		    {
		    case STMODE_MERGE:
//		      if (strcmp(Rom->GameName,GameInfo->Name)!=0)
//			{
			  Rom=AddRomToRomSharedRoms(GameInfo,Rom,NULL);
			  if(Rom==NULL)
			    break;
//			};
		    case STMODE_SPLIT:
		    case STMODE_FULL:
		      AddRomToGameRoms(GameInfo,Rom);
		      break;
		    };
		};
	    };
	  break;
	case tk_disk:
	  Disk=ReadDiskInfo(ROMONLY);
	  if(Disk!=NULL)
	    {
	      GameInfo->Disks=XREALLOC(GameInfo->Disks,s_DiskInfo*,
				       GameInfo->NbrDisks+1);
	      GameInfo->Disks[GameInfo->NbrDisks]=Disk;
	      GameInfo->NbrDisks++;
	    };
	  break;
	case tk_sampleof:
	  TokenID=ReadNextToken(&Token);
	  GameInfo->More->SampleOf=XSTRDUP(Token);
	  break;
	case tk_sample:
	  TokenID=ReadNextToken(&Token);
	  Sample=XSTRDUP(Token);
	  Sample=AddSampleToMameSamples(Sample,GamesList);
	  GameInfo->More->Samples=XREALLOC(GameInfo->More->Samples,
					   unsigned char*,
					   GameInfo->More->NbrSamples+1);

	  GameInfo->More->Samples[GameInfo->More->NbrSamples]=Sample;
	  GameInfo->More->NbrSamples++;
	  break;
	case tk_chip:
	  if(Type!=ROMONLY)
	    {
	      Chip=ReadChipInfo();
	      if(Chip!=NULL)
		{
		  GameInfo->More->Misc->Chips=
		    XREALLOC(GameInfo->More->Misc->Chips,
			     s_ChipInfo*,
			     GameInfo->More->Misc->NbrChips+1);

		  GameInfo->More->Misc->Chips[GameInfo->More->Misc->NbrChips]=Chip;
		  GameInfo->More->Misc->NbrChips++;
		};
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token); 
	  break;
	case tk_display:
	case tk_video:
	  if(Type!=ROMONLY)
	    {
	      GameInfo->More->Misc->Displays=
		XREALLOC(GameInfo->More->Misc->Displays,
			 s_DisplayInfo*,
			 GameInfo->More->Misc->NbrDisplays+1);
	      GameInfo->More->Misc->Displays[GameInfo->More->Misc->NbrDisplays]=
		ReadDisplayInfo();
	      GameInfo->More->Misc->NbrDisplays++;
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_sound:
	  if(Type!=ROMONLY)
	    GameInfo->More->Misc->Sound=ReadSoundInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_input:
	  if(Type!=ROMONLY)
	    GameInfo->More->Misc->Input=ReadInputInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_dipswitch:
	  if(Type!=ROMONLY)
	    {
	      DipSwitch=ReadDipSwitchInfo();
	      if(DipSwitch!=NULL)
		{
		  GameInfo->More->Misc->DipSwitches=
		    XREALLOC(GameInfo->More->Misc->DipSwitches,
			     s_DipSwitchInfo*,
			     GameInfo->More->Misc->NbrDipSwitches+1);

		  GameInfo->More->Misc->DipSwitches[GameInfo->More->Misc->NbrDipSwitches]=DipSwitch;
		  GameInfo->More->Misc->NbrDipSwitches++;
		};
	    }
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_driver:
	  if(Type!=ROMONLY)
	    GameInfo->More->Misc->Driver=ReadDriverInfo();
	  else
	    while(TokenID!=tk_closed_par)
	      TokenID=ReadNextToken(&Token);
	  break;
	case tk_closed_par:
	  EndGameInfo=1;
	  break;
	default:
	  if(strlen(Token)!=0)
	    {
	      NbrUnknownTokens++;
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in ReadGame",Token);
	    }
	};
    };
  
  XFREE(Token);
  XFREE(GameName);
  XFREE(GameDescription);
  return GameInfo;
};


s_GamesListSource *ReadClrMameProHeader(s_GamesList *GamesList)
{
  s_GamesListSource *Source=NULL;
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned int EndClrMameProHeader=0;
  unsigned char *ptr;

  Source=XCALLOC(s_GamesListSource,1);

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  Source->Type=TYPE_FILE;
  Source->StorageMode=STMODE_SPLIT;

  while(!EndClrMameProHeader)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)
	{
	case tk_name:
	  TokenID=ReadNextToken(&Token);
	  Source->Target=XSTRDUP(Token);
	  break;
	case tk_description:
	  TokenID=ReadNextToken(&Token);
	  if(Source->Target!=NULL)
	    {
	      ptr=strstr(Token,Source->Target);
	      if (ptr==NULL)
		Source->TargetVersion=XSTRDUP(Token);
	      else
		{
		  ptr=ptr+strlen(Source->Target);
		  Source->TargetVersion=XSTRDUP(ptr);
		};
	    }
	  else
	    Source->TargetVersion=XSTRDUP(Token);
	  break;
	case tk_category:
	  TokenID=ReadNextToken(&Token);
	  Source->Comment=XSTRDUP(Token);
	  break;
	case tk_version:
	  TokenID=ReadNextToken(&Token);
	  Source->Version=XSTRDUP(Token);
	  break;
	case tk_author:
	  TokenID=ReadNextToken(&Token);
	  Source->Author=XSTRDUP(Token);
	  break; 
	case tk_forcemerging:
	  TokenID=ReadNextToken(&Token);
	  if(strcmp(Token,"merge")==0)
	    Source->ForceStorageMode=STMODE_MERGE;
	  if(strcmp(Token,"split")==0)
	    Source->ForceStorageMode=STMODE_SPLIT;
	  if(strcmp(Token,"full")==0)
	    Source->ForceStorageMode=STMODE_FULL;
	  break; 
	case tk_forcezipping:
	  TokenID=ReadNextToken(&Token);
	  if(strcmp(Token,"yes")==0)
	    Source->ForceStorageFormat=STFMT_ZIP;
	  break;
	case tk_comment:
	  TokenID=ReadNextToken(&Token);
	  break;
	case tk_closed_par:
	  EndClrMameProHeader=1;
	  break;
	default:
	  if(strlen(Token)!=0)
	    NbrUnknownTokens++;
	};
    };
  XFREE(Token);
  return Source;
};
    
s_GamesListSource * ReadMamoryHeader()
{
  s_GamesListSource *Source=NULL;
  e_Token TokenID=0;
  unsigned char *Token=NULL;
  unsigned int EndMamoryHeader=0;
  unsigned int EndManager=0;

  Source=XCALLOC(s_GamesListSource,1);

  TokenID=ReadNextToken(&Token);
  if (TokenID!=tk_opened_par)
    return NULL;

  while(!EndMamoryHeader)
    {
      TokenID=ReadNextToken(&Token);
      switch(TokenID)  
	{
	case tk_type:
	  TokenID=ReadNextToken(&Token);
	  switch(TokenID)
	    {
	    case tk_file:
	      Source->Type=TYPE_FILE;
	      break;
	    case tk_exec:
	      Source->Type=TYPE_EXEC;
	      break;
	    case tk_admameli:
	      Source->Type=TYPE_ADMAMELI;
	      break;
	    case tk_directory:
	      Source->Type=TYPE_DIR;
	      break;
	    };
	  break;
	case tk_source:
	  TokenID=ReadNextToken(&Token);
	  Source->Source=XSTRDUP(Token);
	  break;
	case tk_target:
	  TokenID=ReadNextToken(&Token);
	  Source->Target=XSTRDUP(Token);
	  break;
	case tk_targetver:
	  TokenID=ReadNextToken(&Token);
	  Source->TargetVersion=XSTRDUP(Token);
	  break;
	case tk_author:
	  TokenID=ReadNextToken(&Token);
	  Source->Author=XSTRDUP(Token);
	  break; 
	case tk_version:
	  TokenID=ReadNextToken(&Token);
	  Source->Version=XSTRDUP(Token);
	  break; 
	case tk_comment:
	  TokenID=ReadNextToken(&Token);
	  Source->Comment=XSTRDUP(Token);
	  break;
	case tk_storagemode:
	  TokenID=ReadNextToken(&Token);
	  if(TokenID==tk_opened_par)
	    {
	      TokenID=ReadNextToken(&Token);
	      while(TokenID!=tk_closed_par)
		{
		  if(strcmp(Token,"merge")==0)
		    Source->StorageMode|=STMODE_MERGE;
		  if(strcmp(Token,"split")==0)
		    Source->StorageMode|=STMODE_SPLIT;
		  if(strcmp(Token,"full")==0)
		    Source->StorageMode|=STMODE_FULL;
		  TokenID=ReadNextToken(&Token);
		};
	    }
	  else
	    {
	      if(strcmp(Token,"merge")==0)
		Source->StorageMode=STMODE_MERGE;
	      if(strcmp(Token,"split")==0)
		Source->StorageMode=STMODE_SPLIT;
	      if(strcmp(Token,"full")==0)
		Source->StorageMode=STMODE_FULL;
	    };
	  break;
/*
	case tk_storageformat:
	  TokenID=ReadNextToken(&Token);
	  if(TokenID==tk_opened_par)
	    {
	      TokenID=ReadNextToken(&Token);
	      while(TokenID!=tk_closed_par)
		{
		  if(strcmp(Token,"raw")==0)
		    Source->StorageFormat|=STFMT_RAW;
		  if(strcmp(Token,"zip")==0)
		    Source->StorageFormat|=STFMT_ZIP;
		  TokenID=ReadNextToken(&Token);
		};
	    }
	  else
	    {
	      if(strcmp(Token,"raw")==0)
		Source->StorageFormat=STFMT_RAW;
	      if(strcmp(Token,"zip")==0)
		Source->StorageFormat=STFMT_ZIP;
	    };
	  break;
*/
	case tk_manager:
	  while(!EndManager)
	    {
	      TokenID=ReadNextToken(&Token);

	      TokenID=ReadNextToken(&Token);
	      switch(TokenID)  	  
		{
		case tk_forcestoragemode:
		  TokenID=ReadNextToken(&Token);
		  if(strcmp(Token,"merge")==0)
		    Source->ForceStorageMode=STMODE_MERGE;
		  if(strcmp(Token,"split")==0)
		    Source->ForceStorageMode=STMODE_SPLIT;
		  if(strcmp(Token,"full")==0)
		    Source->ForceStorageMode=STMODE_FULL;
		  break; 
		case tk_forcestorageformat:
		  TokenID=ReadNextToken(&Token);
		  if(strcmp(Token,"zip")==0)
		    Source->ForceStorageFormat=STFMT_ZIP;
		  if(strcmp(Token,"raw")==0)
		    Source->ForceStorageFormat=STFMT_RAW;
		  break;
		case tk_closed_par:
		  EndManager=1;
		  break;
		default:
		  if(strlen(Token)!=0)
		    NbrUnknownTokens++;
		};  
	    }
	  break;
	case tk_closed_par:
	  EndMamoryHeader=1;
	  break;
	default:
	  if(strlen(Token)!=0)
	    {
	      LPRINTF(DEBUG,"Unknown Token \"%s\"",Token);
	      NbrUnknownTokens++;
	    }
	};
    };
  XFREE(Token);
  return Source;
};

s_GamesList *GetGamesListFromAdMAMEliFile(unsigned char *File,
					  unsigned char *Source,
					  unsigned int SourceType)
{
  s_GamesList *GamesList=NULL;
  s_GamesListSource *NewSource=NULL,*DefaultSource=NULL;
  s_GameInfo *GameInfo=NULL;
  s_GameInfo **Games=NULL;
  s_RomInfo** roms;
  unsigned int nbrdata=1,i,j,k,l,m;
  e_Token TokenID=0;
  unsigned char *Token=NULL;

  GamesSourceFile=fopen(File,"r");
  /*
  if(!GamesSourceFile)
    {
      fprintf(stderr,"Can't open file: %s\n",File);
      perror("bouh:");
      fprintf(stderr,"\n");
      return(NULL);
    };
  */
  memset(GamesSourceFileLine,0,BUFSIZx2);
  GamesSourceLinePtr=0;
  
  GamesList=XMALLOC(s_GamesList,1);

  GamesList->MergedGamesList=FALSE;
  GamesList->NbrSources=0;
  GamesList->Sources=NULL;
  DefaultSource=XMALLOC(s_GamesListSource,1);

  DefaultSource->GamesList=GamesList;
  DefaultSource->Type=SourceType;
  DefaultSource->Source=XSTRDUP(Source);
  DefaultSource->Target=NULL;
  DefaultSource->TargetVersion=NULL;
  DefaultSource->Author=NULL;
  DefaultSource->Version=NULL;
  DefaultSource->Comment=NULL;
  DefaultSource->StorageMode=STMODE_SPLIT;
  DefaultSource->ForceStorageMode=STMODE_UNSET;
//  DefaultSource->StorageFormat=STFMT_UNSET;
  DefaultSource->ForceStorageFormat=STFMT_UNSET;
  GamesList->NbrUnknownTokens=0;

  GamesList->NbrResources=0;
  GamesList->Resources=NULL;
  GamesList->NbrGames=0;
  GamesList->Games=NULL;
  GamesList->NbrRomContents=0;
  GamesList->RomContents=NULL;
  GamesList->NbrSoundSamples=0;
  GamesList->SoundSamples=NULL;

  GamesSortBox=InitSortBox(SB_ALPHANUM,NULL);
  RomContentsSortBox=InitSortBox(SB_NUM,NULL);
//  SamplesSortBox=InitSortBox(NULL);

  GamesList->Sources=XREALLOC(GamesList->Sources,
			      s_GamesListSource*,
			      GamesList->NbrSources+1);

  GamesList->Sources[GamesList->NbrSources]=DefaultSource;
  GamesList->NbrSources++;

  TokenID=ReadNextToken(&Token);
  while (Token!=NULL &&
	 (TokenID==tk_game || TokenID==tk_resource || TokenID==tk_clrmamepro ||
	  TokenID==tk_mamory || TokenID==tk_unknown))
    {
      switch(TokenID)
	{
	case tk_mamory:
	  if(DefaultSource!=NULL)
	    {
	      GamesList->Sources[GamesList->NbrSources-1]=NULL;
	      XFREE(GamesList->Sources);
	      GamesList->NbrSources=0;
	      FreeGamesListSource(DefaultSource);
	      DefaultSource=NULL;
	    };
	  NewSource=ReadMamoryHeader();
	  GamesList->Sources=XREALLOC(GamesList->Sources,
				      s_GamesListSource*,
				      GamesList->NbrSources+1);

	  NewSource->GamesList=GamesList;
	  GamesList->Sources[GamesList->NbrSources]=NewSource;
	  GamesList->NbrSources++;
//	  if(GamesList->Source->ForceStorageMode!=STMODE_UNSET)
//	    StorageMode=GamesList->Source->ForceStorageMode;
	  break;
	case tk_clrmamepro:
	  if(DefaultSource!=NULL)
	    {
	      GamesList->Sources[GamesList->NbrSources-1]=NULL;
	      XFREE(GamesList->Sources);
	      GamesList->NbrSources=0;
	      FreeGamesListSource(DefaultSource);
	      DefaultSource=NULL;
	    };
	  NewSource=ReadClrMameProHeader(GamesList);
	  NewSource->GamesList=GamesList;
	  GamesList->Sources=XREALLOC(GamesList->Sources,
				      s_GamesListSource*,
				      GamesList->NbrSources+1);

	  GamesList->Sources[GamesList->NbrSources]=NewSource;
	  if(!NewSource->Source)
	    NewSource->Source=XSTRDUP(File);
	  GamesList->NbrSources++;
//	  if(GamesList->Source->ForceStorageMode!=STMODE_UNSET)
//	    StorageMode=GamesList->Source->ForceStorageMode;
	  break;
	case tk_game:
	  {
	    GameInfo=ReadGameInfo(GamesList,ROMONLY);
//	    printf("Game %s\n",GameInfo->Name);
	    if (GameInfo==NULL)
	      {
		fclose(GamesSourceFile);
		FreeSortBox(RomContentsSortBox);
		RomContentsSortBox=NULL;
		FreeSortBox(GamesSortBox);
		GamesSortBox=NULL;
		XFREE(Token);
		FreeGamesList(GamesList);
		GamesList=NULL;
		return(NULL);
	      }
	    else
	      {
		Games=(s_GameInfo**)SortBox_FindTheWord(GamesSortBox,
							GameInfo->Name,
							(int*)&nbrdata);
		if(nbrdata==0)
		  {
		    GamesList->Games=XREALLOC(GamesList->Games,
					      s_GameInfo*,
					      GamesList->NbrGames+1);

		    GamesList->Games[GamesList->NbrGames]=GameInfo;
		    GamesList->NbrGames++;
		    SortBox_AddOneWord(GamesSortBox,GameInfo->Name,GameInfo);
		  }
		for(i=0;i<nbrdata;i++)
		  Games[i]=NULL;
		XFREE(Games);
	      };
	    break;
	  };
	case tk_resource:
	  {
	    GameInfo=ReadGameInfo(GamesList,ROMONLY);
//	    printf("Resource %s\n",GameInfo->Name);
	    if (GameInfo==NULL)
	      {
		fclose(GamesSourceFile);
		FreeSortBox(RomContentsSortBox);
		RomContentsSortBox=NULL;
		FreeSortBox(GamesSortBox);
		GamesSortBox=NULL;
		XFREE(Token);
		FreeGamesList(GamesList);
		GamesList=NULL;
		return(NULL);
	      }
	    else 
	      {
		GamesList->Games=XREALLOC(GamesList->Games,
					  s_GameInfo*,
					  GamesList->NbrGames+1);

		GamesList->Games[GamesList->NbrGames]=GameInfo;
		GamesList->NbrGames++;
		SortBox_AddOneWord(GamesSortBox,GameInfo->Name,GameInfo);

		for(i=0;i<GamesList->NbrResources;i++)
		  if(strcmp(GameInfo->Name,GamesList->Resources[i]->Name)==0)
		    {
		      GamesList->Resources[i]->Game=GameInfo;
		      break;
		    };

		if(i==GamesList->NbrResources)
		  {
		    GamesList->Resources=XREALLOC(GamesList->Resources,
						  s_ResourceInfo*,
						  GamesList->NbrResources+1);

		    GamesList->Resources[GamesList->NbrResources]=XCALLOC(s_ResourceInfo,1);
		    GamesList->Resources[GamesList->NbrResources]->Name=XSTRDUP(GameInfo->Name);
		    GamesList->Resources[GamesList->NbrResources]->Game=GameInfo;
		    GamesList->NbrResources++;
		  };
	      };
	    break;
	  };
//	default:
//	  TokenID=ReadNextToken(&Token);
	}; 
      TokenID=ReadNextToken(&Token);
    };

  for(i=0;i<NbrTempGames;i++)
    {
//      printf("Game %s",TempGames[i]->Name);
      nbrdata=1;
      Games=SortBox_FindTheWord(GamesSortBox,
				TempGames[i]->More->CloneOf,
				(int*)(&nbrdata));
      if(nbrdata==1)
	{
	  if(Games[0]->More->Resource!=NULL)
	    {
	      TempGames[i]->More->Resource=XSTRDUP(Games[0]->More->Resource);
	      CheckGamesListResources(GamesList,TempGames[i]);
	    };	      
	};
//      printf("\n");
      for(j=0;j<nbrdata;j++)
	Games[j]=NULL;
      XFREE(Games);
    };

  XFREE(TempGames);
  NbrTempGames=0;


  for(i=0;i<GamesList->NbrResources;i++)
    {
      if(GamesList->Resources[i]->Game==NULL)
	{
	  LPRINTF(WARNING,"Data for resource '%s' are missing.",
		  GamesList->Resources[i]->Name);
	  GamesList->Resources[i]->Game=XCALLOC(s_GameInfo,1);
	  GamesList->Resources[i]->Game->Name=
	    XSTRDUP(GamesList->Resources[i]->Name);
	  GamesList->Resources[i]->Game->More=XCALLOC(s_MoreGameInfo,1);
	  GamesList->Games=XREALLOC(GamesList->Games,
				    s_GameInfo*,
				    GamesList->NbrGames+1);
	  GamesList->Games[GamesList->NbrGames]=
	    GamesList->Resources[i]->Game;
	  GamesList->NbrGames++;

	  // Transfert resource rom into the game (Merged -> Rom)
	  for(j=0;j<GamesList->Resources[i]->NbrAssociatedGames;j++)
	    {
	      for(k=0;k<GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms;k++)
		{
		  if(strcmp(GamesList->Resources[i]->AssociatedGames[j]->Name,
			    GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName)==0)
		    {
		      GamesList->Resources[i]->AssociatedGames[j]->Roms=
			XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->Roms,
				 s_RomInfo*,
				 GamesList->Resources[i]->AssociatedGames[j]->NbrRoms+1);
		      GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]=GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k];
		      
		      // Add the game to the rom associated games
		      for(l=0;l<GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames;l++)
			if(GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[l]->Game==GamesList->Resources[i]->AssociatedGames[j])
			  break;
		      
		      if(l==GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames)
			{		      
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames=
			    XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames,
				     s_RomAssociatedGame*,
				     GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames+1);
			  
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]=
			    XCALLOC(s_RomAssociatedGame,1);
			  
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]->NbFiles=1;
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]->Game=GamesList->Resources[i]->AssociatedGames[j];
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames++;
			}
		      else
			GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[l]->NbFiles++;
		      
		      GamesList->Resources[i]->AssociatedGames[j]->NbrRoms++;
		      
		      // remove the rom from the Merged Roms
		      GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]=NULL;
		      GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms=
			XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms,
				 s_RomInfo*,
				 GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms-1);
		      GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms--;
		    };
		};		
	    };
	};
    };


  if(GamesList->Sources[0]->StorageMode==STMODE_SPLIT)
    {
      for(i=0;i<GamesList->NbrGames;i++)
	{
	  if(GamesList->Games[i]->More->RomOf!=NULL)
	    {
	      nbrdata=1;
	      Games=SortBox_FindTheWord(GamesSortBox,
				GamesList->Games[i]->More->RomOf,
				(int*)(&nbrdata));
	      if(nbrdata==0)
		{
		  LPRINTF(WARNING,"Parent game '%s' for '%s' not found",
			  GamesList->Games[i]->More->RomOf,
			  GamesList->Games[i]->Name);
		  continue;
		};

	      if(GamesList->Games[i]->More->NbrMergedRoms==0)
		{
		  for(j=0;j<GamesList->Games[i]->NbrRoms;j++)
		    {
		      for(k=0;k<Games[0]->NbrRoms;k++)
			{
			  if(GamesList->Games[i]->Roms[j]->Content->Size==Games[0]->Roms[k]->Content->Size &&
			     GamesList->Games[i]->Roms[j]->Content->CRC==Games[0]->Roms[k]->Content->CRC)
			    {
			      GamesList->Games[i]->More->MergedRoms=
				XREALLOC(GamesList->Games[i]->More->MergedRoms,
					 s_RomInfo*,
					 GamesList->Games[i]->More->NbrMergedRoms+1);

			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]=GamesList->Games[i]->Roms[j];
			      XFREE(GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName);
			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName=XSTRDUP(GamesList->Games[i]->More->CloneOf);
			      GamesList->Games[i]->More->NbrMergedRoms++;
			      for(l=0;l<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames;l++)
				if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game==GamesList->Games[i])
				  break;

			      if(l!=GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames)
				{
				  GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles--;
				  if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles==0)
				    {
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game=NULL;
				      XFREE(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]);
				      for(m=l;m<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1;m++)
					GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m+1];
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=NULL;
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames=XREALLOC(GamesList->Games[i]->Roms[j]->Content->AssociatedGames,s_RomAssociatedGame*,GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1);
				      GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames--;
				    };
				}
			      else
				{
				  LPRINTF(WARNING,"Game '%s' not found as a romcontent associated game of %x/%i (CRC/size)",
					  GamesList->Games[i]->Name,
					  GamesList->Games[i]->Roms[j]->Content->CRC,
					  GamesList->Games[i]->Roms[j]->Content->Size);
				};
			      GamesList->Games[i]->Roms[j]=NULL;
			      break;
			    };
			};
		    };
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      l++;
		  roms=XCALLOC(s_RomInfo*,l);
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      {
			roms[l]=GamesList->Games[i]->Roms[k];
			l++;
		      };
		  XFREE(GamesList->Games[i]->Roms);
		  GamesList->Games[i]->Roms=roms;
		  GamesList->Games[i]->NbrRoms=l;
		};
	      XFREE(Games);
	    };

	  if(GamesList->Games[i]->More->Resource!=NULL &&
	     strcmp(GamesList->Games[i]->More->Resource,
		    GamesList->Games[i]->More->RomOf)!=0)
	    {
	      nbrdata=1;
	      Games=SortBox_FindTheWord(GamesSortBox,
				GamesList->Games[i]->More->Resource,
				(int*)(&nbrdata));
	      if(nbrdata==0)
		{
		  LPRINTF(WARNING,"Resource '%s' for '%s' not found",
			  GamesList->Games[i]->More->Resource,
			  GamesList->Games[i]->Name);
		  continue;
		}; 
//	      if(GamesList->Games[i]->More->NbrMergedRoms==0)
		{
		  for(j=0;j<GamesList->Games[i]->NbrRoms;j++)
		    {
		      for(k=0;k<Games[0]->NbrRoms;k++)
			{
			  if(GamesList->Games[i]->Roms[j]->Content->Size==Games[0]->Roms[k]->Content->Size &&
			     GamesList->Games[i]->Roms[j]->Content->CRC==Games[0]->Roms[k]->Content->CRC)
			    {
			      GamesList->Games[i]->More->MergedRoms=
				XREALLOC(GamesList->Games[i]->More->MergedRoms,
					 s_RomInfo*,
					 GamesList->Games[i]->More->NbrMergedRoms+1);

			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]=GamesList->Games[i]->Roms[j];
			      XFREE(GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName);
			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName=XSTRDUP(GamesList->Games[i]->More->Resource);
			      GamesList->Games[i]->More->NbrMergedRoms++;
			      for(l=0;l<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames;l++)
				if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game==GamesList->Games[i])
				  break;

			      if(l!=GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames)
				{
				  GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles--;
				  if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles==0)
				    {
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game=NULL;
				      XFREE(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]);
				      for(m=l;m<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1;m++)
					GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m+1];
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=NULL;
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames=XREALLOC(GamesList->Games[i]->Roms[j]->Content->AssociatedGames,s_RomAssociatedGame*,GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1);
				      GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames--;
				    };
				}
			      else
				{
				  LPRINTF(WARNING,"Game '%s' not found as a romcontent associated game of %x/%i (CRC/size)",
					  GamesList->Games[i]->Name,
					  GamesList->Games[i]->Roms[j]->Content->CRC,
					  GamesList->Games[i]->Roms[j]->Content->Size);
				};
			      GamesList->Games[i]->Roms[j]=NULL;
			      break;
			    };
			};
		    };
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      l++;
		  roms=XCALLOC(s_RomInfo*,l);
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      {
			roms[l]=GamesList->Games[i]->Roms[k];
			l++;
		      };
		  XFREE(GamesList->Games[i]->Roms);
		  GamesList->Games[i]->Roms=roms;
		  GamesList->Games[i]->NbrRoms=l;
		};
	      XFREE(Games);
	    };
	};
    };

  for(i=0;i<GamesList->NbrResources;i++)
    {  
      for(j=0;j<GamesList->Resources[i]->NbrAssociatedGames;j++)
	{
	  for(k=0;k<GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms;k++)
	    {
	      for(l=0;l<GamesList->Resources[i]->Game->NbrRoms;l++)
		if(strcmp(GamesList->Resources[i]->Game->Roms[l]->Name,
			  GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->Name)==0)
		  {
		    XFREE(GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName);
		    GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName=XSTRDUP(GamesList->Resources[i]->Name);
		  };
	    };
	};
    };
  

  fclose(GamesSourceFile);
  FreeSortBox(RomContentsSortBox);
  RomContentsSortBox=NULL;
  FreeSortBox(GamesSortBox);
  GamesSortBox=NULL;
  GamesList->NbrUnknownTokens=NbrUnknownTokens;
  XFREE(Token);
  return GamesList;
};

int isAdMAMEliFile(const char *Path)
{
  e_Token TokenID;
  unsigned char *Token=NULL;
  s_GamesList *GamesList=NULL;
  s_GamesListSource *Source=NULL;
  s_GameInfo *Game=NULL;
  unsigned int result=FALSE;

  GamesSourceFile=fopen(Path,"r");
  memset(GamesSourceFileLine,0,BUFSIZx2);
  GamesSourceLinePtr=0;

  GamesSortBox=InitSortBox(SB_ALPHANUM,NULL);
  RomContentsSortBox=InitSortBox(SB_NUM,NULL);
  NbrUnknownTokens=0;

  GamesList=XCALLOC(s_GamesList,1);
  Source=XCALLOC(s_GamesListSource,1);
  Source->GamesList=GamesList;
  Source->Source=XSTRDUP("test");
  GamesList->Sources=XCALLOC(s_GamesListSource*,1);

  GamesList->Sources[0]=Source;
  GamesList->NbrSources++;

  TokenID=ReadNextToken(&Token);
  switch(TokenID)
    {  
    case tk_mamory:
      FreeGamesListSource(Source);
      Source=NULL;
      GamesList->Sources[0]=NULL;
      if(!(Source=ReadMamoryHeader()) ||
	 NbrUnknownTokens!=0)
	result=FALSE;
      else
	result=TRUE;
      FreeGamesListSource(Source);
      Source=NULL;
      break;
    case tk_clrmamepro:
      FreeGamesListSource(Source);
      Source=NULL;
      GamesList->Sources[0]=NULL;
      if(!(Source=ReadClrMameProHeader(GamesList)) ||
	 NbrUnknownTokens!=0)
	result=FALSE;
      else
	result=TRUE;
      FreeGamesListSource(Source);
      Source=NULL;
      break;
    case tk_resource:
    case tk_game:
      if(!(Game=ReadGameInfo(GamesList,ROMONLY)) ||
	 NbrUnknownTokens!=0)
	result=FALSE;
      else
	    result=TRUE;
      break;	  
    };

  FreeGamesList(GamesList);
  fclose(GamesSourceFile);
  FreeSortBox(RomContentsSortBox);
  RomContentsSortBox=NULL;
  FreeSortBox(GamesSortBox);
  GamesSortBox=NULL;
  XFREE(Token);
  return result;
};
