/*
 The MIT License

 Copyright (c) 2010-2020 Paul R. Holser, Jr.

 Permission is hereby granted, free of charge, to any person obtaining
 a copy of this software and associated documentation files (the
 "Software"), to deal in the Software without restriction, including
 without limitation the rights to use, copy, modify, merge, publish,
 distribute, sublicense, and/or sell copies of the Software, and to
 permit persons to whom the Software is furnished to do so, subject to
 the following conditions:

 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

package com.pholser.junit.quickcheck;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assume.assumeThat;
import static org.junit.Assume.assumeTrue;

import com.pholser.junit.quickcheck.runner.JUnitQuickcheck;
import org.junit.runner.RunWith;

/**
 * Swiped from <a href="http://stackoverflow.com/questions/837484/junit-theory-for-hashcode-equals-contract">here</a>.
 */
@RunWith(JUnitQuickcheck.class)
public abstract class EqualsHashCodePropertiesTest<T> {
    @Property public void equalsIsReflexive(T o) {
        assumeThat(o, not(equalTo(null)));

        assertEquals(o, o);
    }

    @Property public void equalsIsSymmetric(T x, T y) {
        assumeThat(x, not(equalTo(null)));
        assumeThat(y, not(equalTo(null)));
        assumeTrue(y.equals(x));

        assertEquals(x, y);
    }

    @Property public void equalsIsTransitive(T x, T y, T z) {
        assumeThat(x, not(equalTo(null)));
        assumeThat(y, not(equalTo(null)));
        assumeThat(z, not(equalTo(null)));
        assumeTrue(x.equals(y) && y.equals(z));

        assertEquals(z, x);
    }

    @Property public void equalsIsConsistent(T x, T y) {
        assumeThat(x, is(not(equalTo(null))));
        boolean alwaysTheSame = x.equals(y);

        for (int i = 0; i < 30; i++)
            assertThat(x.equals(y), is(alwaysTheSame));
    }

    @Property public void equalsReturnsFalseOnNull(T x) {
        assumeThat(x, not(equalTo(null)));

        assertNotEquals(x, null);
    }

    @Property public void hashCodeIsSelfConsistent(T x) {
        assumeThat(x, not(equalTo(null)));
        int alwaysTheSame = x.hashCode();

        for (int i = 0; i < 30; i++)
            assertThat(x.hashCode(), is(alwaysTheSame));
    }

    @Property public void hashCodeIsConsistentWithEquals(T x, T y) {
        assumeThat(x, not(equalTo(null)));
        assumeTrue(x.equals(y));

        assertEquals(x.hashCode(), y.hashCode());
    }

    @Property public void equalsWorks(T x, T y) {
        assumeThat(x, not(equalTo(null)));
        assumeTrue(x == y);

        assertEquals(x, y);
    }

    @Property public void notEqualsWorks(T x, T y) {
        assumeThat(x, not(equalTo(null)));
        assumeTrue(x != y);

        assertNotEquals(x, y);
    }
}
