/* #includes *//*{{{C}}}*//*{{{*/
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 200809L

#include "config.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <assert.h>
#include <ctype.h>
#ifdef HAVE_GETTEXT
#include <libintl.h>
#define _(String) gettext(String)
#else
#define _(String) String
#endif
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "complete.h"
#include "display.h"
#include "macro.h"
#include "misc.h"
#include "msgline.h"
/*}}}*/
/* #defines */ /*{{{*/
#ifdef BROKEN_CURSES_PROTOTYPES
#define CAST_CHAR_PTR (char*)
#else
#define CAST_CHAR_PTR
#endif
/*}}}*/

/* variables *//*{{{*/
static int msg;
/*}}}*/

static int metacharlen(const char *s)/*{{{*/
{
  size_t len;

  for (len=0; *s; s+=bf_rcharlen(s)) ++len;
  return len;
}
/*}}}*/

/* showmsg      -- show a message *//*{{{*/
void showmsg(const char *fmt, ...)
{
  /* variables *//*{{{*/
  va_list ap;
  int x,maxx,maxy;
  char c;
  int oldy,oldx;
  /*}}}*/
  
  if (!msg && fmt==(const char*)0) return;
  (void)(getyx(stdscr,oldy,oldx));
  (void)move(LINES-1,0);
  (void)clrtoeol();
  if (fmt==(const char*)0) msg=0;
  else
  {
    va_start(ap,fmt);
    x=0;
    getmaxyx(stdscr,maxy,maxx); (void)maxy;
    while (*fmt)
    {
      if (*fmt=='%')
      {
        switch (c=*++fmt)
        {
          /* d *//*{{{*/
          case 'd':
          {
            int u;
            char ustr[10],*s;

            u=va_arg(ap,int);
            if (u<0)
            {
              (void)addch('-');
              u=-u;
            }
            s=ustr;
            do { *s=(u%10)+'0'; u/=10; if (u) ++s; } while (u);
            while (x<maxx-1 && s>=ustr) { (void)addch((chtype)((unsigned char)*s--)); ++x; }
            break;
          }
          /*}}}*/
          /* s *//*{{{*/
          case 's':
          {
            char *s;

            s=va_arg(ap,char*);
            while (x<maxx-1 && *s) { (void)addch((chtype)((unsigned char)*s)); ++s; ++x; }
            break;
          }
          /*}}}*/
          /* c *//*{{{*/
          case 'c':
          {
            char ch;

            ch=va_arg(ap,int);
            if (x<maxx-1) { (void)addch((chtype)ch); ++x; }
            break;
          }
          /*}}}*/
          /* o *//*{{{*/
          case 'o':
          {
            char s[32],*p;
            int i;
            unsigned int oc;

            oc=va_arg(ap,unsigned int);
            for (i=0; oc; oc>>=3,++i)
            {
              s[i]=(((unsigned int)((unsigned char)oc))&7)+'0';
            }
            s[i]='\\';
            p=s+i;
            while (x<maxx-1 && p>=s) { (void)addch((chtype)((unsigned char)*p--)); ++x; }
            break;
          }
          /*}}}*/
        }
        ++fmt;
      }
      else if (x<maxx-1)
      {
        size_t len;

        len=bf_rcharlen(fmt);
        while (len--) (void)addch((chtype)((unsigned char)*fmt++));
        ++x;
      }
    }
    va_end(ap);
    msg=1;
  }
  (void)move(oldy,oldx);
}
/*}}}*/
/* editline     -- line editor *//*{{{*/
char *editline(const char *prompt, const char *preset, char *history[HISTORY_LEN])
{
  /* variables */ /*{{{*/
  int promptlen;
  struct Display *d;
  struct Buffer *b;
  int inhist;
  struct MacroChar mc;
  /*}}}*/

  /* asserts */ /*{{{*/
  assert(prompt);
  /*}}}*/
  promptlen=metacharlen(prompt)+1;
  (void)mvwaddstr(stdscr,LINES-1,0,CAST_CHAR_PTR prompt); (void)addch(' ');
  inhist=-1;

  b=bf_alloc();
  if (preset) bf_insertstr(b,preset,preset+strlen(preset));
  d=ds_alloc(b,promptlen,LINES-1,COLS-promptlen,1);

  do
  {
    ds_drawline(d,1,0,0);
    move(d->oriy,d->orix+d->x-d->offx);
    mc_get(&mc,ds_utf8term);
    switch (mc.key)
    {
      /* C-a, KEY_BEG       -- beginning of line         *//*{{{*/
      case '\001':
      case KEY_BEG:
      {
        ds_beginline(d);
        break;
      }  
      /*}}}*/
      /* C-b, KEY_LEFT      -- backward character        *//*{{{*/
      case KEY_LEFT:
      case '\002': ds_backward(d); break;
      /*}}}*/
      /* C-f, KEY_RIGHT     -- forward character         *//*{{{*/
      case '\006':
      case KEY_RIGHT: ds_forward(d); break;
      /*}}}*/
      /* C-h, KEY_BACKSPACE -- delete previous character *//*{{{*/
      case '\010':
      case KEY_BACKSPACE:
      {
        ds_deletech(d,1);
        break;
      }
      /*}}}*/
      /* C-d, KEY_DC        -- delete character          *//*{{{*/
      case '\004':
      case KEY_DC:
      {
        ds_deletech(d,0);
        break;
      }
      /*}}}*/
      /* C-e, KEY_END       -- end of line               *//*{{{*/
      case '\005':
      case KEY_END:
      {
        ds_endline(d);
        break;
      }  
      /*}}}*/
      /* C-i                -- complete file             *//*{{{*/
      case '\t':
      {
        char *path;

        if ((path=strndup(d->buffer->buffer, d->buffer->gapstart-d->buffer->buffer)))
        {
          char *cont;

          if ((cont=completefile(path)))
          {
            bf_insertstr(d->buffer,cont,cont+strlen(cont));
            free(cont);
          }
          free(path);
        }
        break;
      }
      /*}}}*/
      /* C-l                -- redraw screen             *//*{{{*/
      case '\014': (void)touchwin(curscr); (void)wrefresh(curscr); break;
      /*}}}*/
      /* KEY_IC             -- toggle insert mode        *//*{{{*/
      case KEY_IC:
      {
        if (d->mode&OVERWRITE) ds_delmode(d,OVERWRITE);
        else ds_addmode(d,OVERWRITE);
        break;
      }  
      /*}}}*/
      /* KEY_EIC            -- end insert mode           *//*{{{*/
      case KEY_EIC:
      {
        ds_delmode(d,OVERWRITE);
        break;
      }
      /*}}}*/
      /* C-t                -- transpose characters      *//*{{{*/
      case '\024':
      {
        ds_transpose(d);
        break;
      }
      /*}}}*/
      /* C-k, KEY_DL        -- kill to end of line       *//*{{{*/
      case '\013':
      case KEY_DL:
      {
        struct Buffer *killed;

        if ((killed=ds_moveline(d))) bf_free(killed);
        break;
      }
      /*}}}*/
      /* C-p, KEY_UP        -- previous history entry    *//*{{{*/
      case '\020':
      case KEY_UP:
      {
        if (history && inhist<(HISTORY_LEN-1) && history[inhist+1])
        {
          char *h;

          h=history[++inhist];
          ds_beginline(d);
          bf_empty(d->buffer);
          bf_insertstr(d->buffer,h,h+strlen(h));
        }
        break;
      }
      /*}}}*/
      /* C-n, KEY_DOWN      -- next history entry        *//*{{{*/
      case '\016':
      case KEY_DOWN:
      {
        if (history && inhist>0 && history[inhist-1])
        {
          char *h;

          h=history[--inhist];
          ds_beginline(d);
          bf_empty(d->buffer);
          bf_insertstr(d->buffer,h,h+strlen(h));
        }
        break;
      }
      /*}}}*/
      /* C-q                -- quote character           *//*{{{*/
      case '\021':
      {
        struct MacroChar mc2;

        mc_get(&mc,ds_utf8term);
        if (mc.key==KEY_BACKSPACE) mc.key='\010';
        if (FUNCTION_KEY(mc.key)) break;
        if (!mc.key) break;
        if (ds_compose((char)mc.key,'\0'))
        {
          const struct MetaChar *comp;

          mc_get(&mc2,ds_utf8term);
          if (FUNCTION_KEY(mc2.key)) break;
          if (!(comp=ds_compose((char)mc.key,(char)mc2.key))) break;
          mc.ch=*comp;
        }
        ds_insertch(d,&mc.ch);
        break;
      }
      /*}}}*/
      /* default            -- insert printable character*//*{{{*/
      default:
      {
        if (!FUNCTION_KEY(mc.key) && ((unsigned char)ASCII(mc.ch))>=' ') ds_insertch(d,&mc.ch);
        break;
      }
      /*}}}*/
    }
  } while (mc.key!=KEY_ENTER && mc.key!=(chtype)'\n' && mc.key!=(chtype)'\r' && mc.key!='\007' && mc.key!=KEY_CANCEL);
  (void)move(LINES-1,0);
  (void)clrtoeol();
  if (mc.key==KEY_CANCEL || mc.key=='\007')
  {
    ds_free(d);
    return (char*)0;
  }
  else
  {
    size_t len;
    char *ret;

    bf_end(d->buffer);
    len=d->buffer->gapstart-d->buffer->buffer+1;
    if ((ret=malloc(len)))
    {
      memcpy(ret,d->buffer->buffer,len-1);
      ret[len-1]='\0';
      if (history!=(char**)0)
      {
        if (history[HISTORY_LEN-1]) free(history[HISTORY_LEN-1]);
        memmove(history+1,history,sizeof(history[0])*(HISTORY_LEN-1));
        if ((history[0]=malloc(len))) memcpy(history[0],ret,len);
      }
    }
    ds_free(d);
    return ret;
  }
}
/*}}}*/
/* menuline     -- one line menu *//*{{{*/
/* Notes *//*{{{*/
/*

The choices are terminated by the last element having a (const char*)str
field.  Each item can be chosen by tolower(*str) or by the key stored in
the c field.  Use a space as first character of str, if you only want the
function key to work.

*/
/*}}}*/
int menuline(const char *prompt, const MenuChoice *choice, int curx)
{
  /* variables *//*{{{*/
  int maxx,x,offx,width,mx,my;
  struct MacroChar mc;
  size_t promptlen;
  int oldy,oldx;
  /*}}}*/

  /* asserts *//*{{{*/
  assert(prompt!=(const char*)0);
  assert(choice!=(const MenuChoice*)0);
  assert(curx>=0);
  /*}}}*/
  getyx(stdscr,oldy,oldx);
  getmaxyx(stdscr,my,mx); (void)my;
  (void)mvwaddstr(stdscr,LINES-1,0,CAST_CHAR_PTR prompt);
  promptlen=strlen(prompt);
  for (maxx=0; (choice+maxx)->str!=(const char*)0; ++maxx);
  offx=0;
  do
  {
    (void)move(LINES-1,(int)promptlen);
    /* correct offset so choice is visible *//*{{{*/
    if (curx<=offx) offx=curx;
    else do
    {
      for (width=(int)promptlen,x=offx; x<maxx && width+(int)strlen((choice+x)->str+1)+1<=mx; width+=strlen((choice+x)->str)+1,++x);
      --x;
      if (x<curx) ++offx;
    } while (x<curx);
    /*}}}*/
    /* show visible choices *//*{{{*/
    for (width=promptlen,x=offx; x<maxx && width+(int)strlen((choice+x)->str+1)+1<=mx; width+=strlen((choice+x)->str+1)+1,++x)
    {
      (void)addch(' ');
      if (x==curx) (void)attron(A_REVERSE);
      (void)addstr((const char*)(choice+x)->str+1);
      if (x==curx) (void)attroff(A_REVERSE);
    }
    /*}}}*/
    if (width<mx) (void)clrtoeol();
    (void)move(oldy,oldx);
    mc_get(&mc,ds_utf8term);
    switch (mc.key)
    {
      /* KEY_LEFT              -- move to previous item *//*{{{*/
      case '\002':
      case KEY_BACKSPACE:
      case KEY_LEFT: if (curx>0) --curx; else curx=maxx-1; break;
      /*}}}*/
      /* Space, Tab, KEY_RIGHT -- move to next item *//*{{{*/
      case '\006':
      case ' ':
      case '\t':
      case KEY_RIGHT: if (curx<(maxx-1)) ++curx; else curx=0; break;
      /*}}}*/
      /* default               -- search choice keys *//*{{{*/
      default:
      {
        int i;

        for (i=0; (choice+i)->str!=(const char*)0; ++i)
        {
          if ((mc.key<256 && tolower(mc.key)==*((choice+i)->str)) || (choice+i)->c==mc.key)
          {
            mc.key=KEY_ENTER;
            curx=i;
          }
        }
      }
      /*}}}*/
    }
  }
  while (mc.key!=KEY_ENTER && mc.key!='\n' && mc.key!='\r' && mc.key!=KEY_CANCEL && mc.key!='\007');
  (void)move(LINES-1,0);
  (void)clrtoeol();
  (void)move(oldy,oldx);
  return ((mc.key==KEY_CANCEL || mc.key=='\007') ? -1 : curx);
}
/*}}}*/
/* ok           -- one line yes/no menu *//*{{{*/
int ok(const char *prompt, int curx)
{
  /* variables *//*{{{*/
  char *yes, *no;
  MenuChoice menu[3];
  int result;
  /*}}}*/

  /* asserts *//*{{{*/
  assert(curx==0 || curx==1);
  /*}}}*/
  menu[0].str=no=strdup(_("nNo"));  menu[0].c='\0';
  menu[1].str=yes=strdup(_("yYes")); menu[1].c='\0';
  menu[2].str=(const char*)0;
  result=menuline(prompt,menu,curx);
  free(no);
  free(yes);
  return result;
}
/*}}}*/
