/* #includes *//*{{{C}}}*//*{{{*/
#ifndef NO_POSIX_SOURCE
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#ifdef OLD_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
#include <unistd.h>

#include "buffer.h"
#include "misc.h"

#include <string.h>
/*}}}*/
/* #defines *//*{{{*/
#define DEF_BUFLEN 1024

#if 1
#define ASSERT_CONSISTENCY(b) \
assert(b!=(struct Buffer*)0); \
assert(b->buffer!=(char*)0); \
assert(b->gapstart>=b->buffer); \
assert(b->gapend<=b->buffer+b->size-1)
#else
#define ASSERT_CONSISTENCY(b)
#endif

#define BF_FORWARDOK(b) (b->gapend<b->buffer+b->size-1)

/*
In UTF-8 mode the first char encodes the length:
110xxxxx = 2 bytes
1110xxxx = 3 bytes
11110xxx = 4 bytes
*/

#define BF_FORWARD(b) \
do \
{ \
  char bf_prev; \
  ++b->curch; \
  bf_prev=*(b->gapstart++)=*(++b->gapend); \
  if (bf_prev=='\0' && *(b->gapend+1)==QUOTE) \
  { \
    *(b->gapstart++)=*(++b->gapend); \
    *(b->gapstart++)=*(++b->gapend); \
  } \
  else if (bf_prev=='\0') *(b->gapstart++)=*(++b->gapend); \
  else if (bf_prev=='\n') ++b->cury; \
  else if (bf_utf8) \
  { \
    if ((bf_prev & 0xf0)==0xf0) *(b->gapstart++)=*(++b->gapend); \
    if ((bf_prev & 0xe0)==0xe0) *(b->gapstart++)=*(++b->gapend); \
    if ((bf_prev & 0xc0)==0xc0) *(b->gapstart++)=*(++b->gapend); \
  } \
} while (0)

#define BF_BACKWARDOK(b) (b->gapstart>b->buffer)

#define BF_BACKWARD(b) \
do \
{ \
  char bf_cur; \
  bf_cur = *(b->gapend--)=*(--b->gapstart); \
  if (b->gapstart>b->buffer && *(b->gapstart-1)=='\0') \
  { \
    *(b->gapend--)=*(--b->gapstart); \
  } \
  else if (b->gapstart-1>b->buffer && *(b->gapstart-2)=='\0' && *(b->gapstart-1)==QUOTE) \
  { \
    *(b->gapend--)=*(--b->gapstart); \
    *(b->gapend--)=*(--b->gapstart); \
  } \
  else if (bf_cur=='\n') --b->cury; \
  else if ((bf_utf8) && (bf_cur & 0x80)) \
  { \
    while ((bf_cur & 0xc0) != 0xc0) bf_cur = *(b->gapend--)=*(--b->gapstart); \
  } \
  --b->curch; \
} while (0);
/*}}}*/

/* variables *//*{{{*/
int bf_utf8;

const struct MetaChar bf_foldstart = { { '\0', FOLDSTART } };
const struct MetaChar bf_foldend = { { '\0', FOLDEND } };

struct Language language[LANGUAGES]=
{
  { "C", "/*", "*/" },
  { "script", "#", "" },
  { "mail", "# ", "" },
  { "X11", "!", "" },
  { "roff", "\\\"", "" },
  { "SGML", "<!--", "-->" },
  { "Occam", "--", "" },
  { "C++", "//", "" },
  { "TeX", "%", "" },
  { "Texinfo", "@c ", "" },
  { "Lisp", ";", "" },
  { "Pascal", "(*", "*)" },
  { "", "", "" },
  { (const char*)0, (const char*)0, (const char*)0 },
};
/*}}}*/

/* bf_alloc        -- allocate new buffer *//*{{{*/
struct Buffer *bf_alloc(void)
{
  struct Buffer *b;

  if ((b=malloc(sizeof(struct Buffer)))==(struct Buffer*)0) return b;
  b->size=DEF_BUFLEN;
  if ((b->buffer=malloc(DEF_BUFLEN+1))==(char*)0)
  {
    free(b);
    return (struct Buffer*)0;
  }
  *(b->buffer+DEF_BUFLEN)='\0';
  b->gapstart=b->buffer;
  b->gapend=b->buffer+b->size-1;
  b->cury=0;
  b->mode=0;
  b->name=(char*)0;
  b->curch=0;
  b->msz=0;
  b->refcnt=0;
  b->lang=&language[sizeof(language)/sizeof(language[0])-2];
  ASSERT_CONSISTENCY(b);
  return b;
}
/*}}}*/
/* bf_free         -- free buffer *//*{{{*/
void bf_free(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (b->name) free(b->name);
  free(b->buffer);
  b->buffer=b->gapstart=b->gapend=(char*)0;
  b->msz=0;
  free(b);
}
/*}}}*/
/* bf_empty        -- empty buffer */ /*{{{*/
int bf_empty(struct Buffer *b)
{
  int i;

  if (b->mode&READONLY) return 0;
  b->gapstart=b->buffer;
  b->gapend=b->buffer+b->size-1;
  b->cury=0;
  b->mode=0;
  b->curch=0;
  for (i=0; i<b->msz; ++i) b->mark[i]=0;
  b->lang=&language[sizeof(language)/sizeof(language[0])-2];
  ASSERT_CONSISTENCY(b);
  return 1;
}
/*}}}*/
/* bf_forward      -- move one character forward *//*{{{*/
int bf_forward(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (BF_FORWARDOK(b))
  {
    BF_FORWARD(b);
    ASSERT_CONSISTENCY(b);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* bf_backward     -- move one character backward *//*{{{*/
int bf_backward(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (BF_BACKWARDOK(b)) { BF_BACKWARD(b); return 1; }
  else return 0;
}
/*}}}*/
/* bf_lcharlen     -- return length of ending MetaChar *//*{{{*/
size_t bf_lcharlen(const char *s, const char *start)
{
  const char *end;

  end=s;
  if (s>start && *(s-1)=='\0') --s;
  if (s-1>start && *(s-2)=='\0' && *(s-1)==QUOTE) s-=2;
  else if (bf_utf8 && (*s & 0x80))
  {
    for (--s; (*s & 0xc0) != 0xc0; --s);
  }
  return end-s+1;
}
/*}}}*/
/* bf_rcharlen     -- return length of starting MetaChar *//*{{{*/
size_t bf_rcharlen(const char *s)
{
  if (*s=='\0')
  {
    return (*(s+1)==QUOTE ? 3 : 2);
  }
  else if (bf_utf8)
  {
    if (!(*s & 0x80)) return 1;
    if ((*s & 0xe0) == 0xc0) return 2;
    else if ((*s & 0xf0) == 0xe0) return 3;
    else if ((*s & 0xf8) == 0xf0) return 4;
    else assert(0);
  }
  else return 1;
}
/*}}}*/
/* bf_insert       -- insert one character *//*{{{*/
int bf_insert(struct Buffer *b, const struct MetaChar *ch)
{
  int i,len;
  const char *cp;

  ASSERT_CONSISTENCY(b);
  if (b->mode&READONLY) return 0;

  if (b->gapstart>=b->gapend-4) /* increase gap *//*{{{*/
  {
    char *new_buffer;
    size_t new_size,pregap_length,postgap_length;

    new_size=b->size*2;
    if ((new_buffer=malloc(new_size+1))==(char*)0) return 0;
    *(new_buffer+new_size)='\0';
    pregap_length=b->gapstart-b->buffer;
    postgap_length=b->buffer+b->size-1-b->gapend;
    if (pregap_length!=(size_t)0)
    {
      memcpy(new_buffer,b->buffer,pregap_length);
    }
    if (postgap_length!=(size_t)0)
    {
      memcpy(new_buffer+new_size-postgap_length,b->gapend+1,postgap_length);
    }
    free(b->buffer);
    b->buffer=new_buffer;
    b->size=new_size;
    b->gapstart=b->buffer+pregap_length;
    b->gapend=b->buffer+b->size-1-postgap_length;
    ASSERT_CONSISTENCY(b);
  }
  /*}}}*/

  for (i=0; i<b->msz; ++i) if (b->gapstart-b->buffer<=b->mark[i]) ++b->mark[i];

  cp=ch->c;
  if (*cp=='\n') ++b->cury;
  for (i=0,len=bf_rcharlen(cp); i<len; ++i) *(b->gapstart++)=*cp++;
  b->mode|=CHANGED;
  ++b->curch;
  return 1;
}
/*}}}*/
/* bf_delete       -- delete one character *//*{{{*/
int bf_delete(struct Buffer *b)
{
  int i;

  ASSERT_CONSISTENCY(b);
  if (b->mode&READONLY) return 0;
  if (b->gapstart==b->buffer) return 0;
  else
  {
    struct MetaChar ch;

    for (i=0; i<b->msz; ++i) if (b->gapstart-b->buffer<=b->mark[i]) --b->mark[i];

    b->gapstart-=bf_lchar(b,&ch);
    if (ASCII(ch)=='\n')
    {
      --b->cury;
    }
    b->mode|=CHANGED;
    --b->curch;
    return 1;
  }
}
/*}}}*/
/* bf_insertstr    -- insert string of characters (external, no MetaChars) *//*{{{*/
int bf_insertstr(struct Buffer *b, const char *str, const char *end)
{
  assert(str<=end);

  while (str<end)
  {
    struct MetaChar ch;

    if (*str=='\0')
    {
      ch.c[0] = '\0';
      ch.c[1] = NUL;
      ++str;
    }
    else if (bf_utf8)
    {
      size_t len;

      if (!(*str & 0x80))
      {
        ch.c[0] = *str++;
      }
      else if ((*str & 0xe0) == 0xc0 && (str+1<end) && ((*(str+1)&0xc0) == 0x80))
      {
        ch.c[0] = *str++;
        ch.c[1] = *str++;
      }
      else if ((*str & 0xf0) == 0xe0 && (str+2<end) && ((*(str+1)&0xc0) == 0x80) && ((*(str+2)&0xc0) == 0x80))
      {
        ch.c[0] = *str++;
        ch.c[1] = *str++;
        ch.c[2] = *str++;
      }
      else if ((*str & 0xf8) == 0xf0 && (str+3<end) && ((*(str+1)&0xc0) == 0x80) && ((*(str+2)&0xc0) == 0x80) && ((*(str+3)&0xc0) == 0x80))
      {
        ch.c[0] = *str++;
        ch.c[1] = *str++;
        ch.c[2] = *str++;
        ch.c[3] = *str++;
      }
      else
      {
        /* Invalid UTF-8 character */
        ch.c[0] = '\0';
        ch.c[1] = QUOTE;
        ch.c[2] = *str++;
      }
    }
    else
    {
      ch.c[0] = *str++;
    }

    if (!bf_insert(b,&ch)) return 0;
  }
  return 1;
}
/*}}}*/
/* bf_lchar        -- get character before beginning of gap, return length *//*{{{*/
int bf_lchar(const struct Buffer *b, struct MetaChar *ch)
{
  const char *s;

  ASSERT_CONSISTENCY(b);
  if (b->gapstart==b->buffer) return 0;

  s=b->gapstart-1;
  if (s-2>=b->buffer && *(s-2)=='\0' && *(s-1)==QUOTE) s-=2;
  else if (s-1>=b->buffer && *(s-1)=='\0') --s;
  else if (bf_utf8 && (*s & 0x80))
  {
    for (--s; (*s & 0xc0) != 0xc0; --s);
  }
  assert(b->gapstart-s<=4);
  memcpy(ch->c, s, b->gapstart-s);
  return b->gapstart-s;
}
/*}}}*/
/* bf_rchar        -- get character after end of gap, return length *//*{{{*/
int bf_rchar(const struct Buffer *b, struct MetaChar *ch)
{
  const char *s;

  ASSERT_CONSISTENCY(b);
  assert(ch);

  if (b->gapend==b->buffer+b->size-1) return 0;

  s=b->gapend+1;
  if (*s=='\0')
  {
    ++s;
    if (*s==QUOTE) ++s;
  }
  else if (bf_utf8)
  {
    if ((*s & 0xe0) == 0xc0) ++s;
    else if ((*s & 0xf0) == 0xe0) s+=2;
    else if ((*s & 0xf8) == 0xf0) s+=3;
  }

  assert(s-b->gapend<=4);
  memcpy(ch->c, b->gapend+1, s-b->gapend);
  return s-b->gapend;
}
/*}}}*/
/* bf_name         -- set buffer name *//*{{{*/
int bf_name(struct Buffer *b, const char *name)
{
  char *n;

  if (b->name) free(b->name);
  if ((n=malloc(strlen(name)+1))==(char*)0) return 0;
  strcpy(n,name);
  b->name=n;
  b->mode|=CHANGED;
  b->mode&=~READONLY;
  return 1;
}
/*}}}*/
/* bf_lang         -- get/set buffer language *//*{{{*/
struct Language *bf_lang(struct Buffer *b, struct Language *l)
{
  if (l)
  {
    if (b->mode&READONLY) return (struct Language*)0;
    b->lang=l;
    b->mode|=CHANGED;
  }
  return b->lang;
}
/*}}}*/
/* bf_load         -- load buffer from file *//*{{{*/
int bf_load(struct Buffer *b, const char *cmd, const char *file, int filefd, const char *beg, const char *end)
{
  /* variables *//*{{{*/
  int fd[2],eof,readonly=0;
  char buf[1024];
  char *line,*linerun,*bufrun=(char*)0;
  size_t linesize,remaining,begsz,endsz;
  enum { LANGUAGE, FOLDS } search_for;
  char *opening=(char*)0,*closing=(char*)0;
  size_t openingsz=-1,closingsz=-1;
  pid_t pid=0;
  int status;
  /*}}}*/

  assert(b!=(struct Buffer*)0);
  assert(beg!=(const char*)0);
  assert(end!=(const char*)0);
  begsz=strlen(beg);
  endsz=strlen(end);
  search_for=LANGUAGE;
  linesize=(size_t)256;
  remaining=(size_t)0;
  if ((line=malloc(linesize))==(char*)0) return 0;
  linerun=line;
  if (cmd) /*{{{*/
  {
    pipe(fd);
    switch (pid=fork())
    {
      case 0:
      {
        close(0);
        close(1);
        close(fd[0]);
        open(file,O_RDONLY);
        dup(fd[1]);
        close(fd[1]);
        execlp(getshell(),getshell(),"-c",cmd,(char*)0);
        _exit(127);
        break;
      }
      case -1:
      {
        close(fd[0]);
        close(fd[1]);
        break;
      }
      default: close(fd[1]);
    }    
  }
  /*}}}*/
  else /*{{{*/
  {
    struct stat statbuf;

    if (file)
    {
      if ((fd[0]=open(file,O_RDONLY))==-1) return 0;
      if (access(file,W_OK)==-1 && errno==EACCES) readonly=1;
    }
    else { file="(stdin)"; fd[0]=filefd; }
    if (fstat(fd[0],&statbuf)==-1) return 0;
    if (S_ISDIR(statbuf.st_mode)) { errno=EISDIR; return 0; }
  }
  /*}}}*/
  eof=0;
  do
  {
    /* enlarge read buffer if needed *//*{{{*/
    if (linerun+1>=line+linesize)
    {
      char *larger_line;

      if ((larger_line=realloc(line,linesize+256))==(char*)0)
      /* can not allocate more memory, close file and return failure *//*{{{*/
      {
        int oerrno;

        oerrno=errno;
        free(line);
        close(fd[0]);
        if (cmd) while (wait(&status)!=pid);
        errno=oerrno;
        return 0;
      }
      /*}}}*/
      else
      {
        linerun=larger_line+(linerun-line);
        line=larger_line;
        linesize+=256;
      }    
    }
    /*}}}*/
    /* read new block if needed *//*{{{*/
    if (remaining==0 && (remaining=read(fd[0],bufrun=buf,sizeof(buf)))<=0) eof=1;
    /*}}}*/
    --remaining;
    if ((eof || (*linerun++=*bufrun++)=='\n') && linerun>line) /* process line */
    {
      size_t len;

      *linerun='\0'; /* invalidates incomplete UTF-8 sequence */
      len=linerun-line;
      if (search_for==LANGUAGE) /* search for magic mark *//*{{{*/
      {
        char *found_beg,*found_end;

        if ((found_beg=mymemmem(line,len,beg,begsz)) && (found_end=mymemmem(found_beg+begsz,linerun-found_beg-begsz,end,endsz)))
        {
          struct Language *langrun;

          /* search for given language between begin/end */ /*{{{*/
          for (langrun=language; langrun->name!=(const char*)0; ++langrun)
          {
            if (strncmp(found_beg+begsz,langrun->name,strlen(langrun->name))==0)
            {
              /* variables */ /*{{{*/
              char *found;
              /*}}}*/

              /* determine opening and closing fold marks */ /*{{{*/
              if (opening) free(opening);
              if (closing) free(closing);
              opening=malloc((openingsz=strlen(langrun->beg)+begsz+strlen(langrun->end))+1);
              strcpy(opening,langrun->beg);
              strcpy(opening+strlen(langrun->beg),beg);
              memcpy(opening+strlen(langrun->beg)+begsz,langrun->end,strlen(langrun->end));
              closing=malloc((closingsz=strlen(langrun->beg)+endsz+strlen(langrun->end))+1);
              strcpy(closing,langrun->beg);
              strcpy(closing+strlen(langrun->beg),end);
              memcpy(closing+strlen(langrun->beg)+endsz,langrun->end,strlen(langrun->end));
              /*}}}*/
              /* now check if this is also a opening fold */
              if ((found=mymemmem(found_end+endsz,linerun-found_end-endsz,opening,openingsz))!=(char*)0) /* good */ /*{{{*/
              {
                *(found_beg-strlen(langrun->beg))='\0';
                bf_insertstr(b,line,found_beg-strlen(langrun->beg));
                *found='\0';
                bf_insertstr(b,found_end+endsz+strlen(langrun->end), found);
                bf_insert(b,&bf_foldstart);
                bf_insertstr(b,found+openingsz,linerun);
                search_for=FOLDS;
                b->lang=langrun;
                break;
              }
              /*}}}*/
            }
          }

          if (langrun->name==(const char*)0) /* not found, so this was just a regular line */ /*{{{*/
          {
            
            bf_insertstr(b,line,linerun);
          }
          /*}}}*/
          /*}}}*/
        }
        else
        {
          bf_insertstr(b,line,linerun);
        }
      }
      /*}}}*/
      else /* search for folds or insert literally *//*{{{*/
      {
        char *found;
        
        if ((found=mymemmem(line,len,opening,openingsz))!=(char*)0)
        {
          *found='\0';
          bf_insertstr(b,line,found);
          bf_insert(b,&bf_foldstart);
          bf_insertstr(b,found+openingsz,linerun);
        }
        else if ((found=mymemmem(line,len,closing,closingsz))!=(char*)0)
        {
          *found='\0';
          bf_insertstr(b,line,found);
          bf_insert(b,&bf_foldend);
          bf_insertstr(b,found+openingsz,linerun);
        }
        else
        {
          bf_insertstr(b,line,linerun);
        }
      }
      /*}}}*/
      linerun=line;
    }
  } while (!eof);
  close(fd[0]);
  if (cmd) while (wait(&status)!=pid);
  free(line);
  if (opening) free(opening);
  if (closing) free(closing);
  if (readonly) b->mode|=READONLY;
  return 1;
}
/*}}}*/
/* bf_save         -- save buffer into file *//*{{{*/
int bf_save(struct Buffer *b, const char *file, const char *beg, const char *end, unsigned int *chars)
{
  /* variables *//*{{{*/
  char *gapstart;
  char *bufp;
  char buf[1024];
  struct MetaChar ch;
  size_t len;
  int fd;
  char *magic,*begfold,*endfold;
  int first=1;
  /*}}}*/

  assert(file!=(const char*)0);
  assert(beg!=(const char*)0);
  assert(end!=(const char*)0);
  if (chars!=(unsigned int*)0) *chars=0;
  if ((begfold=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(beg)+strlen(b->lang ? b->lang->end : "")+1))==(char*)0) return 0;
  strcpy(begfold,b->lang ? b->lang->beg : "");
  strcat(begfold,beg);
  if (b->lang) strcat(begfold,b->lang->end);
  if ((magic=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(beg)+strlen(b->lang ? b->lang->name : "")+strlen(end)+strlen(b->lang ? b->lang->end : "")+strlen(begfold)+1))==(char*)0)
  {
    free(begfold);
    return 0;
  }
  strcpy(magic,b->lang ? b->lang->beg : "");
  strcat(magic,beg);
  if (b->lang) strcat(magic,b->lang->name);
  strcat(magic,end);
  if (b->lang) strcat(magic,b->lang->end);
  strcat(magic,begfold);
  if ((endfold=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(end)+strlen(b->lang ? b->lang->end : "")+1))==(char*)0)
  {
    free(begfold);
    free(magic);
    return 0;
  }
  strcpy(endfold,b->lang ? b->lang->beg : "");
  strcat(endfold,end);
  if (b->lang) strcat(endfold,b->lang->end);
  if ((fd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0666))==-1) return 0;
  gapstart=b->gapstart;
  bufp=buf;
  bf_begin(b);
  while ((len=bf_rchar(b,&ch)))
  {
    const char *writestr;
    size_t writelen=0;

    if (ASCII(ch)=='\0') switch (MARK(ch))
    {
      case NUL: writestr=""; writelen=1; break;
      case QUOTE: writestr=ch.c+2; writelen=1; break;
      case FOLDSTART: writestr=(first ? magic : begfold); first=0; writelen=strlen(writestr); break;
      case FOLDEND: writestr=endfold; writelen=strlen(writestr); break;
      default: assert(0);
    }
    else
    {
      writestr=ch.c; writelen=len;
    }

    assert(writelen);
    do
    {
      *bufp++=*writestr++;
      --writelen;
      if (bufp==buf+sizeof(buf))
      {
        if (write(fd,buf,sizeof(buf))!=sizeof(buf))
        {
          int oerrno;
      
          oerrno=errno;
          close(fd);
          free(begfold);
          free(magic);
          free(endfold);
          errno=oerrno;
          return 0;
        }
        else if (chars!=(unsigned int*)0) *chars+=sizeof(buf);
        bufp=buf;
      }
    } while (writelen);
    BF_FORWARD(b);
  } while (gapstart<b->gapstart) BF_BACKWARD(b);
  free(begfold);
  free(magic);
  free(endfold);
  if (bufp>buf)
  {
    if (write(fd,buf,bufp-buf)!=bufp-buf)
    {
      int oerrno;
      
      oerrno=errno;
      close(fd);
      errno=oerrno;
      return 0;
    }
    else if (chars!=(unsigned int*)0) *chars+=(bufp-buf);
  }
  if (close(fd)==-1) return 0;
  b->mode&=~CHANGED;
  return 1;
}
/*}}}*/
/* bf_write        -- write buffer without fold marks into file *//*{{{*/
int bf_write(struct Buffer *b, const char *file, unsigned int *chars)
{
  /* variables *//*{{{*/
  char *gapstart;
  char *bufp;
  char buf[1024];
  struct MetaChar ch;
  size_t len;
  int fd;
  /*}}}*/

  assert(file!=(const char*)0);
  if (chars!=(unsigned int*)0) *chars=0;
  if ((fd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0666))==-1) return 0;
  gapstart=b->gapstart;
  bufp=buf;
  bf_begin(b);
  while ((len=bf_rchar(b,&ch)))
  {
    size_t writelen=0;
    char const *writestr;

    if (ASCII(ch)=='\0') switch (MARK(ch))
    {
      case NUL: writestr=""; writelen+1; break;
      case QUOTE: writestr=ch.c+2; writelen=1; break;
      case FOLDSTART: break;
      case FOLDEND: break;
      default: assert(0);
    }
    else
    {
      writestr=ch.c;
      writelen=len;
    }

    while (writelen)
    {
      *bufp++=*writestr++;
      --writelen;

      if (bufp==buf+sizeof(buf))
      {
        if (write(fd,buf,sizeof(buf))!=sizeof(buf))
        {
          int oerrno;
      
          oerrno=errno;
          close(fd);
          errno=oerrno;
          return 0;
        }
        else if (chars!=(unsigned int*)0) *chars+=sizeof(buf);
        bufp=buf;
      }
    }
    BF_FORWARD(b);
  }
  while (gapstart<b->gapstart) BF_BACKWARD(b);
  if (bufp>buf)
  {
    if (write(fd,buf,bufp-buf)!=bufp-buf)
    {
      int oerrno;
      
      oerrno=errno;
      close(fd);
      errno=oerrno;
      return 0;
    }
    else if (chars!=(unsigned int*)0) *chars+=(bufp-buf);
  }
  if (close(fd)==-1) return 0;
  b->mode&=~CHANGED;
  return 1;
}
/*}}}*/
/* bf_linestart    -- go to start of current line *//*{{{*/
int bf_linestart(struct Buffer *b, int realstart)
{
  struct MetaChar ch;
  
  ASSERT_CONSISTENCY(b);
  if (realstart)
  {
    if (b->gapstart==b->buffer) return 0;
    while (bf_lchar(b,&ch) && ASCII(ch)!='\n' && BF_BACKWARDOK(b)) BF_BACKWARD(b);
  }
  else
  {
    int blanks;

    blanks=0;
    if (bf_lchar(b,&ch) && ASCII(ch)!='\n')
    {
      if (bf_rchar(b,&ch) && ASCII(ch)!=' ') blanks=1;
      while (bf_lchar(b,&ch) && ASCII(ch)!='\n' && BF_BACKWARDOK(b))
      {
        BF_BACKWARD(b);
        if (ASCII(ch)==' ' && blanks) ++blanks; else blanks=0;
      }
    }
    if (blanks==0) while (bf_rchar(b,&ch) && ASCII(ch)==' ') BF_FORWARD(b);
  }
  ASSERT_CONSISTENCY(b);
  return 1;
}
/*}}}*/
/* bf_lineend      -- go to end of current line *//*{{{*/
int bf_lineend(struct Buffer *b, int realend)
{
  struct MetaChar ch;
  
  while (bf_rchar(b,&ch) && ASCII(ch)!='\n') BF_FORWARD(b);
  return 1;
}
/*}}}*/
/* bf_isfold       -- is this a fold? *//*{{{*/
int bf_isfold(struct Buffer *b, int fold, int where)
{
  char *gapstart;
  struct MetaChar ch;

  gapstart=b->gapstart;
  if (where&LEFT)
  {
    while (bf_lchar(b,&ch))
    {
      if (ASCII(ch)=='\0' && (MARK(ch)&fold))
      {
        while (b->gapstart<gapstart) BF_FORWARD(b);
        return 1;
      }
      else if (ASCII(ch)=='\n') break;
      if (BF_BACKWARDOK(b)) BF_BACKWARD(b);
    }
    while (b->gapstart<gapstart) BF_FORWARD(b);
  }
  if (where&RIGHT)
  {
    while (bf_rchar(b,&ch))
    {
      if (ASCII(ch)=='\0' && (MARK(ch)&fold))
      {
        while (b->gapstart>gapstart) BF_BACKWARD(b);
        return 1;
      }
      else if (ASCII(ch)=='\n') break;
      BF_FORWARD(b);
    }   
    while (b->gapstart>gapstart) BF_BACKWARD(b);
  }
  return 0;
}
/*}}}*/
/* bf_tofoldstart  -- goto start of this fold *//*{{{*/
int bf_tofoldstart(struct Buffer *b, int level, int *linesback)
{
  char *gapstart;
  struct MetaChar ch;
  int entrylevel;
  int firstline=level;
        
  entrylevel=level;
  gapstart=b->gapstart;
  if (linesback) *linesback=0;
  do
  {
    if (!BF_BACKWARDOK(b) || bf_lchar(b,&ch)==0)
    {
      while (b->gapstart<gapstart) BF_FORWARD(b);
      return 0;
    }
    BF_BACKWARD(b);
    if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND && !firstline) ++level;
    else if (ASCII(ch)=='\0' && MARK(ch)==FOLDSTART && !firstline) --level;
    else if (ASCII(ch)=='\n')
    {
      firstline=0;
      if (level==entrylevel && linesback) ++*linesback;
    }
  } while (!(ASCII(ch)=='\0' && MARK(ch)==FOLDSTART && level==0));
  return 1;
}
/*}}}*/
/* bf_tofoldend    -- goto end of this fold *//*{{{*/
int bf_tofoldend(struct Buffer *b, int level, int *linesforward)
{
  char *gapstart;
  int entrylevel;
        
  if (linesforward) *linesforward=0;
  entrylevel=level;
  gapstart=b->gapstart;
  if (!BF_FORWARDOK(b)) return 0;
  do
  {
    struct MetaChar ch;

    BF_FORWARD(b);
    if (bf_lchar(b,&ch)==0)
    {
      while (b->gapstart>gapstart) BF_BACKWARD(b);
      return 0;
    }
    if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND) --level;
    else if (ASCII(ch)=='\0' && MARK(ch)==FOLDSTART) ++level;
    else if (ASCII(ch)=='\n' && level==entrylevel && linesforward) ++*linesforward;
    if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND && level==0) return 1;
  } while (BF_FORWARDOK(b));
  return 1;
}
/*}}}*/
/* bf_begin        -- goto begin of buffer *//*{{{*/
void bf_begin(struct Buffer *b)
{
  while (BF_BACKWARDOK(b)) BF_BACKWARD(b);
}
/*}}}*/
/* bf_end          -- goto end of buffer *//*{{{*/
void bf_end(struct Buffer *b)
{
  while (BF_FORWARDOK(b)) BF_FORWARD(b);
}
/*}}}*/
/* bf_gotoline     -- goto line of buffer *//*{{{*/
int bf_gotoline(struct Buffer *b, int line)
{
  char *gapstart;

  gapstart=b->gapstart;
  bf_begin(b);
  while (b->cury!=line && bf_forward(b));
  if (b->cury==line) return 1;
  while (b->gapstart<gapstart) bf_forward(b);
  while (b->gapstart>gapstart) bf_backward(b);
  return 0;
}
/*}}}*/
/* bf_strfsearch   -- goto line of forward searched string *//*{{{*/
int bf_strfsearch(struct Buffer *b, const char *needle, size_t needlesize)
{
  char *found;

  if ((b->buffer+b->size-b->gapend-1)>0 && (found=mymemmem(b->gapend+1,b->buffer+b->size-b->gapend-1,needle,needlesize)))
  {
    while (b->gapend+1<found) BF_FORWARD(b);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* bf_regfsearch   -- goto line of forward searched string *//*{{{*/
int bf_regfsearch(struct Buffer *b, const regex_t *needle)
{
  regmatch_t found;
  int forward;
  int failed;
  struct MetaChar ch;

  forward=0;
  failed=REG_NOMATCH;
  while ((failed=regexec(needle,b->gapend+1,1,&found,((b->gapstart==b->buffer || (bf_lchar(b,&ch) && ASCII(ch)=='\n')) ? 0 : REG_NOTBOL)|( b->gapend+strlen(b->gapend)<b->buffer+b->size ? REG_NOTEOL : 0 )))==REG_NOMATCH)
  {
    if (bf_rchar(b,&ch))
    {
      if (ASCII(ch)=='\0')
      {
        bf_forward(b);
        ++forward;
      }
      else do
      {
        bf_forward(b);
        ++forward;
      } while (bf_rchar(b,&ch) && ASCII(ch)!='\0');
    }
    else break;
  }
  if (failed==0)
  {
    if (found.rm_so>0)
    {
      const char *rm_so;

      rm_so=b->gapend+1+found.rm_so;
      while (b->gapend+1<rm_so) BF_FORWARD(b);
    }
    return 1;
  }
  else
  {
    while (forward) { bf_backward(b); --forward; }
    return 0;
  }
}
/*}}}*/
/* bf_strbsearch   -- goto line of backwards searched string *//*{{{*/
int bf_strbsearch(struct Buffer *b, const char *needle, size_t needlesize)
{
  int back=0;

  while (memcmp(b->gapend+1,needle,needlesize))
  {
    if (BF_BACKWARDOK(b)) { ++back; BF_BACKWARD(b); }
    else { while (back>0) { --back; BF_FORWARD(b); } return 0; }
  }
  return 1;
}
/*}}}*/
/* bf_regbsearch   -- goto line of backwards searched string *//*{{{*/
int bf_regbsearch(struct Buffer *b, const regex_t *needle)
{
  regmatch_t found;
  int backward;
  int failed;
  struct MetaChar ch;

  backward=0;
  failed=REG_NOMATCH;
  while ((failed=regexec(needle,b->gapend+1,1,&found,((b->gapstart==b->buffer || (bf_lchar(b,&ch) && ASCII(ch)=='\n')) ? 0 : REG_NOTBOL)|( b->gapend+strlen(b->gapend)<b->buffer+b->size ? REG_NOTEOL : 0 )))==REG_NOMATCH || found.rm_so>0)
  {
    if (bf_backward(b)) ++backward;
    else break;
  }
  if (failed==0)
  {
    return 1;
  }
  else
  {
    while (backward) { bf_forward(b); --backward; }
    return 0;
  }
}
/*}}}*/
/* bf_tomark       -- goto alternate mark *//*{{{*/
void bf_tomark(struct Buffer *b, int to)
{
  assert(to<b->msz);
  while (b->curch<b->mark[to] && BF_FORWARDOK(b)) BF_FORWARD(b);
  while (b->curch>b->mark[to] && BF_BACKWARDOK(b)) BF_BACKWARD(b);
}
/*}}}*/
/* bf_mark         -- set mark *//*{{{*/
int bf_mark(struct Buffer *b, int mark)
{
  if (mark<0)
  {
    for (mark=0; mark<b->msz && b->markused[mark]; ++mark);
    if (mark==b->msz)
    {
      assert(b->msz<BUFMARKS);
      ++b->msz;
    }
    b->markused[mark]=1;
  }
  b->mark[mark]=b->curch;
  return mark;
}
/*}}}*/
/* bf_unmark       -- unset mark *//*{{{*/
void bf_unmark(struct Buffer *b, int mark)
{
  assert(mark>=0);
  assert(b->markused[mark]);
  b->markused[mark]=0;
}
/*}}}*/
/* bf_uc           -- convert meta character to Unicode *//*{{{*/
long bf_uc(const struct MetaChar *ch)
{
  long uc;
  const char *c, *end;

  assert(ch);
  c=ch->c;
  assert(*c);
  if (!(*c & 0x80))
  {
    end=c+1;
    uc=*c;
  }
  else if ((*c & 0xe0) == 0xc0)
  {
    end=c+2;
    uc=*c & 0x1f;
  }
  else if ((*c & 0xf0) == 0xe0)
  {
    end=c+3;
    uc=*c & 0xf;
  }
  else
  {
    assert((*c & 0xf8) == 0xf0);
    end=c+4;
    uc=*c & 0x7;
  }
  while (++c<end)
  {
    uc=(uc<<6)|(*c&0x3f);
  }
  return uc;
}
/*}}}*/
