%% This Source Code Form is subject to the terms of the Mozilla Public
%% License, v. 2.0. If a copy of the MPL was not distributed with this
%% file, You can obtain one at https://mozilla.org/MPL/2.0/.
%%
%% Copyright (c) 2007-2026 Broadcom. All Rights Reserved. The term “Broadcom” refers to Broadcom Inc. and/or its subsidiaries. All rights reserved.
%%

-module(rabbit_federation_exchange_link_sup_sup).

-behaviour(mirrored_supervisor).

-include_lib("rabbit_common/include/rabbit.hrl").
-include_lib("kernel/include/logger.hrl").
-include_lib("rabbitmq_federation/include/logging.hrl").
-define(SUPERVISOR, ?MODULE).

%% Supervises the upstream links for all exchanges (but not queues). We need
%% different handling here since exchanges want a mirrored sup.

-export([start_link/0, start_child/1, adjust/1, stop_child/1]).
-export([init/1]).
-export([id_to_khepri_path/1]).

%%----------------------------------------------------------------------------

start_link() ->
    _ = pg:start_link(),
    %% This scope is used by concurrently starting exchange and queue links,
    %% and other places, so we have to start it very early outside of the supervision tree.
    %% The scope is stopped in stop/1.
    _ = rabbit_federation_pg:start_scope(),
    rabbit_federation_app_state:reset_shutting_down_marker(),
    mirrored_supervisor:start_link({local, ?SUPERVISOR}, ?SUPERVISOR,
                                   ?MODULE, []).

%% Note that the next supervisor down, rabbit_federation_link_sup, is common
%% between exchanges and queues.
start_child(X) ->
    try mirrored_supervisor:start_child(
           ?SUPERVISOR,
           {id(X), {rabbit_federation_link_sup, start_link, [X]},
            transient, ?SUPERVISOR_WAIT, supervisor,
            [rabbit_federation_link_sup]}) of
        {ok, _Pid}               -> ok;
        {error, {already_started, _Pid}} ->
          #exchange{name = ExchangeName} = X,
          ?LOG_DEBUG("Federation link for exchange ~tp was already started",
                     [rabbit_misc:rs(ExchangeName)]),
          ok;
        %% A link returned {stop, gone}, the link_sup shut down, that's OK.
        {error, {shutdown, _}} -> ok
    catch
        exit:{noproc, _} -> ok;
        exit:{shutdown, _} -> ok;
        exit:shutdown -> ok
    end.

adjust({clear_upstream, VHost, UpstreamName}) ->
    _ = [rabbit_federation_link_sup:adjust(Pid, X, {clear_upstream, UpstreamName}) ||
            {#exchange{name = Name} = X, Pid, _, _} <- which_children(),
            Name#resource.virtual_host == VHost],
    ok;
adjust(Reason) ->
    _ = [rabbit_federation_link_sup:adjust(Pid, X, Reason) ||
            {X, Pid, _, _} <- which_children()],
    ok.

which_children() ->
    try
        mirrored_supervisor:which_children(?SUPERVISOR)
    catch
        exit:{noproc, _} -> [];
        exit:{shutdown, _} -> [];
        exit:shutdown -> []
    end.

stop_child(X) ->
    try mirrored_supervisor:terminate_child(?SUPERVISOR, id(X)) of
      ok -> ok;
      {error, Err} ->
        #exchange{name = ExchangeName} = X,
        ?LOG_WARNING("Attempt to stop a federation link for exchange ~tp failed: ~tp",
                     [rabbit_misc:rs(ExchangeName), Err]),
        ok
    catch
        exit:{noproc, _} -> ok;
        exit:{shutdown, _} -> ok;
        exit:shutdown -> ok
    end,
    try
        _ = mirrored_supervisor:delete_child(?SUPERVISOR, id(X)),
        ok
    catch
        exit:{noproc, _} -> ok;
        exit:{shutdown, _} -> ok;
        exit:shutdown -> ok
    end.

%%----------------------------------------------------------------------------

init([]) ->
    logger:set_process_metadata(#{domain => ?RMQLOG_DOMAIN_FEDERATION}),
    {ok, {{one_for_one, 1200, 60}, []}}.

%% See comment in rabbit_federation_queue_link_sup_sup:id/1
id(X = #exchange{policy = Policy}) ->
    X1 = rabbit_exchange:immutable(X),
    X2 = X1#exchange{policy = Policy},
    X2.

id_to_khepri_path(
  #exchange{name = #resource{virtual_host = VHost, name = Name}}) ->
    [exchange, VHost, Name].
