#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <cairo/cairo.h>
#include <librsvg/rsvg.h>
#include <librsvg/rsvg-cairo.h>

#include <gdk/gdkkeysyms.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#include "razertool.h"
#include "razerlib.h"
#include "razerfirmware.h"

static gint razer_errno = 0;
static gboolean razer_errored = FALSE;
static razer_t *razer_udev = NULL;
static struct usb_dev_handle *razer_fw_udev = NULL;

static int firmware_version = -1;
static int profile_orig = 1, profile = 1;
static razer_profile_t profiles_orig[RAZER_MAX_PROFILES];
static razer_profile_t profiles[RAZER_MAX_PROFILES];

#define STATUS_TIMEOUT	5000

static guint st_context_device, st_context_user;

static GtkWidget *window1;

static void
display_macrodialog(gint button_num);

static void
flashdialog_check_status(void);

static void
display_flashdialog(void);

/* ******************************************************** */
/* MISC/DEVICE                                              */
/* ******************************************************** */

static const struct {
	char *name;
	int action;
} action_map[] = {
	{ N_("Left Click"),	RAZER_ACTION_BUTTON1 },
	{ N_("Double Click"),	RAZER_ACTION_DOUBLE_CLICK },
	{ N_("Right Click"),	RAZER_ACTION_BUTTON2 },
	{ N_("Middle Click"),	RAZER_ACTION_BUTTON3 },
	{ N_("Button 4"),	RAZER_ACTION_BUTTON4 },
	{ N_("Button 5"),	RAZER_ACTION_BUTTON5 },
	{ N_("Off"),		RAZER_ACTION_OFF },
	{ N_("On-the-Fly"),	RAZER_ACTION_ON_THE_FLY },
	{ N_("DPI Up"),		RAZER_ACTION_DPI_UP },
	{ N_("DPI Down"),	RAZER_ACTION_DPI_DOWN },
	{ N_("400 DPI"),	RAZER_ACTION_DPI_400 },
	{ N_("800 DPI"),	RAZER_ACTION_DPI_800 },
	{ N_("1600 DPI"),	RAZER_ACTION_DPI_1600 },
	{ N_("2000 DPI"),	RAZER_ACTION_DPI_2000 },
	{ N_("Profile 1"),	RAZER_ACTION_PROFILE1 },
	{ N_("Profile 2"),	RAZER_ACTION_PROFILE2 },
	{ N_("Profile 3"),	RAZER_ACTION_PROFILE3 },
	{ N_("Profile 4"),	RAZER_ACTION_PROFILE4 },
	{ N_("Profile 5"),	RAZER_ACTION_PROFILE5 },
	{ N_("Key..."),		RAZER_ACTION_KEY },
	{ N_("Macro..."),	RAZER_ACTION_MACRO },
	{ NULL,			0 }
};


static int
action_map_lookup(int idx)
{
	return action_map[idx].action;
}


static int
action_map_lookup_action(int action)
{
	int i;

	for (i = 0; action_map[i].name; i++) {
		if (action == action_map[i].action)
			return i;
	}

	return -1;
}


static gint
button_map_idx(gint button_num)
{
	switch (button_num) {
		case 0: return 0;
		case 1: return 2;
		case 2: return 1;
		case 3: return 3;
		case 4: return 4;
		case 5: return 5;
		case 6: return 6;
	}
	g_assert_not_reached();
}

static gint
button_map_pos(gint button_num)
{
	switch (button_num) {
		case 0: return 0;
		case 1: return 1;
		case 2: return 0;
		case 3: return 1;
		case 4: return 2;
		case 5: return 2;
		case 6: return 3;
	}
	g_assert_not_reached();
}

static gboolean
button_map_side_left(gint button_num)
{
	switch (button_num) {
		case 0: case 3: case 4: return TRUE;
		case 1: case 2: case 5: case 6: return FALSE;
	}
	g_assert_not_reached();
}


static void
razer_info_update(void);


static void
status_add(guint context, gint timeout, const gchar *text);


static int
razer_info_load(void)
{
	int status;
	int i;

	status = razer_read_firmware_version(razer_udev, &firmware_version, RAZERLIB_DEFAULT_TIMEOUT);
	if (status) goto err;

	status = razer_read_profile_number(razer_udev, &profile_orig, RAZERLIB_DEFAULT_TIMEOUT);
	if (status) { profile = profile_orig = 1; goto err; }
	profile = profile_orig;

	for (i = 0; i < razer_udev->profiles; i++) {
		status = razer_set_profile(razer_udev, i + 1, RAZERLIB_DEFAULT_TIMEOUT);
		if (status) goto err;

		status = razer_read_profile(razer_udev, &profiles_orig[i], RAZERLIB_DEFAULT_TIMEOUT);
		if (status) goto err;

		memcpy(&profiles[i], &profiles_orig[i], sizeof(profiles[i]));
	}

	status_add(st_context_device, STATUS_TIMEOUT, _("Profiles loaded"));

	razer_info_update();

	return 0;

  err:
	return status;
}


gboolean
on_search_devices_timeout              (gpointer         user_data)
{
	gboolean is_error = GPOINTER_TO_INT(user_data);
	gboolean was_errored;

	if (is_error) {
		if (is_error != 2) {
			gchar *str;
			str = g_strconcat(
				_("There was an error communicating with the device"),
				razer_errno?_(": "):NULL,
				razer_strerror(razer_errno),
				razer_errno==EPERM?_("\nYou may need to change the permissions of the device (see the README)"):NULL,
				NULL
			);
			GtkWidget *warn = gtk_message_dialog_new(
				GTK_WINDOW(window1),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_WARNING,
				GTK_BUTTONS_OK,
				str
			);
			g_signal_connect((gpointer)warn, "response", G_CALLBACK(gtk_widget_destroy), NULL);
			gtk_widget_show(warn);
			g_free(str);

			status_add(st_context_device, STATUS_TIMEOUT, _("Communication error"));
		}

		if (razer_udev) {
			razer_close(razer_udev);
			razer_udev = NULL;
		}

		if (razer_fw_udev) {
			usb_close(razer_fw_udev);
			razer_fw_udev = NULL;
		}

		flashdialog_check_status();
		razer_info_update();
	}

	was_errored = razer_errored;
	razer_errno = 0;
	razer_errored = FALSE;

	if (razer_udev) {
		int status = razer_ping(razer_udev, 8*RAZERLIB_DEFAULT_TIMEOUT);
		if (status) {
			razer_errno = -status;
			razer_errored = TRUE;
			on_search_devices_timeout(GINT_TO_POINTER(2));
		}
		return TRUE;
	} else if (razer_fw_udev) {
		int status = razer_firmware_ping(razer_fw_udev, 8*RAZERLIB_DEFAULT_TIMEOUT);
		if (status) {
			razer_errno = -status;
			razer_errored = TRUE;
			on_search_devices_timeout(GINT_TO_POINTER(2));
		}
		return TRUE;
	} else {
		struct usb_device *dev;

		dev = razer_find(NULL);
		if (was_errored && dev) {
			razer_errored = TRUE;
			dev = NULL;
		}
		if (dev) {
			razer_udev = razer_open(dev);
			if (razer_udev) {
				int status = razer_info_load();
				if (status) {
					razer_errno = -status;
					razer_errored = TRUE;
					on_search_devices_timeout(GINT_TO_POINTER(1));
				} else {
					status_add(st_context_device, STATUS_TIMEOUT, _("Device plugged in"));
				}
				flashdialog_check_status();
				return TRUE;
			}
		}

		dev = razer_find_flash(NULL);
		if (was_errored && dev) {
			razer_errored = TRUE;
			dev = NULL;
		}
		if (dev) {
			razer_fw_udev = usb_open(dev);
			if (razer_fw_udev) {
				razer_info_update();
				status_add(st_context_device, STATUS_TIMEOUT, _("Flash device plugged in"));
				flashdialog_check_status();
				return TRUE;
			}
		}
	}

	return TRUE;
}


/* ******************************************************** */
/* MAIN WINDOW                                              */
/* ******************************************************** */


static GtkTooltips *tooltips;
static GtkWidget *drawingarea1;
static GtkWidget *statusbar1;
static GtkWidget *label_firmware;
static GtkWidget *profile_widget[RAZER_MAX_PROFILES];
static GtkWidget *DPI400, *DPI800, *DPI1600, *DPI2000;
static GtkWidget *HZ125, *HZ500, *HZ1000;
static GtkWidget *button_widget[RAZER_MAX_BUTTONS];
static GtkWidget *button_box[RAZER_MAX_BUTTONS];
static GtkWidget *button_revert, *button_apply;

static RsvgHandle *razer_svg = NULL;
static RsvgDimensionData razer_svg_dims;
static GdkPoint razer_buttonloc[RAZER_MAX_BUTTONS];
static gboolean razer_buttondown[RAZER_MAX_BUTTONS];

static GtkWidget *about = NULL;


static void
razer_info_update(void)
{
	int i;

	if (razer_udev) {
		gchar verstr[128];

		g_snprintf(verstr, sizeof(verstr), "%d.%02d (%s)", firmware_version / 100, firmware_version % 100, razer_udev->name);
		gtk_label_set_text(GTK_LABEL(label_firmware), verstr);

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(profile_widget[profile - 1]), TRUE);
		if (razer_udev) {
			for (i = 0; i < razer_udev->profiles; i++)
				gtk_widget_show(profile_widget[i]);
			for (; i < RAZER_MAX_PROFILES; i++)
				gtk_widget_hide(profile_widget[i]);
		}
		if (!razer_udev || razer_udev->profiles <= 1) {
			gtk_widget_set_sensitive(profile_widget[0], FALSE);
		} else {
			gtk_widget_set_sensitive(profile_widget[0], TRUE);
		}

		switch (profiles[profile-1].dpi) {
		  case RAZER_DPI_400: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(DPI400), TRUE); break;
		  case RAZER_DPI_800: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(DPI800), TRUE); break;
		  case RAZER_DPI_1600: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(DPI1600), TRUE); break;
		  case RAZER_DPI_2000: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(DPI2000), TRUE); break;
		  default: g_assert_not_reached();
		}
	
		switch (profiles[profile-1].hz) {
		  case RAZER_HZ_125: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(HZ125), TRUE); break;
		  case RAZER_HZ_500: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(HZ500), TRUE); break;
		  case RAZER_HZ_1000: gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(HZ1000), TRUE); break;
		  default: g_assert_not_reached();
		}
	
		for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
			razer_button_t *b;
			int idx;

			if (!razer_udev || i >= razer_udev->buttons) {
				gtk_combo_box_set_active(GTK_COMBO_BOX(button_widget[i]), -1);
				gtk_widget_hide_all(button_box[i]);
				gtk_widget_show(button_box[i]);
				continue;
			}

			b = &profiles[profile-1].button[i];
			idx = action_map_lookup_action(b->action);
			gtk_combo_box_set_active(GTK_COMBO_BOX(button_widget[i]), idx);
			gtk_widget_show_all(button_box[i]);
		}
	
		gtk_widget_set_sensitive(button_revert, TRUE);
		gtk_widget_set_sensitive(button_apply, TRUE);
	} else {
		if (razer_fw_udev) {
			gtk_label_set_text(GTK_LABEL(label_firmware), _("?.?? (FLASH MODE)"));
		} else {
			gtk_label_set_text(GTK_LABEL(label_firmware), _("?.?? (NO DEVICE)"));
		}

		gtk_widget_set_sensitive(button_revert, FALSE);
		gtk_widget_set_sensitive(button_apply, FALSE);
	}
}


static gboolean
remove_device_status(gpointer user_data)
{
	guint id = GPOINTER_TO_INT(user_data);
	gtk_statusbar_remove(GTK_STATUSBAR(statusbar1), st_context_device, id);
	return FALSE;
}


static gboolean
remove_user_status(gpointer user_data)
{
	guint id = GPOINTER_TO_INT(user_data);
	gtk_statusbar_remove(GTK_STATUSBAR(statusbar1), st_context_user, id);
	return FALSE;
}


static void
status_add(guint context, gint timeout, const gchar *text)
{
	guint id;

	id = gtk_statusbar_push(GTK_STATUSBAR(statusbar1), context, text);

	if (timeout < 0) return;

	g_timeout_add_full(
		G_PRIORITY_DEFAULT,
		timeout,
		(GSourceFunc)(context==st_context_device?remove_device_status:remove_user_status),
		GINT_TO_POINTER(id),
		NULL
	);
}


void
on_window1_realize                     (GtkWidget       *widget,
                                        gpointer         user_data)
{
	int i;
	GtkWidget *vboxleft, *vboxright, *hboxprofile;
	int vboxleft_children, vboxright_children;
	GSList *profile_group = NULL;

	window1 = widget;

	statusbar1 = lookup_widget(widget, "statusbar1");
	st_context_device = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar1), "Device");
	st_context_user = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar1), "User");

	status_add(st_context_device, -1, PACKAGE " " VERSION);

	drawingarea1 = lookup_widget(widget, "drawingarea1");
	label_firmware = lookup_widget(widget, "label_firmware");

	button_revert = lookup_widget(widget, "button_revert");
	button_apply = lookup_widget(widget, "button_apply");

	if (!tooltips)
		tooltips = gtk_tooltips_new();

	hboxprofile = lookup_widget(widget, "hboxprofile");

	for (i = 0; i < RAZER_MAX_PROFILES; i++) {
		gchar number[] = "X";
		GtkWidget *radio;

		number[0] = i + '1';
		profile_widget[i] = radio = gtk_radio_button_new_with_mnemonic(NULL, number);
		gtk_radio_button_set_group(GTK_RADIO_BUTTON(radio), profile_group);
		profile_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio));

		g_signal_connect((gpointer)radio, "toggled", G_CALLBACK(on_profile_toggled), NULL);

		if (i < 1 || (razer_udev && i < razer_udev->profiles))
			gtk_widget_show(profile_widget[i]);

		gtk_box_pack_start(GTK_BOX(hboxprofile), profile_widget[i], TRUE, FALSE, 0);
	}

	vboxleft = lookup_widget(widget, "vboxleft");
	vboxright = lookup_widget(widget, "vboxright");

	for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
		int j;
		int bn = button_map_idx(i);
		GtkWidget *hbox, *combo, *button, *macro, *image;
		GtkWidget *side = button_map_side_left(bn)?vboxleft:vboxright;

		button_box[bn] = hbox = gtk_hbox_new(FALSE, 0);
		gtk_widget_show(hbox);
		gtk_box_pack_start(GTK_BOX(side), hbox, TRUE, FALSE, 0);

		button_widget[bn] = combo = gtk_combo_box_new_text();
		for (j = 0; action_map[j].name; j++)
			gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _(action_map[j].name));
		gtk_combo_box_set_active(GTK_COMBO_BOX(combo), -1);
		if (i < 3 || (razer_udev && i < razer_udev->buttons))
			gtk_widget_show(combo);
		gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

		/* GtkComboBox is stupid, and doesn't propagate the state change from
		   its own internal widget */
		button = gtk_bin_get_child(GTK_BIN(combo));
		g_assert(button);
		g_signal_connect_swapped(
			(gpointer)button, "state_changed",
			G_CALLBACK(on_buttonXc_state_changed),
			G_OBJECT(combo)
		);

		macro = gtk_button_new();
		if (i < 3 || (razer_udev && i < razer_udev->buttons))
			gtk_widget_show(macro);
		gtk_box_pack_start(GTK_BOX(hbox), macro, FALSE, FALSE, 0);
		gtk_widget_set_sensitive(macro, FALSE);
		gtk_tooltips_set_tip(tooltips, macro, _("Record Key/Macro"), NULL);

		image = gtk_image_new_from_stock("gtk-media-record", GTK_ICON_SIZE_BUTTON);
		gtk_widget_show(image);
		gtk_container_add(GTK_CONTAINER(macro), image);

		g_signal_connect_swapped(
			(gpointer)macro, "clicked",
			G_CALLBACK(on_buttonXkm_clicked),
			G_OBJECT(combo)
		);

		g_signal_connect(
			(gpointer)combo, "changed",
			G_CALLBACK(on_buttonXc_changed),
			G_OBJECT(macro)
		);

		gtk_widget_set_sensitive(GTK_WIDGET(macro), FALSE);
	}

	vboxleft_children = g_list_length(gtk_container_get_children(GTK_CONTAINER(vboxleft)));
	vboxright_children = g_list_length(gtk_container_get_children(GTK_CONTAINER(vboxright)));
	while (vboxleft_children < vboxright_children) {
		GtkWidget *spacer = gtk_hbox_new(FALSE, 0);
		gtk_widget_show(spacer);
		gtk_box_pack_start(GTK_BOX(vboxleft), spacer, TRUE, FALSE, 0);
		++vboxleft_children;
	}
	while (vboxright_children < vboxleft_children) {
		GtkWidget *spacer = gtk_hbox_new(FALSE, 0);
		gtk_widget_show(spacer);
		gtk_box_pack_start(GTK_BOX(vboxright), spacer, TRUE, FALSE, 0);
		++vboxright_children;
	}

	DPI400 = lookup_widget(widget, "DPI400");
	DPI800 = lookup_widget(widget, "DPI800");
	DPI1600 = lookup_widget(widget, "DPI1600");
	DPI2000 = lookup_widget(widget, "DPI2000");

	HZ125 = lookup_widget(widget, "HZ125");
	HZ500 = lookup_widget(widget, "HZ500");
	HZ1000 = lookup_widget(widget, "HZ1000");

	razer_info_update();
}


gboolean
on_window1_map_event                   (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	on_search_devices_timeout(GINT_TO_POINTER(0));
	g_timeout_add(1000, (GSourceFunc)on_search_devices_timeout, G_PRIORITY_DEFAULT);
	return FALSE;
}


void
on_new1_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_open1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_save1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_save_as1_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_update1_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	display_flashdialog();
}


void
on_about1_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	if (about) return;
	about = create_aboutdialog1();
	gtk_widget_show_all(about);
}


void
on_button_revert_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	int i;

	for (i = 0; i < RAZER_MAX_PROFILES; i++)
		memcpy(&profiles[i], &profiles_orig[i], sizeof(profiles[i]));
	profile = profile_orig;

	razer_info_update();

	status_add(st_context_user, STATUS_TIMEOUT, _("Profiles reverted"));
}


void
on_button_apply_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	int status;
	int i;

	for (i = 0; i < razer_udev->profiles; i++) {
		if (!memcmp(&profiles_orig[i], &profiles[i], sizeof(profiles[i])))
			continue;

		status = razer_write_profile(razer_udev, &profiles[i], RAZERLIB_DEFAULT_TIMEOUT);
		if (status) goto err;
	}

	if (profile_orig != profile) {
		status = razer_select_profile(razer_udev, profile, RAZERLIB_DEFAULT_TIMEOUT);
		if (status) goto err;
	}

	status = razer_info_load();
	if (status) goto err;

	status_add(st_context_user, STATUS_TIMEOUT, _("Profiles uploaded"));

	return;

  err:
	razer_errno = -status;
	razer_errored = TRUE;
	on_search_devices_timeout(GINT_TO_POINTER(1));
	return;	
}


void
on_aboutdialog1_destroy                (GtkObject       *object,
                                        gpointer         user_data)
{
	about = NULL;
}


static void
drawingarea1_calc_matrix(GtkWidget          *widget,
                         cairo_matrix_t     *mat)
{
	double x_scale, y_scale, scale;
	
	x_scale = (double)widget->allocation.width / (double)razer_svg_dims.width;
	y_scale = (double)widget->allocation.height / (double)razer_svg_dims.height;
	scale = MIN(x_scale, y_scale);

	mat->xx = scale;
	mat->yy = scale;
	mat->xy = mat->yx = 0;
	mat->x0 = (widget->allocation.width - razer_svg_dims.width*scale) / 2.0;
	mat->y0 = (widget->allocation.height - razer_svg_dims.height*scale) / 2.0;
}


#define BUBBLE_THING_RADIUS	16.0
#define BUBBLE_THING_STROKE	4.0
#define BUBBLE_THING_STROKE_RGB	1.0, 1.0, 1.0
#define BUBBLE_THING_FILL_RGB	0.0, 0.0, 0.0
static void
cairo_draw_bubble_thing(cairo_t *cr, double x0, double y0, double x1, double y1, const gchar *text)
{
	cairo_matrix_t mat;
	cairo_text_extents_t ext;

	cairo_get_matrix(cr, &mat);

	cairo_new_path(cr);
	cairo_move_to(cr, x0, y0);
	cairo_identity_matrix(cr);
	cairo_line_to(cr, x1, y1);
	cairo_set_matrix(cr, &mat);
	cairo_set_source_rgb(cr, BUBBLE_THING_STROKE_RGB);
	cairo_set_line_width(cr, BUBBLE_THING_STROKE*3.0);
	cairo_stroke(cr);

	cairo_new_path(cr);
	cairo_arc(
		cr,
		x0, y0,
		16.0,
		0.0, 2.0*M_PI
	);
	cairo_set_source_rgb(cr, BUBBLE_THING_FILL_RGB);
	cairo_fill_preserve(cr);

	cairo_set_line_width(cr, BUBBLE_THING_STROKE);
	cairo_set_source_rgb(cr, BUBBLE_THING_STROKE_RGB);
	cairo_stroke(cr);

	cairo_new_path(cr);
	cairo_move_to(cr, x0, y0);
	cairo_identity_matrix(cr);
	cairo_line_to(cr, x1, y1);
	cairo_set_matrix(cr, &mat);
	cairo_set_line_width(cr, BUBBLE_THING_STROKE);
	cairo_set_source_rgb(cr, BUBBLE_THING_FILL_RGB);
	cairo_stroke(cr);

	cairo_select_font_face(cr, "sans-serif", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size(cr, 24.0);

	cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
	cairo_text_extents(cr, text, &ext);
	cairo_move_to(
		cr,
		x0 - (ext.width/2.0 + ext.x_bearing),
		y0 - (ext.height/2.0 + ext.y_bearing)
	);
	cairo_show_text(cr, text);
}


gboolean
on_drawingarea1_expose_event           (GtkWidget       *widget,
                                        GdkEventExpose  *event,
                                        gpointer         user_data)
{
	int i;
	cairo_t *cr;
	cairo_matrix_t mat;
	char button[] = "#buttonX";
	int buttons = razer_udev ? razer_udev->buttons : 0;

	cr = gdk_cairo_create(widget->window);

	if (!razer_svg) {
		for (i = 0; i < buttons; i++) {
			int pos;
			gboolean left;
			double x0, x1, y;
			gchar button[2] = "X";

			pos = button_map_pos(i);
			left = button_map_side_left(i);

			x1 = left?0:widget->allocation.width;
			x0 = x1 + (left?1:-1)*(BUBBLE_THING_RADIUS*1.5);
			y = pos*(widget->allocation.height / 4.0) + (widget->allocation.height / 8.0);

			button[0] = i + '1';

			cairo_draw_bubble_thing(
				cr,
				x0, y,
				x1, y,
				button
			);
		}

		return FALSE;
	}

	drawingarea1_calc_matrix(widget, &mat);
	cairo_set_matrix(cr, &mat);

	/* Draw the mouse */
	rsvg_handle_render_cairo_sub(razer_svg, cr, "#razer");

	/* Draw some links from buttons to their combo boxes */
	cairo_select_font_face(cr, "sans-serif", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size(cr, 24.0);
	for (i = 0; i < buttons; i++) {
		int pos;
		gboolean left;
		double x, y;
		char button[2] = "X";

		pos = button_map_pos(i);
		left = button_map_side_left(i);

		x = left?0:widget->allocation.width;
		y = pos*(widget->allocation.height / 4.0) + (widget->allocation.height / 8.0);

		button[0] = i + '1';

		cairo_draw_bubble_thing(
			cr,
			razer_buttonloc[i].x, razer_buttonloc[i].y,
			x, y,
			button
		);
	}

	/* Draw highlighted buttons */
	for (i = 0; i < buttons; i++) {
		if (!razer_buttondown[i]) continue;
		button[sizeof(button)-2] = i + '1';
		rsvg_handle_render_cairo_sub(razer_svg, cr, button);
	}

	cairo_destroy(cr);

	return FALSE;
}


void
on_drawingarea1_destroy                (GtkObject       *object,
                                        gpointer         user_data)
{
	if (razer_svg) {
		g_object_unref(razer_svg);
		razer_svg = NULL;
	}
}


void
on_drawingarea1_size_allocate          (GtkWidget       *widget,
                                        GdkRectangle    *allocation,
                                        gpointer         user_data)
{
	int i;

	if (!razer_svg) {
		const char *desc;

		razer_svg = rsvg_handle_new_from_file(PACKAGE_DATA_DIR "/pixmaps/razer-copperhead.svg", NULL);
		if (!razer_svg)
			return;

		rsvg_handle_get_dimensions(razer_svg, &razer_svg_dims);

		for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
			razer_buttonloc[i].x = 0;
			razer_buttonloc[i].y = 0;
		}

		desc = rsvg_handle_get_desc(razer_svg);
		if (desc) while (*desc) {
			const char *p = desc;
			int len = strcspn(desc, "\r\n");
			int button, x, y;

			if (!len) { ++desc; continue; }
			desc += len;

			if (strncmp(p, "button ", 7)) continue;
			if (sscanf(p+7, "%d %d %d", &button, &x, &y) != 3) continue;
			if (button < 1 || button > RAZER_MAX_BUTTONS) continue;

			razer_buttonloc[button-1].x = x;
			razer_buttonloc[button-1].y = razer_svg_dims.height - y;
		}

		for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
			razer_buttondown[i] = FALSE;
		}
	}
}


void
on_buttonXc_changed                    (GtkComboBox     *combobox,
                                        GtkWidget       *button)
{
	int i, idx, action;

	g_assert(GTK_IS_COMBO_BOX(combobox));
	g_assert(GTK_IS_WIDGET(button));

	idx = gtk_combo_box_get_active(combobox);
	action = action_map_lookup(idx);

	if (action == RAZER_ACTION_KEY || action == RAZER_ACTION_MACRO) {
		gtk_widget_set_sensitive(button, TRUE);
	} else {
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	}

	for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
		razer_button_t *b;
		if (button_widget[i] != (GtkWidget*)combobox) continue;

		b = &profiles[profile-1].button[i];
		if (b->action != action) {
			if (action == RAZER_ACTION_KEY
			 || action == RAZER_ACTION_MACRO)
			{
				b->macro_len = 0;
				memset(b->macro, 0, sizeof(b->macro));
			}

			b->action = action;
		}

		break;
	}
}


void
on_buttonXc_state_changed              (GtkWidget       *widget,
                                        GtkStateType    state,
                                        GtkWidget       *button)
{
	int i;
	gboolean down;

	down = (GTK_WIDGET_STATE(button) == GTK_STATE_PRELIGHT
		|| GTK_WIDGET_STATE(button) == GTK_STATE_SELECTED);

	for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
		if (button_widget[i] != widget) continue;
		if (razer_buttondown[i] != down) {
			razer_buttondown[i] = down;
			gtk_widget_queue_draw(drawingarea1);
		}
		break;
	}
}


void
on_profile_toggled                     (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	int i;

	if (!gtk_toggle_button_get_active(togglebutton))
		return;

	for (i = 0; i < RAZER_MAX_PROFILES; i++) {
		if (profile_widget[i] != (GtkWidget*)togglebutton) continue;
		profile = i + 1;
		break;
	}

	razer_info_update();
}


void
on_DPI_toggled                         (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if (!gtk_toggle_button_get_active(togglebutton))
		return;

	if ((GtkWidget*)togglebutton == DPI400)
		profiles[profile-1].dpi = RAZER_DPI_400;
	else if ((GtkWidget*)togglebutton == DPI800)
		profiles[profile-1].dpi = RAZER_DPI_800;
	else if ((GtkWidget*)togglebutton == DPI1600)
		profiles[profile-1].dpi = RAZER_DPI_1600;
	else if ((GtkWidget*)togglebutton == DPI2000)
		profiles[profile-1].dpi = RAZER_DPI_2000;
}


void
on_HZ_toggled                          (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if (!gtk_toggle_button_get_active(togglebutton))
		return;

	if ((GtkWidget*)togglebutton == HZ125)
		profiles[profile-1].hz = RAZER_HZ_125;
	else if ((GtkWidget*)togglebutton == HZ500)
		profiles[profile-1].hz = RAZER_HZ_500;
	else if ((GtkWidget*)togglebutton == HZ1000)
		profiles[profile-1].hz = RAZER_HZ_1000;
}


void
on_buttonXkm_clicked                   (GtkWidget       *widget,
                                        GtkButton       *button)
{
	int i;

	for (i = 0; i < RAZER_MAX_BUTTONS; i++) {
		if (button_widget[i] != widget) continue;
		display_macrodialog(i);
		break;
	}
}


/* ******************************************************** */
/* KEY/MACRO EDITOR                                         */
/* ******************************************************** */

static GtkWidget *macrodialog = NULL;
static GtkWidget *keysdialog = NULL;

static GtkWidget *macro1;
static GtkListStore *macrolist;
static gint macro_button = 0, macro_max = 0;
static gint macro_count = 0;

static GtkWidget *keysview;
static GtkListStore *keyslist;


enum {
	MACRO_COLUMN_KEY_CODE,
	MACRO_COLUMN_KEY_TEXT,
	MACRO_COLUMN_ACTION_CODE,
	MACRO_COLUMN_ACTION_TEXT,
	MACRO_COLUMN_COUNT
};


enum {
	KEYS_COLUMN_KEY_CODE,
	KEYS_COLUMN_KEY_TEXT,
	KEYS_COLUMN_COUNT
};


static void macrolist_untranslate_key(int key, char *buf, int buflen)
{
	char *slash;

	razer_untranslate_key(key, buf, buflen);

	/* Only display the first two */
	slash = strchr(buf, '/');
	if (!slash) return;
	slash = strchr(slash+1, '/');
	if (!slash) return;
	*slash = '\0';
}

static const gchar *macro_action_to_string(guchar action)
{
	switch (action) {
	  case RAZER_MACRO_KEY_UP: return _("up");
	  case RAZER_MACRO_KEY_DOWN: return _("down");
	  case RAZER_MACRO_KEY_DELAY_AFTER: return _("50ms delay");
	}
	g_assert_not_reached();
}

static void macrolist_append(guchar key, guchar action)
{
	GtkTreeIter iter;
	char keytext[64];
	const gchar *actiontext;
	GtkTreePath *path;

	if (action == RAZER_MACRO_KEY_DELAY_AFTER) {
		if (macro_max == 1) /* special case for keys.. */
			return;

		key = 0;
		keytext[0] = '\0';
		actiontext = macro_action_to_string(action);
	} else {
		if (macro_count >= macro_max)
			return;
		++macro_count;

		macrolist_untranslate_key(key, keytext, sizeof(keytext));

		actiontext = macro_action_to_string(action & RAZER_MACRO_KEY_DOWNUP_MASK);
	}

	gtk_list_store_append(macrolist, &iter);

	gtk_list_store_set(
		macrolist, &iter,
		MACRO_COLUMN_KEY_CODE, key,
		MACRO_COLUMN_KEY_TEXT, keytext,
		MACRO_COLUMN_ACTION_CODE, action,
		MACRO_COLUMN_ACTION_TEXT, actiontext,
		-1
	);

	path = gtk_tree_model_get_path(GTK_TREE_MODEL(macrolist), &iter);
	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(macro1), path, NULL, FALSE, 1.0, 0.0);
	gtk_tree_path_free(path);
}


static void
macro_to_macrolist(gint button_num)
{
	int i;
	razer_button_t *b = &profiles[profile-1].button[button_num];

	gtk_list_store_clear(macrolist);

	macro_count = 0;

	for (i = 0; i < b->macro_len; i++) {
		macrolist_append(b->macro[i].key, b->macro[i].how & RAZER_MACRO_KEY_DOWNUP_MASK);
		if (b->macro[i].how & RAZER_MACRO_KEY_DELAY_AFTER)
			macrolist_append(0, RAZER_MACRO_KEY_DELAY_AFTER);
	}
}


static void
macrolist_to_macro(gint button_num)
{
	GtkTreePath *path = gtk_tree_path_new_first();
	razer_button_t *b = &profiles[profile-1].button[button_num];
	GValue codeval, actionval;

	b->macro_len = 0;
	memset(b->macro, 0, sizeof(b->macro));

	memset(&codeval, 0, sizeof(codeval));
	memset(&actionval, 0, sizeof(actionval));

	for (path = gtk_tree_path_new_first(); path; gtk_tree_path_next(path)) {
		GtkTreeIter iter;
		guchar code, action;

		if (!gtk_tree_model_get_iter(GTK_TREE_MODEL(macrolist), &iter, path))
			break;

		if (G_IS_VALUE(&codeval)) {
			g_value_unset(&codeval);
			g_value_unset(&actionval);
		}

		gtk_tree_model_get_value(GTK_TREE_MODEL(macrolist), &iter, MACRO_COLUMN_KEY_CODE, &codeval);
		gtk_tree_model_get_value(GTK_TREE_MODEL(macrolist), &iter, MACRO_COLUMN_ACTION_CODE, &actionval);

		code = g_value_get_uchar(&codeval);
		action = g_value_get_uchar(&actionval);

		if (action == RAZER_MACRO_KEY_DELAY_AFTER) {
			if (b->macro_len) {
				b->macro[b->macro_len - 1].how |= action;
			}

			continue;
		}

		if (b->macro_len >= b->macro_max_len)
			break;

		b->macro[b->macro_len].key = code;
		b->macro[b->macro_len].how = action;
		++b->macro_len;
	}

	if (G_IS_VALUE(&codeval)) {
		g_value_unset(&codeval);
		g_value_unset(&actionval);
	}

	gtk_tree_path_free(path);
}


static void
display_macrodialog(gint button_num)
{
	GtkWidget *label, *button_delay;
	gchar buffer[1024];

	if (!macrodialog) {
		int i;
		GtkTreeViewColumn *column;

		macrodialog = create_macrodialog();

		macro1 = lookup_widget(macrodialog, "macro1");

		column = gtk_tree_view_column_new();
		gtk_tree_view_column_set_visible(column, FALSE);
		gtk_tree_view_append_column(GTK_TREE_VIEW(macro1), column);

		column = gtk_tree_view_column_new_with_attributes(
			_("Key"), gtk_cell_renderer_text_new(),
			"text", MACRO_COLUMN_KEY_TEXT,
			NULL
		);
		gtk_tree_view_column_set_expand(column, TRUE);
		gtk_tree_view_append_column(GTK_TREE_VIEW(macro1), column);

		column = gtk_tree_view_column_new();
		gtk_tree_view_column_set_visible(column, FALSE);
		gtk_tree_view_append_column(GTK_TREE_VIEW(macro1), column);

		column = gtk_tree_view_column_new_with_attributes(
			_("Action"), gtk_cell_renderer_text_new(),
			"text", MACRO_COLUMN_ACTION_TEXT,
			NULL
		);
		gtk_tree_view_column_set_expand(column, FALSE);
		gtk_tree_view_append_column(GTK_TREE_VIEW(macro1), column);

		macrolist = gtk_list_store_new(MACRO_COLUMN_COUNT, G_TYPE_UCHAR, G_TYPE_STRING, G_TYPE_UCHAR, G_TYPE_STRING);

		gtk_tree_view_set_model(GTK_TREE_VIEW(macro1), GTK_TREE_MODEL(macrolist));

		/* key selector */
		keysdialog = create_keysdialog();

		keysview = lookup_widget(keysdialog, "keysview");

		column = gtk_tree_view_column_new();
		gtk_tree_view_column_set_visible(column, FALSE);
		gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
		gtk_tree_view_append_column(GTK_TREE_VIEW(keysview), column);

		column = gtk_tree_view_column_new_with_attributes(
			_("Key"), gtk_cell_renderer_text_new(),
			"text", KEYS_COLUMN_KEY_TEXT,
			NULL
		);
		gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
		gtk_tree_view_append_column(GTK_TREE_VIEW(keysview), column);

		keyslist = gtk_list_store_new(KEYS_COLUMN_COUNT, G_TYPE_UCHAR, G_TYPE_STRING);
		gtk_tree_view_set_model(GTK_TREE_VIEW(keysview), GTK_TREE_MODEL(keyslist));

		for (i = 4; i < 0xff; i++) {
			char keytext[64];
			GtkTreeIter iter;
			macrolist_untranslate_key(i, keytext, sizeof(keytext));
			gtk_list_store_append(keyslist, &iter);
			gtk_list_store_set(keyslist, &iter,
				KEYS_COLUMN_KEY_CODE, i,
				KEYS_COLUMN_KEY_TEXT, keytext,
				-1
			);
		}
	}

	label = lookup_widget(macrodialog, "label");
	g_snprintf(buffer, sizeof(buffer), _("Editing button %d"), button_num+1);
	gtk_label_set_text(GTK_LABEL(label), buffer);

	macro_button = button_num;
	macro_max = (profiles[profile-1].button[button_num].action == RAZER_ACTION_KEY) ? 1 : profiles[profile-1].button[button_num].macro_max_len;

	macro_to_macrolist(button_num);

	button_delay = lookup_widget(macrodialog, "button_delay");
	gtk_widget_set_sensitive(button_delay, macro_max != 1);

	gtk_widget_grab_focus(macro1);

	gtk_widget_show(macrodialog);
}


static guchar
keyval_to_usb(guint keyval)
{
	if (keyval >= GDK_A && keyval <= GDK_Z)
		return keyval - GDK_A + 4;
	if (keyval >= GDK_a && keyval <= GDK_z)
		return keyval - GDK_a + 4;
	if (keyval >= GDK_F1 && keyval <= GDK_F12)
		return keyval - GDK_F1 + 58;
	if (keyval >= GDK_F13 && keyval <= GDK_F24)
		return keyval - GDK_F13 + 104;
	if (keyval >= GDK_1 && keyval <= GDK_9)
		return keyval - GDK_1 + 30;
	if (keyval >= GDK_KP_1 && keyval <= GDK_KP_9)
		return keyval - GDK_KP_1 + 89;

	switch (keyval) {
	  case GDK_exclam: return 30;
	  case GDK_at: return 31;
	  case GDK_numbersign: return 32;
	  case GDK_dollar: return 33;
	  case GDK_percent: return 34;
	  case GDK_asciicircum: return 35;
	  case GDK_ampersand: return 36;
	  case GDK_asterisk: return 37;
	  case GDK_parenleft: return 38;
	  case GDK_0: case GDK_parenright: return 39;
	  case GDK_Return: return 40;
	  case GDK_Escape: return 41;
	  case GDK_BackSpace: return 42;
	  case GDK_Tab: return 43;
	  case GDK_space: return 44;
	  case GDK_minus: case GDK_underscore: return 45;
	  case GDK_equal: case GDK_plus: return 46;
	  case GDK_bracketleft: case GDK_braceleft: return 47;
	  case GDK_bracketright: case GDK_braceright: return 48;
	  case GDK_backslash: case GDK_bar: return 49;
/*	  case GDK_: return 50;*/
	  case GDK_semicolon: case GDK_colon: return 51;
	  case GDK_apostrophe: case GDK_quotedbl: return 52;
	  case GDK_grave: case GDK_asciitilde: return 53;
	  case GDK_comma: case GDK_less: return 54;
	  case GDK_period: case GDK_greater: return 55;
	  case GDK_slash: case GDK_question: return 56;
	  case GDK_Caps_Lock: return 57;
	  /* F1(58) ... F12(69) */
	  case GDK_Print: return 70; /* ? */
	  case GDK_Scroll_Lock: return 71;
	  case GDK_Pause: return 72;
	  case GDK_Insert: return 73;
	  case GDK_Home: return 74;
	  case GDK_Page_Up: return 75;
	  case GDK_Delete: return 76;
	  case GDK_End: return 77;
	  case GDK_Page_Down: return 78;
	  case GDK_Right: return 79;
	  case GDK_Left: return 80;
	  case GDK_Down: return 81;
	  case GDK_Up: return 82;
	  case GDK_Num_Lock: return 83;
	  case GDK_KP_Divide: return 84;
	  case GDK_KP_Multiply: return 85;
	  case GDK_KP_Subtract: return 86;
	  case GDK_KP_Add: return 87;
	  case GDK_KP_Enter: return 88;
	  case GDK_KP_End: return 89;
	  case GDK_KP_Down: return 90;
	  case GDK_KP_Page_Down: return 91;
	  case GDK_KP_Left: return 92;
	  /* KP_5 */
	  case GDK_KP_Right: return 94;
	  case GDK_KP_Home: return 95;
	  case GDK_KP_Up: return 96;
	  case GDK_KP_Page_Up: return 97;
	  case GDK_KP_0: case GDK_KP_Insert: return 98;
	  case GDK_KP_Decimal: case GDK_KP_Delete: return 99;
/*	  case GDK_: return 100; */
/*	  case GDK_: return 101; */
/*	  case GDK_: return 102; */
	  case GDK_KP_Equal: return 103;
	  /* F13(104) ... F24(115) */
	  case GDK_Execute: return 116;
	  case GDK_Help: return 117;
	  case GDK_Menu: return 118;
	  case GDK_Select: return 119;
/*	  case GDK_: return 120; */
/*	  case GDK_: return 121; */
	  case GDK_Undo: return 122;
/*	  case GDK_: return 123; */
/*	  case GDK_: return 124; */
/*	  case GDK_: return 125; */
	  case GDK_Find: return 126;
/*	  case GDK_: return 127; */
/*	  case GDK_: return 128; */
/*	  case GDK_: return 129; */
/*	  case GDK_: return 130; */
/*	  case GDK_: return 131; */
/*	  case GDK_: return 132; */
/*	  case GDK_: return 133; */
/*	  case GDK_: return 134; */
/*	  case GDK_: return 135; */
/*	  case GDK_: return 136; */
/*	  case GDK_: return 137; */
/*	  case GDK_: return 138; */
/*	  case GDK_: return 139; */
/*	  case GDK_: return 140; */
/*	  case GDK_: return 141; */
/*	  case GDK_: return 142; */
/*	  case GDK_: return 143; */
	  case GDK_Hangul_switch: return 144;
	  case GDK_Hangul_Hanja: return 145;
	  case GDK_Katakana: return 146;
	  case GDK_Hiragana: return 147;
	  case GDK_Zenkaku_Hankaku: return 148;
/*	  case GDK_: return 149; */
/*	  case GDK_: return 151; */
/*	  case GDK_: return 152; */
/*	  case GDK_: return 153; */
	  case GDK_Sys_Req: return 154;
	  case GDK_Cancel: return 155;
	  case GDK_Clear: return 156;
/*	  case GDK_Prior: return 157; */
/*	  case GDK_: return 158; */
	  case GDK_KP_Separator: return 159;
/*	  case GDK_: return 160; */
/*	  case GDK_: return 161; */
/*	  case GDK_: return 162; */
/*	  case GDK_: return 163; */
/*	  case GDK_: return 164; */
	  /* 164 ... 175: resv. */
/*	  case GDK_: return 176; */
/*	  case GDK_: return 177; */
/*	  case GDK_: return 178; */
/*	  case GDK_: return 179; */
/*	  case GDK_: return 180; */
/*	  case GDK_: return 181; */
/*	  case GDK_: return 182; */
/*	  case GDK_: return 183; */
/*	  case GDK_: return 184; */
/*	  case GDK_: return 185; */
	  case GDK_KP_Tab: return 186;
/*	  case GDK_: return 187; */
/*	  case GDK_: return 188; */
/*	  case GDK_: return 189; */
/*	  case GDK_: return 190; */
/*	  case GDK_: return 191; */
/*	  case GDK_: return 192; */
/*	  case GDK_: return 193; */
/*	  case GDK_: return 194; */
/*	  case GDK_: return 195; */
/*	  case GDK_: return 196; */
/*	  case GDK_: return 197; */
/*	  case GDK_: return 198; */
/*	  case GDK_: return 199; */
/*	  case GDK_: return 200; */
/*	  case GDK_: return 201; */
/*	  case GDK_: return 202; */
/*	  case GDK_: return 203; */
/*	  case GDK_: return 204; */
	  case GDK_KP_Space: return 205;
/*	  case GDK_: return 206; */
/*	  case GDK_: return 207; */
/*	  case GDK_: return 208; */
/*	  case GDK_: return 209; */
/*	  case GDK_: return 210; */
/*	  case GDK_: return 211; */
/*	  case GDK_: return 212; */
/*	  case GDK_: return 213; */
/*	  case GDK_: return 214; */
/*	  case GDK_: return 215; */
/*	  case GDK_: return 216; */
/*	  case GDK_: return 217; */
/*	  case GDK_: return 218; */
/*	  case GDK_: return 219; */
/*	  case GDK_: return 220; */
/*	  case GDK_: return 221; */
	  /* 222 ... 223: resv. */
	  case GDK_Control_L: return 224;
	  case GDK_Shift_L: return 225;
	  case GDK_Alt_L: return 226;
	  case GDK_Super_L: return 227; /* "GUI" */
	  case GDK_Control_R: return 228;
	  case GDK_Shift_R: return 229;
	  case GDK_Alt_R: return 230;
	  case GDK_Super_R: return 231;
	  /* 232 ... : resv. */
	}
	return 0;
}


gboolean
on_macro1_key_press_event              (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        GtkTreeView     *treeview)
{
	unsigned char code;

	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
		return FALSE;

	code = keyval_to_usb(event->keyval);
	if (!code) {
		g_warning(_("Unrecognized key pressed: 0x%04x\n"), event->keyval);
		return TRUE;
	}

	macrolist_append(code, RAZER_MACRO_KEY_DOWN);

	return TRUE;
}


gboolean
on_macro1_key_release_event            (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        GtkTreeView     *treeview)
{
	unsigned char code;

	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
		return FALSE;

	code = keyval_to_usb(event->keyval);
	if (!code) {
		g_warning(_("Unrecognized key released: 0x%04x\n"), event->keyval);
		return TRUE;
	}

	macrolist_append(code, RAZER_MACRO_KEY_UP);

	return TRUE;
}


void
on_button_clear_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_list_store_clear(macrolist);
	macro_count = 0;
}


void
on_button_delay_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	macrolist_append(0, RAZER_MACRO_KEY_DELAY_AFTER);
}


void
on_okbutton1_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
	macrolist_to_macro(macro_button);
	status_add(st_context_user, STATUS_TIMEOUT, _("Key/macro updated"));
}


void
on_cancelbutton1_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	status_add(st_context_user, STATUS_TIMEOUT, _("Key/macro not updated"));
}


void
on_revert_macro_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	macro_to_macrolist(macro_button);
}


void
on_macro_moveup_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkTreeSelection *sel;
	GtkTreeIter itera, iterb;
	GtkTreeModel *macromodel = GTK_TREE_MODEL(macrolist);
	GtkTreePath *path;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(macro1));
	if (!gtk_tree_selection_get_selected(sel, &macromodel, &itera))
		return;

	path = gtk_tree_model_get_path(GTK_TREE_MODEL(macrolist), &itera);
	if (!gtk_tree_path_prev(path)) {
		gtk_tree_path_free(path);
		return;
	}
	if (!gtk_tree_model_get_iter(GTK_TREE_MODEL(macrolist), &iterb, path)) {
		gtk_tree_path_free(path);
		return;
	}

	gtk_tree_path_free(path);
	gtk_list_store_swap(macrolist, &itera, &iterb);
}


void
on_macro_movedown_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkTreeSelection *sel;
	GtkTreeIter itera, iterb;
	GtkTreeModel *macromodel = GTK_TREE_MODEL(macrolist);

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(macro1));
	if (!gtk_tree_selection_get_selected(sel, &macromodel, &itera))
		return;

	iterb = itera;
	if (!gtk_tree_model_iter_next(GTK_TREE_MODEL(macrolist), &iterb))
		return;
	gtk_list_store_swap(macrolist, &itera, &iterb);
}


void
on_macro_trash_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkTreeSelection *sel;
	GtkTreeIter iter, iter_sel;
	GtkTreeModel *macromodel = GTK_TREE_MODEL(macrolist);

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(macro1));
	if (!gtk_tree_selection_get_selected(sel, &macromodel, &iter))
		return;

	iter_sel = iter;
	if (gtk_tree_model_iter_next(macromodel, &iter_sel))
		gtk_tree_selection_select_iter(sel, &iter_sel);

	gtk_list_store_remove(macrolist, &iter);
}


void
on_macro_invert_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkTreeSelection *sel;
	GtkTreeIter iter;
	GtkTreeModel *macromodel = GTK_TREE_MODEL(macrolist);
	GValue actionval;
	guchar action;
	const gchar *actiontext;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(macro1));
	if (!gtk_tree_selection_get_selected(sel, &macromodel, &iter))
		return;

	memset(&actionval, 0, sizeof(actionval));

	gtk_tree_model_get_value(macromodel, &iter, MACRO_COLUMN_ACTION_CODE, &actionval);
	action = g_value_get_uchar(&actionval);

	if (action == RAZER_MACRO_KEY_DOWN) {
		action = RAZER_MACRO_KEY_UP;
	} else if (action == RAZER_MACRO_KEY_UP) {
		action = RAZER_MACRO_KEY_DOWN;
	} else {
		return;
	}

	actiontext = macro_action_to_string(action);
	gtk_list_store_set(macrolist, &iter, MACRO_COLUMN_ACTION_CODE, action, MACRO_COLUMN_ACTION_TEXT, actiontext, -1);
}


void
on_macro_addkey_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	gint response;

	do {
		response = gtk_dialog_run(GTK_DIALOG(keysdialog));

		if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_APPLY) do {
			GtkTreeSelection *sel;
			GtkTreeIter iter;
			GtkTreeModel *keysmodel = GTK_TREE_MODEL(keyslist);
			GValue keyval;
			guchar key;
	
			sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(keysview));
			if (!gtk_tree_selection_get_selected(sel, &keysmodel, &iter))
				break;
	
			memset(&keyval, 0, sizeof(keyval));
	
			gtk_tree_model_get_value(keysmodel, &iter, KEYS_COLUMN_KEY_CODE, &keyval);
			key = g_value_get_uchar(&keyval);
	
			macrolist_append(key, RAZER_MACRO_KEY_DOWN);
			macrolist_append(key, RAZER_MACRO_KEY_UP);
		} while(0);
	} while (response != GTK_RESPONSE_OK && response != GTK_RESPONSE_CLOSE);

	gtk_widget_hide(keysdialog);
}


/* ******************************************************** */
/* FIRMWARE UPDATER                                         */
/* ******************************************************** */

static GtkWidget *flashdialog = NULL;
static GtkWidget *firmware_chooser = NULL;
static GtkWidget *firmware_step, *firmware_progress;
static GtkWidget *firmware_cancel, *firmware_apply;
static GtkWidget *firmware_file;
static GtkWidget *firmware_update;

static gboolean flash_updating = FALSE;
static enum {
	FLASH_DOING_NOTHING,
	FLASH_VERIFYING,
	FLASH_VERIFIED,
	FLASH_VERIFY_FAILED,
	FLASH_UPDATING,
	FLASH_UPDATED,
	FLASH_UPDATE_FAILED,
} flash_state;

static const gchar *step_state[] = {
	NULL,
	N_("Verifying flash\n"),
	N_("Flash passed verification\n"),
	N_("Flash failed verification\n(Modified profiles or different firmware)"),
	N_("Flash update is in progress\nPLEASE OBSERVE THE WARNINGS"),
	N_("Flash update completed successfully\n"),
	N_("Flash update FAILED.\n"),
};

static const gchar step1[] = N_("Step 1) Select a firmware file to upload\n");
static const gchar step1noopen[] = N_("Couldn't open firmware: ");
static const gchar step2[] = N_("Step 2) Unplug the mouse\n");
static const gchar step3[] = N_("Step 3) Plug the mouse in, with the profile button depressed\n");
static const gchar step4[] = N_("You're ready to update!\n(Press escape to cancel, or enter to update)");


static void
flashdialog_check_status(void)
{
	const gchar *filename;
	int fd;
	struct stat sb;
	int status;
	razer_firmware_t fw;

	if (!flashdialog)
		return;

	gtk_widget_set_sensitive(firmware_apply, FALSE);

	if (flash_state) {
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			_(step_state[flash_state])
		);
		return;
	}

	/* Step 1: Select a firmware */

	filename = gtk_entry_get_text(GTK_ENTRY(firmware_file));
	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		if (errno == ENOENT || errno == EISDIR || errno == ENOTDIR) {
			gtk_label_set_text(
				GTK_LABEL(firmware_step),
				_(step1)
			);
		} else {
			gchar *step1_err;
			step1_err = g_strconcat(
				_(step1), _(step1noopen),
				razer_strerror(errno),
				NULL
			);
			gtk_label_set_text(
				GTK_LABEL(firmware_step),
				step1_err
			);
			g_free(step1_err);
		}
		return;
	}

	status = fstat(fd, &sb);
	if (status < 0) {
		gchar *step1_err;
		step1_err = g_strconcat(
			_(step1), _(step1noopen),
			_("fstat failed: "),
			razer_strerror(errno),
			NULL
		);
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			step1_err
		);
		g_free(step1_err);
		close(fd);
		return;
	}

	if (S_ISDIR(sb.st_mode)) {
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			_(step1)
		);
		close(fd);
		return;
	} else if (!S_ISREG(sb.st_mode)) {
		gchar *step1_err;
		step1_err = g_strconcat(
			_(step1), _(step1noopen),
			_("Not a file"),
			NULL
		);
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			step1_err
		);
		g_free(step1_err);
		close(fd);
		return;
	}

	status = razer_firmware_parse_srec(fd, &fw);
	close(fd);
	if (status) {
		gchar *step1_err;
		step1_err = g_strconcat(
			_(step1), _(step1noopen),
			_("Firmware is invalid"),
			NULL
		);
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			step1_err
		);
		g_free(step1_err);
		return;
	}

	/* Step 2: Unplug the mouse */
	if (!razer_fw_udev && razer_udev) {
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			_(step2)
		);

		if (razer_udev->firmware_mode) {
			razer_firmware_mode(razer_udev);
			razer_udev = NULL;
		}
		return;
	}

	/* Step 3: Plug the mouse in */
	if (!razer_fw_udev) {
		gtk_label_set_text(
			GTK_LABEL(firmware_step),
			_(step3)
		);
		return;
	}

	/* Step 4: Update! */
	gtk_label_set_text(
		GTK_LABEL(firmware_step),
		_(step4)
	);

	gtk_widget_set_sensitive(firmware_apply, TRUE);
	gtk_widget_grab_focus(firmware_apply);

	gdk_window_raise(GTK_WIDGET(flashdialog)->window);
	gtk_widget_grab_focus(GTK_WIDGET(flashdialog));
}


static void
display_flashdialog(void)
{
	if (!flashdialog) {
		flashdialog = create_flashdialog();
		firmware_step = lookup_widget(flashdialog, "firmware_step");
		firmware_progress = lookup_widget(flashdialog, "firmware_progress");
		firmware_cancel = lookup_widget(flashdialog, "firmware_cancel");
		firmware_apply = lookup_widget(flashdialog, "firmware_apply");
		firmware_file = lookup_widget(flashdialog, "firmware_file");
		firmware_update = lookup_widget(flashdialog, "firmware_update");
		firmware_chooser = gtk_file_chooser_dialog_new(
			_("Select Firmware"),
			GTK_WINDOW(flashdialog),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_OK,
			NULL
		);
		gtk_window_set_destroy_with_parent(GTK_WINDOW(firmware_chooser), TRUE);
	}

	if (!flash_updating) {
		flash_state = FLASH_DOING_NOTHING;
		gtk_progress_bar_set_text(GTK_PROGRESS_BAR(firmware_progress), "");
		gtk_button_set_label(GTK_BUTTON(firmware_cancel), "gtk-cancel");
		gtk_widget_set_sensitive(firmware_update, TRUE);
		gtk_widget_set_sensitive(firmware_file, TRUE);
	}

	flashdialog_check_status();

	gtk_widget_show(flashdialog);
}


void
on_flashdialog_destroy                 (GtkObject       *object,
                                        gpointer         user_data)
{
	flashdialog = NULL;
}


gboolean
on_flashdialog_delete_event            (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	if (flash_updating)
		return TRUE;

	gtk_widget_destroy(widget);

	return TRUE;
}


void
on_firmware_file_changed               (GtkEditable     *editable,
                                        gpointer         user_data)
{
	flashdialog_check_status();
}


void
on_firmware_open_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	if (gtk_dialog_run(GTK_DIALOG(firmware_chooser)) == GTK_RESPONSE_OK) {
		const gchar *filename;

		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(firmware_chooser));
		gtk_entry_set_text(GTK_ENTRY(firmware_file), filename);
	}

	gtk_widget_hide(firmware_chooser);
}


static void
flash_progress(int action, int start_block, int end_block, double fraction)
{
	const char *actionstr = "";
	double old_fraction;

	if (action == RAZER_PROGRESS_VERIFY_ERROR) {
		g_warning(_("Verification failed: 0x%04x-0x%04x\n"), start_block, end_block);
		return;
	}

	switch (action) {
	  case RAZER_PROGRESS_DONE: actionstr = _("Done"); break;
	  case RAZER_PROGRESS_READING: actionstr = _("Reading"); break;
	  case RAZER_PROGRESS_VERIFYING: actionstr = _("Verifying"); break;
	  case RAZER_PROGRESS_ERASING: actionstr = _("Erasing"); break;
	  case RAZER_PROGRESS_PROGRAMMING: actionstr = _("Programming"); break;
	}

	gtk_progress_bar_set_text(GTK_PROGRESS_BAR(firmware_progress), actionstr);
	old_fraction = gtk_progress_bar_get_fraction(GTK_PROGRESS_BAR(firmware_progress));
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(firmware_progress), fraction);

	if (floor(30.0*fraction) == floor(30.0*old_fraction))
		return;

	while (gtk_events_pending())
		gtk_main_iteration_do(FALSE);
}


void
on_firmware_apply_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *filename;
	int fd;
	int status;
	razer_firmware_t fw;
	gboolean update;

	if (!razer_fw_udev) {
		flashdialog_check_status();
		return;
	}

	filename = gtk_entry_get_text(GTK_ENTRY(firmware_file));
	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		flashdialog_check_status();
		return;
	}

	status = razer_firmware_parse_srec(fd, &fw);
	close(fd);
	if (status) {
		flashdialog_check_status();
		return;
	}

	flash_updating = TRUE;

	gtk_widget_set_sensitive(firmware_file, FALSE);
	gtk_widget_set_sensitive(firmware_update, FALSE);
	gtk_widget_set_sensitive(firmware_cancel, FALSE);
	gtk_widget_set_sensitive(firmware_apply, FALSE);

	update = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(firmware_update));
	if (!update) {
		flash_state = FLASH_VERIFYING;

		status = razer_firmware_verify_flash(razer_fw_udev, &fw, flash_progress);
		if (status) {
			flash_state = FLASH_VERIFY_FAILED;
		} else {
			flash_state = FLASH_VERIFIED;
		}
	} else {
		flash_state = FLASH_UPDATING;

		status = razer_firmware_write_flash(razer_fw_udev, &fw, flash_progress);
		if (status) {
			flash_state = FLASH_UPDATE_FAILED;
		} else {
			flash_state = FLASH_UPDATED;
		}
	}

	flash_updating = FALSE;

	gtk_progress_bar_set_text(GTK_PROGRESS_BAR(firmware_progress), "");
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(firmware_progress), 0.0);

	flashdialog_check_status();

	gtk_button_set_label(GTK_BUTTON(firmware_cancel), "gtk-close");
	gtk_widget_set_sensitive(firmware_cancel, TRUE);
	gtk_widget_grab_focus(firmware_cancel);
}

