/*
 * ot.h - OpenType Organizer header file.
 *
    Copyright (C) 2001  Yao Zhang

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * yaoz@users.sourceforge.net
 */
#ifndef OT_H
#define OT_H 1

#include <stdint.h>

#include "u2c.h"

#define GET2B(p)        ( (p) += 2, ((*((p)-2))<<8)  |  (*((p)-1)) )
#define GET4B(p)        ( (p) += 4, ((*((p)-4))<<24) | ((*((p)-3))<<16) \
			          | ((*((p)-2))<<8)  |  (*((p)-1)) )

#define SET2B(p, n)     ( *(p)++ = ((n)>>8) &0xFF, *(p)++ =  (n)     &0xFF )
#define SET4B(p, n)     ( *(p)++ = ((n)>>24)&0xFF, *(p)++ = ((n)>>16)&0xFF, \
                          *(p)++ = ((n)>>8) &0xFF, *(p)++ =  (n)     &0xFF )

/* OT tags */
#define CMAP_TAG	"cmap"
#define GLYF_TAG	"glyf"
#define HEAD_TAG	"head"
#define HHEA_TAG	"hhea"
#define HMTX_TAG	"hmtx"
#define LOCA_TAG	"loca"
#define MAXP_TAG	"maxp"
#define NAME_TAG	"name"
#define POST_TAG	"post"

#define CVT__TAG	"cvt "
#define FPGM_TAG	"fpgm"
#define HDMX_TAG	"hdmx"
#define KERN_TAG	"hern"
#define OS_2_TAG	"OS/2"
#define PREP_TAG	"prep"

/* Font Directory */
typedef struct {
    uint32_t version;
    uint16_t numTables;
    uint16_t searchRange;
    uint16_t entrySelector;
    uint16_t rangeShift;
} OffsetSubtable;

typedef struct {
    uint8_t tag[4];
    uint32_t checkSum;
    uint32_t offset;
    uint32_t length;
} TableDirectoryEntry;

typedef struct {
    uint32_t length;
    uint8_t *buffer;
    OffsetSubtable offsetSubtable;
    TableDirectoryEntry *tableDirectory;
} FontDirectory;

/* Name Table */
typedef struct {
    uint16_t format;
    uint16_t count;
    uint16_t stringOffset;
} NameTableHeader;

typedef struct {
    uint16_t platformID;
    uint16_t platformSpecificID;
    uint16_t languageID;
    uint16_t nameID;
    uint16_t length;
    uint16_t offset;
} NameRecordEntry;

typedef struct {
    NameRecordEntry nameRecord;
    uint8_t *name;
} NameArray;

typedef struct {
    uint32_t length;
    uint8_t *buffer;
    NameTableHeader nameTableHeader;
    NameArray *nameArray;
} NameTable;

/* CMap Table */
typedef struct {
    uint16_t version;
    uint16_t numberSubtables;
} CMapIndex;

typedef struct {
    uint16_t platformID;
    uint16_t platformSpecificID;
    uint32_t offset;
} CMapEncodingSubtable;

typedef struct {
    uint16_t format;
    uint16_t length;
    uint16_t language;
} CMapFormat0;

typedef struct {
    uint16_t firstCode;
    uint16_t entryCount;
    int16_t idDelta;
    uint16_t idRangeOffset;
} F2SubHeader;

typedef struct {
    uint16_t format;
    uint16_t length;
    uint16_t language;
    uint16_t subHeaderKeys[256];
    F2SubHeader *subHeaders;
} CMapFormat2;

typedef struct {
    uint16_t format;
    uint16_t length;
    uint16_t language;
    uint16_t segCountX2;
    uint16_t searchRange;
    uint16_t entrySelector;
    uint16_t rangeShift;
} CMapFormat4;

typedef struct {
    uint16_t format;
    uint16_t length;
    uint16_t language;
    uint16_t firstCode;
    uint16_t entryCount;
} CMapFormat6;

typedef struct {
    uint16_t format;
    uint16_t reserved;
    uint32_t length;
    uint32_t language;
} CMapFormat8;

typedef struct {
    uint16_t format;
    uint16_t reserved;
    uint32_t length;
    uint32_t language;
    uint32_t startCharCode;
    uint32_t numChars;
} CMapFormat10;

typedef struct {
    uint16_t format;
    uint16_t reserved;
    uint32_t length;
    uint32_t language;
    uint32_t nGroups;
} CMapFormat12;

typedef struct {
    union {
        CMapFormat0 f0;
        CMapFormat2 f2;
        CMapFormat4 f4;
        CMapFormat6 f6;
        CMapFormat8 f8;
        CMapFormat10 f10;
        CMapFormat12 f12;
    } format;
    U2C *u2c;
#if 0
    uint16_t glyphIndexArray[65536];
#endif
} CMapSubtable;

typedef struct {
    CMapEncodingSubtable cmapEncodingSubtable;
    CMapSubtable cmapSubtable;
} CMapArray;

typedef struct {
    uint32_t length;
    uint8_t *buffer;
    CMapIndex cmapIndex;
    CMapArray *cmapArray;
} CMapTable;

__BEGIN_DECLS

extern void read_table(uint8_t *p, uint32_t length,
                       char *file, uint32_t offset);
extern uint32_t calc_table_checksum(uint8_t *p, uint32_t length);
extern int to_utf8_string(char *from, char *buffer, int length,
                                      char *utf8, int utf8bytes);
extern int from_utf8_string(char *to, char *utf8, int utf8bytes,
                                      char *buffer, int length);
extern char *platform_string(int platformID);
extern char *platform_specific_string(int platformID, int platformSpecificID);
extern char *language_string(int platformID, int languageID);
extern char *name_string(int nameID);

extern FontDirectory *new_font_directory(void);
extern void delete_font_directory(FontDirectory *fontDirectory);
extern void read_font_directory(char *file, FontDirectory *fontDirectory);
extern void print_offset_subtable(const OffsetSubtable *offsetSubtable);
extern void print_table_directory_entry(const TableDirectoryEntry *tableDirectoryEntry);

extern NameTable *new_name_table(void);
extern void delete_name_table(NameTable *nameTable);
extern void read_name_table(char *file, FontDirectory *fontDirectory,
                                        NameTable *nameTable);
extern void print_name_table_header(const NameTableHeader *nameTableHeader);
extern void print_name_record_entry(const NameRecordEntry *nameRecordEntry);
extern void print_name_string(NameArray *nameEntry, int in_hex);
extern void recreate_name(NameTable *nameTable);
extern void add_utf8_names(NameTable *nameTable);
extern void replace_names(NameTable *nameTable, char *naming_spec);

extern CMapTable *new_cmap_table(void);
extern void delete_cmap_table(CMapTable *cmapTable);
extern void read_cmap_table(char *file, FontDirectory *fontDirectory,
                                        CMapTable *cmapTable);
extern void print_cmap_index(CMapIndex *cmapIndex);
extern void print_cmap_encoding_and_subtable(CMapArray *cmapArrayEntry);
extern uint16_t f2_glyph_index(uint8_t b1, uint8_t b2,
                               CMapTable *cmapTable, int n);
extern void cmap_format_4(U2C *u2c, uint16_t language,
                          uint8_t **bufferp, uint32_t *lengthp);
extern void add_unicode_cmap(CMapTable *cmapTable,
                             uint8_t **bufferp, uint32_t *lengthp);

__END_DECLS

#endif /* !OT_H */
