%{

#include "headers.h"
#include "cfgvar.h"
#include "servers.h"
#include "rsh.h"
#include "opt.h"
#include "storage.h"

int yylex();
int yyerror();

extern int ipcacfglineno;

%}

/*
 * Token value definition
 */
%union {
	int tv_int;
	char *tv_char;
	struct in_addr tv_ip;
	struct { 
		int is_ip_aggr;
		char *s1;
		char *s2;
		char *s3;
	} tv_aggr;
}

/*
 * Token types returned by scanner
 */
%token	<tv_char>	TOK_STRING
%token	<tv_int>	TOK_INTEGER

/*
 * Reserved words
 */
%token				ERROR
%token				EQ
%token				AT
%token				SEMICOLON
%token				SLASH
%token				IFACE
%token				IFILE
%token				IPQ
%token				ULOG
%token				GROUP
%token				DIVERT
%token				TEE
%token				PORT
%token				INONLY
%token				PROMISC
%token				NETFLOW_SAMPLED
%token				NETFLOW_DISABLE
%token				FILTER

%token				CAPTURE_PORTS

%token				MEMSIZE
%token				BUFFERS
%token				RSH
%token				DUMP
%token				CHROOT
%token				UID
%token				GID
%token				AGGR
%token				STRIP
%token				INTO
%token				DENY
%token				ALLOW
%token				TIMEOUT
%token				TTL
%token				PIDFILE

%token				ADMIN
%token				BACKUP
%token				DEFAULT
%token				VIEW_ONLY

%token				NETFLOW
%token				EXPORT
%token				DESTINATION
%token				VERSION
%token				ACTIVE
%token				INACTIVE
%token				ENGINE_TYPE
%token				ENGINE_ID
%token				SAMPLING_MODE
%token				PACKET_INTERVAL
%token				IFCLASS
%token				MAPTO

%type	<tv_int>	privlevel
%type	<tv_int>	optIFlags IFlags IFlag
%type	<tv_char>	IFilter
%type	<tv_int>	ulog_gmask
%type	<tv_ip>		ip_addr at_ip
%type	<tv_aggr>	aggr_values

%%
cfg_file:
	sequence;

sequence: block ps
	| sequence block ps
	;

ps:
	| SEMICOLON;

privlevel:
	{ $$ = 2; }
	| ADMIN	{ $$ = 5; }
	| BACKUP	{ $$ = 3; }
	| DEFAULT	{ $$ = 2; }
	| VIEW_ONLY { $$ = 1; }
	| DENY { $$ = 0; }
	;

IFlag:
	INONLY { $$ = IFLAG_INONLY; }
	| PROMISC { $$ = IFLAG_PROMISC; }
	| NETFLOW_SAMPLED { $$ = IFLAG_NF_SAMPLED; }
	| NETFLOW_DISABLE { $$ = IFLAG_NF_DISABLE; }
	;

IFlags:
	IFlag { $$ = $1; }
	| IFlags IFlag { $$ = $1 | $2; }
	;

optIFlags:	/* Interface flags, may be omitted */
	{ $$ = 0; }
	| IFlags { $$ = $1; }
	;

IFilter:
	FILTER TOK_STRING { $$ = $2; }
	;

ulog_gmask:
	GROUP TOK_INTEGER {
		$$ = (1 << ($2 - 1));
	}
	| ulog_gmask ',' GROUP TOK_INTEGER {
		$$ = $1 | (1 << ($4 - 1));
	}
	;

block:
	PIDFILE EQ TOK_STRING {
		if(conf->pidfile)
			return yyerror("Pidfile already defined");
		conf->pidfile = $3;
	}
	| IFACE IFILE TOK_STRING optIFlags {
		if(cfg_add_iface("file", $4, $3) == NULL)
			exit(EX_NOINPUT);
	}
	| IFACE TOK_STRING optIFlags {
		if(cfg_add_iface($2, $3, NULL) == NULL)
			exit(EX_NOINPUT);
		free($2);
	}
	| IFACE TOK_STRING optIFlags IFilter {
		if(cfg_add_iface($2, $3, $4) == NULL)
			exit(EX_NOINPUT);
		free($2);
	}
	| IFACE ULOG ulog_gmask optIFlags {
		static int ulog_already_defined;
		char *p;
		if(ulog_already_defined++) {
			return yyerror("Duplicate ULOG interface specification.\nUse \"interface ulog group X, group Y;\" format.\n");
		}
		p = malloc(16);
		assert(p);
		snprintf(p, 16, "%u", $3);
		if(cfg_add_iface("ulog", $4, p) == NULL)
			exit(EX_NOINPUT);
	}
	| IFACE IPQ optIFlags {
		if(cfg_add_iface("ipq", $3, NULL) == NULL)
			exit(EX_NOINPUT);
	}
	| IFACE DIVERT PORT TOK_INTEGER optIFlags {
		char *p;
		if($4 < 0 || $4 > 65535)
			return yyerror("Port must be in range 0..65535");
		p = malloc(16);
		assert(p);
		snprintf(p, 16, "%u", $4);
		if(cfg_add_iface("divert", $5, p) == NULL)
			exit(EX_NOINPUT);
	}
	| IFACE TEE PORT TOK_INTEGER optIFlags {
		char *p = malloc(16);
		assert(p);
		snprintf(p, 16, "%u", $4);
		if(cfg_add_iface("tee", $5, p) == NULL)
			exit(EX_NOINPUT);
	}
	| DUMP EQ TOK_STRING {
		if( conf->dump_table_file )
			free(conf->dump_table_file);
		conf->dump_table_file = $3;
	}
	| CAPTURE_PORTS ALLOW {
		conf->capture_ports = 1;
	}
	| CAPTURE_PORTS DENY {
		conf->capture_ports = 0;
	}
	| RSH ALLOW at_ip {
		if(add_server(rsh_server, "RSH Server", &($3), 514))
			return yyerror("Failed to install RSH server");
		fprintf(stderr, "Configured RSH Server listening at %s\n",
			inet_ntoa($3));
	}
	| RSH DENY at_ip {
		fprintf(stderr, "Warning: Option at line %d has no effect\n",
			ipcacfglineno);
	}
	| RSH TIMEOUT EQ TOK_STRING {
		int to_ms;
		to_ms = atoi($4);
		free($4);
		if(to_ms < 0)
			to_ms = -1;	/* INFTIM */
		else
			to_ms = to_ms * 1000;
		rsh_rw_timeout = to_ms;
	}
	| RSH TOK_STRING privlevel {
		cfg_add_rsh_host("", $2, $3);
		free($2);
	}
	| RSH AT TOK_STRING privlevel {
		cfg_add_rsh_host("", $3, $4);
		free($3);
	}
	| RSH TOK_STRING AT TOK_STRING privlevel {
		cfg_add_rsh_host($2, $4, $5);
		free($2); free($4);
	}
	| RSH TTL EQ TOK_STRING {
		conf->rsh_ttl = atoi($4);
		free($4);
	}
	| TTL EQ TOK_STRING {
		fprintf(stderr, "WARNING: \"ttl = %s;\" at line %d: "
			"Obsolete syntax. Please use \"rsh ttl = %s;\"\n",
			$3, ipcacfglineno, $3);
		conf->rsh_ttl = atoi($3);
		free($3);
	}
	| NetFlow
	| CHROOT EQ TOK_STRING {
		if(conf->chroot_to)
			free(conf->chroot_to);
		conf->chroot_to = $3;
	}
	| UID EQ TOK_STRING {
		int id = $3 ? atoi($3) : -1;
		free($3);
		if(id < 0 || id > 65535)
			return yyerror("Inappropriate UID value");
		conf->set_uid = id;
	}
	| GID EQ TOK_STRING {
		int id = $3 ? atoi($3) : -1;
		free($3);
		if(id < 0 || id > 65535)
			return yyerror("Inappropriate UID value");
		conf->set_gid = id;
	}

	| MEMSIZE EQ TOK_STRING {
		char *p = $3;

		conf->memsize = atoi($3);
		if(conf->memsize <= 0) {
			free($3);
			return yyerror("Invalid memory limit");
		}
		while(*++p) {
			switch(*p) {
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				continue;
			case ' ':
				break;
			case 'k':
				conf->memsize *= 1024;
				break;
			case 'm':
				conf->memsize *= 1024 * 1024;
				break;
			case 'g':
				conf->memsize *= 1024 * 1024 * 1024;
				break;
			case 'e':
				conf->memsize *= sizeof(flow_el_t);
				break;
			default:
				fprintf(stderr,
					"Invalid memory limit at line %d "
					"near '%c': %s\n",
					*p, ipcacfglineno, $3);
				return -1;
			}
		}
		free($3);
	}

	| BUFFERS EQ TOK_STRING {
		char *p = $3;

		conf->bufsize = atoi($3);
		if(conf->bufsize <= 0) {
			free($3);
			return yyerror("Invalid buffers size");
		}
		while(*++p) {
			switch(*p) {
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				continue;
			case ' ':
				break;
			case 'k':
				conf->bufsize *= 1024;
				break;
			case 'm':
				conf->bufsize *= 1024 * 1024;
				break;
			case 'g':
				conf->bufsize *= 1024 * 1024 * 1024;
				break;
			default:
				fprintf(stderr,
					"Invalid buffers size at line %d "
					"near '%c': %s\n",
					*p, ipcacfglineno, $3);
				return -1;
			}
		}
		free($3);
	}

	| AGGR TOK_STRING aggr_values {
		int ret;
		$3.s1 = $2;
		if($3.is_ip_aggr) {
			ret = cfg_add_atable($3.s1, $3.s2, $3.s3);
		} else {
			ret = cfg_add_aggregate_ports_table(
				atoi($3.s1),
				atoi($3.s2?$3.s2:$3.s1),
				atoi($3.s3));
		}

		if($3.s1) free($3.s1);
		if($3.s2) free($3.s2);
		if($3.s3) free($3.s3);
		if(ret)
			return yyerror("Parse error");
	}
	;

aggr_values:
	SLASH TOK_STRING STRIP TOK_STRING {
		$$.is_ip_aggr = 1;
		$$.s2 = $2;
		$$.s3 = $4;
	}
	| '-' TOK_STRING INTO TOK_STRING {
		$$.is_ip_aggr = 0;
		$$.s2 = $2;
		$$.s3 = $4;
	}
	| INTO TOK_STRING {
		$$.is_ip_aggr = 0;
		$$.s2 = NULL;
		$$.s3 = $2;
	}
	;

at_ip:
	{ $$.s_addr = INADDR_ANY; }
	| AT TOK_STRING {
		if(inet_aton($2, &($$)) != 1)
			return yyerror("IP address expected");
		free($2);
	}
	;

ip_addr:
	TOK_STRING {
		if(inet_aton($1, &($$)) != 1)
			return yyerror("IP address expected");
		free($1);
	}
	;

NetFlow:
	NETFLOW EXPORT DESTINATION ip_addr TOK_INTEGER {
		conf->netflow_enabled = 1;
		if(add_server(netflow_exporter,
				"NetFlow destination", &($4), $5))
			return yyerror("Failed to install NetFlow destination");
		fprintf(stderr, "Configured NetFlow destination at %s:%d\n",
			inet_ntoa($4), $5);
	}
	| NETFLOW EXPORT VERSION TOK_INTEGER {
		if($4 != 5 && $4 != 1)
			yyerror("NetFlow version: "
				"only versions 1 and 5 are supported");
		conf->netflow_version = $4;
	}
	| NETFLOW TIMEOUT ACTIVE TOK_INTEGER {
		if($4 < 1 || $4 > 60)
			return yyerror("Netflow active timeout "
				"out of range 1..60");
		conf->netflow_timeout_active = $4 * 60;
	}
	| NETFLOW TIMEOUT INACTIVE TOK_INTEGER {
		if($4 < 1 || $4 > 600)
			return yyerror("Netflow inactive timeout "
				"out of range 1..600");
		conf->netflow_timeout_inactive = $4;
	}
	| NETFLOW ENGINE_TYPE TOK_INTEGER {
		if($3 < 0 || $3 > 255)
			return yyerror("Netflow engine-type "
				"out of range 0..255");
		conf->netflow_engine_type = $3;
	}
	| NETFLOW ENGINE_ID TOK_INTEGER {
		if($3 < 0 || $3 > 255)
			return yyerror("Netflow engine-id "
				"out of range 0..255");
		conf->netflow_engine_id = $3;
	}
	| NETFLOW SAMPLING_MODE PACKET_INTERVAL TOK_INTEGER {
		if($4 < 10 || $4 > 16382)
			return yyerror("Netflow packet interval "
				"out of range 10..16382");
		conf->netflow_packet_interval = $4;
	}
	| NETFLOW IFCLASS TOK_STRING MAPTO TOK_INTEGER '-' TOK_INTEGER {
		if($5 > $7)
			return yyerror("NetFlow invalid range: %d..%d",
				$5, $7);
		if(cfg_add_ifclass_map($3, $5, $7)) {
			free($3);
			return yyerror("Cannot add NetFlow interface class to SNMP id mapping");
		}
	}
	;

%%

int
yyerror(char *s) {
	fprintf(stderr,
		"Config parse error near line %d: %s\n",
		ipcacfglineno, s);
	return -1;
};

