package container

import (
	"github.com/jfrog/jfrog-cli-core/artifactory/utils"
	"github.com/jfrog/jfrog-cli-core/artifactory/utils/container"
	"github.com/jfrog/jfrog-cli-core/utils/config"
)

type BuildDockerCreateCommand struct {
	ContainerManagerCommand
	containerManagerType container.ContainerManagerType
	manifestSha256       string
}

func NewBuildDockerCreateCommand() *BuildDockerCreateCommand {
	return &BuildDockerCreateCommand{containerManagerType: container.Kaniko}
}

// Set tag and manifest sha256 of an image in Artifactory.
// This file can be generated by Kaniko using the'--image-name-with-digest-file' flag.
// Kaniko is a tool to build & push container images from a Dockerfile, inside a container or Kubernetes cluster.
// Tag and Sha256 will be used later on to search the image in Artifactory.
func (bpc *BuildDockerCreateCommand) SetImageNameWithDigest(filePath string) (err error) {
	bpc.imageTag, bpc.manifestSha256, err = container.GetImageTagWithDigest(filePath)
	return
}

func (bpc *BuildDockerCreateCommand) Run() error {
	serverDetails, err := bpc.ServerDetails()
	if err != nil {
		return err
	}
	cm := container.NewManager(bpc.containerManagerType)
	image := container.NewImage(bpc.imageTag)
	buildName := bpc.BuildConfiguration().BuildName
	buildNumber := bpc.BuildConfiguration().BuildNumber
	project := bpc.BuildConfiguration().Project
	if err := utils.SaveBuildGeneralDetails(buildName, buildNumber, project); err != nil {
		return err
	}
	serviceManager, err := utils.CreateServiceManager(serverDetails, -1, false)
	if err != nil {
		return err
	}
	builder, err := container.NewKanikoBuildInfoBuilder(image, bpc.Repo(), buildName, buildNumber, project, serviceManager, container.Push, cm, bpc.manifestSha256)
	if err != nil {
		return err
	}
	buildInfo, err := builder.Build(bpc.BuildConfiguration().Module)
	if err != nil {
		return err
	}
	return utils.SaveBuildInfo(buildName, buildNumber, project, buildInfo)
}

func (pc *BuildDockerCreateCommand) CommandName() string {
	return "rt_build_docker_create"
}

func (pc *BuildDockerCreateCommand) ServerDetails() (*config.ServerDetails, error) {
	return pc.serverDetails, nil
}
