﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/DataSourceRunStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DataZone {
namespace Model {

/**
 */
class ListDataSourceRunsRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API ListDataSourceRunsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDataSourceRuns"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  AWS_DATAZONE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The identifier of the data source.</p>
   */
  inline const Aws::String& GetDataSourceIdentifier() const { return m_dataSourceIdentifier; }
  inline bool DataSourceIdentifierHasBeenSet() const { return m_dataSourceIdentifierHasBeenSet; }
  template <typename DataSourceIdentifierT = Aws::String>
  void SetDataSourceIdentifier(DataSourceIdentifierT&& value) {
    m_dataSourceIdentifierHasBeenSet = true;
    m_dataSourceIdentifier = std::forward<DataSourceIdentifierT>(value);
  }
  template <typename DataSourceIdentifierT = Aws::String>
  ListDataSourceRunsRequest& WithDataSourceIdentifier(DataSourceIdentifierT&& value) {
    SetDataSourceIdentifier(std::forward<DataSourceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone domain in which to invoke the
   * <code>ListDataSourceRuns</code> action.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  ListDataSourceRunsRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of runs to return in a single call to
   * <code>ListDataSourceRuns</code>. When the number of runs to be listed is greater
   * than the value of <code>MaxResults</code>, the response contains a
   * <code>NextToken</code> value that you can use in a subsequent call to
   * <code>ListDataSourceRuns</code> to list the next set of runs.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListDataSourceRunsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the number of runs is greater than the default value for the
   * <code>MaxResults</code> parameter, or if you explicitly specify a value for
   * <code>MaxResults</code> that is less than the number of runs, the response
   * includes a pagination token named <code>NextToken</code>. You can specify this
   * <code>NextToken</code> value in a subsequent call to
   * <code>ListDataSourceRuns</code> to list the next set of runs.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListDataSourceRunsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the data source.</p>
   */
  inline DataSourceRunStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(DataSourceRunStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListDataSourceRunsRequest& WithStatus(DataSourceRunStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dataSourceIdentifier;
  bool m_dataSourceIdentifierHasBeenSet = false;

  Aws::String m_domainIdentifier;
  bool m_domainIdentifierHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  DataSourceRunStatus m_status{DataSourceRunStatus::NOT_SET};
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
