﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/codecommit/model/PullRequestEventType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class DescribePullRequestEventsRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API DescribePullRequestEventsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribePullRequestEvents"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The system-generated ID of the pull request. To get this ID, use
   * <a>ListPullRequests</a>.</p>
   */
  inline const Aws::String& GetPullRequestId() const { return m_pullRequestId; }
  inline bool PullRequestIdHasBeenSet() const { return m_pullRequestIdHasBeenSet; }
  template <typename PullRequestIdT = Aws::String>
  void SetPullRequestId(PullRequestIdT&& value) {
    m_pullRequestIdHasBeenSet = true;
    m_pullRequestId = std::forward<PullRequestIdT>(value);
  }
  template <typename PullRequestIdT = Aws::String>
  DescribePullRequestEventsRequest& WithPullRequestId(PullRequestIdT&& value) {
    SetPullRequestId(std::forward<PullRequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional. The pull request event type about which you want to return
   * information.</p>
   */
  inline PullRequestEventType GetPullRequestEventType() const { return m_pullRequestEventType; }
  inline bool PullRequestEventTypeHasBeenSet() const { return m_pullRequestEventTypeHasBeenSet; }
  inline void SetPullRequestEventType(PullRequestEventType value) {
    m_pullRequestEventTypeHasBeenSet = true;
    m_pullRequestEventType = value;
  }
  inline DescribePullRequestEventsRequest& WithPullRequestEventType(PullRequestEventType value) {
    SetPullRequestEventType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user whose actions resulted in the
   * event. Examples include updating the pull request with more commits or changing
   * the status of a pull request.</p>
   */
  inline const Aws::String& GetActorArn() const { return m_actorArn; }
  inline bool ActorArnHasBeenSet() const { return m_actorArnHasBeenSet; }
  template <typename ActorArnT = Aws::String>
  void SetActorArn(ActorArnT&& value) {
    m_actorArnHasBeenSet = true;
    m_actorArn = std::forward<ActorArnT>(value);
  }
  template <typename ActorArnT = Aws::String>
  DescribePullRequestEventsRequest& WithActorArn(ActorArnT&& value) {
    SetActorArn(std::forward<ActorArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An enumeration token that, when provided in a request, returns the next batch
   * of the results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribePullRequestEventsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A non-zero, non-negative integer used to limit the number of returned
   * results. The default is 100 events, which is also the maximum number of events
   * that can be returned in a result.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribePullRequestEventsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_pullRequestId;
  bool m_pullRequestIdHasBeenSet = false;

  PullRequestEventType m_pullRequestEventType{PullRequestEventType::NOT_SET};
  bool m_pullRequestEventTypeHasBeenSet = false;

  Aws::String m_actorArn;
  bool m_actorArnHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
