﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudfront/CloudFront_EXPORTS.h>
#include <aws/cloudfront/model/ResponseHeadersPolicyConfig.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace CloudFront {
namespace Model {

/**
 * <p>A response headers policy.</p> <p>A response headers policy contains
 * information about a set of HTTP response headers.</p> <p>After you create a
 * response headers policy, you can use its ID to attach it to one or more cache
 * behaviors in a CloudFront distribution. When it's attached to a cache behavior,
 * the response headers policy affects the HTTP headers that CloudFront includes in
 * HTTP responses to requests that match the cache behavior. CloudFront adds or
 * removes response headers according to the configuration of the response headers
 * policy.</p> <p>For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/modifying-response-headers.html">Adding
 * or removing HTTP headers in CloudFront responses</a> in the <i>Amazon CloudFront
 * Developer Guide</i>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2020-05-31/ResponseHeadersPolicy">AWS
 * API Reference</a></p>
 */
class ResponseHeadersPolicy {
 public:
  AWS_CLOUDFRONT_API ResponseHeadersPolicy() = default;
  AWS_CLOUDFRONT_API ResponseHeadersPolicy(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_CLOUDFRONT_API ResponseHeadersPolicy& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_CLOUDFRONT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>The identifier for the response headers policy.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  ResponseHeadersPolicy& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the response headers policy was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  ResponseHeadersPolicy& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A response headers policy configuration.</p>
   */
  inline const ResponseHeadersPolicyConfig& GetResponseHeadersPolicyConfig() const { return m_responseHeadersPolicyConfig; }
  inline bool ResponseHeadersPolicyConfigHasBeenSet() const { return m_responseHeadersPolicyConfigHasBeenSet; }
  template <typename ResponseHeadersPolicyConfigT = ResponseHeadersPolicyConfig>
  void SetResponseHeadersPolicyConfig(ResponseHeadersPolicyConfigT&& value) {
    m_responseHeadersPolicyConfigHasBeenSet = true;
    m_responseHeadersPolicyConfig = std::forward<ResponseHeadersPolicyConfigT>(value);
  }
  template <typename ResponseHeadersPolicyConfigT = ResponseHeadersPolicyConfig>
  ResponseHeadersPolicy& WithResponseHeadersPolicyConfig(ResponseHeadersPolicyConfigT&& value) {
    SetResponseHeadersPolicyConfig(std::forward<ResponseHeadersPolicyConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedTime{};
  bool m_lastModifiedTimeHasBeenSet = false;

  ResponseHeadersPolicyConfig m_responseHeadersPolicyConfig;
  bool m_responseHeadersPolicyConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFront
}  // namespace Aws
