/*

Copyright 2006 Suzanne Skinner, John Spray

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <limits.h>
#include <dirent.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libintl.h>
#define _(String) gettext (String)

#include "lucidlife.h"
#include "life.h"
#include "ewmh.h"
#include "util.h"



/*** Constants ***/

static point  null_point = {-1, -1};
static rect   null_rect  = {{-1, -1}, {-1, -1}};

char* file_format_names[NUM_FORMATS] = {"GLF (GtkLife)", "RLE", "Life 1.05", "Life 1.06", "XLife"};
char* default_file_extensions[]      = {"glf", "rle", "lif", "l", ""};
char* color_names[NUM_COLORS]        = {"background", "live cells", "grid", "selection"};

static char*  component_short_names[NUM_COMPONENTS] = {
	"toolbar",
	"sidebar",
	"scrollbars",
	"statusbar"
};

pattern_coll  pattern_collections[NUM_COLLECTIONS] = {
	{"lpa",     "Life Pattern Archive - Alan Hensel"},
	{"jslife",  "JSLife - Jason Summers"},
};

gchar *stock_icon_files[] = {"zoom_16.png", "description.png",
                             "draw.png", "select.png", "grab.png"};
gchar *stock_names[] = {"zoom_16", "description",
                        "draw", "select", "grab"};

static const char *ui_description =
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu name = 'FileMenu' action='FileMenu'>"
"      <menuitem action='New'/>"
"      <menuitem action='Open'/>"
"      <separator/>"
"      <menuitem action='Save'/>"
"      <menuitem action='SaveAs'/>"
"      <menuitem action='Revert'/>"
"      <separator/>"
"      <menuitem action='ChooseCollection'/>"
"      <menuitem action='Description' name ='ItemBeforeRecentList'/>"
"      <separator/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu name='EditMenu' action='EditMenu'>"
"      <menuitem action='Cut'/>"
"      <menuitem action='Copy'/>"
"      <menuitem action='Paste'/>"
"      <menuitem action='Clear'/>"
"      <menuitem action='Move'/>"
"      <separator/>"
"      <menuitem action='CancelPaste'/>"
"      <separator/>"
"      <menu action='ToolMenu'>"
"        <menuitem action='Draw'/>"
"        <menuitem action='Select'/>"
"        <menuitem action='Grab'/>"
"      </menu>"
"      <separator/>"
"      <menuitem action='Preferences'/>"
"    </menu>"
"    <menu action='ViewMenu'>"
"      <menuitem action='ZoomIn'/>"
"      <menuitem action='ZoomOut'/>"
"      <menu action='ZoomMenu'>"
"        <menuitem action='Zoom1:1'/>"
"        <menuitem action='Zoom2:1'/>"
"        <menuitem action='Zoom4:1'/>"
"        <menuitem action='Zoom8:1'/>"
"        <menuitem action='Zoom16:1'/>"
"      </menu>"
"      <separator/>"
"      <menuitem action='Recenter'/>"
"      <menuitem action='FindActiveCells'/>"
"      <separator/>"
"      <menuitem action='Fullscreen'/>"
"      <menuitem action='ShowToolbar'/>"
"      <menuitem action='ShowSidebar'/>"
"      <menuitem action='ShowStatusbar'/>"
"      <menuitem action='ShowScrollbars'/>"
"    </menu>"
"    <menu action='RunMenu'>"
"      <menuitem action='Play'/>"
"      <menuitem action='Step'/>"
"      <menuitem action='Jump'/>"
"      <separator/>"
"      <menuitem action='Speed'/>"
"      <menuitem action='Faster'/>"
"      <menuitem action='Slower'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='HelpContents'/>"
"      <menuitem action='PatternArchive'/>"
"      <menuitem action='FileFormat'/>"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"  <toolbar name='MainToolBar'>"
"    <separator/>"
"    <toolitem action='New'/>"
"    <toolitem action='Open'/>"
"    <toolitem action='Save'/>"
"    <separator/>"
"    <toolitem action='Cut'/>"
"    <toolitem action='Copy'/>"
"    <toolitem action='Paste'/>"
"    <toolitem action='CancelPaste'/>"
"    <separator/>"
"    <toolitem action='Draw'/>"
"    <toolitem action='Select'/>"
"    <toolitem action='Grab'/>"
"    <separator/>"
"    <toolitem action='ZoomIn'/>"
"    <toolitem action='ZoomOut'/>"
"    <separator/>"
"    <toolitem action='Play'/>"
"    <toolitem action='Step'/>"
"  </toolbar>"

"  <accelerator action='Draw'/>"
"  <accelerator action='Select'/>"
"  <accelerator action='Grab'/>"

"  <accelerator action='ScrollLeft'/>"
"  <accelerator action='ScrollRight'/>"
"  <accelerator action='ScrollUp'/>"
"  <accelerator action='ScrollDown'/>"
"  <accelerator action='ScrollNW'/>"
"  <accelerator action='ScrollNE'/>"
"  <accelerator action='ScrollSW'/>"
"  <accelerator action='ScrollSE'/>"
"  <accelerator action='PageLeft'/>"
"  <accelerator action='PageRight'/>"
"  <accelerator action='PageUp'/>"
"  <accelerator action='PageDown'/>"
"  <accelerator action='PageNW'/>"
"  <accelerator action='PageNE'/>"
"  <accelerator action='PageSW'/>"
"  <accelerator action='PageSE'/>"
"</ui>";

/* Normal items */
static GtkActionEntry ui_entries[] = {
	{ "FileMenu", NULL, ("_File") },
	{ "EditMenu", NULL, "_Edit" },
	{ "ToolMenu", NULL, "_Tool" },
	{ "ViewMenu", NULL, "_View" },
	{ "ZoomMenu", NULL, "_Zoom" },
	{ "RunMenu", NULL, "_Run" },
	{ "HelpMenu", NULL, "_Help" },

	{ "New", GTK_STOCK_NEW, NULL, NULL, "New file", file_new },
	{ "Open", GTK_STOCK_OPEN, NULL, NULL, "Open a file", file_open },
	{ "Revert", GTK_STOCK_REVERT_TO_SAVED, NULL, "R", "Revert to saved file", file_reopen },
	{ "Save", GTK_STOCK_SAVE, NULL, NULL, "Save the file", file_save },
	{ "SaveAs", GTK_STOCK_SAVE_AS, NULL, NULL, "Save the file under a different name", file_save_as },
	{ "Description", "description", "_Description...", "D", "Edit the pattern's description", file_description },
	{ "ChooseCollection", NULL, "_Choose Pattern Collection...", "C", ("Choose from pattern collections or select a custom pattern folder"), file_change_collection },
	{ "Quit", GTK_STOCK_QUIT, NULL, NULL, "Exit the program", file_quit },

	{ "Cut", GTK_STOCK_CUT, NULL, NULL, ("Cut the selection to the clipboard"), edit_cut},
	{ "Copy", GTK_STOCK_COPY, NULL, NULL, ("Copy the selection to the clipboard"), edit_copy},
	{ "Paste", GTK_STOCK_PASTE, NULL, NULL, ("Paste the last selection"), edit_paste},
	{ "Clear", GTK_STOCK_CLEAR, NULL, NULL, ("Cancel the current pasting operation"), edit_clear},
	{ "Move", "", "_Move", "M", NULL, edit_move},
	{ "CancelPaste", GTK_STOCK_CANCEL, "C_ancel Paste", "<control>G", NULL, edit_cancel_paste},
	{ "Preferences", GTK_STOCK_PREFERENCES, NULL, "P", NULL, edit_preferences },

	{ "ZoomIn", GTK_STOCK_ZOOM_IN, NULL, "<control>plus", "Zoom into the image", view_zoom_in },
	{ "ZoomOut", GTK_STOCK_ZOOM_OUT, NULL, "<control>minus", "Zoom away from the image", view_zoom_out },
	{ "Recenter", NULL, "_Recenter", "KP_5", "", view_recenter },
	{ "FindActiveCells", GTK_STOCK_ZOOM_FIT, "_Find Active Cells", "0", "", view_find_active_cells },

	{ "Step", GTK_STOCK_MEDIA_NEXT, "S_tep", "T", "Step ahead one generation", run_step },
	{ "Jump", GTK_STOCK_JUMP_TO, "_Jump to...", "J", "Jump ahead some number of generations", run_jump },
	{ "Faster", GTK_STOCK_MEDIA_FORWARD, "_Faster", "period", "", run_faster },
	{ "Slower", GTK_STOCK_MEDIA_REWIND, "Slo_wer", "comma", "", run_slower },
	{ "Speed", NULL, "_Speed...", "<alt>s", "", run_speed },

	{ "HelpContents", GTK_STOCK_HELP, NULL, NULL, "", help_help },
	{ "PatternArchive", NULL, "_Pattern Archive", NULL, "", help_pattern_archive },
	{ "FileFormat", NULL, "_GLF File Format", NULL, "", help_glf_file_format },
	{ "About", GTK_STOCK_ABOUT, "_About " TITLE, NULL, "", help_about },

	{ "ScrollLeft",  NULL, "", "KP_4", NULL, position_scroll_left },
	{ "ScrollRight",  NULL, "", "KP_6", NULL, position_scroll_right },
	{ "ScrollUp",  NULL, "", "KP_8", NULL, position_scroll_up },
	{ "ScrollDown",  NULL, "", "KP_2", NULL, position_scroll_down },
	{ "ScrollNW",  NULL, "", "KP_7", NULL, position_scroll_nw },
	{ "ScrollNE",  NULL, "", "KP_9", NULL, position_scroll_ne },
	{ "ScrollSW",  NULL, "", "KP_1", NULL, position_scroll_sw },
	{ "ScrollSE",  NULL, "", "KP_3", NULL, position_scroll_se },
	{ "PageLeft",  NULL, "", "<control>KP_4", NULL, position_page_left },
	{ "PageRight",  NULL, "", "<control>KP_6", NULL, position_page_right },
	{ "PageUp",  NULL, "", "<control>KP_8", NULL, position_page_up },
	{ "PageDown",  NULL, "", "<control>KP_2", NULL, position_page_down },
	{ "PageNW",  NULL, "", "<control>KP_7", NULL, position_page_nw },
	{ "PageNE",  NULL, "", "<control>KP_9", NULL, position_page_ne },
	{ "PageSW",  NULL, "", "<control>KP_1", NULL, position_page_sw },
	{ "PageSE",  NULL, "", "<control>KP_3", NULL, position_page_se },
};

/* Radio items */
static GtkRadioActionEntry ui_zoom_radios[] = {
	{ "Zoom1:1", GTK_STOCK_ZOOM_100, "_1:1", "1", "Zoom all the way out", 1 },
	{ "Zoom2:1", NULL, "_2:1", "2", "NULL", 2 },
	{ "Zoom4:1", NULL, "_4:1", "4", "NULL", 4 },
	{ "Zoom8:1", NULL, "_8:1", "8", "NULL", 8 },
	{ "Zoom16:1", "zoom_16", "1_6:1", "Z", "Zoom all the way in", 16 },
};

static GtkRadioActionEntry ui_tool_radios[] = {
	{ "Draw", "draw", ("Draw"), "q", "Draw on the pattern", TOOL_DRAW },
	{ "Select", "select", ("Select"), "w", "Select regions of the pattern for copying", TOOL_SELECT },
	{ "Grab", "grab", ("Grab"), "e", "Use the mouse to explore the pattern", TOOL_GRAB },
};

/* Toggle items */
static GtkToggleActionEntry ui_toggle_entries[] = {
	{ "Fullscreen", NULL, "_Fullscreen", "F11", "Switch between full screen and windowed mode", G_CALLBACK (view_fullscreen_cb), FALSE },
	{ "ShowToolbar", NULL, "_Show Toolbar", "<control><shift>T", "Show the toolbar or not", G_CALLBACK (view_showtoolbar_cb), TRUE },
	{ "ShowSidebar", NULL, "S_how Sidebar", NULL, "Show the sidebar or not", G_CALLBACK (view_showsidebar_cb), FALSE },
	{ "ShowScrollbars", NULL, "Sh_ow Scrollbars", NULL, "Show the scrollbars or not", G_CALLBACK (view_showscrollbars_cb), TRUE },
	{ "ShowStatusbar", NULL, "Sho_w Status Bar", NULL, "Show the status bar or not", G_CALLBACK (view_showstatusbar_cb), TRUE },
	{ "Play", GTK_STOCK_MEDIA_PLAY, "_Playing", "S", "Start/stop the game", G_CALLBACK (run_play_cb), FALSE },
};

/*** Globals ***/

static char*  user_dir = NULL;

/* Preferences */
static struct {
	char*            collection;
	int32            window_width;
	int32            window_height;
	int32            zoom;
	boolean          visible_components[NUM_COMPONENTS];
	boolean          fullscreen;
	uint32           colors[NUM_COLORS];
	int32            speed;
	boolean          skip_frames;
	int32            speed_max;
} config = {
	NULL,
	DEFAULT_WINDOW_WIDTH,
	DEFAULT_WINDOW_HEIGHT,
	DEFAULT_ZOOM,
	{DEFAULT_SHOW_TOOLBAR, DEFAULT_SHOW_SIDEBAR, DEFAULT_SHOW_SCROLLBARS, DEFAULT_SHOW_STATUSBAR},
	DEFAULT_FULLSCREEN,
	{DEFAULT_BG_COLOR, DEFAULT_CELL_COLOR, DEFAULT_GRID_COLOR, DEFAULT_SELECT_COLOR},
	DEFAULT_SPEED,
	DEFAULT_SKIP_FRAMES,
	DEFAULT_SPEED_MAX
};

/* Some general state globals */
static struct {
	/* Directories and pattern paths */
	char*             home_dir;
	char*             pattern_path;
	file_format_type  file_format;
	char*             current_collection;
	coll_status_type  collection_status;
	char*             current_dir;
	pattern_file*     sidebar_files;
	int32             num_sidebar_files;
	pattern_file*     sub_sidebar_files;
	int32             num_sub_sidebar_files;
	recent_file       recent_files[MAX_RECENT_FILES];

	/* Running pattern stats */
	boolean  pattern_running;
	int32    speed;
	boolean  skip_frames;
	uint32   start_tick;
	uint64   start_time;
	int32    skipped_frames;
	boolean  jump_cancelled;

	/* Data related to mouse tracking, mouse dragging and cut/paste */
	tracking_mouse_type  tracking_mouse;
	point       hover_point;
	point       last_drawn;
	rect        selection;
	uint64      selection_start_time;
	cage_type*  copy_buffer;
	rect        copy_rect;
	rect        paste_box;
	boolean     moving;
	tool_id     active_tool;

	/* Data related to statusbar messages */
	boolean     temp_message_active;
	uint64      temp_message_start_time;
	char*       original_message;
} state;

/* Globals related to the GUI display state */
static struct {
	/* Zoom level, where [zoom]:1 is the display ratio */
	int32        zoom;

	/* Physical canvas and logical viewport. Canvas is measured in pixels,
	 * viewport in cells */
	dimension    canvas_size;
	dimension    eff_canvas_size;   /* "effective" canvas size--minus any unused space */
	dimension    viewport_sizes[MAX_ZOOM+1];
	dimension    viewport_size;
	rect         viewport;

	/* Clipping rectangle, in pixels */
	rect         update;

	/* Offscreen pixmap data */
	uint8*       life_pixmap;
	uint32       life_pixmap_alloc_len;
	GdkRgbCmap*  life_pixmap_colormap;

	/* Which parts of the GUI are visible */
	boolean      visible_components[NUM_COMPONENTS];
	boolean      sub_sidebar_visible;
	boolean      fullscreen;
} dstate;

/* GUI objects that need to be global */
static struct {
	GtkWidget*   window;
	GtkWidget*   menubar;
	GtkTooltips* menu_tooltips;
	GtkWidget*   toolbar;
	GtkWidget*   sidebar;
	GtkWidget*   main_sidebar;
	GtkWidget*   sub_sidebar;
	GtkWidget*   patterns_clist;
	GtkWidget*   sub_patterns_clist;
	GtkWidget*   canvas;
	GtkWidget*   vscrollbar;
	GtkWidget*   hscrollbar;
	GtkWidget*   statusbar;
	GtkWidget*   recent_files_separator;
	GtkWidget*   speed_label;
	GtkWidget*   speed_slider;
	GtkWidget*   hover_point_label;
	GtkWidget*   status_message_label;
	GtkWidget*   tick_label;
	GtkWidget*   population_label;

	GtkWidget*   description_dialog;
	GtkTextBuffer*   description_textbuffer;

	GtkUIManager *ui_manager;
	GtkActionGroup *action_group;
} gui;

/*** Main ***/

int main(int argc, char *argv[])
{
	struct stat  statbuf;
	char*        resolved_path;

	set_prog_name(PROG);
	if (argc > 2)
		usage_abort();
	state.home_dir = get_home_directory();
	create_user_dir();

	load_preferences();
	state_from_config();
	load_recent_files_list();

	gnome_vfs_init();
	gtk_init(&argc, &argv);
	init_gui();

	if (argc > 1) {
		resolved_path = get_canonical_path(argv[1]);
		if (!resolved_path)
		    error_dialog("Bad path on command line:\n\"%s\" does not exist", argv[1]);
		else if (stat(resolved_path, &statbuf) == 0 && S_ISDIR(statbuf.st_mode)) {
		    validate_and_set_collection_dir(resolved_path, TRUE);
		    state.collection_status = COLL_OKAY;
		} else {
		    attempt_load_pattern(resolved_path);
		    set_current_dir_from_file(resolved_path);
		}
		free(resolved_path);
	}

	if (state.collection_status == COLL_BAD)
		error_dialog("Warning: the configured pattern collection is no\n"
		             "longer valid, restoring default collection.");
	else if (state.collection_status == COLL_DEFAULT_BAD)
		error_dialog("Warning: The default pattern collection is unavailable.\n"
		             "(Has %s been properly installed?)", TITLE);

	main_loop();

	gnome_vfs_shutdown();

	return 0;
}

void usage_abort(void)
{
	fprintf(stderr, "Usage: %s <pattern file> or %s <pattern directory>\n", PROG, PROG);
	exit(1);
}

/* Main loop functions */

/* Handle events and run the Life pattern
 */
void main_loop(void)
{
	while (1) {
		/* Block if no pattern is running and we don't need to track mouse movement, else poll */
		while (gtk_events_pending())
		    gtk_main_iteration_do(FALSE);
		if (state.temp_message_active &&
		    get_time_milliseconds() - state.temp_message_start_time >= TEMP_MESSAGE_INTERVAL)
		    restore_status_message();
		if (state.pattern_running)
		    tick_and_update();
		else
		    usleep(10000);   /* don't eat up all the processor*/
	}
}

/* If we're in sync with the requested speed (or the pattern is being stepped through manually),
 * process the next Life tick, update the pixmap and redraw the screen. If going too fast, don't do
 * anything. If too slow and state.skip_frames is TRUE, and state.skipped_frames is not maxed out
 * yet, process the next tick and update the pixmap, but don't redraw the screen.
 */
void tick_and_update(void)
{
	int32  running_fps = 0;

	if (state.pattern_running) {
		running_fps = ROUND((double)(tick - state.start_tick) * 1000.0 /
		                    (double)(get_time_milliseconds() - state.start_time));
		if (running_fps > state.speed) {
			usleep(10000);   /* don't eat up all the processor*/
			return;
		}
	}
	if (!state.skipped_frames) {
		dstate.update.start.x = dstate.canvas_size.width  - 1;
		dstate.update.start.y = dstate.canvas_size.height - 1;
		dstate.update.end.x = dstate.update.end.y = 0;
	}

	next_tick(TRUE);

	if (state.pattern_running && running_fps < state.speed && state.skip_frames &&
		state.skipped_frames < MAX_SKIPPED_FRAMES)
		state.skipped_frames++;
	else {
		trigger_canvas_update();
		update_tick_label();
		update_population_label();
	}
}

/*** Functions for reading/saving user settings ***/

/* Create the user directory (if necessary), and record its full path.
 */
void create_user_dir(void)
{
	user_dir = dsprintf("%s/%s", state.home_dir, USER_DIR);
	if (mkdir(user_dir, 0777) < 0 && errno != EEXIST) {
		sys_warn("can't create user directory %s", user_dir);
		return;
	}
}

/* Read and apply saved user preferences
 */
void load_preferences(void)
{
	FILE*  f;
	char*  path;
	char*  line;
	char*  key;
	char*  val;
	char*  equal_pos;
	int32  component;

	config.collection = dsprintf("%s/patterns/%s/",
	                             DATADIR,
		                           pattern_collections[DEFAULT_COLLECTION].dir);

	path = dsprintf("%s/preferences", user_dir);
	f = fopen(path, "r");
	free(path);
	if (!f)
		return;

	while ((line = dgets(f)) != NULL) {
		equal_pos = strchr(line, '=');
		if (!equal_pos) {
		    warn("Invalid line '%s' found in preferences file, ignoring", line);
		    free(line);
		    continue;
		}
		*equal_pos = '\0';
		key = line;
		val = equal_pos+1;

		if (strstr(key, "collection")) {
		    free(config.collection);
		    config.collection = safe_strdup(val);
		} else if (strstr(key, "window_width"))
		    config.window_width = atoi(val);
		else if (strstr(key, "window_height"))
		    config.window_height = atoi(val);
		else if (STR_EQUAL(key, "zoom"))
		    config.zoom = atoi(val);
		else if (STR_STARTS_WITH(key, "show_") &&
		         (component = get_component_by_short_name(key + strlen("show_"))) >= 0)
		    config.visible_components[component] = atoi(val);
		else if (STR_EQUAL(key, "fullscreen"))
		    config.fullscreen = atoi(val);
		else if (STR_STARTS_WITH(key, "color_") && is_numeric(key+6) && atoi(key+6) < NUM_COLORS)
		    config.colors[atoi(key+6)] = strtoul(val, NULL, 16);
		else if (STR_EQUAL(key, "speed"))
		    config.speed = atoi(val);
		else if (STR_EQUAL(key, "speed_max"))
		    config.speed_max = atoi(val);
		else if (STR_EQUAL(key, "skip_frames"))
		    config.skip_frames = atoi(val);
		else
		    warn("Invalid key '%s' found in preferences file, ignoring", key);

		free(line);
	}

	fclose(f);
}

/* Apply default values to various globals, based on preferences and/or system defaults.
 */
void state_from_config(void)
{
	char*  default_dir;

	if (!validate_and_set_collection_dir(config.collection, FALSE)) {
		state.collection_status = COLL_BAD;
		default_dir = dsprintf("%s/patterns/%s/", DATADIR,
		                       pattern_collections[DEFAULT_COLLECTION].dir);
		if (!validate_and_set_collection_dir(default_dir, FALSE)) {
		    state.collection_status = COLL_DEFAULT_BAD;
		    set_collection_dir(default_dir, FALSE);
		}
		free(default_dir);
	}

	state.speed       = config.speed;
	state.skip_frames = config.skip_frames;
	dstate.zoom       = config.zoom;

	memcpy(dstate.visible_components, config.visible_components,
		   NUM_COMPONENTS * sizeof(boolean));
	
	dstate.fullscreen = config.fullscreen;

	state.selection = state.copy_rect = state.paste_box = null_rect;
	state.hover_point = state.last_drawn = null_point;
}

void config_from_state(void)
{
	// strdup to be more robust?
	config.collection = state.current_collection;

	if (!dstate.fullscreen) {
		gtk_window_get_size (GTK_WINDOW (gui.window),
			                     &config.window_width,
			                     &config.window_height);
	}

	config.zoom = dstate.zoom;
	
	memcpy(config.visible_components, dstate.visible_components,
		NUM_COMPONENTS * sizeof(boolean));

	config.fullscreen = dstate.fullscreen;
	config.speed = state.speed;
	config.skip_frames = state.skip_frames;

	/*
	ONLY STORED IN CONFIG ANYWAY:
		uint32           colors[NUM_COLORS];
	*/
}

/* Read the recent file list to put in the "File" menu
 */
void load_recent_files_list(void)
{
	FILE*  f;
	char*  rf_path;
	char*  full_path;
	char*  filename;
	char*  slash_pos;
	int32  i;

	memset(state.recent_files, 0, sizeof(recent_file) * MAX_RECENT_FILES);
	rf_path = dsprintf("%s/recent", user_dir);
	f = fopen(rf_path, "r");
	free(rf_path);
	if (!f)
		return;
	i = 0;
	while (i < MAX_RECENT_FILES && (full_path = dgets(f)) != NULL) {
		slash_pos = strrchr(full_path, '/');
		filename = safe_strdup(slash_pos ? slash_pos+1 : full_path);
		state.recent_files[i].full_path = full_path;
		state.recent_files[i].filename  = filename;
		i++;
	}
	fclose(f);
}

/* Save current preferences to a file in the user directory.
 */
void save_preferences(void)
{
	FILE*  f;
	char*  path;
	int32  i;

	path = dsprintf("%s/preferences", user_dir);
	f = fopen(path, "w");
	free(path);
	if (!f) {
		sys_warn("can't open %s/preferences for writing", user_dir);
		return;
	}

	fprintf(f, "collection=%s\n",       config.collection);
	fprintf(f, "window_width=%d\n",     config.window_width);
	fprintf(f, "window_height=%d\n",    config.window_height);
	fprintf(f, "zoom=%d\n",             config.zoom);
	for (i=0; i < NUM_COMPONENTS; i++)
		fprintf(f, "show_%s=%d\n", component_short_names[i],
		        config.visible_components[i]);
	fprintf(f, "fullscreen=%d\n",       config.fullscreen);
	for (i=0; i < NUM_COLORS; i++)
		fprintf(f, "color_%d=%06X\n", i, config.colors[i]);
	fprintf(f, "speed=%d\n",            config.speed);
	fprintf(f, "speed_max=%d\n",            config.speed_max);
	fprintf(f, "skip_frames=%d\n",      config.skip_frames);

	if (fclose(f) != 0)
		sys_warn("close failed after writing to ~/%s/preferences", USER_DIR);
}


/* Save the list of recently accessed pattern files
 */
void save_recent_files_list(void)
{
	FILE*  f;
	char*  path;
	int32  i;

	path = dsprintf("%s/recent", user_dir);
	f = fopen(path, "w");
	free(path);
	if (!f) {
		sys_warn("can't open %s/recent for writing", user_dir);
		return;
	}
	for (i=0; i < MAX_RECENT_FILES && state.recent_files[i].full_path; i++)
		fprintf(f, "%s\n", state.recent_files[i].full_path);
	if (fclose(f) != 0)
		sys_warn("close failed after writing to ~/%s/recent", USER_DIR);
}

/*** Bound Functions ***/

void file_new(void)
{
	if (state.pattern_running)
		start_stop();
	free(state.pattern_path);
	state.pattern_path = NULL;
	gtk_window_set_title(GTK_WINDOW(gui.window), TITLE);
	GtkAction *action = gtk_action_group_get_action (gui.action_group, "Revert");
	gtk_action_set_sensitive(action, FALSE);
	state.last_drawn = null_point;
	deactivate_selection(FALSE);
	deactivate_paste(FALSE);
	g_message ("file_new: >> clear_world");
	clear_world();
	g_message ("file_new: << clear_world");
	update_tick_label();
	update_population_label();
	g_message ("file_new: >> update_description_textbox");
	update_description_textbox(FALSE);
	g_message ("file_new: << update_description_textbox");
	sidebar_unselect();
	view_recenter();    /* will trigger a canvas redraw */
}

void file_open(void)
{
	GtkWidget *chooser;

	chooser = gtk_file_chooser_dialog_new (
		"Open Pattern",
		GTK_WINDOW (gui.window),
		GTK_FILE_CHOOSER_ACTION_OPEN,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
		NULL);

	gtk_dialog_set_default_response(
		GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);

	GtkFileFilter *filter;
	char *extension;
	int i;
	for (i=0; i < NUM_FORMATS; i++) {
		filter = gtk_file_filter_new ();
		extension = g_strconcat("*.", default_file_extensions[i], NULL);
		gtk_file_filter_add_pattern (filter, extension);
		g_free (extension);
		gtk_file_filter_set_name (filter, file_format_names[i]);
		gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (chooser), filter);
	}
	filter = gtk_file_filter_new ();
	gtk_file_filter_add_pattern (filter, "*");
	gtk_file_filter_set_name (filter, "All files");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (chooser), filter);
	
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), state.current_dir);

	if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT) {
		const gchar*       path;

		path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (chooser));
		if (attempt_load_pattern(path)) {
		    sidebar_unselect();
		    set_current_dir_from_file(state.pattern_path);
		}
	}

	gtk_widget_destroy (chooser);
}

void file_reopen(void)
{
	attempt_load_pattern(state.pattern_path);
}

void file_save(void)
{
	if (state.pattern_path)
		attempt_save_pattern(state.pattern_path, state.file_format);
	else
		file_save_as();
}

void file_save_as(void)
{
	GtkWidget*  chooser;
	int32       i;

	GList*      format_list = NULL;
	GtkWidget*  format_combo;
	GtkWidget*  hbox;
	char *basename;

	chooser = gtk_file_chooser_dialog_new (
		"Save Pattern",
		GTK_WINDOW (gui.window),
		GTK_FILE_CHOOSER_ACTION_SAVE,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
		NULL);

	gtk_dialog_set_default_response(
		GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);

	if (state.pattern_path) {
		basename = g_path_get_basename(state.pattern_path);

		gtk_file_chooser_set_filename(
			GTK_FILE_CHOOSER(chooser), state.pattern_path);
		gtk_file_chooser_set_current_name(
			GTK_FILE_CHOOSER(chooser), basename);

		g_free(basename);
	}

	/* Create and add the format-selection combo box */
	format_combo = gtk_combo_new();
	gtk_object_set_data(GTK_OBJECT(chooser), "format", format_combo);
	for (i=0; i < NUM_FORMATS; i++)
		format_list = g_list_append(format_list, file_format_names[i]);
	gtk_combo_set_popdown_strings(GTK_COMBO(format_combo), format_list);
	g_list_free(format_list);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("File Format:"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), format_combo, FALSE, FALSE, 0);

	gtk_widget_show_all (hbox);

	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (chooser), hbox);

	if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT) {
		struct stat statbuf;
		const char* path;
		int format;
		const char *format_str;

		path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(chooser));

	 	format_str = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(format_combo)->entry));
		for (i=0; i < NUM_FORMATS; i++) {
			if (STR_EQUAL(format_str, file_format_names[i]))
		    break;
		}
		format = ((i < NUM_FORMATS) ? i : FORMAT_GLF);

		basename = g_path_get_basename (path);
		if (strchr (basename, '.') == NULL)
		{
			char *strtmp;



			//cast is to discard constness to shut up compiler
			strtmp = (char*)path;
	    path = g_strconcat (path, ".", default_file_extensions[format], NULL);
	    g_message ("path = \"%s\"", path);
		  g_free (strtmp);
		}
		g_free (basename);

		if (stat(path, &statbuf) == 0) {  // file exists
			char *message;
			GtkWidget *dialog;
			const gchar *path;

			dialog = gtk_dialog_new_with_buttons (
				"File Exists",
				GTK_WINDOW (chooser),
				GTK_DIALOG_MODAL,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				"_Overwrite", GTK_RESPONSE_ACCEPT,
				NULL);

			path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
			message = dsprintf("<b><big>Overwrite file?</big></b>\n\nThe file '%s'"
				" exists, do you wish to overwrite it?", path);
			GtkWidget *label = gtk_label_new (message);
			// TODO: add big question icon
			gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
			gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
			gtk_container_set_border_width (GTK_CONTAINER (dialog), 12);
			gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
			//gtk_label_set_markup (GTK_LABEL (label), message);
			//gtk_label_set_text (GTK_LABEL (label), message);
			gtk_widget_show (label);
			free(message);
			gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);

			if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
				set_current_dir_from_file(path);
				if (attempt_save_pattern(path, format))
					sidebar_unselect();
			}

			gtk_widget_destroy (dialog);
		} else {
			set_current_dir_from_file(path);
			if (attempt_save_pattern(path, format))
				sidebar_unselect();
		}
	}

	gtk_widget_destroy (chooser);
}

void file_description(void)
{
	GtkWidget*  dialog;
	GtkWidget*  hbox;
	GtkWidget*  textview;
	GtkTextBuffer*  textbuffer;
	GtkTextTagTable*  texttags;
	GtkWidget*  scrollwin;

	if (gui.description_dialog)
		dialog = gui.description_dialog;
	else {
		/* Setup the dialog window */
		dialog = gui.description_dialog =
			gtk_dialog_new_with_buttons (
				_("Pattern Description"),
				GTK_WINDOW (gui.window),
				GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CLOSE,
				GTK_RESPONSE_ACCEPT,
				NULL);


		/* Set up the textbox with scrollbars */
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), 12);

		scrollwin=gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_shadow_type (
			GTK_SCROLLED_WINDOW (scrollwin),
			GTK_SHADOW_IN);
		gtk_scrolled_window_set_policy (
			GTK_SCROLLED_WINDOW (scrollwin),
			GTK_POLICY_AUTOMATIC,
			GTK_POLICY_ALWAYS);

		texttags=gtk_text_tag_table_new();
		textbuffer = gui.description_textbuffer = gtk_text_buffer_new(texttags);

		g_signal_connect (textbuffer,
		                  "end-user-action",
		                  G_CALLBACK (file_description_changed),
		                  NULL);

		/* Add the text */
		update_description_textbox(TRUE);

		textview = gtk_text_view_new_with_buffer(textbuffer);
		gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), TRUE);
		gtk_container_add(GTK_CONTAINER(scrollwin),textview);
		gtk_box_pack_start(GTK_BOX(hbox), scrollwin, TRUE, TRUE, 0);
		gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);
	}

	gtk_widget_show_all (dialog);
	/* FIXME: is this naughty in the case of clicking the WM close button? */
	g_signal_connect (
		dialog, "response", G_CALLBACK (gtk_widget_hide), (gpointer) dialog);
}

void file_description_changed(GtkTextBuffer *buffer, gpointer user_data)
{
	char*  desc;
	GtkTextIter start,end;

	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	desc = gtk_text_buffer_get_text(
		GTK_TEXT_BUFFER(buffer),
		&start,
		&end,
		TRUE);
	set_description(desc);
	free(desc);
}

void file_change_collection(void)
{
	GtkWidget *dialog;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *radio;
	GtkWidget *label;
	GtkWidget *radio_buttons[NUM_COLLECTIONS + 1];
	GtkWidget *chooserbutton;
	boolean  found_collection = FALSE;
	char*    buf;
	int32    i;

	dialog = gtk_dialog_new_with_buttons (
		_("Choose Pattern Collection"),
		GTK_WINDOW (gui.window),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_REJECT,
		GTK_STOCK_OK,
		GTK_RESPONSE_ACCEPT,
		NULL);
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	label = gtk_label_new(_("Collection:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	for (i=0; i < NUM_COLLECTIONS; i++) {
		hbox = gtk_hbox_new(FALSE, 3);
		gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("     "), FALSE, FALSE, 0);
		if (i == 0)
		    radio = gtk_radio_button_new_with_label(NULL, pattern_collections[i].title);
		else
		    radio = gtk_radio_button_new_with_label_from_widget(
		        GTK_RADIO_BUTTON(radio_buttons[0]), pattern_collections[i].title);
		if (!found_collection) {
		    buf = dsprintf("%s/patterns/%s/", DATADIR, pattern_collections[i].dir);
		    if (STR_EQUAL(state.current_collection, buf)) {
		        found_collection = TRUE;
		        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);
		    }
		    free(buf);
		}
		gtk_box_pack_start(GTK_BOX(hbox), radio, FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
		radio_buttons[i] = radio;
	}

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("     "), FALSE, FALSE, 0);
	radio = gtk_radio_button_new_with_mnemonic_from_widget(
		GTK_RADIO_BUTTON(radio_buttons[0]), _("C_ustom:"));
	if (!found_collection)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);


	gtk_box_pack_start(GTK_BOX(hbox), radio, FALSE, FALSE, 0);
	radio_buttons[i] = radio;

	chooserbutton = gtk_file_chooser_button_new (
		_("Choose custom pattern collection directory"),
		GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooserbutton), state.current_collection);
	gtk_widget_set_sensitive (
		chooserbutton,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)));

	// Update chooserbutton sensitivity on radio toggle
	g_signal_connect(radio,
	                 "toggled",
		               G_CALLBACK(file_change_coll_toggle_custom),
		               chooserbutton);

	gtk_box_pack_start (GTK_BOX (hbox), chooserbutton, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);


	gtk_widget_show_all (dialog);
	gint result = gtk_dialog_run (GTK_DIALOG (dialog));
	if (result == GTK_RESPONSE_ACCEPT) {
		char*    dir;
		int succeeded;
		for (i=0; i < NUM_COLLECTIONS; i++) {
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio_buttons[i])))
			    break;
		}
		if (i < NUM_COLLECTIONS)
			dir = dsprintf("%s/patterns/%s", DATADIR, pattern_collections[i].dir);
		else {
			dir = safe_strdup(gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooserbutton)));
			g_message ("Setting collection dir=%s", dir);
		}

		succeeded = validate_and_set_collection_dir(dir, TRUE);
		if (!succeeded)
			error_dialog (_("Invalid pattern directory: \"%s\""), dir);
		free(dir);
		if (dstate.sub_sidebar_visible) {
		    dstate.sub_sidebar_visible = FALSE;
		    if (dstate.visible_components[COMPONENT_SIDEBAR])
		        gtk_widget_hide_all(gui.sub_sidebar);
		}
		sidebar_unselect();
		force_sidebar_resize();
	}

	gtk_widget_destroy (dialog);
}

void file_recent(GtkMenuItem* menu_item, gpointer user_data)
{
	int32  num;

	num = *((int32*)user_data);
	if (state.recent_files[num].full_path) {
		if (attempt_load_pattern(state.recent_files[num].full_path))
		    sidebar_unselect();
	}
}

void file_quit(void)
{
	config_from_state();
	save_preferences();
	save_recent_files_list();

	exit(0);
}

void view_zoom_in(void)
{
	if (dstate.zoom < MAX_ZOOM)
		view_zoom (dstate.zoom * 2);
}

void view_zoom_out(void)
{
	if (dstate.zoom > MIN_ZOOM)
		view_zoom (dstate.zoom / 2);
}

void view_recenter(void)
{
	point  pt;

	pt.x = pt.y = WORLD_SIZE/2;
	set_viewport_position(&pt, TRUE);
	adjust_scrollbar_values();
	full_canvas_redraw();
}

void view_find_active_cells(void)
{
	point  pt;

	if (!find_active_cell(&pt.x, &pt.y)) {
		error_dialog("The grid is empty!");
		return;
	}
	set_viewport_position(&pt, TRUE);
	adjust_scrollbar_values();
}

void view_showtoolbar_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	boolean visible = gtk_toggle_action_get_active (toggleaction);
	view_show (COMPONENT_TOOLBAR, visible);
}

void view_showsidebar_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	boolean visible = gtk_toggle_action_get_active (toggleaction);
	view_show (COMPONENT_SIDEBAR, visible);
}

void view_showstatusbar_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	boolean visible = gtk_toggle_action_get_active (toggleaction);
	view_show (COMPONENT_STATUSBAR, visible);
}

void view_showscrollbars_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	boolean visible = gtk_toggle_action_get_active (toggleaction);
	view_show (COMPONENT_SCROLLBARS, visible);
}

void view_fullscreen_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	if (dstate.fullscreen == gtk_toggle_action_get_active (toggleaction))
		return;

	dstate.fullscreen = gtk_toggle_action_get_active (toggleaction);

	if (dstate.fullscreen) {
		gtk_widget_hide_all(gui.menubar);
		gtk_widget_hide_all(gui.toolbar);
		gtk_widget_hide_all(gui.sidebar);
		gtk_widget_hide(gui.vscrollbar);
		gtk_widget_hide(gui.hscrollbar);
		set_status_message(FULLSCREEN_MESSAGE, FALSE);
	} else {
		gtk_widget_show_all(gui.menubar);
		if (dstate.visible_components[COMPONENT_TOOLBAR])
		    gtk_widget_show_all(gui.toolbar);
		if (dstate.visible_components[COMPONENT_SIDEBAR]) {
		    gtk_widget_show_all(gui.sidebar);
		    if (!dstate.sub_sidebar_visible)
		        gtk_widget_hide(gui.sub_sidebar);
		} if (dstate.visible_components[COMPONENT_SCROLLBARS]) {
		    gtk_widget_show(gui.vscrollbar);
		    gtk_widget_show(gui.hscrollbar);
		}
		if (state.tracking_mouse == PASTING)
		    set_status_message((state.moving ? MOVING_MESSAGE : PASTING_MESSAGE), FALSE);
		else
		    set_status_message("", FALSE);
	}

	ewmh_toggle_fullscreen(gui.window);
}


void view_zoom_changed(GtkRadioAction *action, GtkRadioAction *current, gpointer user_data)
{
	view_zoom (gtk_radio_action_get_current_value (current));
}

void position_scroll_left(void)
{
	position_scroll_generic(-1, 0, SCROLL_STEP);
}

void position_scroll_right(void)
{
	position_scroll_generic(1, 0, SCROLL_STEP);
}

void position_scroll_up(void)
{
	position_scroll_generic(0, -1, SCROLL_STEP);
}

void position_scroll_down(void)
{
	position_scroll_generic(0, 1, SCROLL_STEP);
}

void position_scroll_nw(void)
{
	position_scroll_generic(-1, -1, SCROLL_STEP);
}

void position_scroll_ne(void)
{
	position_scroll_generic(1, -1, SCROLL_STEP);
}

void position_scroll_sw(void)
{
	position_scroll_generic(-1, 1, SCROLL_STEP);
}

void position_scroll_se(void)
{
	position_scroll_generic(1, 1, SCROLL_STEP);
}

void position_page_left(void)
{
	position_scroll_generic(-1, 0, SCROLL_PAGE);
}

void position_page_right(void)
{
	position_scroll_generic(1, 0, SCROLL_PAGE);
}

void position_page_up(void)
{
	position_scroll_generic(0, -1, SCROLL_PAGE);
}

void position_page_down(void)
{
	position_scroll_generic(0, 1, SCROLL_PAGE);
}

void position_page_nw(void)
{
	position_scroll_generic(-1, -1, SCROLL_PAGE);
}

void position_page_ne(void)
{
	position_scroll_generic(1, -1, SCROLL_PAGE);
}

void position_page_sw(void)
{
	position_scroll_generic(-1, 1, SCROLL_PAGE);
}

void position_page_se(void)
{
	position_scroll_generic(1, 1, SCROLL_PAGE);
}

void edit_cut(void)
{
	selection_copy_clear(TRUE, TRUE);
	deactivate_selection(FALSE);
	full_canvas_redraw();
	update_population_label();
}

void edit_copy(void)
{
	selection_copy_clear(TRUE, FALSE);
	deactivate_selection(TRUE);
}

void edit_clear(void)
{
	selection_copy_clear(FALSE, TRUE);
	deactivate_selection(FALSE);
	full_canvas_redraw();
	update_population_label();
}

void edit_paste(void)
{
	point  pt;
	rect   r;

	if (state.tracking_mouse)
		return;

	/* FIXME: this behaviour is unexpected: the if the user pastes with a selection 
	   the existing contents of the clipboard are erased.  Facilitates UNIX-style 
	   copying, but is different from the behaviour in most applications */
	if (SELECTION_ACTIVE())
		selection_copy_clear(TRUE, FALSE);
	state.tracking_mouse = PASTING;
	set_action_sensitive ("CancelPaste", TRUE);
	set_status_message((state.moving ? MOVING_MESSAGE : PASTING_MESSAGE), FALSE);
	gtk_widget_get_pointer(gui.canvas, &pt.x, &pt.y);
	get_logical_coords(&pt, &r.start);
	r.end.x = r.start.x + (state.copy_rect.end.x - state.copy_rect.start.x);
	r.end.y = r.start.y + (state.copy_rect.end.y - state.copy_rect.start.y);
	if (r.end.x < WORLD_SIZE && r.end.y < WORLD_SIZE)
		screen_box_update(&state.paste_box, &r, TRUE);
}

void edit_move(void)
{
	if (state.tracking_mouse)
		return;

	state.moving = TRUE;
	edit_paste();
}

void edit_cancel_paste(void)
{
	deactivate_paste(TRUE);
}

void edit_preferences(void)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *align;
	GtkWidget *table;
	GtkWidget *colorbutton;
	GdkColor color;
	GtkWidget *toggle;
	GtkWidget *hbox;
	GtkObject *adj;
	GtkWidget *spin;
	guint i;

	dialog = gtk_dialog_new_with_buttons (
		_("Preferences"),
		GTK_WINDOW (gui.window),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE,
		GTK_RESPONSE_ACCEPT,
		NULL);

	char* color_ui_names[NUM_COLORS] = {
		"Background:",
		"Live cells:",
		"Grid lines:",
		"Selection box:"};

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Colors</b>"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.0f);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), label);

	align = gtk_alignment_new (0.0f, 0.0f, 1.0f, 1.0f);
	gtk_alignment_set_padding (GTK_ALIGNMENT (align), 0.0f, 12.0f, 12.0f, 0.0f);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), align);

	table = gtk_table_new (NUM_COLORS + 1, 2, FALSE);
	gtk_container_add (GTK_CONTAINER (align), table);
	
	for (i = 0; i < NUM_COLORS; ++i) {
		label = gtk_label_new (color_ui_names[i]);
		gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.0f);
		gtk_table_attach (
			GTK_TABLE (table),
			label,
			0, 1,
			i, i + 1,
			GTK_FILL | GTK_EXPAND, GTK_SHRINK,
			6.0f, 3.0f);
		
		colorbutton = gtk_color_button_new();
		gtk_table_attach (
			GTK_TABLE (table),
			colorbutton,
			1, 2,
			i, i + 1,
			GTK_SHRINK, GTK_SHRINK,
			6.0f, 3.0f);

		color.red = (config.colors[i] >> 16) << 8;
		color.green = ((config.colors[i] - (color.red << 8)) >> 8) << 8;
		color.blue = (config.colors[i] - (color.red << 8) - (color.green)) << 8;

		gtk_color_button_set_color (GTK_COLOR_BUTTON (colorbutton), &color);
		g_signal_connect (colorbutton, "color-set", G_CALLBACK (prefs_color_cb), (gpointer)i);
	}

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Speed</b>"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.0f);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), label);	

	align = gtk_alignment_new (0.0f, 0.0f, 1.0f, 1.0f);
	gtk_alignment_set_padding (GTK_ALIGNMENT (align), 6.0f, 3.0f, 12.0f, 0.0f);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), align);
	
	toggle = gtk_check_button_new_with_label (_("Skip frames to achieve speed"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), state.skip_frames);
	gtk_container_add (GTK_CONTAINER (align), toggle);
	g_signal_connect (toggle, "toggled", G_CALLBACK (prefs_skip_cb), NULL);

	align = gtk_alignment_new (0.0f, 0.0f, 1.0f, 1.0f);
	gtk_alignment_set_padding (GTK_ALIGNMENT (align), 3.0f, 12.0f, 12.0f, 0.0f);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), align);
	
	hbox = gtk_hbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (align), hbox);
	
	label = gtk_label_new (_("Maximum speed (Gen/s):"));
	gtk_box_pack_start_defaults (GTK_BOX (hbox), label);	
	
	adj = gtk_adjustment_new (config.speed_max, 1, 999999, 10, 100, 10);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
	g_signal_connect (adj, "value-changed", G_CALLBACK (prefs_speed_max_cb), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), spin);

	gtk_widget_show_all (dialog);
	gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 12.0f);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void run_step(void)
{
	if (!state.pattern_running)
		tick_and_update();
}

void run_jump(void)
{
	GtkWidget*  dialog;
	GtkWidget*  hbox;
	GtkWidget*  jumpspin;
	GtkObject *jumpadj;

	/* Setup the dialog window */
	dialog = gtk_dialog_new_with_buttons (
		_("Jump"),
		GTK_WINDOW (gui.window),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_REJECT,
		"Jump",
		GTK_RESPONSE_ACCEPT,
		NULL);
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 12);

	jumpadj = gtk_adjustment_new ((double)tick, (double)tick, 999999.0, 10.0, 1000.0, 10.f);
	jumpspin = gtk_spin_button_new (GTK_ADJUSTMENT (jumpadj), 1.0f, 0);
	gtk_entry_set_activates_default (GTK_ENTRY (jumpspin), TRUE);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (jumpspin), TRUE);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Jump to tick:")), FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), jumpspin, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 5);

	/* Show the dialog */
	gtk_widget_show_all(dialog);
	gint result = gtk_dialog_run (GTK_DIALOG (dialog));
	if (result == GTK_RESPONSE_ACCEPT) {
		gtk_widget_hide (dialog);
		run_jump_execute (
			(int)gtk_adjustment_get_value (GTK_ADJUSTMENT (jumpadj)));
	}
	gtk_widget_destroy (dialog);
}

void run_faster(void)
{
	set_speed(state.speed + SPEED_INCREMENT);
}

void run_slower(void)
{
	set_speed(state.speed - SPEED_INCREMENT);
}

void run_speed(void)
{
	GtkWidget *dialog = gtk_dialog_new_with_buttons (
		_("Adjust Speed"),
		GTK_WINDOW (gui.window),
		GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE,
		GTK_RESPONSE_ACCEPT,
		NULL);
	
	GtkWidget *vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	
	GtkAdjustment *adj = gtk_range_get_adjustment (GTK_RANGE (gui.speed_slider));
	GtkWidget *slider = gtk_hscale_new (adj);
	gtk_scale_set_digits (GTK_SCALE (slider), 0);	
	gtk_box_pack_start (GTK_BOX (vbox), slider, FALSE, FALSE, 0);
	
	GtkWidget *check = gtk_check_button_new_with_mnemonic (
		_("_Skip frames to achieve speed"));
	g_signal_connect (check, "toggled", G_CALLBACK (prefs_skip_cb), NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), state.skip_frames);
	gtk_box_pack_start (GTK_BOX (vbox), check, FALSE, FALSE, 0);		
	
	GtkWidget *label = gtk_label_new ("");
	gtk_label_set_markup (
		GTK_LABEL (label),
		_("<i>The speed set here is a target: the actual speed is limited "
		"by the power of your computer.  Skipping frames will allow a greater "
		"speed to be achieved at the expense of graphical smoothness.</i>"));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_FILL);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	/* FIXME: is this naughty in the case of clicking the WM close button? */
	g_signal_connect (
		dialog, "response", G_CALLBACK (gtk_widget_destroy), (gpointer) dialog);
		
	gtk_widget_show_all (dialog);
}

void help_help(void)
{
	help_view_page("index.html");
}

void help_pattern_archive(void)
{
	help_view_page("patterns.html");
}

void help_glf_file_format(void)
{
	help_view_page("glf_format.html");
}

void help_about(void)
{
	GtkWidget*  dialog;

	/* Set up the dialog window */
	dialog = gtk_about_dialog_new();

	gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (dialog), TITLE);
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (dialog), VERSION);
	gtk_about_dialog_set_license (GTK_ABOUT_DIALOG (dialog), "GNU General Public License");
	gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (dialog), "Copyright © 2005 John Spray, Suzanne Britton\n" TITLE " is a derivative of GtkLife, by Suzanne Britton");

	gtk_about_dialog_set_url_hook (activate_url, NULL, NULL);
	gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (dialog), "http://icculus.org/~jcspray/LucidLife/");

	/* Add the banner */
	GdkPixbuf* pixbuf;

	pixbuf = gdk_pixbuf_new_from_file (DATADIR "/graphics/logo.png", NULL);

	if (pixbuf)
		gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (dialog), pixbuf);

	/* Show the dialog */
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void activate_url(GtkAboutDialog *about, const gchar *url, gpointer data)
{
	gnome_vfs_url_show (url);
}

/*** Bound Function Helpers ***/


/* Change to the given zoom level. */
void view_zoom(int32 new_zoom)
{
	point  center;
	GtkAction *action = NULL;

	if (new_zoom == dstate.zoom)
		return;

	/* Set the new zoom level */
	dstate.zoom = new_zoom;

	/* Determine the original viewport center point */
	center.x = dstate.viewport.start.x + dstate.viewport_size.width/2;
	center.y = dstate.viewport.start.y + dstate.viewport_size.height/2;

	/* Record new viewport dimensions at the current zoom level */
	dstate.viewport_size.width  = dstate.viewport_sizes[new_zoom].width;
	dstate.viewport_size.height = dstate.viewport_sizes[new_zoom].height;

	/* Determine the effective canvas size for the new zoom level */
	set_effective_canvas_size();

	/* Set new viewport start to center around the same point as before */
	set_viewport_position(&center, TRUE);

	/* Adjust scrollbars for the new viewport size and position */
	adjust_scrollbars();

	/* Draw the pixmap and generate a canvas expose */
	full_canvas_redraw();

	/* Set whether "zoom in" and "zoom out" menu items are grayed */
	set_zoom_sensitivities();

	switch (new_zoom) {
		case 1:
		action = gtk_action_group_get_action (gui.action_group, "Zoom1:1");
		break;
		case 2:
		action = gtk_action_group_get_action (gui.action_group, "Zoom2:1");
		break;
		case 4:
		action = gtk_action_group_get_action (gui.action_group, "Zoom4:1");
		break;
		case 8:
		action = gtk_action_group_get_action (gui.action_group, "Zoom8:1");
		break;
		case 16:
		action = gtk_action_group_get_action (gui.action_group, "Zoom16:1");
		break;
	}
	if (action)
		gtk_action_activate (action);
}

/* Toggle display of the given main window component. */
void view_show(component_type component, boolean is_visible)
{
	GtkWidget*  widget1;
	GtkWidget*  widget2;

	if (is_visible == dstate.visible_components[component])
		return;

	dstate.visible_components[component] = is_visible;

	/*widget1 = widget2 = NULL;
	if (component == COMPONENT_TOOLBAR)
		widget1 = gui.toolbar;
	else if (component == COMPONENT_SIDEBAR)
		widget1 = gui.sidebar;
	else if (component == COMPONENT_STATUSBAR)
		widget1 = gui.statusbar;
	else if (component == COMPONENT_SCROLLBARS) {
		widget1 = gui.hscrollbar;
		widget2 = gui.vscrollbar;
	}*/

	get_component_widgets (component, &widget1, &widget2);

	if (is_visible) {
		gtk_widget_show_all(widget1);
		if (widget2)
		    gtk_widget_show_all(widget2);
		if (component == COMPONENT_SIDEBAR && !dstate.sub_sidebar_visible)
		    gtk_widget_hide_all(gui.sub_sidebar);
	} else {
		gtk_widget_hide_all(widget1);
		if (widget2)
		    gtk_widget_hide_all(widget2);
	}
}

/* A helper for the various scroll functions. xdir indicates how to scroll along the x axis (-1 for
 * left, 0 for none, 1 for right), ydir indicates the y axis scrolling, and how_far indicates how
 * much to scroll (this parameter may take the special values SCROLL_STEP and SCROLL_PAGE, to
 * use the scrollbar step and page increments). This function just adjusts the scrollbar(s), and
 * lets the scrollbar event handlers take care of the rest.
 */
void position_scroll_generic(int32 xdir, int32 ydir, int32 how_far)
{
	GtkWidget*      scrollbar;
	GtkAdjustment*  sb_state;
	int32           dir;
	int32           i;

	for (i=0; i < 2; i++) {
		if (i == 0) {
		    dir = xdir;
		    scrollbar = gui.hscrollbar;
		} else {
		    dir = ydir;
		    scrollbar = gui.vscrollbar;
		}
		if (!dir)
		    continue;
		sb_state = gtk_range_get_adjustment(GTK_RANGE(scrollbar));
		if (how_far == SCROLL_STEP)
		    how_far = sb_state->step_increment;
		else if (how_far == SCROLL_PAGE)
		    how_far = sb_state->page_increment;
		gtk_adjustment_set_value(sb_state, ROUND(sb_state->value) + dir * how_far);
	}
}

/* Called when the user chooses where they want to paste after typing "ctrl-v".
 */
void edit_paste_win_style(const point* pt)
{
	if (!edit_paste_verify_target(pt))
		return;
	if (state.moving)
		selection_copy_clear(FALSE, TRUE);
	edit_paste_perform(pt);
	deactivate_selection(FALSE);
	deactivate_paste(FALSE);
	full_canvas_redraw();
	update_population_label();
}

/* Called when the user middle-clicks on a cell to paste.
 */
void edit_paste_unix_style(const point* pt)
{
	if (state.tracking_mouse)
		return;

	if (SELECTION_ACTIVE())
		selection_copy_clear(TRUE, FALSE);
	else if (rects_identical(&state.copy_rect, &null_rect))
		return;
	if (!edit_paste_verify_target(pt))
		return;
	edit_paste_perform(pt);
	//deactivate_selection(FALSE);
	full_canvas_redraw();
	update_population_label();
}

/* Verify that the current copy buffer can be pasted to the given point (i.e., it won't go off
 * the end of the world). Display an error dialog and return FALSE if not, otherwise return TRUE.
 */
boolean edit_paste_verify_target(const point* pt)
{
	if (pt->x + (state.copy_rect.end.x - state.copy_rect.start.x) < WORLD_SIZE &&
		pt->y + (state.copy_rect.end.y - state.copy_rect.start.y) < WORLD_SIZE)
		return TRUE;
	else {
		error_dialog("There's not enough room to paste there.");
		return FALSE;
	}
}

/* Paste the current copy buffer at the given location.
 */
void edit_paste_perform(const point* pt)
{
	int32       xstart, ystart;
	int32       xoff, yoff;
	cage_type*  c;

	for (c=state.copy_buffer; c; c=c->next) {
		xstart = pt->x + (c->x * CAGE_SIZE) - state.copy_rect.start.x;
		ystart = pt->y + (c->y * CAGE_SIZE) - state.copy_rect.start.y;
		for (yoff=0; yoff < 4; yoff++) {
		    for (xoff=0; xoff < 4; xoff++) {
		        if (c->bnw[0] & (1 << (yoff*4 + xoff)))
		            draw_cell(xstart+xoff, ystart+yoff, DRAW_SET);
		    }
		}
		for (yoff=0; yoff < 4; yoff++) {
		    for (xoff=0; xoff < 4; xoff++) {
		        if (c->bne[0] & (1 << (yoff*4 + xoff)))
		            draw_cell(xstart+4+xoff, ystart+yoff, DRAW_SET);
		    }
		}
		for (yoff=0; yoff < 4; yoff++) {
		    for (xoff=0; xoff < 4; xoff++) {
		        if (c->bsw[0] & (1 << (yoff*4 + xoff)))
		            draw_cell(xstart+xoff, ystart+4+yoff, DRAW_SET);
		    }
		}
		for (yoff=0; yoff < 4; yoff++) {
		    for (xoff=0; xoff < 4; xoff++) {
		        if (c->bse[0] & (1 << (yoff*4 + xoff)))
		            draw_cell(xstart+4+xoff, ystart+4+yoff, DRAW_SET);
		    }
		}
	}
}

/* View the given file in the web browser. */
void help_view_page(const char* filename)
{
	char*  url;

	url = dsprintf("file://%s/%s", DOCDIR, filename);
	gnome_vfs_url_show(url);
	free(url);
}

/*** GUI initialization functions ***/

/* Setup the window and widgets
 */
void init_gui(void)
{
	GtkWidget*  vbox;
	GtkWidget*  hbox;
	GtkWidget*  widget1;
	GtkWidget*  widget2;

	int32       i;

	init_rgb();
	init_window();

	init_stock_icons();

	/* Create the global GtkActionGroup */
	gui.action_group = gtk_action_group_new ("MenuActions");
	/* Add simple actions */
	gtk_action_group_add_actions (gui.action_group, ui_entries,
		G_N_ELEMENTS (ui_entries), gui.window);
	/* Add toggle actions */
	gtk_action_group_add_toggle_actions (gui.action_group, ui_toggle_entries,
		G_N_ELEMENTS (ui_toggle_entries), gui.window);
	/* Add zoom radio group */
	gtk_action_group_add_radio_actions (gui.action_group, ui_zoom_radios,
		G_N_ELEMENTS (ui_zoom_radios), 0, G_CALLBACK (view_zoom_changed), gui.window);
	/* Add tool selection radio group */
	gtk_action_group_add_radio_actions (gui.action_group, ui_tool_radios,
		G_N_ELEMENTS (ui_tool_radios), 0, G_CALLBACK (edit_tool_changed), gui.window);

	/* Create the global GtkUIManager */
	gui.ui_manager = gtk_ui_manager_new ();
	/* Associate it with the global GtkActionGroup */
	gtk_ui_manager_insert_action_group (gui.ui_manager, gui.action_group, 0);

	/* Associate the GtkUIManager's accelerator group with the main window */
	GtkAccelGroup *accel_group = gtk_ui_manager_get_accel_group (gui.ui_manager);
	gtk_window_add_accel_group (GTK_WINDOW (gui.window), accel_group);

	/* Load the UI string for menus, toolbars and accelerators, checking for errors */
	GError *error = NULL;
	if (!gtk_ui_manager_add_ui_from_string (
		gui.ui_manager, ui_description, -1, &error)) {
			g_message ("building menus failed: %s", error->message);
			g_error_free (error);
			exit (EXIT_FAILURE);
	}

	vbox = gtk_vbox_new(FALSE, 0);
	init_menubar(vbox);
	init_toolbar(vbox);
	init_sensitivities();
	
	//GtkWidget *paned = gtk_hpaned_new();
	
	hbox = gtk_hbox_new(FALSE, 0);
	init_sidebar(hbox);
	/*gtk_paned_add1 (GTK_PANED (paned), hbox);
	hbox = gtk_hbox_new(FALSE, 0);*/
	init_canvas(hbox);
	//gtk_paned_add2 (GTK_PANED (paned), hbox);
	
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	//gtk_box_pack_start(GTK_BOX(vbox), paned, TRUE, TRUE, 0);

	init_statusbar(vbox);
	gtk_container_add(GTK_CONTAINER(gui.window), vbox);
	gtk_widget_show_all(gui.window);

	/* Hide any GUI components the user doesn't want to see */
	for (i=0; i < NUM_COMPONENTS; i++) {
		if (!dstate.visible_components[i]) {
		    get_component_widgets(i, &widget1, &widget2);
		    gtk_widget_hide_all(widget1);
		    if (widget2)
		        gtk_widget_hide_all(widget2);
		}
	}

	/* Hide the sub-patterns sidebar initially */
	if (dstate.visible_components[COMPONENT_SIDEBAR])
		gtk_widget_hide_all(gui.sub_sidebar);

	/* Hide unused recent-file entries. If there are no entries, hide the separator too. */
	for (i=0; i < MAX_RECENT_FILES; i++) {
		if (!state.recent_files[i].filename)
		    gtk_widget_hide(state.recent_files[i].menu_item);
	}
	if (!state.recent_files[0].filename)
		gtk_widget_hide(gui.recent_files_separator);

	/* Put keyboard focus on the speed slider */
	gtk_widget_grab_focus(gui.speed_slider);

	/* Go to fullscreen if we're meant to be*/
	if (dstate.fullscreen) {
		// Crack to get visibility of components to update properly
		ewmh_toggle_fullscreen(gui.window);
		set_toggle_action_active ("Fullscreen", FALSE);
		set_toggle_action_active ("Fullscreen", TRUE);
	}
			
	/* Default to drawing tool, calling this to set up cursor */
	set_active_tool (TOOL_DRAW);
}

/* Initialize the GdkRGB subsystem for image handling. Also create the colormap we'll be using.
 */
void init_rgb(void)
{
	gdk_rgb_init();
	gtk_widget_set_default_colormap(gdk_rgb_get_cmap());
	gtk_widget_set_default_visual(gdk_rgb_get_visual());
	dstate.life_pixmap_colormap = gdk_rgb_cmap_new(config.colors, NUM_COLORS);
}


void init_stock_icons(void)
{
	GtkIconFactory *icon_factory;
	GtkIconSet *icon_set;
	GtkIconSource *icon_source;
	gint i;
	gchar *path;

	icon_factory = gtk_icon_factory_new ();

	const gint n_icons = sizeof (stock_icon_files) / sizeof (gchar*);

	for (i = 0; i < n_icons; ++i) {
		icon_set = gtk_icon_set_new ();
		icon_source = gtk_icon_source_new ();
		path = g_strconcat (DATADIR, "/icons/", stock_icon_files[i], NULL);
		gtk_icon_source_set_filename (icon_source, path);
		g_free (path);
		gtk_icon_set_add_source (icon_set, icon_source);
		gtk_icon_source_free (icon_source);
		gtk_icon_factory_add (icon_factory, stock_names[i], icon_set);
		gtk_icon_set_unref (icon_set);
	}

	gtk_icon_factory_add_default (icon_factory);

	g_object_unref (icon_factory);
}


/* Initialize the main application window
 */
void init_window(void)
{
	gui.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(gui.window), "destroy",
		               GTK_SIGNAL_FUNC(handle_main_window_destroy), NULL);
	gtk_widget_set_name(GTK_WIDGET(gui.window), PROG);
	gtk_window_set_title(GTK_WINDOW(gui.window), TITLE);
	gtk_window_set_default_size(GTK_WINDOW(gui.window), config.window_width,
		                        config.window_height);
	gtk_widget_realize(gui.window);
}

/* Setup the top menubar
 */
void init_menubar(GtkWidget* containing_box)
{
	gui.menubar = gtk_ui_manager_get_widget (gui.ui_manager, "/MainMenu");
	gtk_box_pack_start (GTK_BOX (containing_box), gui.menubar, FALSE, FALSE, 0);

	// Set up toggles from state
	set_toggle_action_active (
		"ShowToolbar", dstate.visible_components[COMPONENT_TOOLBAR]);
	set_toggle_action_active (
		"ShowSidebar", dstate.visible_components[COMPONENT_SIDEBAR]);
	set_toggle_action_active (
		"ShowStatusbar", dstate.visible_components[COMPONENT_STATUSBAR]);
	set_toggle_action_active (
		"ShowScrollbars", dstate.visible_components[COMPONENT_SCROLLBARS]);
	set_toggle_action_active (
		"Fullscreen", dstate.fullscreen);

	switch (dstate.zoom) {
		case 1:
			set_toggle_action_active ("Zoom1:1", TRUE);
			break;
		case 2:
			set_toggle_action_active ("Zoom2:1", TRUE);
			break;
		case 4:
			set_toggle_action_active ("Zoom4:1", TRUE);
			break;
		case 8:
			set_toggle_action_active ("Zoom8:1", TRUE);
			break;
		default:
			set_toggle_action_active ("Zoom16:1", TRUE);
			break;
	}


	/* Setup recent files list */
	GtkWidget *file_menu = NULL;
	int i;
	char *str;
	GtkWidget *item = NULL;
	int *num;
	
	file_menu = gtk_widget_get_parent (
		gtk_ui_manager_get_widget (
			gui.ui_manager, "/MainMenu/FileMenu/ItemBeforeRecentList"
			)
		);
	for (i=0; i < MAX_RECENT_FILES; i++) {
		if (state.recent_files[i].filename)
		    str = dsprintf("_%d. %s", i+1, state.recent_files[i].filename);
		else
		    str = dsprintf("_%d.", i+1);
		item = gtk_menu_item_new_with_mnemonic(str);
		free(str);
		state.recent_files[i].menu_item = item;
		state.recent_files[i].label     = GTK_BIN(item)->child;
		gtk_label_set_pattern(GTK_LABEL(state.recent_files[i].label), "_");
		gtk_widget_add_accelerator(
			item,
			"activate",
			gtk_ui_manager_get_accel_group (GTK_UI_MANAGER (gui.ui_manager)),
			(i+1) + 0x30,
			GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
		num = safe_malloc(sizeof(int32));
		*num = i;
		gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(file_recent), num);
		gtk_menu_shell_insert(GTK_MENU_SHELL(file_menu), item, RECENT_FILES_INSERT_POS+i);
	}

	/* Insert separator after recent files */
	gui.recent_files_separator = gtk_menu_item_new();
	gtk_menu_shell_insert(GTK_MENU_SHELL(file_menu), gui.recent_files_separator,
		                  RECENT_FILES_INSERT_POS+i);
	gtk_widget_show_all (file_menu);

}

/* Setup the toolbar, including speed slider to the right of the buttons. Place the toolbar button
 * widgets in gui.command_widgets[].toolbar_button.
 */
void init_toolbar(GtkWidget* containing_box)
{
	gui.toolbar = gtk_ui_manager_get_widget (gui.ui_manager, "/MainToolBar");
	gtk_toolbar_set_style(gui.toolbar, GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (containing_box), gui.toolbar, FALSE, FALSE, 0);

	/* Create the Gen/s slider */
	GtkToolItem *item = gtk_separator_tool_item_new();
	gtk_toolbar_insert (GTK_TOOLBAR (gui.toolbar), item, -1);

	GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
	init_speed_box(hbox);
	item = gtk_tool_item_new();
	gtk_tool_item_set_expand (item, TRUE);
	gtk_container_add (GTK_CONTAINER (item), hbox);
	gtk_toolbar_insert (GTK_TOOLBAR (gui.toolbar), item, -1);
}

/* Set whether various menu items and toolbar buttons are initially enabled.
 */
void init_sensitivities(void)
{
	set_action_sensitive ("Revert", FALSE);

	set_zoom_sensitivities();

	set_action_sensitive ("Cut", FALSE);
	set_action_sensitive ("Copy", FALSE);
	set_action_sensitive ("Clear", FALSE);
	set_action_sensitive ("Paste", FALSE);
	set_action_sensitive ("Move", FALSE);
	set_action_sensitive ("CancelPaste", FALSE);
}

/* Setup the speed slider box
 */
void init_speed_box(GtkWidget* containing_box)
{
	GtkWidget*  label;
	GtkObject*  slider_params;
	
	gtk_box_set_spacing (GTK_BOX (containing_box), 6);

	/* GtkAdjustment for speed setting */
	slider_params = gtk_adjustment_new(state.speed, SPEED_MIN,
		                               config.speed_max, 1, SPEED_INCREMENT, 0);
	gtk_signal_connect(GTK_OBJECT(slider_params), "value_changed",
		               GTK_SIGNAL_FUNC(handle_speed_changed), NULL);

	/* Speed: Label */
	label = gtk_label_new(_("Speed:"));
	gtk_box_pack_start(GTK_BOX(containing_box), label, FALSE, FALSE, 0);

	/* Slider */
	gui.speed_slider = gtk_hscale_new ( GTK_ADJUSTMENT(slider_params));
	gtk_scale_set_draw_value (GTK_SCALE (gui.speed_slider), TRUE);
	gtk_scale_set_digits (GTK_SCALE (gui.speed_slider), 0);
	gtk_box_pack_start(GTK_BOX(containing_box), gui.speed_slider, TRUE, TRUE, 0);

	/* FIXME: can't we get the tooltips object that 
	 * already exists for the toolbar? */
	gtk_tooltips_set_tip (
		gtk_tooltips_new (),
		gui.speed_slider,
		_("Speed of game in generations per second"),
		NULL);

	/* Units label */
	/*label = gtk_label_new(_("Gen/s"));
	gtk_box_pack_start(GTK_BOX(containing_box), label, FALSE, FALSE, 0);*/
}

/* Setup the pattern loader sidebar and sub-patterns sidebar
 */
void init_sidebar(GtkWidget* containing_box)
{
	GtkWidget**  sb;
	GtkWidget**  clist;
	char*        list_header;
	boolean      is_main_sidebar;
	int32        i;

	gui.sidebar = gtk_vbox_new(FALSE, 0);

	for (i=0; i < 2; i++) {
		if (i == 0) {
		    is_main_sidebar = TRUE;
		    sb = &gui.main_sidebar;
		    clist = &gui.patterns_clist;
		    list_header = "Patterns";
		} else {
		    is_main_sidebar = FALSE;
		    sb = &gui.sub_sidebar;
		    clist = &gui.sub_patterns_clist;
		    list_header = "[None]";
		}

		/* Create the list, its contents and its selection signal handler */
		*clist = gtk_clist_new_with_titles(1, &list_header);
		if (is_main_sidebar)
		    update_sidebar_generic(TRUE, state.current_collection);
		gtk_clist_set_selection_mode(GTK_CLIST(*clist), GTK_SELECTION_SINGLE);
		gtk_signal_connect(GTK_OBJECT(*clist), "select_row",
		                   GTK_SIGNAL_FUNC(handle_sidebar_select), (void*)is_main_sidebar);

		/* For the main sidebar, set the event handler for clicking the column header */
		if (is_main_sidebar) {
		    gtk_clist_column_titles_active(GTK_CLIST(*clist));
		    gtk_signal_connect(GTK_OBJECT(*clist), "click-column",
		                       GTK_SIGNAL_FUNC(handle_sidebar_header_click), NULL);
		} else
		    gtk_clist_column_titles_passive(GTK_CLIST(*clist));

		/* Give it scrollbars */
		*sb = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(*sb), GTK_POLICY_AUTOMATIC,
		                               GTK_POLICY_AUTOMATIC);
		gtk_container_add(GTK_CONTAINER(*sb), *clist);

		/* Pack it into the sidebar vbox */
		gtk_box_pack_start(GTK_BOX(gui.sidebar), *sb, TRUE, TRUE, 0);
	}

	gtk_box_pack_start(GTK_BOX(containing_box), gui.sidebar, FALSE, FALSE, 0);
}

/* Setup the Life pattern drawing area, including scrollbars
 */
void init_canvas(GtkWidget* containing_box)
{
	GtkWidget*  table;
	GtkObject*  scroll_params;

	table = gtk_table_new (2, 2, FALSE);

	/* Create and frame the canvas */
	gui.canvas = gtk_drawing_area_new();
	gtk_signal_connect(GTK_OBJECT(gui.canvas), "configure_event",
		               GTK_SIGNAL_FUNC(handle_canvas_resize), NULL);
	gtk_signal_connect(GTK_OBJECT(gui.canvas), "expose_event",
		               GTK_SIGNAL_FUNC(handle_canvas_expose), NULL);
	gtk_signal_connect(GTK_OBJECT(gui.canvas), "button_press_event",
		               GTK_SIGNAL_FUNC(handle_mouse_press), NULL);
	g_signal_connect (gui.canvas, "scroll-event", G_CALLBACK (handle_mouse_scroll), NULL);
	g_signal_connect (gui.canvas, "motion-notify-event", G_CALLBACK (handle_mouse_movement), NULL);
	gtk_signal_connect(GTK_OBJECT(gui.canvas), "button_release_event",
		               GTK_SIGNAL_FUNC(handle_mouse_release), NULL);
	gtk_widget_set_events(gui.canvas,
		GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK);

	gtk_table_attach (GTK_TABLE (table), gui.canvas, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

	/* Create a vertical scrollbar for the canvas. A number of these settings won't get properly
	 * set until handle_canvas_resize is called, so use dummy values for now.
	 */
	scroll_params = gtk_adjustment_new(INIT_Y_CENTER, 0, WORLD_SIZE-1, 1, 1, 0);
	gui.vscrollbar = gtk_vscrollbar_new(GTK_ADJUSTMENT(scroll_params));
	gtk_table_attach (GTK_TABLE (table), gui.vscrollbar, 1, 2, 0, 1, 0, GTK_EXPAND | GTK_FILL, 0, 0);

	/* Create and add a horizontal scrollbar for the canvas. */
	scroll_params = gtk_adjustment_new(INIT_X_CENTER, 0, WORLD_SIZE-1, 1, 1, 0);
	gui.hscrollbar = gtk_hscrollbar_new(GTK_ADJUSTMENT(scroll_params));
	gtk_table_attach (GTK_TABLE (table), gui.hscrollbar, 0, 1, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	/* Put the canvas+scrollbars into the box */
	gtk_box_pack_start(GTK_BOX(containing_box), table, TRUE, TRUE, 0);
}

/* Initialize the bottom statusbar, showing current cell, tick, and population.
 */
void init_statusbar(GtkWidget* containing_box)
{
	GtkWidget*  hbox;
	GtkWidget*  hbox2;
	GtkWidget*  hseparator;
	GtkWidget*  alignment;

	gui.statusbar = gtk_vbox_new(FALSE, 0);
	hseparator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(gui.statusbar), hseparator, FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 0);

	hbox2 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), gtk_label_new("  "), FALSE, FALSE, 0);
	gui.hover_point_label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox2), gui.hover_point_label, FALSE, FALSE, 0);
	LEFT_ALIGN(hbox2);
	gtk_box_pack_start(GTK_BOX(hbox), alignment, FALSE, FALSE, 0);

	gui.status_message_label = gtk_label_new("");
	CENTER_ALIGN(gui.status_message_label);
	gtk_box_pack_start(GTK_BOX(hbox), alignment, TRUE, TRUE, 0);

	hbox2 = gtk_hbox_new(FALSE, 0);
	init_tick_display(hbox2);
	init_population_display(hbox2);
	RIGHT_ALIGN(hbox2);
	gtk_box_pack_start(GTK_BOX(hbox), alignment, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(gui.statusbar), hbox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(containing_box), gui.statusbar, FALSE, FALSE, 0);
}

/* Set up the tick display
 */
void init_tick_display(GtkWidget* containing_box)
{
	gtk_box_pack_start(GTK_BOX(containing_box), gtk_label_new(" Tick:  "), FALSE, FALSE, 0);
	gui.tick_label = gtk_label_new("0");
	gtk_box_pack_start(GTK_BOX(containing_box), gui.tick_label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(containing_box), gtk_label_new("   "), FALSE, FALSE, 0);
}

/* Set up the population display
 */
void init_population_display(GtkWidget* containing_box)
{
	gtk_box_pack_start(GTK_BOX(containing_box), gtk_label_new("Population:  "), FALSE, FALSE, 0);
	gui.population_label = gtk_label_new("0");
	gtk_box_pack_start(GTK_BOX(containing_box), gui.population_label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(containing_box), gtk_label_new("  "), FALSE, FALSE, 0);
}

/*** Event Handlers ***/

/* Called when all or some of the Life display needs to be drawn onscreen. Redraw the exposed
 * rectangle from the offscreen pixmap.
 */
gboolean handle_canvas_expose(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{
	gdk_draw_indexed_image(gui.canvas->window,
		             gui.canvas->style->fg_gc[GTK_WIDGET_STATE(gui.canvas)],
		             event->area.x, event->area.y, event->area.width, event->area.height,
		             GDK_RGB_DITHER_NONE,
		             dstate.life_pixmap + event->area.y*dstate.canvas_size.width + event->area.x,
		             dstate.canvas_size.width, dstate.life_pixmap_colormap);
	return TRUE;
}

/* Called when the window dimensions are changed (including at startup): Set up various globals
 * dependent on window size, and recreate and redraw the canvas pixmap. Since an expose event will
 * already have been generated, we don't generate one here.
 */
gboolean handle_canvas_resize(GtkWidget* widget, GdkEventConfigure* event, gpointer user_data)
{
	point  center;
	int32  i;

	/* Get canvas dimensions (in pixels) */
	dstate.canvas_size.width  = gui.canvas->allocation.width;
	dstate.canvas_size.height = gui.canvas->allocation.height;

	/* Determine viewport dimensions (in cells) at all zoom levels for the new window size */
	for (i=MIN_ZOOM; i <= MAX_ZOOM; i *= 2) {
		if (DRAW_GRID(i)) {
		    dstate.viewport_sizes[i].width  = (dstate.canvas_size.width  - 1) / (i + 1);
		    dstate.viewport_sizes[i].height = (dstate.canvas_size.height - 1) / (i + 1);
		}
		else {
		    dstate.viewport_sizes[i].width  = dstate.canvas_size.width  / i;
		    dstate.viewport_sizes[i].height = dstate.canvas_size.height / i;
		}
	}

	/* Determine the original viewport center point. If the program has just started up, pick the
	 * middle of the world.
	 */
	if (!dstate.life_pixmap) {   /* program just started */
		center.x = INIT_X_CENTER;
		center.y = INIT_Y_CENTER;
	} else {
		center.x = dstate.viewport.start.x + dstate.viewport_size.width/2;
		center.y = dstate.viewport.start.y + dstate.viewport_size.height/2;
	}

	/* Record new viewport dimensions at the current zoom level */
	dstate.viewport_size.width  = dstate.viewport_sizes[dstate.zoom].width;
	dstate.viewport_size.height = dstate.viewport_sizes[dstate.zoom].height;

	/* Determine the new effective canvas size */
	set_effective_canvas_size();

	/* Set new viewport start to center around the same point as before */
	set_viewport_position(&center, TRUE);

	/* Adjust scrollbars for the new viewport size and position */
	adjust_scrollbars();

	/* Create and draw a new pixmap for the new viewport */
	if (dstate.canvas_size.width * dstate.canvas_size.height > dstate.life_pixmap_alloc_len) {
		dstate.life_pixmap_alloc_len = dstate.canvas_size.width * dstate.canvas_size.height;
		dstate.life_pixmap = safe_realloc(dstate.life_pixmap, dstate.life_pixmap_alloc_len);
	}
	draw_life_pixmap();

	return TRUE;
}

/* Called when a dialog is about to be mapped to the screen, in order to center it on the main
 * application window.
 */
void handle_child_window_realize(GtkWidget* child, gpointer user_data)
{
	int32  x, y, w, h;
	int32  cw, ch;

	gdk_window_get_root_origin(gui.window->window, &x, &y);
	gdk_window_get_geometry(gui.window->window, NULL, NULL, &w, &h, NULL);
	cw = child->allocation.width;
	ch = child->allocation.height;

	x += w/2 - cw/2;
	y += h/2 - ch/2;
	if (x < 0)
		x = 0;
	else if (x + cw >= gdk_screen_width())
		x = gdk_screen_width() - cw - 1;
	if (y < 0)
		y = 0;
	else if (y + ch >= gdk_screen_height())
		y = gdk_screen_height() - ch - 1;

	gtk_widget_set_uposition(child, x, y);
}

/* Called when the user closes the main application window.
 */
void handle_main_window_destroy(GtkObject* object, gpointer user_data)
{
	file_quit();
}

/* Handle mouse movement over the Life canvas.
 */
gboolean handle_mouse_movement (GtkWidget *canvas,
                                GdkEventMotion *event,
                                gpointer user_data)
{
	GdkModifierType  mouse_mask;
	point  pt, lpt;
	static point oldpt;
	rect   r;
	int32  scroll_increment;

	gdk_window_get_pointer(gui.canvas->window, &pt.x, &pt.y, &mouse_mask);

	if (pt.x >= 0 && pt.x < dstate.eff_canvas_size.width &&
		pt.y >= 0 && pt.y < dstate.eff_canvas_size.height)
		get_logical_coords(&pt, &lpt);
	else
		lpt = null_point;

	if (dstate.visible_components[COMPONENT_STATUSBAR] &&
		!points_identical(&lpt, &state.hover_point)) {
		state.hover_point = lpt;
		update_hover_point_label();
	}

	// FIXME: this is a bit off when zoomed in
	if (state.tracking_mouse == GRABBING) {
		GtkAdjustment *adj;
		adj = gtk_range_get_adjustment (GTK_RANGE (gui.vscrollbar));
		gtk_adjustment_set_value (adj, gtk_adjustment_get_value (adj)
			- (double)(pt.y - oldpt.y)/(double)dstate.zoom);
		adj = gtk_range_get_adjustment (GTK_RANGE (gui.hscrollbar));
		gtk_adjustment_set_value (adj, gtk_adjustment_get_value (adj)
			- (double)(pt.x - oldpt.x)/(double)dstate.zoom);
	}
	oldpt = pt;
	if (state.tracking_mouse == GRABBING)
		return TRUE;

	if (!state.tracking_mouse)
		return TRUE;

	if (state.tracking_mouse != PASTING && !(mouse_mask & GDK_BUTTON1_MASK)) {
		/* They released the mouse button while we weren't looking */
		state.tracking_mouse = NORMAL;
		state.last_drawn = null_point;
		return TRUE;
	}

	/* If they're dragging the mouse and are at or off the end of the canvas, scroll with them */
	if (state.tracking_mouse != PASTING) {
		scroll_increment = MAX_ZOOM / dstate.zoom;
		if (state.tracking_mouse == SELECTING)   /* scroll faster if selecting */
		    scroll_increment *= 2;
		if (pt.x <= 0)
		    position_scroll_generic(-1, 0, scroll_increment);
		else if (pt.x >= dstate.eff_canvas_size.width-1)
		    position_scroll_generic(1, 0, scroll_increment);
		if (pt.y <= 0)
		    position_scroll_generic(0, -1, scroll_increment);
		else if (pt.y >= dstate.eff_canvas_size.height-1)
		    position_scroll_generic(0, 1, scroll_increment);
		get_logical_coords(&pt, &lpt);
	}

	if (state.tracking_mouse == DRAWING) {
		if (points_identical(&state.last_drawn, &null_point))
		    user_draw(&lpt);
		else if (!points_identical(&lpt, &state.last_drawn))
		    draw_from_to(&state.last_drawn, &lpt);
	} else if (state.tracking_mouse == SELECTING) {
		r.start = state.selection.start;
		r.end   = lpt;
		screen_box_update(&state.selection, &r, TRUE);
	} else {  /* pasting */
		if (points_identical(&lpt, &null_point))
		    r = null_rect;
		else {
		    r.start = lpt;
		    r.end.x = r.start.x + (state.copy_rect.end.x - state.copy_rect.start.x);
		    r.end.y = r.start.y + (state.copy_rect.end.y - state.copy_rect.start.y);
		    if (r.end.x >= WORLD_SIZE || r.end.y >= WORLD_SIZE)
		        r = null_rect;
		}
		screen_box_update(&state.paste_box, &r, TRUE);
	}

	return TRUE;
}

/* Handle a mouse button click on the Life canvas.
 */
gboolean handle_mouse_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
	point  pt, lpt;

	pt.x = event->x;
	pt.y = event->y;
	get_logical_coords(&pt, &lpt);
	if (state.tracking_mouse == PASTING && event->button < 3)
		/* They've chosen where to paste a copied region */
		edit_paste_win_style(&lpt);
	else if (event->button == 1) {
		/* If they held the shift key down when clicking, start selecting a region. Otherwise
		 * start drawing. */
		if (event->state & GDK_SHIFT_MASK || state.active_tool == TOOL_SELECT)
			activate_selection(&lpt);
		else if (state.active_tool == TOOL_DRAW)
			user_draw(&lpt);
		else if (state.active_tool == TOOL_GRAB)
			state.tracking_mouse = GRABBING;
	} else if (event->button == 2) {
		/* They're pasting a region unix-style by clicking the middle mouse button */
		edit_paste_unix_style(&lpt);
	} else if (event->button == 3) {
		/* Recenter at this point */
		set_viewport_position(&lpt, TRUE);
		adjust_scrollbar_values();
	}

	return TRUE;
}

/* Handle a mouse button release on the Life canvas.
 */
gboolean handle_mouse_release(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
	if (event->button == 1 &&
		(state.tracking_mouse == DRAWING || state.tracking_mouse == SELECTING)) {
		/* If they just quickly shift-clicked the left mouse button, deactivate the current
		 * selection. */
		if (state.tracking_mouse == SELECTING &&
		    state.selection.start.x == state.selection.end.x &&
		    state.selection.start.y == state.selection.end.y &&
		    get_time_milliseconds() - state.selection_start_time < 250) {
		    deactivate_selection(TRUE);
		}
		state.tracking_mouse = NORMAL;
		state.last_drawn = null_point;
	}

	if (state.tracking_mouse == GRABBING)
		state.tracking_mouse = NORMAL;

	return TRUE;
}

gboolean handle_mouse_scroll(GtkWidget* widget, GdkEventScroll* event, gpointer user_data)
{
	int haveshift = (event->state & GDK_SHIFT_MASK);
	int havecontrol = (event->state & GDK_CONTROL_MASK);

	switch (event->direction) {
	case GDK_SCROLL_UP:
		if (havecontrol) {
			view_zoom_in();
		} else {
			if (haveshift)
				position_scroll_left();
			else
				position_scroll_up();
		}
		break;
	case GDK_SCROLL_DOWN:
		if (havecontrol) {
			view_zoom_out();
		} else {
			if (haveshift)
				position_scroll_right();
			else
				position_scroll_down();
		}
		break;
	case GDK_SCROLL_LEFT:
		position_scroll_left();
		break;
	case GDK_SCROLL_RIGHT:
		position_scroll_right();
		break;
	}

	return TRUE;
}

/* Called when the horizontal scrollbar is adjusted (by the user or by our program). Adjust the
 * viewport start, and do a full redraw.
 */
void handle_hscrollbar_change(GtkAdjustment* adjustment, gpointer user_data)
{
	point  pt;

	pt.x = ROUND(adjustment->value);
	pt.y = dstate.viewport.start.y;

	if (!(state.tracking_mouse == GRABBING)) {
		state.hover_point = pt;
		update_hover_point_label ();
	}
	
	set_viewport_position(&pt, FALSE);
	full_canvas_redraw();
}

/* Called when the vertical scrollbar is adjusted (by the user or by our program). Adjust the
 * viewport start, and do a full redraw.
 */
void handle_vscrollbar_change(GtkAdjustment* adjustment, gpointer user_data)
{
	point  pt;

	pt.x = dstate.viewport.start.x;
	pt.y = ROUND(adjustment->value);
	
	if (!(state.tracking_mouse == GRABBING)) {
		state.hover_point = pt;
		update_hover_point_label ();
	}
	
	set_viewport_position(&pt, FALSE);
	full_canvas_redraw();
}

/* Called when the user selects a pattern file from the sidebar or sub-sidebar. */
void handle_sidebar_select(GtkCList* clist, gint row, gint column, GdkEventButton* event,
		                   gpointer user_data)
{
	boolean        is_main_sidebar;
	pattern_file*  pattern_list;
	int32          num_patterns;

	is_main_sidebar = (boolean)user_data;
	if (is_main_sidebar) {
		pattern_list = state.sidebar_files;
		num_patterns = state.num_sidebar_files;
	} else {
		pattern_list = state.sub_sidebar_files;
		num_patterns = state.num_sub_sidebar_files;
	}

	if (row >= 0 && row < num_patterns) {
		if (pattern_list[row].is_directory)
		    update_sub_sidebar_contents(row);
		else {
		    if (is_main_sidebar && dstate.sub_sidebar_visible) {
		        /* If they selected a non-directory in the main sidebar, and the sub-sidebar
		         * is currently visible, hide it. */
		        gtk_widget_hide_all(gui.sub_sidebar);
		        dstate.sub_sidebar_visible = FALSE;
		        force_sidebar_resize();
		    }
		    attempt_load_pattern(pattern_list[row].path);
		}
	}
}

/* Called when the user clicks the header on the main patterns sidebar: initiate the menu command
 * File->Change Pattern Collection.
 */
void handle_sidebar_header_click(GtkCList* clist, gint column, gpointer user_data)
{
	file_change_collection();
}

/* Called when the speed slider has been adjusted (by the user or by our program). Set the new
 * speed, update the speed label, and reset the FPS measurement if the pattern is running.
 */
void handle_speed_changed(GtkAdjustment* adjustment, gpointer user_data)
{
	state.speed = ROUND(adjustment->value);
	reset_fps_measure();
}

/* Called when the user toggles the "Custom" pattern collection option in the
 * File->Change-Pattern-Collection dialog.
 */
void file_change_coll_toggle_custom(GtkToggleButton* toggle, gpointer user_data)
{
	boolean sensitive;

	GtkWidget *chooserbutton = (GtkWidget*)user_data;

	sensitive = gtk_toggle_button_get_active(toggle);
	gtk_widget_set_sensitive(chooserbutton,  sensitive);
}

/* Called when a different TOOL_ is selected 
 */
void edit_tool_changed(GtkRadioAction *action, GtkRadioAction *current, gpointer user_data)
{
	set_active_tool (gtk_radio_action_get_current_value (current));
}

/* Called when the checkbox for skipping frames in the preferences 
 * dialog is toggled
 */
void prefs_skip_cb (GtkWidget *widget, gpointer user_data)
{
	state.skip_frames =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
}

/* Called when the speed_max adjustment in the preferences dialog is changed
 */
void prefs_speed_max_cb (GtkAdjustment *adj, gpointer user_data)
{
	config.speed_max = gtk_adjustment_get_value (adj);
	
	GtkAdjustment *speedadj = gtk_range_get_adjustment (GTK_RANGE (gui.speed_slider));
	speedadj->upper = config.speed_max;
	gtk_adjustment_changed (speedadj);
	
	if (gtk_adjustment_get_value (speedadj) > config.speed_max)
		gtk_adjustment_set_value (speedadj, (double)config.speed_max);
}

/* Called when one of the color buttons in the preferences dialog is used
 */
void prefs_color_cb (GtkColorButton *button, gpointer user_data)
{
	GdkColor color;
	uint32 newcolor;
	
	guint i = (guint) user_data;
	gtk_color_button_get_color (button, &color);

	newcolor = ((color.red & 0xff00) << 8) 
	           + (color.green  & 0xff00)
	           + ((color.blue  & 0xff00) >> 8);

	if (config.colors[i] != newcolor) {
		config.colors[i] = newcolor;
		dstate.life_pixmap_colormap = gdk_rgb_cmap_new(config.colors, NUM_COLORS);
		state.skipped_frames = 0;
		gtk_widget_queue_draw(gui.canvas);
	}
}

/* Called when the user clicks "Okay" from the jump-to-tick dialog. */
void run_jump_execute(gint target_tick)
{
	GtkWidget*  jumping_dialog;
	GtkWidget*  table;
	GtkWidget*  tick_label;
	GtkWidget*  pop_label;
	GtkWidget*  label;
	GtkWidget*  hbox;
	GtkWidget*  alignment;
	char*       str;
	uint32      tick_offset;

	/* Determine the tick to jump to */
	if (target_tick < tick) {
		error_dialog("You can't jump backwards.");
		return;
	}

	/* Setup and display the modal progress dialog */
	jumping_dialog = gtk_dialog_new_with_buttons (
		"Jumping...",
		GTK_WINDOW (gui.window),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_CANCEL,
		NULL);

	gtk_window_set_default_size(GTK_WINDOW(jumping_dialog), 200, 0);
		
	g_signal_connect (jumping_dialog, "response", G_CALLBACK (run_jump_abort), NULL);

	table = gtk_table_new(2, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_table_set_row_spacings(GTK_TABLE(table), 5);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);

	label = gtk_label_new("Tick:");
	LEFT_ALIGN(label);
	gtk_table_attach(GTK_TABLE(table), alignment, 0, 1, 0, 1, TABLE_OPTS, TABLE_OPTS, 0, 0);
	hbox = gtk_hbox_new(FALSE, 0);
	tick_label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), tick_label, FALSE, FALSE, 0);
	str = dsprintf(" / %u", target_tick);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(str), FALSE, FALSE, 0);
	free(str);
	LEFT_ALIGN(hbox);
	gtk_table_attach(GTK_TABLE(table), alignment, 1, 2, 0, 1, TABLE_OPTS, TABLE_OPTS, 0, 0);

	label = gtk_label_new("Population:");
	LEFT_ALIGN(label);
	gtk_table_attach(GTK_TABLE(table), alignment, 0, 1, 1, 2, TABLE_OPTS, TABLE_OPTS, 0, 0);
	pop_label = gtk_label_new("");
	LEFT_ALIGN(pop_label);
	gtk_table_attach(GTK_TABLE(table), alignment, 1, 2, 1, 2, TABLE_OPTS, TABLE_OPTS, 0, 0);

	CENTER_ALIGN(table);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(jumping_dialog)->vbox), alignment, TRUE, TRUE, 0);
	gtk_widget_show_all(jumping_dialog);

	state.jump_cancelled = FALSE;
	while (gtk_events_pending())
		gtk_main_iteration_do(FALSE);

	/* Perform the jump */
	for (tick_offset = 0;
		 tick < target_tick && !state.jump_cancelled;
		 tick_offset++) {
		if (tick_offset % 100 == 0) {
		    str = dsprintf("%u", tick);
		    gtk_label_set_text(GTK_LABEL(tick_label), str);
		    free(str);
		    str = dsprintf("%u", population);
		    gtk_label_set_text(GTK_LABEL(pop_label), str);
		    free(str);
		}
		next_tick(FALSE);
		gtk_main_iteration_do(FALSE);
	}

	/* it's already gone if the user closed it with the WM */
	if (GTK_IS_WIDGET (jumping_dialog))
		gtk_widget_destroy(jumping_dialog);

	/* Jump complete: update the screen */
	full_canvas_redraw();
	update_tick_label();
	update_population_label();
	reset_fps_measure();
}

void run_play_cb(GtkToggleAction *toggleaction, gpointer user_data)
{
	int playing = gtk_toggle_action_get_active (toggleaction);
	if (playing != state.pattern_running) {
		if (state.pattern_running) {
			if (state.skipped_frames) {
			    trigger_canvas_update();
			    update_tick_label();
			    update_population_label();
			}
		} else if (!points_identical(&state.hover_point, &null_point)) {
			state.hover_point = null_point;
			update_hover_point_label();
		}

		start_stop();
	}
}

/* Called when the user halts a jump-in-progress by hitting the Cancel button or closing the
 * dialog.
 */
void run_jump_abort(GtkDialog* dialog, gint arg1, gpointer user_data)
{
	state.jump_cancelled = TRUE;
}


/*** Functions for drawing the offscreen pixmap ***/

/* Fully redraw the Life pixmap: backdrop, grid, selection box, and live cells.
 */
void draw_life_pixmap(void)
{
	cage_type*  c;
	int32       x, y;

	/* Draw the backdrop */
	memset(dstate.life_pixmap, BG_COLOR_INDEX, dstate.canvas_size.width*dstate.canvas_size.height);

	/* Draw the live cells */
	while ((c = loop_cages_onscreen()) != NULL) {
		x = c->x * CAGE_SIZE;
		y = c->y * CAGE_SIZE;
		if (c->bnw[parity])
		    draw_life_block(c->bnw[parity], x, y, TRUE);
		if (c->bne[parity])
		    draw_life_block(c->bne[parity], x + BLOCK_SIZE, y, TRUE);
		if (c->bsw[parity])
		    draw_life_block(c->bsw[parity], x, y + BLOCK_SIZE, TRUE);
		if (c->bse[parity])
		    draw_life_block(c->bse[parity], x + BLOCK_SIZE, y + BLOCK_SIZE, TRUE);
	}

	draw_grid_and_boxes();
}

/* Redraw just the zoom grid and boxes (selection, paste) on the life pixmap, if applicable.
 */
void draw_grid_and_boxes(void)
{
	int32  x, y;

	/* Draw the grid, if necessary */
	if (DRAW_GRID(dstate.zoom)) {
		/* Draw the horizontal lines */
		for (y=0; y < dstate.canvas_size.height; y += (dstate.zoom+1))
		    memset(dstate.life_pixmap + y*dstate.canvas_size.width, GRID_COLOR_INDEX,
		           dstate.eff_canvas_size.width);
		/* Draw the vertical lines */
		for (x=0; x < dstate.canvas_size.width; x += (dstate.zoom+1)) {
		    for (y=0; y < dstate.eff_canvas_size.height; y++)
		        dstate.life_pixmap[y*dstate.canvas_size.width + x] = GRID_COLOR_INDEX;
		}
	}

	/* Draw the selection box and/or paste-indicator box */
	draw_screen_box(&state.selection);
	draw_screen_box(&state.paste_box);
}

/* Draw the given box (selection or paste indicator) to the pixmap.
 */
void draw_screen_box(const rect* box)
{
	dimension  boxsize;
	rect       nbox;
	rect       r, cr;
	int32      y;

	if (box->start.x < 0)
		return;
	nbox = *box;
	normalize_rectangle(&nbox);
	if (nbox.start.x > dstate.viewport.end.x || nbox.start.y > dstate.viewport.end.y ||
		nbox.end.x < dstate.viewport.start.x || nbox.end.y < dstate.viewport.start.y) {
		return;
	}

	get_screen_rectangle(&nbox, &r);
	r.end.x += dstate.zoom-1;
	r.end.y += dstate.zoom-1;
	if (DRAW_GRID(dstate.zoom)) {
		r.start.x--;
		r.start.y--;
		r.end.x++;
		r.end.y++;
	}

	cr.start.x = MAX(0, r.start.x);
	cr.start.y = MAX(0, r.start.y);
	cr.end.x   = MIN(dstate.eff_canvas_size.width-1,  r.end.x);
	cr.end.y   = MIN(dstate.eff_canvas_size.height-1, r.end.y);
	get_rect_dimensions(&cr, &boxsize);

	if (r.start.y >= 0)
		memset(dstate.life_pixmap + r.start.y*dstate.canvas_size.width + cr.start.x,
		       SELECT_COLOR_INDEX, boxsize.width);
	if (r.end.y < dstate.eff_canvas_size.height)
		memset(dstate.life_pixmap + r.end.y*dstate.canvas_size.width + cr.start.x,
		       SELECT_COLOR_INDEX, boxsize.width);
	if (r.start.x >= 0) {
		for (y=cr.start.y; y <= cr.end.y; y++)
		    dstate.life_pixmap[y*dstate.canvas_size.width + r.start.x] = SELECT_COLOR_INDEX;
	}
	if (r.end.x < dstate.eff_canvas_size.width) {
		for (y=cr.start.y; y <= cr.end.y; y++)
		    dstate.life_pixmap[y*dstate.canvas_size.width + r.end.x] = SELECT_COLOR_INDEX;
	}
}

/* Draw the given 4x4 cell block onto the Life pixmap, at the given starting coordinates.
 * If full_update is TRUE, draw only live cells (because the screen has been cleared beforehand),
 * otherwise draw all cells. If full_update is FALSE, update the current clipping rectangle
 * (dstate.update.start.x, etc.) to include this block. This function may be called by the
 * backend.
 */
void draw_life_block(uint16 block, int32 xstart, int32 ystart, boolean full_update)
{
	int32     xend, yend;
	int32     minx, miny, maxx, maxy;
	int32     real_minx, real_miny, real_maxx, real_maxy;
	int32     realx, realy;
	int32     grid_block_size;
	int32     start_bit;
	int32     bit;
	int32     x, y, i;
	uint8     color;
	uint8*    spos;
	uint8*    pos;

	grid_block_size = (DRAW_GRID(dstate.zoom) ? dstate.zoom+1 : dstate.zoom);

	/* Figure out the screen offset, in blocks */
	xstart -= dstate.viewport.start.x;
	ystart -= dstate.viewport.start.y;
	xend = xstart + BLOCK_SIZE - 1;
	yend = ystart + BLOCK_SIZE - 1;
	minx = MAX(xstart, 0);
	miny = MAX(ystart, 0);
	maxx = MIN(xend, dstate.viewport_size.width-1);
	maxy = MIN(yend, dstate.viewport_size.height-1);
	if (minx > maxx || miny > maxy)
		return;

	/* Update the clipping rectangle, if necessary */
	if (!full_update) {
		real_minx = minx * grid_block_size;
		real_miny = miny * grid_block_size;
		real_maxx = maxx * grid_block_size + (dstate.zoom - 1);
		real_maxy = maxy * grid_block_size + (dstate.zoom - 1);
		if (DRAW_GRID(dstate.zoom)) {
		    real_minx++;
		    real_miny++;
		    real_maxx++;
		    real_maxy++;
		}
		if (real_minx < dstate.update.start.x)
		    dstate.update.start.x = real_minx;
		if (real_miny < dstate.update.start.y)
		    dstate.update.start.y = real_miny;
		if (real_maxx > dstate.update.end.x)
		    dstate.update.end.x = real_maxx;
		if (real_maxy > dstate.update.end.y)
		    dstate.update.end.y = real_maxy;
	}

	start_bit = (miny - ystart) * BLOCK_SIZE + (minx - xstart);
	/* Draw the cells */
	if (dstate.zoom > 1) {
		for (y=miny; y <= maxy; y++, start_bit += BLOCK_SIZE) {
		    for (x=minx, bit=start_bit; x <= maxx; x++, bit++) {
		        color = ((block & (1 << bit)) ? CELL_COLOR_INDEX : BG_COLOR_INDEX);
		        if (full_update && color == BG_COLOR_INDEX)
		            continue;
		        realx = x * grid_block_size;
		        realy = y * grid_block_size;
		        if (DRAW_GRID(dstate.zoom)) {
		            realx++;
		            realy++;
		        }
		        for (i=0, pos=dstate.life_pixmap + realy*dstate.canvas_size.width + realx;
		             i < dstate.zoom;
		             i++, pos += dstate.canvas_size.width)
		            memset(pos, color, dstate.zoom);
		    }
		}
	} else {
		spos = dstate.life_pixmap + miny * dstate.canvas_size.width + minx;
		for (y=miny; y <= maxy; y++, spos += dstate.canvas_size.width, start_bit += BLOCK_SIZE) {
		    for (x=minx, bit=start_bit, pos=spos; x <= maxx; x++, pos++, bit++) {
		        color = ((block & (1 << bit)) ? CELL_COLOR_INDEX : BG_COLOR_INDEX);
		        *pos = color;
		    }
		}
	}
}

/* Set/unset the cell at the given absolute (logical) coordinates, then update the Life state and
 * the screen. If the coordinate are outside of the viewport, adjust them. If the user just clicked
 * the mouse, check the color of the cell to determine whether to set or erase. If they've been
 * dragging the mouse, continue either setting or erasing.
 */
void user_draw(const point* pt)
{
	static uint8  color;
	point   npt;
	point   spt;
	uint8*  pos;
	uint8   cur_color;
	int32   i;

	npt = *pt;
	if (npt.x < dstate.viewport.start.x)
		npt.x = dstate.viewport.start.x;
	else if (npt.x > dstate.viewport.end.x)
		npt.x = dstate.viewport.end.x;
	if (npt.y < dstate.viewport.start.y)
		npt.y = dstate.viewport.start.y;
	else if (npt.y > dstate.viewport.end.y)
		npt.y = dstate.viewport.end.y;
	state.last_drawn = npt;
	get_screen_coords(&npt, &spt);

	cur_color = dstate.life_pixmap[spt.y*dstate.canvas_size.width + spt.x];
	if (state.tracking_mouse == DRAWING) {
		if (cur_color == color)
		    return;
	} else
		color = ((cur_color == CELL_COLOR_INDEX) ? BG_COLOR_INDEX : CELL_COLOR_INDEX);

	draw_cell(npt.x, npt.y, (color == BG_COLOR_INDEX ? DRAW_UNSET : DRAW_SET));
	if (dstate.zoom == 1)
		dstate.life_pixmap[spt.y*dstate.canvas_size.width + spt.x] = color;
	else {
		for (i=0, pos = dstate.life_pixmap + spt.y*dstate.canvas_size.width + spt.x;
		     i < dstate.zoom;
		     i++, pos += dstate.canvas_size.width)
		    memset(pos, color, dstate.zoom);
	}

	gtk_widget_queue_draw_area(gui.canvas, spt.x, spt.y, dstate.zoom, dstate.zoom);
	state.tracking_mouse = DRAWING;

	update_population_label();
}

/* Call user_draw repeatedly to draw/erase a line between the points start and end.
 */
void draw_from_to(const point* start, const point* end)
{
	point   s, e, pt;
	point   temp;
	double  slope, realx, realy;

	s = *start;
	e = *end;

	if (abs(e.x - s.x) > abs(e.y - s.y)) {
		if (s.x > e.x)
		    SWAP(s, e);
		slope = (double)(e.y - s.y) / (double)(e.x - s.x);
		pt = s;
		while (pt.x <= e.x) {
		    user_draw(&pt);
		    pt.x++;
		    realy = (double)s.y + slope * (double)(pt.x - s.x);
		    if (abs(pt.y - realy) >= 0.5)
		        (slope < 0) ? pt.y-- : pt.y++;
		}
	} else {
		if (s.y > e.y)
		    SWAP(s, e);
		slope = (double)(e.x - s.x) / (double)(e.y - s.y);
		pt = s;
		while (pt.y <= e.y) {
		    user_draw(&pt);
		    pt.y++;
		    realx = (double)s.x + slope * (double)(pt.y - s.y);
		    if (abs(pt.x - realx) >= 0.5)
		        (slope < 0) ? pt.x-- : pt.x++;
		}
	}

	state.last_drawn = *end;
}

/*** Misc. GUI functions ***/

/* Adjust horizontal and vertical scrollbars for the current viewport start and size.
 */
void adjust_scrollbars(void)
{
	GtkObject*  scroll_params;

	/* Necessary to say WORLD_SIZE instead of WORLD_SIZE-1 to get full range. Why? */

	scroll_params = gtk_adjustment_new(dstate.viewport.start.x, 0, WORLD_SIZE,
		                               dstate.viewport_size.width/8, dstate.viewport_size.width,
		                               dstate.viewport_size.width);
	gtk_range_set_adjustment(GTK_RANGE(gui.hscrollbar), GTK_ADJUSTMENT(scroll_params));
	gtk_signal_connect(GTK_OBJECT(scroll_params), "value_changed",
		               GTK_SIGNAL_FUNC(handle_hscrollbar_change), NULL);

	scroll_params = gtk_adjustment_new(dstate.viewport.start.y, 0, WORLD_SIZE,
		                               dstate.viewport_size.height/8, dstate.viewport_size.height,
		                               dstate.viewport_size.height);
	gtk_range_set_adjustment(GTK_RANGE(gui.vscrollbar), GTK_ADJUSTMENT(scroll_params));
	gtk_signal_connect(GTK_OBJECT(scroll_params), "value_changed",
		               GTK_SIGNAL_FUNC(handle_vscrollbar_change), NULL);
	gtk_widget_queue_draw(gui.hscrollbar);
	gtk_widget_queue_draw(gui.vscrollbar);
}

/* Adjust horizontal and vertical scrollbars for the current viewport start */
void adjust_scrollbar_values(void)
{
	GtkAdjustment*  sb_state;

	sb_state = gtk_range_get_adjustment(GTK_RANGE(gui.hscrollbar));
	gtk_adjustment_set_value(sb_state, dstate.viewport.start.x);
	sb_state = gtk_range_get_adjustment(GTK_RANGE(gui.vscrollbar));
	gtk_adjustment_set_value(sb_state, dstate.viewport.start.y);
}

/* Set up a signal handler so that, when mapped, the given window will be centered on the main
 * application window.
 */
void center_child_window(GtkWidget* child)
{
	gtk_signal_connect(GTK_OBJECT(child), "realize", GTK_SIGNAL_FUNC(handle_child_window_realize),
		               NULL);
}

/* Display an error/warning dialog with the given message and an "Okay" button. The message may be
 * specified in printf-style fashion, with format string (pango markup) and arguments.
 */
void error_dialog(const char* format, ...)
{
	va_list     args;
	GtkWidget*  dialog;
	char*       str;

	va_start(args, format);
	str = vdsprintf(format, args);
	va_end(args);
	
	dialog = gtk_message_dialog_new_with_markup (
		GTK_WINDOW (gui.window),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_MESSAGE_ERROR,
		GTK_BUTTONS_CLOSE,
		str);

	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

/* Force the sidebar to shrink down to minimum via a hide/show shuffle.
 */
void force_sidebar_resize(void)
{
	if (dstate.fullscreen || !dstate.visible_components[COMPONENT_SIDEBAR])
		return;

	if (dstate.sub_sidebar_visible) {
		gtk_widget_hide_all(gui.sidebar);
		gtk_widget_show_all(gui.sidebar);
	} else {
		gtk_widget_hide_all(gui.main_sidebar);
		gtk_widget_show_all(gui.main_sidebar);
		gtk_widget_hide(gui.sidebar);
		gtk_widget_show(gui.sidebar);
	}
}

/* Redraw the entire Life pixmap, then trigger an expose event on the canvas. Also set
 * state.skipped_frames back to 0.
 */
void full_canvas_redraw(void)
{
	state.skipped_frames = 0;
	draw_life_pixmap();
	gtk_widget_queue_draw(gui.canvas);
}

/* Put the current filename (taken from state.recent_files[0]) into the main window title */
void put_filename_in_window_title(void)
{
	char*  title;

	title = dsprintf("%s - %s", state.recent_files[0].filename, TITLE);
	gtk_window_set_title(GTK_WINDOW(gui.window), title);
	free(title);
}

/* Erase a temporary status message and restore the original one (if any).
 */
void restore_status_message(void)
{
	state.temp_message_active = FALSE;
	gtk_label_set_text(GTK_LABEL(gui.status_message_label), state.original_message);
	free(state.original_message);
	state.original_message = NULL;
}

void set_action_sensitive (gchar *action, gboolean sensitive)
{
	gtk_action_set_sensitive (
		gtk_action_group_get_action (gui.action_group, action),
		sensitive
	);
}

/* Set the active TOOL_ and update cursor/selection
 */
void set_active_tool (tool_id newtool)
{
	state.active_tool = newtool;

	if (state.active_tool != TOOL_SELECT)
		deactivate_selection (TRUE);
	
	GdkCursor *newcursor = NULL;
	if (state.active_tool == TOOL_GRAB) {
		newcursor = gdk_cursor_new (GDK_FLEUR);
	} else if (state.active_tool == TOOL_SELECT) {
		newcursor = gdk_cursor_new (GDK_CROSSHAIR);
	} else {
		newcursor = gdk_cursor_new (GDK_PENCIL);
	}

	gdk_window_set_cursor (gui.canvas->window, newcursor);
	gdk_cursor_unref (newcursor);	
}

void set_toggle_action_active (gchar *action, gboolean active)
{
	gtk_toggle_action_set_active (
		GTK_TOGGLE_ACTION (gtk_action_group_get_action (gui.action_group, action)),
		active
	);
}

/* Record the current effective canvas size (canvas size minus any unused space) */
void set_effective_canvas_size(void)
{
	if (DRAW_GRID(dstate.zoom)) {
		dstate.eff_canvas_size.width  = dstate.viewport_size.width  * (dstate.zoom+1) + 1;
		dstate.eff_canvas_size.height = dstate.viewport_size.height * (dstate.zoom+1) + 1;
	} else {
		dstate.eff_canvas_size.width  = dstate.viewport_size.width  * dstate.zoom;
		dstate.eff_canvas_size.height = dstate.viewport_size.height * dstate.zoom;
	}
}

/* Adjust horizontal widget size for the speed label, based on max speed */
void set_speed_label_size(int32 max_speed)
{
	char*  speed_str;

	speed_str = dsprintf("%d", max_speed);
	gtk_widget_set_usize(GTK_WIDGET(gui.speed_label),
		gdk_string_width(gtk_style_get_font(gtk_widget_get_style(gui.window))
			, speed_str) + 6, 0);
	free(speed_str);
}

/* Put a message in the statusbar. If is_temporary is TRUE, the message will only be shown for
 * TEMP_MESSAGE_INTERVAL seconds, then the original message will be restored. New temporary
 * messages permanently supplant old ones, while leaving the original non-temporary intact.
 */
void set_status_message(char* msg, boolean is_temporary)
{
	char*  org_msg;

	if (is_temporary) {
		if (!state.temp_message_active) {
		    gtk_label_get(GTK_LABEL(gui.status_message_label), &org_msg);
		    state.original_message = safe_strdup(org_msg);
		    state.temp_message_active = TRUE;
		}
		state.temp_message_start_time = get_time_milliseconds();
	} else {     /* dump any current temporary message */
		state.temp_message_active = FALSE;
		free(state.original_message);
		state.original_message = NULL;
	}

	gtk_label_set_text(GTK_LABEL(gui.status_message_label), msg);
}

/* Set the global variables for viewport location, and let the backend know about the new
 * viewport. If center_around is true, attempt to center around the given point, otherwise try to
 * set the given point at the upper left corner. If the viewport would go off the edge of the
 * world, readjust it.
 */
void set_viewport_position(const point* pt, boolean center_around)
{
	/* Set viewport start */
	dstate.viewport.start.x = pt->x;
	dstate.viewport.start.y = pt->y;
	if (center_around) {
		dstate.viewport.start.x -= dstate.viewport_size.width/2;
		dstate.viewport.start.y -= dstate.viewport_size.height/2;
	}

	/* Readjust viewport start if the current viewport goes off the edge of the world */
	if (dstate.viewport.start.x < 0)
		dstate.viewport.start.x = 0;
	else if (dstate.viewport.start.x + dstate.viewport_size.width > WORLD_SIZE)
		dstate.viewport.start.x = WORLD_SIZE - dstate.viewport_size.width;
	if (dstate.viewport.start.y < 0)
		dstate.viewport.start.y = 0;
	else if (dstate.viewport.start.y + dstate.viewport_size.height > WORLD_SIZE)
		dstate.viewport.start.y = WORLD_SIZE - dstate.viewport_size.height;

	/* Set viewport end */
	dstate.viewport.end.x = dstate.viewport.start.x + dstate.viewport_size.width  - 1;
	dstate.viewport.end.y = dstate.viewport.start.y + dstate.viewport_size.height - 1;

	/* Let the backend know about the new viewport */
	set_viewport(dstate.viewport.start.x, dstate.viewport.start.y, dstate.viewport.end.x,
		         dstate.viewport.end.y);
}

/* Set the sensitivity of zoom menu items and buttons based on current zoom level.
 */
void set_zoom_sensitivities(void)
{
	GtkAction *zoominaction = NULL;
	GtkAction *zoomoutaction = NULL;
	zoominaction = gtk_action_group_get_action (gui.action_group, "ZoomIn");
	zoomoutaction = gtk_action_group_get_action (gui.action_group, "ZoomOut");
	gtk_action_set_sensitive (zoominaction, (dstate.zoom < MAX_ZOOM));
	gtk_action_set_sensitive (zoomoutaction, (dstate.zoom > MIN_ZOOM));
}

/* Trigger an expose event on the portion of the Life canvas which has changed over the last tick,
 * setting state.skipped_frames back to 0.
 */
void trigger_canvas_update(void)
{
	dimension  size;

	state.skipped_frames = 0;
	get_rect_dimensions(&dstate.update, &size);
	if (size.width > 0 && size.height > 0)
		gtk_widget_queue_draw_area(gui.canvas, dstate.update.start.x, dstate.update.start.y,
		                           size.width, size.height);
}

/* Update the description box if the File->Description dialog is open. If first_time is TRUE, the
 * dialog has just been created.
 */
void update_description_textbox(boolean first_time)
{
	int32  desc_width = 0;
	int32  desc_height = 0 ;
	int32  w;
	int32  i;

	if (!gui.description_dialog)
		return;

	/* Determine the width and height of the current description text, and set default size of
	 * description dialog accordingly */

	for (i=0; i < desc_num_lines; i++) {
		w=gdk_string_width(gtk_style_get_font(gtk_widget_get_style(gui.window))
			,pattern_description[i])+80;
		if (w > desc_width)
		    desc_width = w;
	}

	if (desc_width == 0) {
		desc_width = 400;
		desc_height = 200;
	} else {
		if (desc_width < 300)
		    desc_width = 300;
		desc_height = MIN(400, 30 * desc_num_lines + 100);
	}
	gtk_window_resize (GTK_WINDOW (gui.description_dialog), desc_width, desc_height);

	/* Set the description text */
	gtk_text_buffer_set_text(gui.description_textbuffer, "", 0);

	for (i=0; i < desc_num_lines; i++) {
		gtk_text_buffer_insert_at_cursor (GTK_TEXT_BUFFER (gui.description_textbuffer),
		                pattern_description[i], -1);
		gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(gui.description_textbuffer),
											"\n", -1);
	}
}

/* Update the onscreen label indicating where the mouse is hovering. */
void update_hover_point_label(void)
{
	char*  hover_point_str;

	if (points_identical(&state.hover_point, &null_point))
		gtk_label_set_text(GTK_LABEL(gui.hover_point_label), "");
	else {
		hover_point_str = dsprintf("(%d, %d)", state.hover_point.x - WORLD_SIZE/2,
		                           state.hover_point.y - WORLD_SIZE/2);
		gtk_label_set_text(GTK_LABEL(gui.hover_point_label), hover_point_str);
		free(hover_point_str);
	}
}

/* Update the onscreen population label to match the current population. */
void update_population_label(void)
{
	char  str[20];

	sprintf(str, "%u", population);
	gtk_label_set_text(GTK_LABEL(gui.population_label), str);
}

/* Regenerate the list of patterns for the sidebar. */
void update_sidebar_contents(void)
{
	update_sidebar_generic(TRUE, state.current_collection);
}

/* Regenerate the list of patterns for the sub-sidebar, based on the given selected row in the
 * main sidebar. Also set the new sub-sidebar header. If the sub-sidebar is not currently
 * visible, show it.*/
void update_sub_sidebar_contents(int32 sidebar_selection)
{
	char*  header;

	update_sidebar_generic(FALSE, state.sidebar_files[sidebar_selection].path);
	header = safe_strdup(state.sidebar_files[sidebar_selection].title);
	if (header[strlen(header)-1] == '/')
		header[strlen(header)-1] = '\0';
	gtk_clist_set_column_title(GTK_CLIST(gui.sub_patterns_clist), 0, header);
	free(header);
	if (!dstate.sub_sidebar_visible) {
		gtk_widget_show_all(gui.sub_sidebar);
		dstate.sub_sidebar_visible = TRUE;
	}
}

/* Update either the main or sub-sidebar, loading patterns from the given directory. If updating
 * the sub-sidebar and it's currently hidden, unhide it.
 */
void update_sidebar_generic(boolean is_main_sidebar, const char* new_dir)
{
	GtkWidget*      sb;
	GtkWidget*      clist;
	pattern_file**  patterns;
	int32*          num_patterns;
	boolean         include_subdirs;
	int32           width_request;
	int32           i;

	if (is_main_sidebar) {
		sb              = gui.main_sidebar;
		clist           = gui.patterns_clist;
		patterns        = &state.sidebar_files;
		num_patterns    = &state.num_sidebar_files;
		include_subdirs = TRUE;
	} else {
		sb              = gui.sub_sidebar;
		clist           = gui.sub_patterns_clist;
		patterns        = &state.sub_sidebar_files;
		num_patterns    = &state.num_sub_sidebar_files;
		include_subdirs = FALSE;
	}

	/* Load the new file list */
	load_pattern_directory(new_dir, include_subdirs, patterns, num_patterns);

	/* Clear the list display */
	gtk_clist_freeze(GTK_CLIST(clist));
	gtk_clist_clear(GTK_CLIST(clist));

	/* Add the list items */
	for (i=0; i < *num_patterns; i++)
		gtk_clist_append(GTK_CLIST(clist), &((*patterns)[i].title));

	/* Set a reasonable list width request */
	width_request = gtk_clist_optimal_column_width(GTK_CLIST(clist), 0) + 30;
	gtk_widget_set_usize(clist, width_request, -1);

	/* Display the new list */
	gtk_clist_thaw(GTK_CLIST(clist));
}

/* Update the onscreen tick label to match the current tick. */
void update_tick_label(void)
{
	char  str[20];

	sprintf(str, "%u", tick);
	gtk_label_set_text(GTK_LABEL(gui.tick_label), str);
}

/* Create an integral, snap-to-ticks spinbutton with the given initial, minimum and maximum values.
 */
GtkWidget* create_spinbox(int32 init_value, int32 min, int32 max)
{
	GtkWidget*  spinbox;
	GtkObject*  adjustment;
	char*       max_str;

	adjustment = gtk_adjustment_new(init_value, min, max, 1, 1, 0);
	spinbox = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment), 1, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinbox), TRUE);
	gtk_spin_button_set_snap_to_ticks(GTK_SPIN_BUTTON(spinbox), TRUE);
	max_str = dsprintf("%d", max);
	/*TESTME*/
	gtk_widget_set_usize(spinbox,
		gdk_string_width(gtk_style_get_font(gtk_widget_get_style(gui.window))
			, max_str) + 25, 0);
	free(max_str);

	return spinbox;
}

/*** Miscellaneous Functions ***/

/* Add the current file to the top of the recent files list, and update the File menu accordingly.
 * If it is already in the list, move it to the top.
 */
void add_to_recent_files(void)
{
	char*    filename;
	char*    label_text;
	boolean  maxed_out = FALSE;
	int32    num_recent;
	int32    old_pos;
	int32    start;
	int32    i;

	/* Is it already at the top of recent files? */
	if (state.recent_files[0].full_path &&
		STR_EQUAL(state.pattern_path, state.recent_files[0].full_path))
		return;

	/* Determine filename from full_path */
	split_path(state.pattern_path, NULL, &filename);

	/* Determine the current # of recent files, and the position of this one if it's already in
	 * the list.
	 */
	for (i=0, old_pos=0; i < MAX_RECENT_FILES && state.recent_files[i].full_path; i++) {
		if (STR_EQUAL(state.recent_files[i].full_path, state.pattern_path))
		    old_pos = i;
	}
	num_recent = i;

	/* If the recent files list is at maximum and this is a new entry, drop off the last item */
	if (num_recent == MAX_RECENT_FILES && !old_pos) {
		maxed_out = TRUE;
		num_recent--;
	}

	/* Make room for the new item at the top of the list */
	start = (old_pos ? old_pos : num_recent);
	free(state.recent_files[start].full_path);
	free(state.recent_files[start].filename);
	for (i=start; i > 0; i--) {
		state.recent_files[i].full_path = state.recent_files[i-1].full_path;
		state.recent_files[i].filename  = state.recent_files[i-1].filename;
		/*gtk_tooltips_set_tip(gui.menu_tooltips, state.recent_files[i].menu_item,
		                     state.recent_files[i].full_path, NULL);*/
		label_text = dsprintf("%d. %s", i+1, state.recent_files[i].filename);
		gtk_label_set_text(GTK_LABEL(state.recent_files[i].label), label_text);
		free(label_text);
	}

	/* Put the new item at the top */
	state.recent_files[0].full_path = safe_strdup(state.pattern_path);
	state.recent_files[0].filename  = filename;
	/*gtk_tooltips_set_tip(gui.menu_tooltips, state.recent_files[0].menu_item,
		                 state.recent_files[0].full_path, NULL);*/
	label_text = dsprintf("1. %s", filename);
	gtk_label_set_text(GTK_LABEL(state.recent_files[i].label), label_text);
	free(label_text);

	/* If we've increased the length of the list, show the menu item which was hidden before */
	if (!maxed_out && !old_pos)
		gtk_widget_show(state.recent_files[num_recent].menu_item);
	/* If the list was empty before this, show the menu separator which was hidden before */
	if (!num_recent)
		gtk_widget_show(gui.recent_files_separator);
}

/* Attempt to load a pattern file from the given path (making it canonical first), displaying an
 * error dialog and returning FALSE on failure, otherwise returning TRUE. Extensions from
 * default_file_extensions[] will be tried if the original path doesn't exist. If the load was
 * successful, this function will stop any running pattern, update the tick label, and recenter and
 * redraw the canvas.
 */
boolean attempt_load_pattern(const char* path)
{
	char*             resolved_path;
	char*             final_path;
	load_result_type  load_result;
	struct stat       statbuf;

	resolved_path = get_canonical_path(path);
	if (!resolved_path) {
		error_dialog("Open failed: \"%s\" does not exist", path);
		return FALSE;
	}
	final_path = find_life_file(resolved_path);
	free(resolved_path);
	if (!final_path) {
		error_dialog("Open failed: \"%s\" does not exist", path);
		return FALSE;
	}
	if (stat(final_path, &statbuf) == 0 && S_ISDIR(statbuf.st_mode)) {
		error_dialog("Open failed: \"%s\" is a directory", path);
		free(final_path);
		return FALSE;
	}

	load_result = load_pattern(final_path, &state.file_format);
	if (load_result != LOAD_SUCCESS) {
		switch (load_result) {
		case LOAD_SYS_ERROR:
		    error_dialog("Open failed: %s", strerror(errno));
		    break;
		case LOAD_UNRECOGNIZED_FORMAT:
		    error_dialog("<b><big>Open failed</big></b>\n\nUnrecognized file format");
		    break;
		case LOAD_BAD_GLF_VERSION:
		    error_dialog("Open failed: invalid GLF version in file");
		    break;
		case LOAD_BAD_LIF_VERSION:
		    error_dialog("Open failed: invalid LIF version in file\n"
		                 "(neither 1.05 nor 1.06)");
		    break;
		default:   /* LOAD_STRUCTURED_XLIFE */
		    error_dialog("This appears to be a structured-format XLife file\n"
		                 "(containing includes and/or sub-blocks). This type\n"
		                 "of XLife file is not currently supported.");
		    break;
		}
		free(final_path);
		return FALSE;
	}

	if (!state.pattern_path || !STR_EQUAL(final_path, state.pattern_path)) {
		free(state.pattern_path);
		state.pattern_path = final_path;
		add_to_recent_files();
		put_filename_in_window_title();
	} else
		free(final_path);

	if (state.pattern_running)
		start_stop();
	state.last_drawn = null_point;
	deactivate_selection(FALSE);
	deactivate_paste(FALSE);
	update_tick_label();
	update_population_label();
	update_description_textbox(FALSE);
	GtkAction *action = gtk_action_group_get_action (gui.action_group, "Revert");
	gtk_action_set_sensitive (action, TRUE);
	view_recenter();   /* will trigger a canvas redraw */
	set_status_message(FILE_LOADED_MESSAGE, TRUE);

	return TRUE;
}

/* Attempt to save a pattern file to the given path (first canonicalizing it), displaying an error
 * dialog and returning FALSE on failure, otherwise returning TRUE.
 */
boolean attempt_save_pattern(const char* path, file_format_type format)
{
	save_result_type  save_result;
	char*  resolved_path;

	resolved_path = get_canonical_path(path);
	if (!resolved_path) {
		error_dialog("Save failed: invalid path \"%s\"", path);
		return FALSE;
	}

	save_result = save_pattern(resolved_path, format);
	if (save_result == SAVE_SUCCESS) {
		if (!state.pattern_path || !STR_EQUAL(resolved_path, state.pattern_path)) {
		    free(state.pattern_path);
		    state.pattern_path = resolved_path;
		    add_to_recent_files();
		    put_filename_in_window_title();
				GtkAction *action = gtk_action_group_get_action (gui.action_group, "Revert");
				gtk_action_set_sensitive(action, TRUE);
		} else
		    free(resolved_path);
		state.file_format = format;
		set_status_message(FILE_SAVED_MESSAGE, TRUE);
		return TRUE;
	}
	else {
		free(resolved_path);
		if (save_result == SAVE_SYS_ERROR)
		    error_dialog("Save failed: %s", strerror(errno));
		else if (format == FORMAT_RLE)
		    error_dialog("The description for an RLE-type file must be\n"
		                 "no more than %d characters wide. Please fix\n"
		                 "and re-save, or choose a different file format.",
		                 RLE_DESC_MAX_COLS);
		else
		    error_dialog("The description for a LIF-type file must be no\n"
		                 "more than %d lines long and %d characters wide.\n"
		                 "Please fix and re-save, or choose a different\n"
		                 "file format.",
		                 LIF_DESC_MAX_LINES, LIF_DESC_MAX_COLS);
		return FALSE;
	}
}

/* Begin a selection box starting at the given point, and redraw the canvas.
 */
void activate_selection(const point* pt)
{
	rect          r;

	state.tracking_mouse = SELECTING;
	state.selection_start_time = get_time_milliseconds();
	r.start = r.end = *pt;
	screen_box_update(&state.selection, &r, TRUE);
	set_action_sensitive ("Cut", TRUE);
	set_action_sensitive ("Copy", TRUE);
	set_action_sensitive ("Clear", TRUE);
	set_action_sensitive ("Move", TRUE);
	set_action_sensitive ("Paste", TRUE);
}

/* Clear all memory associated with the given pattern list, setting list to NULL and num_patterns
 * to 0.
 */
void clear_pattern_list(pattern_file** patterns, int32* num_patterns)
{
	int32  i;

	for (i=0; i < *num_patterns; i++) {
		free((*patterns)[i].filename);
		free((*patterns)[i].path);
		free((*patterns)[i].title);
	}
	free(*patterns);
	*patterns = NULL;
	*num_patterns = 0;
}

/* Deactivate any existing selection box. If redraw is TRUE, update the screen accordingly.
 */
void deactivate_selection(boolean redraw)
{
	if (rects_identical(&state.selection, &null_rect))
		return;

	if (state.tracking_mouse == SELECTING)
		state.tracking_mouse = NORMAL;
	screen_box_update(&state.selection, &null_rect, redraw);

	set_action_sensitive ("Cut", FALSE);
	set_action_sensitive ("Copy", FALSE);
	set_action_sensitive ("Clear", FALSE);
	set_action_sensitive ("Move", FALSE);
	set_action_sensitive ("Paste", COPY_BUFFER_ACTIVE());
}

/* If we are in paste-mode or move-mode, deactivate it. If redraw is TRUE, update the screen
 * accordingly.
 */
void deactivate_paste(boolean redraw)
{
	if (state.tracking_mouse != PASTING)
		return;

	state.tracking_mouse = NORMAL;
	state.moving = FALSE;
	screen_box_update(&state.paste_box, &null_rect, redraw);
	set_status_message((dstate.fullscreen ? FULLSCREEN_MESSAGE : ""), FALSE);
	set_action_sensitive ("CancelPaste", FALSE);
}

/* Attempt to locate a life file based on the given path. Try the path by itself, then try
 * appending the extensions in default_file_extensions[]. Return the first match found
 * (dynamically allocated), or NULL.
 */
char* find_life_file(const char* path)
{
	struct stat  statbuf;
	char*  newpath;
	int32  i;

	if (stat(path, &statbuf) == 0)
		return safe_strdup(path);
	for (i=0; default_file_extensions[i]; i++) {
		newpath = dsprintf("%s.%s", path, default_file_extensions[i]);
		if (stat(newpath, &statbuf) == 0)
		    return newpath;
		free(newpath);
	}

	return NULL;
}

/* Return the number that cooresponds to the given main window component. If there is no such
 * component, return -1.
 */
int32 get_component_by_short_name(const char* name)
{
	int32  i;

	for (i=0; i < NUM_COMPONENTS; i++) {
		if (STR_EQUAL(component_short_names[i], name))
		    return i;
	}
	return -1;
}

void get_component_widgets(component_type component, GtkWidget** widget1, GtkWidget** widget2)
{
	*widget1 = ((component == COMPONENT_TOOLBAR)    ? gui.toolbar :
		        (component == COMPONENT_SIDEBAR)    ? gui.sidebar :
		        (component == COMPONENT_SCROLLBARS) ? gui.hscrollbar :
		                                              gui.statusbar);
	*widget2 = ((component == COMPONENT_SCROLLBARS) ? gui.vscrollbar : NULL);
}

/* Determine the Life coordinates (between 0 and WORLD_SIZE-1) that correspond to the given screen
 * coordinates (relative to the canvas). If the given coordinates are out of bounds, use the
 * nearest point on the canvas.
 */
void get_logical_coords(const point* p, point* logical_p)
{
	point  pt;
	int32  grid_block_size;

	pt = *p;
	if (pt.x < 0)
		pt.x = 0;
	else if (pt.x >= dstate.eff_canvas_size.width)
		pt.x = dstate.eff_canvas_size.width - 1;
	if (pt.y < 0)
		pt.y = 0;
	else if (pt.y >= dstate.eff_canvas_size.height)
		pt.y = dstate.eff_canvas_size.height - 1;

	grid_block_size = (DRAW_GRID(dstate.zoom) ? dstate.zoom+1 : dstate.zoom);
	logical_p->x = MIN(dstate.viewport.end.x, dstate.viewport.start.x + pt.x / grid_block_size);
	logical_p->y = MIN(dstate.viewport.end.y, dstate.viewport.start.y + pt.y / grid_block_size);
}

/* Determine the screen coordinates that correspond to the given Life coordinates. Specifically,
 * determine the location of the upper left hand pixel of the onscreen block (past any grid
 * lines).
 */
void get_screen_coords(const point* p, point* screen_p)
{
	int32  grid_block_size;

	grid_block_size = (DRAW_GRID(dstate.zoom) ? dstate.zoom+1 : dstate.zoom);
	screen_p->x = (p->x - dstate.viewport.start.x) * grid_block_size;
	screen_p->y = (p->y - dstate.viewport.start.y) * grid_block_size;
	if (DRAW_GRID(dstate.zoom)) {
		screen_p->x++;
		screen_p->y++;
	}
}

/* Call get_screen_coords to get screen coordinates for the Life coordinates of the given
 * rectangle.
 */
void get_screen_rectangle(const rect* r, rect* screen_r)
{
	get_screen_coords(&(r->start), &(screen_r->start));
	get_screen_coords(&(r->end),   &(screen_r->end));
}

/* Read a directory of patterns into a sorted, dynamically-allocated array of pattern_file's,
 * including any files with extensions in default_file_extensions[]. If include_subdirs is true,
 * also include subdirectories of dir in the array. The number of items loaded is placed in
 * num_patterns.
 *
 * *patterns and *num_patterns should reflect the original state of the list. If *patterns is
 * non-NULL, the old list will be cleared via clear_pattern_list() before loading the new one.
 */
void load_pattern_directory(const char* dir, boolean include_subdirs, pattern_file** patterns,
		                    int32* num_patterns)
{
	pattern_file*   list;
	pattern_file    new_file;
	char*           dir_prefix;
	char*           path;
	char*           filename;
	char*           dot_pos;
	int32           num;
	int32           alloc_num;
	DIR*            dir_reader;
	struct dirent*  file_ent;
	struct stat     statbuf;
	int32           i;

	if (*patterns)
		clear_pattern_list(patterns, num_patterns);
	list = *patterns;
	num  = *num_patterns;

	dir_reader = opendir(dir);
	if (!dir_reader)
		return;
	if (dir[strlen(dir)-1] == '/')
		dir_prefix = safe_strdup(dir);
	else
		dir_prefix = dsprintf("%s/", dir);

	alloc_num = 50;
	list = safe_malloc(alloc_num * sizeof(pattern_file));
	while ((file_ent = readdir(dir_reader)) != NULL) {
		filename = file_ent->d_name;
		if (filename[0] == '.')
		    continue;
		new_file.filename = NULL;
		path = dsprintf("%s%s", dir_prefix, filename);
		if (stat(path, &statbuf) == 0 && S_ISDIR(statbuf.st_mode)) {
		    if (include_subdirs) {
		        new_file.filename = safe_strdup(filename);
		        new_file.is_directory = TRUE;
		    }
		} else {
		    for (i=0; default_file_extensions[i]; i++) {
		        if ((dot_pos = strrchr(filename, '.')) != NULL &&
		            STR_EQUAL_I(dot_pos+1, default_file_extensions[i])) {
		            new_file.filename = safe_strdup(filename);
		            new_file.is_directory = FALSE;
		            break;
		        }
		    }
		}
		if (new_file.filename) {
		    new_file.path = path;
		    set_pattern_title(&new_file);
		    if (num == alloc_num) {
		        alloc_num += 50;
		        list = safe_realloc(list, alloc_num * sizeof(pattern_file));
		    }
		    memcpy(&list[num++], &new_file, sizeof(pattern_file));
		} else
		    free(path);
	}
	closedir(dir_reader);
	qsort(list, num, sizeof(pattern_file), qsort_pattern_files);
	free(dir_prefix);

	*patterns = list;
	*num_patterns = num;
}

/* If the pattern is running, reset the frames-per-second  measurement.
 */
void reset_fps_measure(void)
{
	if (state.pattern_running) {
		state.start_tick = tick;
		state.start_time = get_time_milliseconds();
	}
}

/* Process an update of an onscreen rectangle (selection or paste). If redraw is true, redraw
 * the offscreen pixmap and invalidate portions of the screen to account for the visible change.
 * Pass &null_rect for newr to indicate that the rectangle has been deactivated.
 */
void screen_box_update(rect* oldr, const rect* newr, boolean redraw)
{
	rect       r, or, sr, cr;
	dimension  dim;
	int32      i;

	if (rects_identical(oldr, newr))
		return;
	or = *oldr;
	*oldr = *newr;
	if (!redraw)
		return;

	if (DRAW_GRID(dstate.zoom))
		draw_grid_and_boxes();
	else
		draw_life_pixmap();

	for (i=0; i < 2; i++) {
		if (i == 0) {
		    if (or.start.x < 0)
		        continue;
		    r = or;
		} else {
		    if (newr->start.x < 0)
		        continue;
		    r = *newr;
		}

		normalize_rectangle(&r);
		get_screen_rectangle(&r, &sr);
		sr.end.x += dstate.zoom-1;
		sr.end.y += dstate.zoom-1;
		if (DRAW_GRID(dstate.zoom)) {
		    sr.start.x--;
		    sr.start.y--;
		    sr.end.x++;
		    sr.end.y++;
		}
		cr.start.x = MAX(0, sr.start.x);
		cr.start.y = MAX(0, sr.start.y);
		cr.end.x   = MIN(dstate.eff_canvas_size.width-1, sr.end.x);
		cr.end.y   = MIN(dstate.eff_canvas_size.height-1, sr.end.y);
		get_rect_dimensions(&cr, &dim);

		if (sr.start.x >= 0)
		    gtk_widget_queue_draw_area(gui.canvas, sr.start.x, cr.start.y, 1, dim.height);
		if (sr.end.x < dstate.eff_canvas_size.width)
		    gtk_widget_queue_draw_area(gui.canvas, sr.end.x, cr.start.y, 1, dim.height);
		if (sr.start.y >= 0)
		    gtk_widget_queue_draw_area(gui.canvas, cr.start.x, sr.start.y, dim.width, 1);
		if (sr.end.y < dstate.eff_canvas_size.height)
		    gtk_widget_queue_draw_area(gui.canvas, cr.start.x, sr.end.y, dim.width, 1);
	}
}

/* Clear and/or copy the current selection. */
void selection_copy_clear(boolean copy, boolean clear)
{
	static uint16 left_col_masks[8]   = {0xFFFF, 0xEEEE, 0xCCCC, 0x8888, 0, 0, 0, 0};
	static uint16 right_col_masks[8]  = {0xFFFF, 0x7777, 0x3333, 0x1111, 0, 0, 0, 0};
	static uint16 top_col_masks[8]    = {0xFFFF, 0xFFF0, 0xFF00, 0xF000, 0, 0, 0, 0};
	static uint16 bottom_col_masks[8] = {0xFFFF, 0x0FFF, 0x00FF, 0x000F, 0, 0, 0, 0};
	cage_type*  c;
	cage_type*  newcage;
	rect        r;
	int32       left_offset, right_offset, top_offset, bottom_offset;
	uint16      nw_mask, ne_mask, sw_mask, se_mask;
	int32       x, y;

	r = state.selection;
	normalize_rectangle(&r);
	if (copy) {
		state.copy_rect = r;
		clear_cage_list(&state.copy_buffer);
		set_action_sensitive ("Paste", TRUE);
	}

	while ((c = loop_cages()) != NULL) {
		if (IS_EMPTY(c))
		    continue;
		x = c->x * CAGE_SIZE;
		y = c->y * CAGE_SIZE;
		if (r.end.x >= x && r.start.x < x+CAGE_SIZE &&
		    r.end.y >= y && r.start.y < y+CAGE_SIZE) {
		    nw_mask = ne_mask = sw_mask = se_mask = 0xFFFF;
		    if (r.start.x > x || r.start.y > y ||
		        r.end.x < x+CAGE_SIZE-1 || r.end.y < y+CAGE_SIZE-1) {

		        left_offset   = MAX(0, r.start.x - x);
		        right_offset  = MAX(0, x + CAGE_SIZE - 1 - r.end.x);
		        top_offset    = MAX(0, r.start.y - y);
		        bottom_offset = MAX(0, y + CAGE_SIZE - 1 - r.end.y);

		        nw_mask &= left_col_masks[left_offset];
		        sw_mask &= left_col_masks[left_offset];
		        ne_mask &= left_col_masks[MAX(left_offset-4, 0)];
		        se_mask &= left_col_masks[MAX(left_offset-4, 0)];
		        ne_mask &= right_col_masks[right_offset];
		        se_mask &= right_col_masks[right_offset];
		        nw_mask &= right_col_masks[MAX(right_offset-4, 0)];
		        sw_mask &= right_col_masks[MAX(right_offset-4, 0)];
		        nw_mask &= top_col_masks[top_offset];
		        ne_mask &= top_col_masks[top_offset];
		        sw_mask &= top_col_masks[MAX(top_offset-4, 0)];
		        se_mask &= top_col_masks[MAX(top_offset-4, 0)];
		        sw_mask &= bottom_col_masks[bottom_offset];
		        se_mask &= bottom_col_masks[bottom_offset];
		        nw_mask &= bottom_col_masks[MAX(bottom_offset-4, 0)];
		        ne_mask &= bottom_col_masks[MAX(bottom_offset-4, 0)];
		    }
		    if (copy) {
		        newcage = safe_malloc(sizeof(cage_type));
		        newcage->x = c->x;
		        newcage->y = c->y;
		        newcage->bnw[0] = c->bnw[parity] & nw_mask;
		        newcage->bne[0] = c->bne[parity] & ne_mask;
		        newcage->bsw[0] = c->bsw[parity] & sw_mask;
		        newcage->bse[0] = c->bse[parity] & se_mask;
		        newcage->next = state.copy_buffer;
		        state.copy_buffer = newcage;
		    }
		    if (clear)
		        mask_cage(c, ~nw_mask, ~ne_mask, ~sw_mask, ~se_mask);
		}
	}
}

/* Set the pattern collection directory to the given path. If have_gui is TRUE, update the onscreen
 * sidebar accordingly.
 */
void set_collection_dir(const char* path, boolean have_gui)
{
	if (!state.current_collection || !STR_EQUAL(path, state.current_collection)) {
		free(state.current_collection);
		state.current_collection = safe_strdup(path);
		if (have_gui)
		    update_sidebar_contents();
	}
	free(state.current_dir);
	state.current_dir = safe_strdup(state.current_collection);
}

/* Set state.current_dir (default load directory) to the parent directory of the given file
 * (specified by absolute path).
 */
void set_current_dir_from_file(const char* filepath)
{
	char*  parent_dir;
	char*  slash_pos;

	parent_dir = safe_strdup(filepath);
	slash_pos = strrchr(parent_dir, '/');
	if (!slash_pos) {
		free(parent_dir);
		return;
	}
	*(slash_pos+1) = '\0';
	free(state.current_dir);
	state.current_dir = parent_dir;
}

/* Set the display title (file->title) for the given pattern file structure. Title is derived from
 * filename as follows: For a regular file, strip off the extension. For a directory, append a '/'
 * character. In all cases, replace underscores with spaces and capitalize words.
 */
void set_pattern_title(pattern_file* file)
{
	char*  p;

	if (file->is_directory)
		file->title = dsprintf("%s/", file->filename);
	else {
		file->title = safe_strdup(file->filename);
		if ((p = strrchr(file->title, '.')) != NULL)
		    *p = '\0';
	}

	file->title[0] = toupper(file->title[0]);
	for (p=file->title; *p; p++) {
		if (*p == '_') {
		    *p = ' ';
		    *(p+1) = toupper(*(p+1));
		}
	}
}

/* Set the pattern run speed, by some means other than the speed slider. This function will
 * readjust the slider, thus triggering an event, so handle_speed_slider_change will take care of
 * everything else.
 */
void set_speed(int32 new_speed)
{
	GtkAdjustment*  slider_state;

	state.speed = new_speed;
	if (state.speed < 1)
		state.speed = 1;
	else if (state.speed > MAX_SPEED)
		state.speed = MAX_SPEED;

	slider_state = gtk_range_get_adjustment(GTK_RANGE(gui.speed_slider));
	if (state.speed < slider_state->lower) {
		slider_state->lower = state.speed;
		gtk_adjustment_changed(slider_state);
	} else if (state.speed > slider_state->upper) {
		slider_state->upper = state.speed;
		gtk_adjustment_changed(slider_state);
		set_speed_label_size(state.speed);
	}
	gtk_adjustment_set_value(slider_state, state.speed);
}

/* Deselect any selected item in the sidebar (or the sub-sidebar, if it is active).
 */
void sidebar_unselect(void)
{
	GtkWidget*  clist;

	clist = (dstate.sub_sidebar_visible ? gui.sub_patterns_clist : gui.patterns_clist);
	if (GTK_CLIST(clist)->selection || GTK_CLIST(clist)->focus_row > -1) {
		gtk_clist_unselect_all(GTK_CLIST(clist));
		GTK_CLIST(clist)->focus_row = -1;
		gtk_widget_queue_draw(clist);
	}
}

/* Toggle state.pattern_running, and update menus, toolbar, etc. accordingly, but do *not* update
 * the canvas or the tick label.
 */
void start_stop(void)
{
	state.pattern_running = !state.pattern_running;

	GtkAction *action = gtk_action_group_get_action (gui.action_group, "Play");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), state.pattern_running);

	if (state.pattern_running) {
		state.skipped_frames = 0;
		state.start_tick = tick;
		state.start_time = get_time_milliseconds();
		g_object_set (action, "stock-id", GTK_STOCK_MEDIA_PAUSE, NULL);
	} else {
		state.skipped_frames = 0;
		g_object_set (action, "stock-id", GTK_STOCK_MEDIA_PLAY, NULL);
	}

}

/* Validate the given pattern collection directory, returning a canonical form (dynamically
 * allocated) if valid, and NULL otherwise. If validation fails and have_gui is TRUE, display an
 * error dialog.
 */
char* validate_collection_dir(const char* path, boolean have_gui)
{
	DIR*   dir;
	char*  resolved_path;
	char*  final_path;;

	resolved_path = get_canonical_path(path);
	if (!resolved_path || !(dir = opendir(resolved_path))) {
		if (have_gui) {
		    if (!resolved_path || errno == ENOENT)
		        error_dialog("Invalid pattern collection directory:\n%s does not exist", path);
		    else
		        error_dialog("Invalid pattern collection directory\n(%s)", strerror(errno));
		}
		free(resolved_path);
		return NULL;
	} else {
		closedir(dir);
		final_path = append_trailing_slash(resolved_path);
		free(resolved_path);
		return final_path;
	}
}

/* Canonicalize, validate and, if valid, set the given pattern collection dir. If have_gui is
 * true, display an error dialog for a validation error, and update the sidebar display after
 * setting the collection. Otherwise do everything quietly.
 *
 * Return TRUE if validation succeeded, FALSE otherwise
 */
boolean validate_and_set_collection_dir(const char* path, boolean have_gui)
{
	char*  resolved_path;

	if ((resolved_path = validate_collection_dir(path, have_gui)) != NULL) {
		set_collection_dir(resolved_path, have_gui);
		free(resolved_path);
		return TRUE;
	} else
		return FALSE;
}

/*** Utility Functions ***/

/* Return a dynamically allocated copy of the given path, with a trailing slash appended unless
 * it already has one.
 */
char* append_trailing_slash(const char* path)
{
	return (path[strlen(path)-1] == '/') ? safe_strdup(path) : dsprintf("%s/", path);
}

/* Return an absolute, canonical pathname for the given path, dynamically allocated. All symlinks
 * and references to . and .. will be resolved. References to ~ will be resolved by referring to
 * state.home_dir; references to ~user will be resolved via getpwnam. If the path cannot be
 * resolved, return NULL.
 *
 * This fuction will resolve a file that does not exist, as long as its parent directory exists.
 * Trailing symlinks will not be translated, but intermediate directory symlinks will.
 */
char* get_canonical_path(const char* path)
{
	struct  passwd*  passwd_entry;
	char   path_buf[PATH_MAX];
	const char*  endptr;
	char*  p;
	char*  user;
	char*  dir;
	char*  file;
	char*  realpath_result;
	int32  username_len;

	if (!path || IS_EMPTY_STRING(path))
		return NULL;

	/* First, render the past absolute */
	if (STR_STARTS_WITH(path, "/"))                /* already an absolute path */
		p = safe_strdup(path);
	else if (STR_STARTS_WITH(path, "~")) {         /* home-relative path */
		if (path[1] == '/' || path[1] == '\0')
		    p = dsprintf("%s%s", state.home_dir, path+1);
		else {
		    endptr = strchr(path+1, '/');
		    if (!endptr)
		        endptr = path + strlen(path);
		    username_len = endptr - (path+1);
		    user = safe_malloc((username_len + 1) * sizeof(char));
		    strncpy(user, path+1, username_len);
		    user[username_len] = '\0';
		    passwd_entry = getpwnam(user);
		    free(user);
		    if (!passwd_entry || !(passwd_entry->pw_dir) || !strlen(passwd_entry->pw_dir))
		        return NULL;
		    p = dsprintf("%s%s", safe_strdup(passwd_entry->pw_dir), endptr);
		}
	} else {                                       /* cwd-relative path */
		if (!getcwd(path_buf, PATH_MAX))
		    return NULL;
		p = dsprintf("%s/%s", path_buf, path);
	}
	if (p[0] != '/') {  /* sanity check */
		free(p);
		return NULL;
	}

	/* Cut off trailing slashes */
	while (!STR_EQUAL(p, "/") && p[strlen(p)-1] == '/')
		p[strlen(p)-1] = '\0';

	/* Usually we want to resolve the parent directory instead of the whole path */
	split_path(p, &dir, &file);
	if (!dir) {}
	else if (STR_EQUAL(file, ".") || STR_EQUAL(file, "..")) {
		realpath_result = realpath(p, path_buf);
		free(p);
		if (realpath_result)
		    p = safe_strdup(path_buf);
		else
		    p = NULL;
	} else {
		realpath_result = realpath(dir, path_buf);
		free(p);
		if (realpath_result)
		    p = join_path(path_buf, file);
		else
		    p = NULL;
	}

	free(dir);
	free(file);
	return p;
}

/* Return the user's home directory, dynamically allocated. If no home directory is found, "/" is
 * assumed.
 */
char* get_home_directory(void)
{
	const char*  home_dir;

	home_dir = g_get_home_dir();
	if (home_dir && strlen(home_dir))
		return ((home_dir[strlen(home_dir)-1] == '/') ? safe_strdup(home_dir) :
		                                                dsprintf("%s/", home_dir));
	else {
		warn("No home directory found! Assuming '/'.");
		return safe_strdup("/");
	}
}

/* Return a dimension structure representing the width and height of the given rectangle.
 */
void get_rect_dimensions(const rect* r, dimension* dim)
{
	dim->width  = r->end.x - r->start.x + 1;
	dim->height = r->end.y - r->start.y + 1;
}

/* Get the current time in milliseconds, as a 64-bit uint
 */
uint64 get_time_milliseconds(void)
{
	struct timeval   t;
	struct timezone  tz;

	gettimeofday(&t, &tz);
	return (uint64)(t.tv_sec) * 1000ULL + (uint64)(t.tv_usec) / 1000ULL;
}

/* Join the given parent directory and filename into a path.
 */
char* join_path(const char* dir, const char* file)
{
	char*  divider;

	divider = ((dir[strlen(dir)-1] == '/') ? "" : "/");
	return dsprintf("%s%s%s", dir, divider, file);
}

/* Insure that r->start is the upper-left-hand corner of the rectangle.
 */
void normalize_rectangle(rect* r)
{
	int32  temp;

	if (r->start.x > r->end.x)
		SWAP(r->start.x, r->end.x);
	if (r->start.y > r->end.y)
		SWAP(r->start.y, r->end.y);
}

/* Return TRUE if the two point structures are identical.
 */
boolean points_identical(const point* p1, const point* p2)
{
	return (p1->x == p2->x && p1->y == p2->y);
}

/* A qsort helper to sort an array of pattern files by title, ignoring case.
 */
int32 qsort_pattern_files(const void* pat1, const void* pat2)
{
	return strcasecmp(((const pattern_file*)pat1)->title, ((const pattern_file*)pat2)->title);
}

/* Return TRUE if the two rect structures are identical.
 */
boolean rects_identical(const rect* p1, const rect* p2)
{
	return (points_identical(&(p1->start), &(p2->start)) &&
		    points_identical(&(p1->end), &(p2->end)));
}

/* Return a dynamically allocated string which is a copy of str with any underscore character
 * removed.
 */
char* remove_underscore(const char* str)
{
	char*  newstr;
	char*  us_pos;

	newstr = safe_strdup(str);
	us_pos = strchr(newstr, '_');
	if (us_pos)
		memmove(us_pos, us_pos+1, strlen(us_pos+1)+1);

	return newstr;
}

/* Split the given absolute path into parent directory and file, which will be dynamically
 * allocated and placed in the the corresponding paramters. If path is "/", the return values are
 * NULL and "/".
 *
 * If you don't need dir or file, pass the unneeded parameter as NULL.
 */
void split_path(const char* path, char** dir, char** file)
{
	char*  slash_pos;
	char*  p;
	char*  d;
	char*  f;

	p = safe_strdup(path);

	if (STR_EQUAL(p, "/")) {
		d = NULL;
		f = "/";
	} else {
		slash_pos = strrchr(p, '/');
		if (slash_pos == p) {
		    d = "/";
		    f = p+1;
		} else {
		    *slash_pos = '\0';
		    d = p;
		    f = slash_pos+1;
		}
	}

	if (dir)
		*dir  = (d ? safe_strdup(d) : NULL);
	if (file)
		*file = safe_strdup(f);
	free(p);
}
