/* Copyright (C) 2016 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <glib.h>
#include <vector>
#include <functional>  // for std::bind and std::ref

#include <c++-gtk-utils/async_channel.h>
#include <c++-gtk-utils/thread.h>
#include <c++-gtk-utils/cgu_config.h>

using namespace Cgu;

class Test {
  int i;
public:
  int get_i() {return i;}
  Test& operator=(const Test& t) {i = t.i; return *this;}
  Test& operator=(Test&& t) {i = t.i; return *this;}
  Test(const Test& t) {i = t.i;}
  Test(Test&& t) {i = t.i;}
  explicit Test(int val): i(val) {}
  ~Test() {}
};

typedef AsyncChannel<Test, 8> BufferedChannel;
typedef AsyncChannel<Test, 1> UnbufferedChannel;

template <class Channel>
void send_func(Channel& c, int n) {
  int count = 0;
  while (count < n) {
    // emplace
    c.emplace(count);
    ++count;
    // push with rvalue
    if (count < n) {
      c.push(Test{count});
      ++count;
    }
    // push with lvalue
    if (count < n) {
      Test t{count};
      c.push(t);
      ++count;
    }
  }
}

template <class Channel>
void send_func_with_close(Channel& c, int n) {
  send_func(c, n);
  c.close();
}

template <class Channel>
std::vector<Test> receive_func(Channel& c, int n) {
  int count = 0;
  Test t{-1};
  std::vector<Test> ret;
  while (count < n) {
    bool more = false;
    if (count % 2)
      more = c.pop(t);
    else
      more = c.move_pop(t);
    if (!more)
      break;
    ret.push_back(t);
    ++count;
  }
  return ret;
}

extern "C" {
static void test_async_channel_buffered() {
  // test push and pop (this also tests the destructor of AsyncChannel
  // because it is destroyed with 2 items in it)
  {
    BufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
      Thread::Thread::start(Callback::make<BufferedChannel&>(send_func<BufferedChannel>,
							     c, 20),
			    true);
    auto vec = receive_func(c, 18);

    g_assert_cmpuint(vec.size(), ==, 18);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
    g_assert_cmpint(vec[17].get_i(), ==, 17);

    t->join();
  }

  // test channel closing - 1
  {
    BufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
      Thread::Thread::start(Callback::make<BufferedChannel&>(send_func_with_close<BufferedChannel>,
							     c, 20),
			    true);
    auto vec = receive_func(c, 100);

    Test test{-1};
    bool ret = c.pop(test);

    g_assert_cmpuint((unsigned int)ret, ==, 0);
    g_assert_cmpint(test.get_i(), ==, -1);
    g_assert_cmpuint(vec.size(), ==, 20);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
    g_assert_cmpint(vec[19].get_i(), ==, 19);
    
    t->join();

    ret = c.emplace(0);
    g_assert_cmpuint((unsigned int)ret, ==, 0);
    ret = c.push(Test{0});
    g_assert_cmpuint((unsigned int)ret, ==, 0);
    ret = c.push(test);
    g_assert_cmpuint((unsigned int)ret, ==, 0);
  }

  // test channel closing - 2
  {
    BufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
      Thread::Thread::start(Callback::make<BufferedChannel&>(send_func_with_close<BufferedChannel>,
							     c, 7),
			    true);

    t->join();

    auto vec = receive_func(c, 100);
    g_assert_cmpuint(vec.size(), ==, 7);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
    g_assert_cmpint(vec[6].get_i(), ==, 6);
  }
}

static void test_async_channel_unbuffered() {
  // test push and pop (this also tests the destructor of AsyncChannel
  // because it is destroyed with an item in it)
  {
    UnbufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
      Thread::Thread::start(Callback::make<UnbufferedChannel&>(send_func<UnbufferedChannel>,
							       c, 19),
			    true);
    auto vec = receive_func(c, 18);

    g_assert_cmpuint(vec.size(), ==, 18);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
    g_assert_cmpint(vec[17].get_i(), ==, 17);

    t->join();
  }

  // test channel closing - 1
  {
    UnbufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
      Thread::Thread::start(Callback::make<UnbufferedChannel&>(send_func_with_close<UnbufferedChannel>,
							       c, 20),
			    true);
    auto vec = receive_func(c, 100);

    Test test{-1};
    bool ret = c.pop(test);

    g_assert_cmpuint((unsigned int)ret, ==, 0);
    g_assert_cmpint(test.get_i(), ==, -1);
    g_assert_cmpuint(vec.size(), ==, 20);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
    g_assert_cmpint(vec[19].get_i(), ==, 19);
    
    t->join();

    ret = c.emplace(0);
    g_assert_cmpuint((unsigned int)ret, ==, 0);
    ret = c.push(Test{0});
    g_assert_cmpuint((unsigned int)ret, ==, 0);
    ret = c.push(test);
    g_assert_cmpuint((unsigned int)ret, ==, 0);
  }

  // test channel closing - 2
  {
    UnbufferedChannel c;

    auto t =
      // gcc-4.4 does not support lambda expressions
       Thread::Thread::start(Callback::make<UnbufferedChannel&>(send_func_with_close<UnbufferedChannel>,
							       c, 1),
			    true);

    t->join();

    auto vec = receive_func(c, 100);
    g_assert_cmpuint(vec.size(), ==, 1);
    g_assert_cmpint(vec[0].get_i(), ==, 0);
  }
}
} // extern "C"


int main (int argc, char* argv[]) {
  g_test_init(&argc, &argv, static_cast<void*>(0));

  g_test_add_func("/async_channel/buffered", test_async_channel_buffered); 
  g_test_add_func("/async_channel/unbuffered", test_async_channel_unbuffered); 

  return g_test_run();
}
