//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file PrimaryGeneratorAction.cc
/// \brief Implementation of the PrimaryGeneratorAction class

#include "PrimaryGeneratorAction.hh"

#include "DetectorConstruction.hh"
#include "HistoManager.hh"

#include "G4Event.hh"
#include "G4GenericMessenger.hh"
#include "G4ParticleDefinition.hh"
#include "G4ParticleGun.hh"
#include "G4ParticleTable.hh"
#include "G4SystemOfUnits.hh"
#include "Randomize.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

PrimaryGeneratorAction::PrimaryGeneratorAction(DetectorConstruction* det) : fDetector(det)
{
  G4int n_particle = 1;
  fParticleGun = new G4ParticleGun(n_particle);
  SetDefaultKinematic();

  // define commands for this class
  DefineCommands();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

PrimaryGeneratorAction::~PrimaryGeneratorAction()
{
  delete fParticleGun;
  delete fMessenger;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OO ooo........oooOO0OOooo......

void PrimaryGeneratorAction::SetDefaultKinematic()
{
  G4ParticleTable* particleTable = G4ParticleTable::GetParticleTable();
  G4String particleName;
  G4ParticleDefinition* particle = particleTable->FindParticle(particleName = "proton");
  fParticleGun->SetParticleDefinition(particle);
  fParticleGun->SetParticleMomentumDirection(G4ThreeVector(1., 0., 0.));
  fParticleGun->SetParticleEnergy(5. * GeV);
  G4double position = -0.5 * (fDetector->GetWorldSizeX()) + 1 * nm;
  fParticleGun->SetParticlePosition(G4ThreeVector(position, 0. * cm, 0. * cm));
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PrimaryGeneratorAction::GeneratePrimaries(G4Event* anEvent)
{
  // this function is called at the begining of event
  //
  // randomize the beam, if requested.
  if (fRndmBeam > 0.) {
    G4ThreeVector oldPosition = fParticleGun->GetParticlePosition();
    G4double rbeam = 0.5 * (fDetector->GetCalorSizeYZ()) * fRndmBeam;
    G4double x0 = oldPosition.x();
    G4double y0 = oldPosition.y() + (2 * G4UniformRand() - 1.) * rbeam;
    G4double z0 = oldPosition.z() + (2 * G4UniformRand() - 1.) * rbeam;
    fParticleGun->SetParticlePosition(G4ThreeVector(x0, y0, z0));
    fParticleGun->GeneratePrimaryVertex(anEvent);
    fParticleGun->SetParticlePosition(oldPosition);
  }
  else
    fParticleGun->GeneratePrimaryVertex(anEvent);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PrimaryGeneratorAction::DefineCommands()
{
  // Define /testhadr/gun command directory using generic messenger class
  fMessenger = new G4GenericMessenger(this, "/testhadr/gun/", "gun control");

  // default kinematic command
  auto& defaultCmd =
    fMessenger->DeclareMethod("setDefault", &PrimaryGeneratorAction::SetDefaultKinematic,
                              "set/reset kinematic in PrimaryGenerator");

  defaultCmd.SetStates(G4State_PreInit, G4State_Idle);

  // randomize beam extension command
  auto& rndmCmd = fMessenger->DeclareProperty("rndm", fRndmBeam);

  rndmCmd.SetGuidance("lateral size of the beam, in fraction of sizeYZ ");
  rndmCmd.SetParameterName("rBeam", false);
  rndmCmd.SetRange("rBeam>=0.&&rBeam<=1.");
  rndmCmd.SetDefaultValue("0.");
  rndmCmd.SetStates(G4State_Idle);
}

//..oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
