/* $Id$ */
/*
 * makes X11 BDF font bold-italic.
 * by Daisuke NISHIKAWA
 */
#include <ctype.h>
#include "common.h"
#include "bold.h"
#include "italic.h"
#include "bolditalic.h"
#include "utils.h"

#define BUFLEN 256

static void mkbolditalic(int bdir, int pile, int correct, int dy, Property *prop, int verbose, char *filename);
static void get_option(int *bdir, int *pile, int *correct, int *dy, Property *prop, int *verbose, char **filename, int argc, char *argv[]);
static void usage(FILE *fp, int status);


int main(int argc, char *argv[])
{
  Property prop;
  int bdir = TRUE;  /* left shifted image */
  int pile = FALSE; /* right edge */
  int correct = PIXEL_CORRECTION;
  int dy = SLANT_DEEP;
  int verbose = FALSE;
  char *filename = NULL;

  prop.weight[0] = "Medium";
  prop.weight[1] = "Bold";
  prop.slant[0] = "R";
  prop.slant[1] = "I";
  get_option(&bdir, &pile, &correct, &dy, &prop, &verbose, &filename, argc, argv);
  mkbolditalic(bdir, pile, correct, dy, &prop, verbose, filename);
  return EXIT_SUCCESS;
}

static void mkbolditalic(int bdir, int pile, int correct, int dy, Property *prop, int verbose, char *filename)
{
  FILE *fp;
  char buf[BUFLEN], outbuf[BUFLEN], keyword[BUFLEN];
  Pattern *ptable;
  Bitmap *bmap;
  size_t count;
  int width, height, dw, width_max, height_max;
  int bitmap, rc;
  ProgressMeter *meter;
  int max_chars;

  width = height = dw = 0;
  prop->width = &width;
  prop->height = &height;
  prop->dw = &dw;
  prop->dy = dy;
  prop->recast = recast_bbx;
  fp = file_open(filename);
  max_chars = replace_property(fp, BUFLEN, prop);
  if (max_chars < 0) {
    fclose(fp);
    return;
  }
  meter = NULL;
  if (verbose) {
    meter = progress_meter_new(max_chars);
  }
  width_max = width+dw;
  height_max = height;
  ptable = pattern_new(correct);
  bmap = bitmap_new(((width-1)/8+1)*8+dw, height_max);
  count = 0;
  bitmap = FALSE;
  while (fgets(buf, BUFLEN, fp) != NULL) {
    if (isalpha(buf[0])) {
      sscanf(buf, "%s", keyword);
      if (strcmp(keyword, "BBX") == 0) {
        /* recast the metric information from the bounding box */
        rc = recast_bbx(outbuf, &width, &height, &dw, dy, buf);
        if (!rc || (width+dw > width_max) || (height > height_max)) {
          fclose(fp);
          fprintf(stderr, "Incorrect bounding box data: %s\n", buf);
          exit(EXIT_FAILURE);
        }
        fputs(outbuf, stdout);
        continue;
      }
      if (strcmp(keyword, "BITMAP") == 0) {
        bitmap = TRUE;
        count = 0;
        fputs(buf, stdout);
        continue;
      }
      if (strcmp(keyword, "ENDCHAR") == 0) {
        bitmap = FALSE;
        if (correct > 0) {
          bitmap_correct_pixel(bmap, ptable, width, height, dy);
        }
        bitmap_make_slant(bmap, width, height, dy);
        bitmap_make_bold_output(stdout, bmap, width+dw, height, bdir, pile);
        fputs(buf, stdout);
        if (verbose) {
          progress_meter_print(meter);
        }
        continue;
      }
    }

    if (bitmap) {
      bitmap_get_line(bmap, width, count, buf);
      if (count < height) {
        count++;
      }
      continue;
    }
    fputs(buf, stdout);
  }
  fclose(fp);
  pattern_delete(ptable);
  bitmap_delete(bmap);

  if (verbose) {
    progress_meter_clear(meter);
    progress_meter_delete(meter);
  }
}

static void get_option(int *bdir, int *pile, int *correct, int *dy, Property *prop, int *verbose, char **filename, int argc, char *argv[])
{
  int i;

  i = 1;
  while (i < argc) {
    if (*argv[i] == '-') {
      switch (*(argv[i]+1)) {
      case 'r':
        *bdir = FALSE;
        break;
      case 'l':
        *bdir = TRUE;
        break;
      case 'R':
        *pile = FALSE;
        break;
      case 'L':
        *pile = TRUE;
        break;
      case 'p':
        print_ptable(stdout, *correct);
        exit(EXIT_SUCCESS);
        break;
#if DEBUG
      case 'W':
        if (++i < argc) {
          prop->weight[0] = argv[i];
        }
        break;
      case 'w':
        if (++i < argc) {
          prop->weight[1] = argv[i];
        }
        break;
      case 'S':
        if (++i < argc) {
          prop->slant[0] = argv[i];
        }
        break;
      case 's':
        if (++i < argc) {
          prop->slant[1] = argv[i];
        }
        break;
      case 'c':
        if (++i < argc) {
          *correct = atoi(argv[i]);
        }
        break;
      case 'd':
        if (++i < argc) {
          *dy = atoi(argv[i]);
        }
        break;
#endif /* DEBUG */
      case 'V':
        *verbose = TRUE;
        break;
      case 'h':
        usage(stdout, EXIT_SUCCESS);
        break;
      }
    } else {
      *filename = argv[i];
      break;
    }
    i++;
  }
}

static void usage(FILE *fp, int status)
{
  fprintf(fp,
          "Usage: mkbolditalic [OPTION] [FILE]\n\n"
          "Write processed X11 BDF file to standard output.\n"
          "With no FILE, read standard input.\n"
          "\nOptions:\n"
          "  -l left shifted image OR original (default)\n"
          "  -r right shifted image OR original\n"
          "  -L reflect left edge\n"
          "  -R reflect right edge (default)\n"
          "  -p print a pattern table for pixel correction used by slanting\n"
          "  -V verbose output\n"
          "  -h display this help and exit\n"
          );
  exit(status);
}
