"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chalk_1 = require("chalk");
const semverCompare = require("semver-compare");
const lib_1 = require("../lib");
const versions_1 = require("../lib/actions/versions");
const strings_1 = require("../lib/util/strings");
// ----------------------------------------------------------------------------
// Helpers
// ----------------------------------------------------------------------------
const { log } = console;
const identical = (val) => chalk_1.default `{bold.magenta ${val}}`;
const similar = (val) => chalk_1.default `{bold.blue ${val}}`;
const warning = (val) => chalk_1.default `{bold.yellow ${val}}`;
const error = (val) => chalk_1.default `{bold.red ${val}}`;
// `~/different-foo/~/foo` + highlight last component.
const shortPath = (filePath, pkgName) => {
    let short = filePath.replace(/node_modules/g, "~");
    // Color last part of package name.
    const lastPkgIdx = short.lastIndexOf(pkgName);
    if (lastPkgIdx > -1) {
        short = chalk_1.default `${short.substring(0, lastPkgIdx)}{cyan ${pkgName}}`;
    }
    return short;
};
// `duplicates-cjs@1.2.3 -> different-foo@1.1.1 -> foo@3.3.3`
const pkgNamePath = (pkgParts) => pkgParts.reduce((m, part) => `${m}${m ? " -> " : ""}${part.name}@${part.range}`, "");
// Organize duplicates by package name.
const getDuplicatesByFile = (files) => {
    const dupsByFile = {};
    Object.keys(files).forEach((fileName) => {
        files[fileName].sources.forEach((source) => {
            source.modules.forEach((mod) => {
                dupsByFile[mod.fileName] = {
                    baseName: mod.baseName || mod.fileName,
                    bytes: mod.size.full,
                    isIdentical: source.meta.extraSources.num > 1,
                };
            });
        });
    });
    return dupsByFile;
};
// Return object of asset names keyed to sets of package names with duplicates.
const getDuplicatesPackageNames = (data) => {
    const names = {};
    Object.keys(data.assets).forEach((assetName) => {
        // Convert to package names.
        const pkgNames = Object.keys(data.assets[assetName].files).map(versions_1._packageName);
        // Unique names.
        const uniqPkgNames = new Set(pkgNames);
        names[assetName] = uniqPkgNames;
    });
    return names;
};
// Return a new versions object with _only_ duplicates packages included.
exports._getDuplicatesVersionsData = (dupData, pkgDataOrig) => {
    // Start with a clone of the data.
    const pkgData = JSON.parse(JSON.stringify(pkgDataOrig));
    const assetsToDupPkgs = getDuplicatesPackageNames(dupData);
    // Iterate the data and mutate meta _and_ resultant entries.
    Object.keys(pkgData.assets).forEach((assetName) => {
        const dupPkgs = assetsToDupPkgs[assetName] || new Set();
        const { meta, packages } = pkgData.assets[assetName];
        Object.keys(packages)
            // Identify the packages that are not duplicates.
            .filter((pkgName) => !dupPkgs.has(pkgName))
            // Mutate packages and meta.
            // Basically, unwind exactly everything from `versions.ts`.
            .forEach((pkgName) => {
            const pkgVersions = Object.keys(packages[pkgName]);
            // Unwind stats.
            meta.packages.num -= 1;
            meta.resolved.num -= pkgVersions.length;
            pkgData.meta.packages.num -= 1;
            pkgData.meta.resolved.num -= pkgVersions.length;
            pkgVersions.forEach((version) => {
                const pkgVers = packages[pkgName][version];
                Object.keys(pkgVers).forEach((filePath) => {
                    meta.files.num -= pkgVers[filePath].modules.length;
                    meta.depended.num -= pkgVers[filePath].skews.length;
                    meta.installed.num -= 1;
                    pkgData.meta.files.num -= pkgVers[filePath].modules.length;
                    pkgData.meta.depended.num -= pkgVers[filePath].skews.length;
                    pkgData.meta.installed.num -= 1;
                });
            });
            // Remove package.
            delete packages[pkgName];
        });
    });
    return pkgData;
};
// ----------------------------------------------------------------------------
// Plugin
// ----------------------------------------------------------------------------
class DuplicatesPlugin {
    constructor(opts) {
        opts = opts || {};
        this.opts = {};
        this.opts.verbose = opts.verbose === true; // default `false`
        this.opts.emitErrors = opts.emitErrors === true; // default `false`
        this.opts.emitHandler = typeof opts.emitHandler === "function" ? opts.emitHandler : undefined;
    }
    apply(compiler) {
        if (compiler.hooks) {
            // Webpack4 integration
            compiler.hooks.emit.tapPromise("inspectpack-duplicates-plugin", this.analyze.bind(this));
        }
        else {
            // Webpack1-3 integration
            compiler.plugin("emit", this.analyze.bind(this));
        }
    }
    analyze(compilation, callback) {
        const { errors, warnings } = compilation;
        const stats = compilation.getStats().toJson();
        const { emitErrors, emitHandler, verbose } = this.opts;
        // Stash messages for output to console (success) or compilation warnings
        // or errors arrays on duplicates found.
        const msgs = [];
        const addMsg = (msg) => msgs.push(msg);
        return Promise.all([
            lib_1.actions("duplicates", { stats }).then((a) => a.getData()),
            lib_1.actions("versions", { stats }).then((a) => a.getData()),
        ])
            .then((datas) => {
            const [dupData, pkgDataOrig] = datas;
            const header = chalk_1.default `{bold.underline Duplicate Sources / Packages}`;
            // No duplicates.
            if (dupData.meta.extraFiles.num === 0) {
                log(chalk_1.default `\n${header} - {green No duplicates found. 🚀}\n`);
                return;
            }
            // Filter versions/packages data to _just_ duplicates.
            const pkgData = exports._getDuplicatesVersionsData(dupData, pkgDataOrig);
            // Choose output format.
            const fmt = emitErrors ? error : warning;
            // Have duplicates. Report summary.
            // tslint:disable max-line-length
            addMsg(chalk_1.default `${header} - ${fmt("Duplicates found! ⚠️")}

* {yellow.bold.underline Duplicates}: Found ${strings_1.numF(dupData.meta.extraFiles.num)} ${similar("similar")} files across ${strings_1.numF(dupData.meta.extraSources.num)} code sources (both ${identical("identical")} + similar)
  accounting for ${strings_1.numF(dupData.meta.extraSources.bytes)} bundled bytes.
* {yellow.bold.underline Packages}: Found ${strings_1.numF(pkgData.meta.packages.num)} packages with ${strings_1.numF(pkgData.meta.resolved.num)} {underline resolved}, ${strings_1.numF(pkgData.meta.installed.num)} {underline installed}, and ${strings_1.numF(pkgData.meta.depended.num)} {underline depended} versions.
`);
            // tslint:enable max-line-length
            Object.keys(pkgData.assets).forEach((dupAssetName) => {
                const pkgAsset = pkgData.assets[dupAssetName];
                let dupsByFile = {};
                if (dupData.assets[dupAssetName] &&
                    dupData.assets[dupAssetName].files) {
                    dupsByFile = getDuplicatesByFile(dupData.assets[dupAssetName].files);
                }
                const { packages } = pkgAsset;
                const pkgNames = Object.keys(packages);
                // Only add asset name when duplicates.
                if (pkgNames.length) {
                    addMsg(chalk_1.default `{gray ## ${dupAssetName}}`);
                }
                pkgNames.forEach((pkgName) => {
                    // Calculate stats / info during maps.
                    let latestVersion;
                    let numPkgInstalled = 0;
                    const numPkgResolved = Object.keys(packages[pkgName]).length;
                    let numPkgDepended = 0;
                    const versions = Object.keys(packages[pkgName])
                        .sort(semverCompare)
                        .map((version) => {
                        // Capture
                        latestVersion = version; // Latest should be correct bc of `semverCompare`
                        numPkgInstalled += Object.keys(packages[pkgName][version]).length;
                        let installs = Object.keys(packages[pkgName][version]).map((installed) => {
                            const skews = packages[pkgName][version][installed].skews
                                .map((pkgParts) => pkgParts.map((part, i) => (Object.assign({}, part, { name: chalk_1.default[i < pkgParts.length - 1 ? "gray" : "cyan"](part.name) }))))
                                .map(pkgNamePath)
                                .sort(strings_1.sort);
                            numPkgDepended += skews.length;
                            if (!verbose) {
                                return chalk_1.default `  {green ${version}} {gray ${shortPath(installed, pkgName)}}
    ${skews.join("\n    ")}`;
                            }
                            const duplicates = packages[pkgName][version][installed].modules
                                .map((mod) => dupsByFile[mod.fileName])
                                .filter(Boolean)
                                .map((mod) => {
                                const note = mod.isIdentical ? identical("I") : similar("S");
                                return chalk_1.default `{gray ${mod.baseName}} (${note}, ${strings_1.numF(mod.bytes)})`;
                            });
                            return chalk_1.default `    {gray ${shortPath(installed, pkgName)}}
      {white * Dependency graph}
        ${skews.join("\n        ")}
      {white * Duplicated files in }{gray ${dupAssetName}}
        ${duplicates.join("\n        ")}
`;
                        });
                        if (verbose) {
                            installs = [chalk_1.default `  {green ${version}}`].concat(installs);
                        }
                        return installs;
                    })
                        .reduce((m, a) => m.concat(a)); // flatten.
                    // tslint:disable-next-line max-line-length
                    addMsg(chalk_1.default `{cyan ${pkgName}} (Found ${strings_1.numF(numPkgResolved)} {underline resolved}, ${strings_1.numF(numPkgInstalled)} {underline installed}, ${strings_1.numF(numPkgDepended)} {underline depended}. Latest {green ${latestVersion || "NONE"}}.)`);
                    versions.forEach(addMsg);
                    if (!verbose) {
                        addMsg(""); // extra newline in terse mode.
                    }
                });
            });
            // tslint:disable max-line-length
            addMsg(chalk_1.default `
* {gray.bold.underline Understanding the report}: Need help with the details? See:
  https://github.com/FormidableLabs/inspectpack/#diagnosing-duplicates
* {gray.bold.underline Fixing bundle duplicates}: An introductory guide:
  https://github.com/FormidableLabs/inspectpack/#fixing-bundle-duplicates
`.trimLeft());
            // tslint:enable max-line-length
            // Drain messages into custom handler or warnings/errors.
            const report = msgs.join("\n");
            if (emitHandler) {
                emitHandler(report);
            }
            else {
                const output = emitErrors ? errors : warnings;
                output.push(new Error(report));
            }
        })
            // Handle old plugin API callback.
            .then(() => {
            if (callback) {
                return void callback();
            }
        })
            .catch((err) => {
            // Ignore error from old webpack.
            if (callback) {
                return void callback();
            }
            throw err;
        });
    }
}
exports.DuplicatesPlugin = DuplicatesPlugin;
