/*
 * Getdate: Network Date/Time Query and Set Local Date/Time.
 *
 *   Michael Hamilton (michael@actrix.gen.nz) Oct 1995.
 *
 *   Based on rdate by Andy Tefft (teffta@crypt.erie.ge.com) which was
 *   based on rdate by Lee Moore at University of Rochester.  
 *
 * Retrieves the current date/time of host using RFC 868.
 *
 * Can set local time by copying from other hosts. 
 *
 * Querying the time:
 *
 *   Reports the time on each host.
 *
 * Setting time:
 *
 *   Tries a list of hosts until one query succeeds or until the list
 *   is exhausted.
 *
 *   Obtains two samples from each host.  Allows for travelling time
 *   by adding half the interval between the pairs of samples to the
 *   new time.
 *
 *   Will reject a remote host's value if the time between samples
 *   exceeds a user defined limit.
 *
 *   Will reject a remotes host's value if its difference from local
 *   time exceeds a supplied limit.
 *
 * Copyright (C) 1995  Michael Hamilton (michael@actrix.gen.nz)
 *   
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *   
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>


#include <time.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#define MAX_DATE_STR  50
#define MAX_HOSTNAME 500

#define UNIXEPOCH 2208988800UL	/* difference between Unix time and net time 
				 * The UNIX EPOCH starts in 1970.
				 */
#define USAGE \
 printf("usage: %s HOST1 [ HOST2... ]\n", argv[0]); \
 printf("       %s -adjust MAX_SAMPLE_INTERVAL MAX_TIME_CHANGE HOST1 [ HOST2... ]\n", argv[0]); \
 printf("       %s -set    MAX_SAMPLE_INTERVAL MAX_TIME_CHANGE HOST1 [ HOST2... ]\n", argv[0]);

static time_t remote_time(const char * const inet_address, char * const hostname)
{
  struct hostent *host_entry;	/* host details returned by the DNS */
  struct servent *time_server;	/* service file entry - for the port number */
  struct sockaddr_in sin;	/* socket to the remote host */
  int fd;			/* network file descriptor - for reading the time */

  time_t unix_time;		/* remote time in Unix format */
  u_char net_time[4];		/* remote time in network format */

  if ((host_entry = gethostbyname(inet_address)) == NULL) {
    fprintf(stderr, "getdate: error - %s - unknown host.\n", inet_address);
    return -1;
  }

  if (hostname) {
    strncpy(hostname, host_entry->h_name, MAX_HOSTNAME - 1);
    hostname[MAX_HOSTNAME - 1] = '\0';
  }

  if ((time_server = getservbyname("time","tcp")) == NULL) {
    fprintf(stderr, "getdate: error - time/tcp - unknown service.\n");
    return -1;
  }

  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    fprintf(stderr,
	    "getdate: error - %s - %s\n", host_entry->h_name, strerror(errno));
    return -1;
  }

  sin.sin_family = host_entry->h_addrtype;
  bcopy(host_entry->h_addr, (caddr_t)&sin.sin_addr, host_entry->h_length);
  sin.sin_port = time_server->s_port;

  if (connect(fd, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
    fprintf(stderr, 
	    "getdate: error - %s - %s\n", host_entry->h_name, strerror(errno));
    close(fd);
    return -1;
  }

  if (read(fd, net_time, 4) != 4) { 	/* get remote date */
    fprintf(stderr,
	    "getdate: error - %s - read error %s\n", 
	    host_entry->h_name, strerror(errno));
    close(fd);
    return -1;
  }

  close(fd);

  unix_time = ntohl(* (long *) net_time) - UNIXEPOCH;

  return unix_time;
}

static char * time_to_str(const time_t t) 
{				/* Get rid of the newline that ctime adds */
  static char buf[MAX_DATE_STR] ;
  strcpy(buf, ctime(&t));
  buf[strlen(buf) - 1] = '\0'; 
  return buf;
}


static void print_samples(const char * const host,
			  const time_t first_sample,
			  const time_t second_sample)
{
  char buf1[MAX_DATE_STR], buf2[MAX_DATE_STR];
  char *ptr;

  strcpy(buf1, time_to_str(first_sample));
  strcpy(buf2, time_to_str(second_sample));
  printf("getdate: %s\t%s, %s\n", host, buf1, buf2);
}


#ifdef NEED_STIME
static stime(time_t *new_time)
{
  struct timeval tv;

  tv.tv_sec = *new_time;
  tv.tv_usec = 0;
  settimeofday(&tv, NULL);
}
#endif
 

static int copy_time(const char * const host, 
		     const time_t max_interval, 
		     const time_t max_change,
		     const int    use_stime) 
{
  time_t first_sample, second_sample, new_time, old_time, diff;

  first_sample = remote_time(host, NULL); 

  if (first_sample == -1) {
    fprintf(stderr, "getdate: error - %s first sample failed - time not set.\n",
	    host);
    return 0;
  }

  second_sample = remote_time(host, NULL);	/* Same host, second sample */

  if (second_sample == -1) {	
    fprintf(stderr, "getdate: error %s second sample failed - time not set\n",
	    host);
    return 0;
  }
  diff = (second_sample - first_sample);
  if (diff > max_interval) {	/* The sampling took too long. */
    print_samples(host, first_sample, second_sample);
    fprintf(stderr, 
	    "getdate: info - %s second sample took more than %d second(s), time not set.\n",
	    host, max_interval);
    return 0;
  }
				/* Assume half the round trip time. */
  new_time = second_sample + ((double) diff + 0.5 / 2.0);
  old_time = time(NULL);

  if (abs(new_time - old_time) > max_change) { /* Diff is too ridical. */
    print_samples(host, first_sample, second_sample);
    fprintf(stderr, 
	    "getdate: info - %s difference is too large (abs(%+d) > %d), time not set.\n",
	    host,
	    new_time - old_time,
	    max_change);
    return 0;
  }

  /* Checking all done - proceed with the change. */

  if (use_stime) { 
    				/* Dangerous!  Could upset cron and other
				 * timer related events.
				 */
    stime(&new_time);
    print_samples(host, first_sample, second_sample);
    printf("getdate: set time to %s to match host %s\n", 
	   time_to_str(new_time),
	   host) ;
  } 
  else {			/* Safely creep the time into sync. */
#ifdef HAVE_ADJTIME  
    struct timeval adjustment, old_adjustment;
    adjustment.tv_sec  = new_time - old_time;
    adjustment.tv_usec = 0;
    if (adjtime(&adjustment, &old_adjustment) == -1) {
      fprintf(stderr, 
	      "getdate: error - adjust time error - %s\n", strerror(errno));
    }
    print_samples(host, first_sample, second_sample);
    if (old_adjustment.tv_sec != 0 || adjustment.tv_usec != 0) {
      printf("getdate: replaced existing adjustment of %d second(s), %d usec(s).\n",
	     old_adjustment.tv_sec,
	     old_adjustment.tv_usec);
    }
    printf("getdate: time will adjust slowly by %d second(s) to match %s.\n",
	   adjustment.tv_sec,
	   host);
#else
    fprintf(stderr, 
	    "getdate: error - adjtime() not available on this operating system.\n");
    exit(EXIT_FAILURE);    
#endif
  }
  
  return new_time;
}


main (int argc, char *argv[])
{
  int i;
  int adjust_time = 0;
  int set_time = 0;

  if (argc == 1) {
    USAGE;
    exit(EXIT_FAILURE);
  }
  
  adjust_time = strcmp(argv[1], "-adjust") == 0 ;
  set_time    = strcmp(argv[1], "-set")    == 0 ;

  if (adjust_time || set_time) {

    if (argc < 5) {
      USAGE;
      exit(EXIT_FAILURE);
    }
    else {
      time_t max_interval, max_change, new_time;
      uid_t uid = geteuid();
      char *host;

      if (uid != 0) {
	fprintf(stderr, 
		"getdate: error - only root can alter the date/time.\n",
		argv[0]);
	exit(EXIT_FAILURE);
      }
      errno = 0;
      max_interval = (time_t) strtol(argv[2], NULL, 10);
      if (errno == ERANGE || max_interval < 0) {
	fprintf(stderr, "getdate: error - invalid max interval %s.\n", argv[2]);
	exit(EXIT_FAILURE);
      }
      errno = 0;
      max_change = (time_t) strtol(argv[3], NULL, 10);
      if (errno == ERANGE || max_change < 0) {
	fprintf(stderr, "getdate: error - invalid max change %s.\n", argv[3]);
	exit(EXIT_FAILURE);
      }

				/* Try until we succeed or run out of hosts */
      for( i = 4; i < argc; i++ ) { 
	new_time = copy_time(argv[i], max_interval, max_change, set_time);
	if (new_time != 0) {	/* Success - quit now */
	  break;
	}
      }
      if (new_time == 0) {
	exit(EXIT_FAILURE);
      }
    }
  }
  else {  /* Just a query */
    for( i = 1; i < argc; i++ ) {
      char hostname[MAX_HOSTNAME];
      time_t unix_time = remote_time(argv[i], hostname);
      if (unix_time != -1) {
	printf("%s:\t(%+d)\t%s\n", 
	       hostname, 
	       unix_time - time(NULL),
	       time_to_str(unix_time));
      }
    }
  }
  exit(EXIT_SUCCESS);
}

