#!/usr/bin/env python3

"""
This script processes events generated by the extaction plugin and displays
notifications of these events on the screen.
"""

import base64
import html
import os
import sys

import gi
gi.require_version('GdkPixbuf', '2.0')
gi.require_version('Notify', '0.7')
from gi.repository import GdkPixbuf, Notify


# The module requires html.escape() and os.environb which first appeared in
# Python 3.2.
if sys.hexversion < 0x03020000:
    print("This program requires at least Python 3.2.", file=sys.stderr)
    sys.exit(1)


def get_env(key):
    bkey = key.encode()
    if bkey not in os.environb:
        print("Environment variable '{}' is not set.".format(key))
        raise KeyError
    return os.environb[bkey].decode()


def main():
    # Make the parameters saved in environment variables easier accessible.
    try:
        event_type = get_env('EVENT_TYPE')
        # This script can handle only the msg type.
        if event_type != 'msg':
            print("EVENT_TYPE='{}' is not recognized.".format(event_type))
            sys.exit(1)

        # event_network = get_env('EVENT_NETWORK')
        # event_local_user = get_env('EVENT_LOCAL_USER')
        event_remote_user = get_env('EVENT_REMOTE_USER')
        event_message = get_env('EVENT_MESSAGE')
        # event_message_html = get_env('EVENT_MESSAGE_HTML')
    except KeyError as e:
        # Some necessary parameter is missing.
        sys.exit(1)

    if not Notify.init("Extaction-plugin handler"):
        print("Initialization of libnotify failed.")
        sys.exit(1)

    title = "Message from {}:".format(event_remote_user)
    # Select first 256 characters from the message.
    body = event_message[:256]
    # Escape the '&', '<', '>' characters.
    body = html.escape(body)

    notification = Notify.Notification.new(title, body)
    if b'EVENT_REMOTE_USER_ICON' in os.environb:
        # The icon is encoded in base64, decode it first.
        icon_encoded = os.environb[b'EVENT_REMOTE_USER_ICON']
        icon_decoded = base64.b64decode(icon_encoded)

        # Create a pixbuf loader.
        loader = GdkPixbuf.PixbufLoader.new()
        loader.set_size(48, 48)
        loader.write(icon_decoded)
        loader.close()

        # Set icon from the pixbuf.
        pixbuf = loader.get_pixbuf()
        notification.set_icon_from_pixbuf(pixbuf)

    # Get the notification on the screen.
    notification.show()

    Notify.uninit()

if __name__ == '__main__':
    main()

# vim: set tabstop=4 shiftwidth=4 textwidth=79 expandtab :
