/*-------------------------------------------------------------------------- 
  Filename        : sgcommon.c
  Abstract        : Common subroutines for SCSI Generic Utilities
  Operation System: Linux 2.2 or greater
  Author  	  : Andy Cress   <arcress@users.sourceforge.net>
  Copyright (c) 2002-2008 Intel Corporation 
  Copyright (c) 2009, Kontron America, Inc.

  ----------- Change History -----------------------------------------------
  07/26/02  ARCress  created 
  09/03/02  ARCress  streamlined error display in get_scsi_info.
  10/31/02  ARCress  handle DevFS names in make_dev_name, etc.
  11/15/02  ARCress  implement tree walking for DevFS in make_dev_name
		     (contribution from jpuhlman@mvista.com)
  03/10/03  ARCress  show inquiry data for emulated usb-storage devs
  05/08/03  ARCress  handle some 12-byte sernum cases
  08/01/03  ARCress  detect DevFS if host1 & not host0 in make_dev_name
  09/29/03  ARCress  make sure getchar() values are int for 64-bit compiles
  05/11/05  ARCress  included patch for more dev types from Nate Dailey
  ----------- Description --------------------------------------------------
  Subroutines included:
    closefd 
    closelog 
    closeall 
    quit    
    itoh
    findmatch
    get_ndev
    get_idev
    get_mdev
    get_func
    do_pause
    showit
    showmsg
    dumpbufr
    make_dev_name
    get_ival
    get_scsi_info
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002, Intel Corporation
Copyright (c) 2009, Kontron America, Inc.
All rights reserved.
Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:
  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include "sgsub.h"
#include "sgcommon.h"

/* Global data definitions */

extern char fdebug;		/* =1 for debug messages */
extern FILE *fdlog;		/* log file descriptor */
extern FILE *fdmsg;		/* file descriptor for messages  */
extern FILE *dbgout;		/* debug output file descriptor */
extern char filesok;		/* =1 if ok to write to files */
extern char flogopen;		/* =1 if log file is open */
extern char logfile[]; 
extern int ndev;
extern DEVLIST_TYPE devlist[];

char fdevfs = 0;
#define NDEVTYP  15
char *devtype_array[NDEVTYP] = {
    "Disk", /*0*/
    "Tape", /*1*/
    "Prtr", /*2*/
    "Proc", /*3*/
    "WORM", /*4*/
    "CDRM", /*5*/
    "Scan", /*6*/
    "Opti", /*7*/
    "MChg", /*8*/
    "Comm", /*9*/
    "Emul", /*10*/
    "Un11", /*11*/
    "RAID", /*12*/
    "Encl", /*13*/
    "Unkn"  /*14*/
};
char HeaderStr[] = "\nNum Name [bus:ch:id:lun] Type Vendor Device_Model     FW   Serial#  Size\n";

/* Local subroutines */
void closefd(void)
{
    int i, sts;
    for (i = 0; i < ndev; i++) {	/* close any open scsi devices */
	if (devlist[i].sgfd > 0) {
	    sts = set_sg_debug(devlist[i].sgfd, 0);
	    if (sts && fdebug)
		showlog("[%d] cant clear debug flag, sts = %d\n", i,sts);
	    close(devlist[i].sgfd);	/*opened in beforegd() */
	    devlist[i].sgfd = 0;
	    if (fdebug)
		showlog("[%d] closefd, sgfd=%x\n", i, devlist[i].sgfd);
	}
    }
}

void closelog(void)
{
    if (flogopen) fclose(fdlog);	/* close the log file */
    flogopen = 0;
    fdlog = stderr;
    dbgout = stderr;
}

void closeall(void)
{
    closefd();
    closelog();
}

void quit(int rc)
{
    closeall();
    /* close any open sg_fd handles */
    /* free any allocated memory and close any open handles */
    exit(rc);
}

void itoh(uchar * chp, int len, char *str)
{
    uchar rgascii[] = { "0123456789abcdef" };
    int i, j;
    uchar k;
    for (i = 0, j = 0; i < len; i++) {
	k = (chp[i] & 0xf0) >> 4;
	str[j++] = rgascii[k];
	k = chp[i] & 0x0f;
	str[j++] = rgascii[k];
	str[j++] = ' ';
    }
    str[j] = 0;			/* stringify */
}

/* 
 * findmatch 
 * returns offset in buffer of the match, if found, 
 * or -1 if not found.  
 */
int findmatch(char *buffer, int sbuf, char *pattern, int spattern, 
		char figncase)
{
    int c, i, j, imatch;

    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) return(-1);
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if ((figncase == 1) &&
                   ((c & 0x5f) == (pattern[imatch] & 0x5f))) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else {
            if (imatch > 0) {
               imatch = 0;
               if (j > 0) j--; /* try again with the first match char */
	    }
        }
        if (imatch == spattern) break;
    }
    if (imatch == spattern) {
        i = (j+1) - imatch;  /*buffer[i] is the match */
        return(i);
    } else return (-1);  /*not found*/
}

uchar get_ndev(int first)
{				/* Get next device index. */
    if (first + 1 > ndev)
	return ((uchar) 0xff);	/* done, no matches */
    else
	return (first + 1);
}

uchar get_idev(void)
{
    int i, j;
    int ch;
    char cstr[6];
    printf("\n");
    printf("Device Num (0 to %d) : ", ndev - 1);
    ch = getchar();
    for (j = 0; j <= 5 && ch != EOF && ch != 0x0a; )
    {
      if (ch > 0x20 && ch < 0x7f) cstr[j++] = ch;
      ch = getchar();
    }
    cstr[j] = 0;
    if (cstr[0] > 0x39) i = cstr[0];   /* character */
    else i = atoi(cstr);
    if (fdebug) showlog("str: %s, i = %d\n",cstr,i);
    if (i == 0x71 || i == 0x51) {	/* 'q' or 'Q' = quit */
	quit(0);
	return (0xff);
    }
    if ((i < 0) || (i >= ndev)) {
	printf("Invalid index (%d)\n", i);
	return (0xff);
    }
    return ((uchar) i);
}

static int prev_idev = -1;
uchar 
get_mdev(char *model)
{
    struct SCSI_INQUIRY *scsi_inq;
    int i, sz;
    /* Get first device index that matches the selected model. */
    sz = strlen(model);  /* won't include trailing spaces */
    for (i = 0; i < ndev; i++) {
        if (i > prev_idev) {
            scsi_inq = (struct SCSI_INQUIRY *) devlist[i].inq;
            if (strncmp(model,(char *)scsi_inq->productid,sz) == 0) {
                prev_idev = i;
                return ((uchar)i);     /* matches */
            }
        }
    }
    return ((uchar) 0xff);      /* no matches */
}

uchar 
get_nextdev(void)
{
    static int previ = -1;
    int i;
    i = previ + 1;
    if ((i < 0) || (i >= ndev)) i = 0xff;
    else previ = i;
    return((uchar)i);
}

char get_func(void)
{   /* Get the function character */
    int c;
    char response = 0;
    c = getchar();
    while (c != EOF && c != 0x0a)
    {
      if (c > 0x20 && c < 0x7f) response = (char)c;
      if (fdebug)
	showlog("response=%c (%02x), c=%02x\n", response, response, c);
      c = getchar();
    }
    if (response == 0) response = (char)c;
    return(response);
}

void do_pause(void)
{
    int ch = 0;
    printf("Press ENTER to continue\n");
    while (ch != 0x0a) {
	ch = getchar();
    }
}

void showit(char *buf)
{
    if (filesok && !flogopen) {	/* log is not open */
	fdlog = fopen(logfile, "a+");
	if (fdebug) printf("opening %s, errno = %d\n", logfile, errno);
	if (fdlog != NULL) {
	    flogopen = 1;
            dbgout = fdlog;
	} else {
	    printf("Cannot open %s, errno = %d\n", logfile, errno);
	    fdlog = stderr;
	}
    }
    if (flogopen) fprintf(fdlog, buf);	/*write to log */
    fprintf(fdmsg, buf);	/*defaults to stdout */
}

void showlog(const char * format, ...) 
{
    va_list vptr; 
    FILE *fd;
    if (flogopen) fd = fdlog;
    else fd = fdmsg;
    va_start(vptr, format);
    vfprintf(fd, format, vptr);
    va_end(vptr);
}

void showmsg(const char * format, ...) 
{
    static char msgshow[MSG_LEN]; 
    va_list vptr; 
    va_start(vptr, format);
    vsnprintf(msgshow, MSG_LEN, format, vptr);
    va_end(vptr);
    showit(msgshow);
}

#ifdef OLD
void dumpbufr(FILE * fdout, uchar * bufp, int mlen, char *hdr)
{		/* now defunct, use dump_buf or dump_log */
    int i;
    char abuf[5];
    if (fdout == NULL) return;
    if (fdebug) {
	fprintf(fdout, "mlen = %d\n", mlen);
	fprintf(fdout, "bufp: %02x %02x %02x %02x\n",
		bufp[0], bufp[1], bufp[2], bufp[3]);
    } 
    if (hdr == NULL)
	hdr = "Buffer";
    fprintf(fdout, "\n%s (len=%d): ", hdr, mlen);
    for (i = 0; i < mlen; i++) {
	if (i % 16 == 0)	/* start a new line */
	    fprintf(fdout, "\n%04x: ", i);
	itoh(&bufp[i], 1, abuf);
	fprintf(fdout, "%s", abuf);
    }
    fprintf(fdout, "\n");
    return;
}				/*end dumpbufr() */
#endif

void dump_buf(FILE * fdout, uchar * pbuf, int sz, char *tag, char fshowascii)
{
   uchar line[17];
   uchar a;
   int i, j;

   if (fdout == NULL) return;
   if (tag == NULL) tag = "Buffer";
   line[0] = 0; line[16] = 0;
   j = 0;
   fprintf(fdout,"%s (len=%d): ", tag,sz);
   for (i = 0; i < sz; i++) {
      if (i % 16 == 0) { j = 0; fprintf(fdout,"%s\n  %04x: ",line,i); }
      if (fshowascii) {
         a = pbuf[i];
         if (a < 0x20 || a > 0x7f) a = '.';
         line[j++] = a;
      }
      fprintf(fdout,"%02x ",pbuf[i]);
   }
   fprintf(fdout,"%s\n",line);
   return;
} 

void dump_log(uchar * pbuf, int sz, char *tag, char fshowascii)
{
    FILE *fd;
    if (flogopen) fd = fdlog;
    else fd = fdmsg;
    dump_buf(fd,pbuf,sz,tag,fshowascii);
}

static int find_lun(DEVFS_LIST * devfsnums)
{
    	char *devfs_lun = "/dev/scsi/host%d/bus%d/target%d/lun%d";
	char devfs_lun_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->lun; i < 16; i++)
	{
		sprintf(devfs_lun_buf, devfs_lun, devfsnums->host, devfsnums->bus, devfsnums->target, i);
		if(stat(devfs_lun_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode)){
				devfsnums->lun=i;
				break;
			}
		}
	}	
	if(i >= 15){
		devfsnums->lun=0;
		return -1;
	}
	return 1;
}

static int find_target(DEVFS_LIST * devfsnums)
{
    	char *devfs_target = "/dev/scsi/host%d/bus%d/target%d";
	char devfs_target_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->target; i < 16; i++)
	{
		sprintf(devfs_target_buf, devfs_target, devfsnums->host, devfsnums->bus, i);
		if(stat(devfs_target_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->target=i;
				if (find_lun(devfsnums) == 1)
					break;
		}
	}
	if(i >= 16){
		devfsnums->target=0;
		return -1;	
	}
	return 1;
}

static int find_bus(DEVFS_LIST * devfsnums)
{
    	char *devfs_bus = "/dev/scsi/host%d/bus%d";
	char devfs_bus_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->bus ; i < 16; i++)
	{
		sprintf(devfs_bus_buf, devfs_bus, devfsnums->host,i);
		if(stat(devfs_bus_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->bus=i;
				if (find_target(devfsnums) == 1)
					break;
		}
	}	
	if(i >= 16){
		devfsnums->bus=0;
		return -1;
	}
	return 1;
}

static int find_host(DEVFS_LIST * devfsnums)
{
    	char *devfs_host = "/dev/scsi/host%d";
	char devfs_host_buf[80];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->host ; i < 16; i++)
	{
		sprintf(devfs_host_buf, devfs_host, i);
		if(stat(devfs_host_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->host=i;
				if (find_bus(devfsnums) == 1)
					break;
		}
	}	
	if(i >= 16)
		return -1;
	return 1;
}

static int get_first_dev_sg(DEVFS_LIST * devfsnums)
{
	if (find_host(devfsnums) == -1){
		devfsnums->all_leaves_searched=1;
		return 0; 
	}
	else
		return 1;

}

void make_dev_name(char *fname, int k, int fnumeric, DEVFS_LIST *devfsnums)
{
    char buff[64];
    int big, little;
    char *devfs_pat = "/dev/scsi/host%d/bus%d/target%d/lun%d/generic";

    if (k == 0) {  /* first time, detect if DevFS */
	struct stat stbuf;
	/* if /dev/scsi/host0 directory exists, we are using DevFS */
	if (stat("/dev/scsi/host0",&stbuf) == 0) {
	    if (S_ISDIR(stbuf.st_mode))  fdevfs = 1; 
	} else if (stat("/dev/scsi/host1",&stbuf) == 0) {
	    if (S_ISDIR(stbuf.st_mode))  fdevfs = 1; 
	}
    }
    if (fdevfs) {  /* DevFS */
	if (get_first_dev_sg(devfsnums)) {
		sprintf(buff,devfs_pat,
			devfsnums->host,devfsnums->bus,
			devfsnums->target,devfsnums->lun++); /* ~46 chars*/
		strcpy(fname,buff);  /*currently sizeof fname is 64 chars */
	} else  strcpy(fname,"/dev/scsi/end");  /* something, but invalid */
    } else {              /* normal dev names */
      strcpy(fname, "/dev/sg");
      if (fnumeric) {     /* numeric */
	sprintf(buff, "%d", k);
	strcat(fname, buff);
      } else {            /* alpha */
	if (k < 26) {
	    buff[0] = 'a' + (char) k;
	    buff[1] = '\0';
	    strcat(fname, buff);
	} else if (k <= 255) {	/* assumes sequence goes x,y,z,aa,ab,ac etc */
	    big = k / 26;
	    little = k - (26 * big);
	    big = big - 1;
	    buff[0] = 'a' + (char) big;
	    buff[1] = 'a' + (char) little;
	    buff[2] = '\0';
	    strcat(fname, buff);
	} else
	    strcat(fname, "xxxx");
      }
    }
}

int get_ival(char *valstr, char fhex)
{
    int i, ret;
    int ch;
    if (fhex) {
       printf("\n%s (hex): ", valstr);
       ret = scanf("%x", &i);
    } else {
       printf("\n%s (dec): ", valstr);
       ret = scanf("%d", &i);
    }
    ch = getchar();		/* get the newline char too */
    if (ret < 0) {
	printf("Invalid entry, error = %d\n", errno);
	i = 0;
    }
    return (i);
}

static char *sts_str[7] = {
/* 0*/ "OK",
/*-1*/ "Error",
/*-2*/ "Sense Error", /*Check Condition*/
/*-3*/ "Write Error",
/*-4*/ "Read Error",
/*-5*/ "ioctl Error",
/*-6*/ "Bad Inquiry" };

char *show_sts(int sts)
{
   int i;
   switch(sts)
   {
	case 0:   i=0; break;
	case -1:  i=1; break;
	case -2:  i=2; break;
	case -3:  i=3; break;
	case -4:  i=4; break;
	case -5:  i=5; break;
	case -6:  i=6; break;
	default:  i=1;
   }
   return(sts_str[i]);
}

int
handle_sense(int sgfd, int sts)
{
	uchar sense_buffer[80];	/* buffer for sense data */
	int k,a,q;
	int x;
        if (sts == SCHECK_CND) {  /* check for certain key/asc/ascq values */
	  x = get_sense(sts, sense_buffer);
	  k = sense_buffer[2] & 0x0f;	/*Sense Key */
	  a = sense_buffer[12]; 	/*ASC*/ 
	  q = sense_buffer[13]; 	/*ASCQ*/ 
	  showlog("Sense error %x:%x:%x\n", k,a,q);
	  if (k == 6 && a == 0x29 && q == 2) { /* reset occurred, clear it */
            sts = test_unit_ready(sgfd, 0); /* clear & ignore errors */
	  } else if (k == 2 && a == 4 && q == 2) {
	    /* 02-04-02 means it needs a start command, so issue it. */
	    showit("Sending start_unit.\n");
	    sts = start_unit(sgfd);
	    if (fdebug) showlog("Start Unit: sts=%d \n", sts);
	  } else {    /* other sense errors */
	    sts = 0;  /* continue anyway */
	    /* e.g. 0b-48-00 means aborted command w error, but keep going */
	  }
	  /* if (sts == 0) here, then ok to retry command */
	}  /*endif SCHECK_CND*/
	return(sts);
}  /*end handle_sense*/

int fillstr(char *to, uchar *from,int len)
{
	int i;
	uchar c;
	for (i = 0; i < len; i++) {
		c = from[i];
		if (c == 0) c = ' ';
		to[i] = c;
	}
	to[i++] = ' ';
	to[i] = 0;
	return(i);
}

#define PATH_MAX   64
#include <dirent.h>

int get_dev_maj_min(uchar bus, uchar ch, uchar id, uchar lun, 
			int *pmaj, int *pmin)
{
   char sysdev_file[64];
   char mmstr[16];
   FILE *fp;
   char *sp;

   sprintf(sysdev_file,"/sys/bus/scsi/devices/%d:%d:%d:%d/block/dev", 
		bus,ch,id,lun);
   fp = fopen(sysdev_file, "r");
   /* printf("fopen(%s) fp = %p\n",sysdev_file,fp);     *++++*/
   if (fp == NULL) return -1;
   sp = fgets(mmstr,sizeof(mmstr),fp);
   /* printf("fget() sp = %p, mmstr: %s\n",sp,mmstr);   *++++*/
   if (sp == NULL) return -1;
   sscanf(mmstr, "%u:%u", pmaj, pmin);
   return 0;
}

int find_mmdev(int maj, int min, char *devname)
{
   int dmaj, dmin;
   int match = 0;
   char *dev_dir = "/dev";
   char device_path[ PATH_MAX + 1];
   DIR *dirp;
   struct dirent *dep;
   struct stat stats;
   int rv = -1;

   dirp = opendir (dev_dir);
   if (dirp == NULL) return(rv);
   while (1)
   {
      dep = readdir (dirp);
      if (dep == NULL) break;
      snprintf (device_path, PATH_MAX, "%s/%s", dev_dir, dep->d_name);
      device_path [PATH_MAX] = '\0';
      if (lstat(device_path, &stats)) continue;
 
      /* Skip non-block/char files. */
      if ( (!S_ISBLK(stats.st_mode)) && (!S_ISCHR(stats.st_mode)) )
          continue;
      dmaj = major(stats.st_rdev);
      dmin = minor(stats.st_rdev);
      if ((maj == dmaj) && (min == dmin)) {
          strcpy(devname,device_path);
          match = 1; 
	  rv = 0;
          break;
      }
   }  /*end while*/
   return(rv);
}

/* 
 * GET SCSI DEVICE INFO: 
 * Displays SCSI Inquiry and other device data.
 */
int get_scsi_info(int sgfd, int idx, char *fname, int fservo)
{
    int i, x, y, ch, bus, dv, lun;
    int sts, ret;
    struct SCSI_INQUIRY *scsi_inq;
    My_scsi_idlun my_idlun;
    My_sg_scsi_id m_id;		/* compatible with Sg_scsi_id in sg.h */
    int host_no;
    int emul;
    ulong dcap = 0;		/* capacity */
    char output[MSG_LEN];	/* message output buffer */
    char output2[MSG_LEN];	/* extra message output buffer */
    uchar devstat[80];		/* used for inquiry & device status */
    uchar devstat2[120];	/* used for Seagate inquiry2, etc.  */
    int f, ilen, slen;

    host_no = 0;
    my_idlun.dev_id = 0;
    if (fdebug) 
	showlog("get_scsi_info: sgfd = %d\n", sgfd);
    /* Get extra info */
    sts = ioctl(sgfd, SCSI_IOCTL_GET_IDLUN, &my_idlun);
    if (sts < 0) sts= SIOCTL_ERR;
    else sts = ioctl(sgfd, SCSI_IOCTL_GET_BUS_NUMBER, &host_no);
    if (sts < 0) sts= SIOCTL_ERR;
    else sts = ioctl(sgfd, SG_EMULATED_HOST, &emul);
    if (sts < 0) sts= SIOCTL_ERR;
    else sts = ioctl(sgfd, SG_GET_SCSI_ID, &m_id);
    if (sts < 0) sts= SIOCTL_ERR;
    bus = host_no;
    ch = (my_idlun.dev_id >> 16) & 0xff;
    dv = my_idlun.dev_id & 0xff;
    lun = (my_idlun.dev_id >> 8) & 0xff;
    if (fdebug) {
	showlog("%s: scsi%d channel=%d id=%d lun=%d emul=%x sts=%d \n",
	       fname, bus, ch, dv, lun, emul, sts);
    }
    /* Start the device information string */
    if (fdevfs)   /* change display format because of longer names */
	sprintf(output, "%2d %s\n            [%d:%d:%d:%d] ", 
		idx,fname,bus,ch,dv,lun);
    else 
	sprintf(output, "%2d %s [%d:%d:%d:%d] ", idx,fname,bus,ch,dv,lun);
    if (sts < 0) {
	sprintf(output2,"%s          (%s)\n",output,show_sts(sts));
	showit(output2);  /* show the partial info */
	return (sts);
	}

    /* Set SCSI to blocking mode */
    f = fcntl(sgfd, F_GETFL);
    if (fdebug) showlog( "info: after fcntl, f = 0x%x\n", f);
    if (f & O_NONBLOCK) {       /* only do this if currently non-blocking */
        f = f & (~O_NONBLOCK);  /* set to blocking mode */
        sts = fcntl(sgfd, F_SETFL, f);
        if (fdebug)
	    showlog("info: after fcntl(2), f = 0x%x, ret = %d\n", f, sts);
    }

    /* Get SCSI INQUIRY information */
    sts = scsi_inquiry(sgfd, devstat, 80);
    if (sts) {
	sts = handle_sense(sgfd,sts);
	if (sts == 0) sts = scsi_inquiry(sgfd, devstat, 80);
    } /*endif sts*/

    /* Parse SCSI INQUIRY information */
    scsi_inq = (struct SCSI_INQUIRY *) devstat;
    if (fdebug) {
	/* Note that inquiry buffer is dumped in scsi_inquiry if debug */
	showlog("inq len = %d, devtype = %x, sts = %d\n",
	       scsi_inq->len, scsi_inq->dev_type, sts);
    }
    ilen = scsi_inq->len; 
    if (ilen < 24) {
	/* bogus inquiry data, disk went away */
	if (fdebug) {
		sprintf(output2,"Error, bad inquiry: len=%d, type=%d\n",
			ilen,devstat[0]);
		showit(output2);
	}
	sts = SIZE_ERR;  /* bad inquiry */
    } else if (ilen > 48) ilen = 48;      /* only use 48 bytes of it anyway */

    if (sts) {
	  sprintf(output2,"%s          (%s)\n",output,show_sts(sts));
	  showit(output2);  /* show the partial info */
	  showlog("[%d] Error %d getting SCSI Inquiry information\n", idx,sts);
	  return (sts);
	}

    /* Add device type string */
    y = scsi_inq->dev_type & 0x1F;
    if (y > DEVT_MAX) y = DEVT_UNKN;	/* if bad, set type to Unknown */
    strcat(output, devtype_array[y]);
    strcat(output, " ");
    x = strlen(output);
    if (emul) { y = DEVT_EMUL; } /*Emul = 10., usu usb-storage*/
    devlist[idx].devtype = y;

    x += fillstr(&output[x],scsi_inq->vendorid,8);
    x += fillstr(&output[x],scsi_inq->productid,16);
    x += fillstr(&output[x],scsi_inq->productrev,4);
    slen = 8;
    if (ilen > 36) {  /* have a sernum */
        if ((ilen > 44) && (scsi_inq->serialnum[8] > ' ') && (y == 0)) {
	    slen = 12; /*apparently a long sernum disk*/ }
        x += fillstr(&output[x],scsi_inq->serialnum,slen);
        /* Fix a Fujitsu disk firmware bug which reports ilen==41, 
         * and has a 10-byte sernum, so ilen should be 46. */
        if (ilen < (36+slen)) ilen = 46; 
    }
    if (emul) strcat(output," [em]");

    if (sts == 0 && scsi_inq->dev_type == 0x00 && emul == 0) { 
	/* For valid disks, get capacity */
	ret = read_capacity(sgfd, &dcap);
        if (ret) ret = handle_sense(sgfd,ret);
        if (!fservo ) {
	  if (ret == 0) {
	    long mb;
	    mb = 4 * 1024;  /*(1024*1024)/256*/
	    // output[x++] = ' ';
	    // output[x] = 0;	/*stringify */
	    sprintf(output2, "%ldMB", dcap / mb);
	    strcat(output, output2);
	    x += strlen(output2);
	  } else {
	    // output[x] = 0;	/*stringify */
	    sprintf(output2,"[%d] read_capacity error = %d\n",idx,ret);
	    showit(output2);
	  }
        }
    } /* endif disk */

    if (sts == 0 && emul == 0 &&
	strncmp((char *)scsi_inq->vendorid,"SEAGAT",6) == 0) {
	/* get & show Seagate servo version */
	sts = seagate_inquiry(sgfd, devstat2, 120);
        if (sts) sts = handle_sense(sgfd,sts);
	if (sts) {
	    sprintf(output2,
		    "[%d] Error %d getting SCSI Inquiry2 information\n",
		    idx, sts);
	    showit(output2);
	    for (i = 0; i < 8; i++) devlist[idx].srev[i] = ' ';
	} else {  /* success, copy servo string */
	  memcpy(devlist[idx].srev, &devstat2[16], 8);  /*copy servo string*/
	}
	if (fservo) 
		for (i = 0; i < 8; i++) output[x++] = devlist[idx].srev[i];
	else 
		output[x++] = 's';  /* indicate servo but dont show whole str */
	output[x] = 0;		/*stringify */
    } else {   /*not Seagate, so blank string */
	for (i = 0; i < 8; i++) devlist[idx].srev[i] = ' ';
    }

    if (sts == 0) {		/* if here, valid, so fill in info */
	/* fill in not just disks, tapes, etc. too */
	devlist[idx].bus = bus;
	devlist[idx].chn = ch;
	devlist[idx].tgt = dv;
	devlist[idx].lun = lun;
	memcpy(devlist[idx].inq, &devstat[0], ilen);  /*copy full inquiry buf*/
	devlist[idx].inq[44] = 0;
	strcpy(devlist[idx].fname, fname);         /*copy fname string*/
	devlist[idx].sgfd = sgfd;
	devlist[idx].dcap = dcap;
    }				/*endif valid */
    devlist[idx].sts = sts;
    strcat(output,"\n");	/* newline, stringified */
    x++;
    showit(output);
    fflush(fdmsg);
    return (sts);
}				/*end get_scsi_info() */

