// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- > Lambda0 Sigmabar0 + c.c.
  class BESIII_2023_I2686032 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2023_I2686032);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      book(_nLS, "/TMP/nLS" );
    }

    void findChildren(const Particle & p,map<long,int> & nRes, int &ncount) {
      for(const Particle &child : p.children()) {
	if(child.children().empty()) {
	  nRes[child.pid()]-=1;
	  --ncount;
	}
	else
	  findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // total hadronic and muonic cross sections
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
	nCount[p.pid()] += 1;
	++ntotal;
      }
      
      // find the Lambdas and Sigmas
      const UnstableParticles & ufs = apply<UnstableParticles>(event, "UFS");
      for(const Particle & p1 : ufs.particles(Cuts::abspid==3122)) {
      	bool matched = false;
      	// check fs
      	bool fs = true;
      	for(const Particle & child : p1.children()) {
      	  if(child.pid()==p1.pid()) {
      	    fs = false;
      	    break;
      	  }
      	}
      	if(!fs) continue;
      	// find the children
      	map<long,int> nRes = nCount;
      	int ncount = ntotal;
      	findChildren(p1,nRes,ncount);
	int sign = p1.pid()/3122;
       	for(const Particle & p2 : ufs.particles(Cuts::pid==-sign*3212)) {
      	  // check fs
      	  bool fs = true;
      	  for(const Particle & child : p2.children()) {
      	    if(child.pid()==p2.pid()) {
      	      fs = false;
      	      break;
      	    }
      	  }
      	  if(!fs) continue;
      	  map<long,int> nRes2 = nRes;
      	  int ncount2 = ncount;
      	  findChildren(p2,nRes2,ncount2);
      	  if(ncount2!=0) continue;
      	  matched=true;
      	  for(auto const & val : nRes2) {
      	    if(val.second!=0) {
      	      matched = false;
      	      break;
      	    }
      	  }
      	  if(matched) {
	    _nLS->fill();
      	    break;
      	  }
       	}
       	if(matched) break;
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      double fact  = crossSection()/ sumOfWeights() /picobarn;
      pair<double,double> sigma = make_pair(_nLS->val()*fact,_nLS->err()*fact);
      double mL = 1.115683;
      double mS = 1.192642;
      double tau  = sqr(sqrtS()/(mL+mS));
      double beta = sqrt(1.-2.*(sqr(mS)+sqr(mL))/sqr(sqrtS())
			 +sqr((sqr(mS)-sqr(mL))/sqr(sqrtS())));
      double alpha = 7.2973525693e-3;
      double GeV2pb = 0.3893793721e9;
      double sigma0 = 4.*M_PI*sqr(alpha/sqrtS())*beta*GeV2pb;
      // F(s)
      pair<double,double> FF = make_pair(sqrt(3.*sigma.first*tau/(sigma0*(0.5 + 0.5*tau))),
					 sqrt(1.5*tau/sigma.first/sigma0/(1.+2.*tau))*sigma.second);
      for(unsigned int ix=0;ix<2;++ix) {
       	Scatter2D temphisto(refData(1, 1, 1+ix));
      	Scatter2DPtr  mult;
        book(mult, 1, 1, 1+ix);
       	for (size_t b = 0; b < temphisto.numPoints(); b++) {
      	  const double x  = temphisto.point(b).x();
	  pair<double,double> ex = temphisto.point(b).xErrs();
	  pair<double,double> ex2 = ex;
	  if(ex2.first ==0.) ex2. first=0.0001;
	  if(ex2.second==0.) ex2.second=0.0001;
	  if (inRange(sqrtS()/GeV, x-ex2.first, x+ex2.second)) {
	    if(ix==0)
	      mult->addPoint(x, sigma.first, ex, make_pair(sigma.second,sigma.second));
	    else
	      mult->addPoint(x, FF   .first, ex, make_pair(FF   .second,FF   .second));
      	  }
      	  else {
      	    mult->addPoint(x, 0., ex, make_pair(0.,.0));
      	  }
      	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    CounterPtr _nLS;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2023_I2686032);

}
