/* wmpasman
 * Copyright © 1999-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#define SECRET_API_SUBJECT_TO_CHANGE "Grr..."
#include <libsecret/secret.h>

#include "die.h"
#include "secureentry.h"

G_DEFINE_TYPE(SecureEntryBuffer, secure_entry_buffer, GTK_TYPE_ENTRY_BUFFER);

static const gchar *secure_entry_buffer_get_text(GtkEntryBuffer *buf, gsize *len) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    if(len) {
        *len = b->blen;
    }
    if (b->data) {
        return secret_value_get(b->data, NULL);
    } else {
        return NULL;
    }
}

static guint secure_entry_buffer_get_length(GtkEntryBuffer *buf) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    return b->clen;
}

static guint secure_entry_buffer_insert_text(GtkEntryBuffer *buf, guint pos, const gchar *chars, guint n_chars) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);

    gsize len = g_utf8_offset_to_pointer(chars, n_chars) - chars;
    while(len + b->blen + 1 > b->datalen) {
        if(b->datalen == 0) b->datalen = 16;
        b->datalen *= 2;

        // This is a bit annoying: we need to pass in an insecure buffer of the
        // appropriate length instead of passing NULL, for it to uselessly
        // memcpy from.
        gchar *dummy = g_malloc(b->datalen);
        SecretValue *v = secret_value_new(dummy, b->datalen, "text/plain; charset=utf-8");
        g_free(dummy);

        if (!v) {
            die("Could not allocate %ld bytes of secure memory", b->datalen);
        }
        if (b->data) {
            memcpy((gchar *)secret_value_get(v, NULL), secret_value_get(b->data, NULL), b->blen);
            secret_value_unref(b->data);
        }
        b->data = v;
    }

    gchar *data = (gchar *)secret_value_get(b->data, NULL);
    gsize at = g_utf8_offset_to_pointer(data, pos) - data;
    g_memmove(data+at+len, data+at, b->blen-at);
    memcpy(data+at, chars, len);

    b->blen += len;
    b->clen += n_chars;
    data[b->blen] = '\0';

    gtk_entry_buffer_emit_inserted_text(buf, pos, chars, n_chars);
    return n_chars;
}

static guint secure_entry_buffer_delete_text(GtkEntryBuffer *buf, guint pos, guint n_chars) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);

    if(pos > b->clen) pos=b->clen;
    if(pos+n_chars > b->clen) n_chars=b->clen - pos;
    if(n_chars > 0) {
        gchar *data = (gchar *)secret_value_get(b->data, NULL);
        gsize start = g_utf8_offset_to_pointer(data, pos) - data;
        gsize end = g_utf8_offset_to_pointer(data, pos+n_chars) - data;
        g_memmove(data+start, data+end, b->blen+1-end);
        b->clen -= n_chars;
        b->blen -= end - start;
        gtk_entry_buffer_emit_deleted_text(buf, pos, n_chars);
    }

    return n_chars;
}

static void secure_entry_buffer_init(SecureEntryBuffer *b) {
    b->data = NULL;
    b->allow_insecure_memory = FALSE;
    b->datalen = 0;
    b->blen = 0;
    b->clen = 0;
}

static void secure_entry_buffer_finalize(GObject *o) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(o);
    if(b->data) secret_value_unref(b->data);
    b->data = NULL;
    b->datalen = 0;
    G_OBJECT_CLASS(secure_entry_buffer_parent_class)->finalize(o);
}

static void secure_entry_buffer_class_init(SecureEntryBufferClass *c) {
    GObjectClass *go = G_OBJECT_CLASS(c);
    go->finalize = secure_entry_buffer_finalize;

    GtkEntryBufferClass *bc = GTK_ENTRY_BUFFER_CLASS(c);
    bc->get_text = secure_entry_buffer_get_text;
    bc->get_length = secure_entry_buffer_get_length;
    bc->insert_text = secure_entry_buffer_insert_text;
    bc->delete_text = secure_entry_buffer_delete_text;
}

GtkEntryBuffer *secure_entry_buffer_new(void) {
    return g_object_new(TYPE_SECURE_ENTRY_BUFFER, NULL);
}

gboolean secure_entry_buffer_get_allow_insecure_memory(GtkEntryBuffer *buf) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    return b->allow_insecure_memory;
}

void secure_entry_buffer_set_allow_insecure_memory(GtkEntryBuffer *buf, gboolean allow) {
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    b->allow_insecure_memory = allow;
}

SecretValue *secure_entry_buffer_get_secret_value(SecureEntryBuffer *buf) {
    return buf->data;
}

