#include "config.h"

#include <gtk/gtk.h>

#include "wmpasman.h"
#include "clipboard.h"
#include "keyring.h"
#include "die.h"

static GRecMutex mutex;
static GtkClipboard *clipboard, *primary;
static gboolean clipboard_owned, primary_owned;

void clipboard_init(void) {
    g_rec_mutex_init(&mutex);

    g_rec_mutex_lock(&mutex);
    clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    primary = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
    clipboard_owned = FALSE;
    primary_owned = FALSE;
    g_rec_mutex_unlock(&mutex);

    if (!clipboard || !primary) {
        die("Could not access clipboard");
    }
}

static gboolean app_timeout(gpointer p G_GNUC_UNUSED) {
    app_deactivate();
    return FALSE;
}

static void clipboard_get(GtkClipboard *clip G_GNUC_UNUSED, GtkSelectionData *selection_data, guint info G_GNUC_UNUSED, gpointer d G_GNUC_UNUSED) {
    warn(DEBUG_DEBUG, "clipboard_get, appstate=%d, clipboard=%s", appstate, clip == clipboard ? "clipboard" : clip == primary ? "primary" : "unknown");
    if (appstate == ACTIVE) {
        password_to_selection(selection_data);
    }
    g_timeout_add(100, app_timeout, NULL);
}

static void clipboard_cleared(GtkClipboard *clip, gpointer data G_GNUC_UNUSED) {
    g_rec_mutex_lock(&mutex);
    if (clip == clipboard) {
        clipboard_owned = FALSE;
    }
    if (clip == primary) {
        primary_owned = FALSE;
    }
    g_rec_mutex_unlock(&mutex);
    app_deactivate();
}

gboolean clipboard_activate(GError **err) {
    g_warn_if_fail(err == NULL || *err == NULL);

    static const GtkTargetEntry targets[] = {
        { "STRING", 0, 1 },
        { "TEXT", 0, 2 }, 
        { "COMPOUND_TEXT", 0, 3 },
        { "UTF8_STRING", 0, 4 }
    };

    g_rec_mutex_lock(&mutex);

    if (!gtk_clipboard_set_with_data(primary, targets, G_N_ELEMENTS(targets), &clipboard_get, &clipboard_cleared, NULL)) {
        g_set_error(err, APP_GENERIC_ERROR, 0, "Failed to obtain primary selection");
        g_rec_mutex_unlock(&mutex);
        return FALSE;
    }

    if (!gtk_clipboard_set_with_data(clipboard, targets, G_N_ELEMENTS(targets), &clipboard_get, &clipboard_cleared, NULL)) {
        g_set_error(err, APP_GENERIC_ERROR, 0, "Failed to obtain clipboard selection");
        gtk_clipboard_clear(primary);
        g_rec_mutex_unlock(&mutex);
        return FALSE;
    }

    primary_owned = TRUE;
    clipboard_owned = TRUE;

    g_rec_mutex_unlock(&mutex);

    return TRUE;
}

void clipboard_deactivate(void) {
    g_rec_mutex_lock(&mutex);
    if (clipboard_owned) {
        gtk_clipboard_clear(clipboard);
        clipboard_owned = FALSE;
    }
    if (primary_owned) {
        gtk_clipboard_clear(primary);
        primary_owned = FALSE;
    }
    g_rec_mutex_unlock(&mutex);
}
