#!/usr/bin/env python

from __future__ import annotations

import argparse
import subprocess
import sys
from pathlib import Path
from typing import Generator


def get_symbols(path: Path) -> Generator[str]:
    for line in subprocess.run(["nm", "-D", "-P", str(path)],
                               stdout=subprocess.PIPE,
                               check=True).stdout.decode().splitlines():
        sym, typ, _ = line.split(" ", 2)
        if typ in ("T", "W"):
            yield sym


def main():
    argp = argparse.ArgumentParser()
    argp.add_argument("--flexiblas",
                      required=True,
                      type=Path,
                      help="Path to flexiblas library")
    argp.add_argument("--lib",
                      required=True,
                      type=Path,
                      action="append",
                      help="BLAS / LAPACK library to process")
    argp.add_argument("--output",
                      required=True,
                      type=Path,
                      help="Output directory")
    args = argp.parse_args()
    args.output.mkdir(exist_ok=True)

    flexiblas_sym = set(get_symbols(args.flexiblas))
    for lib in args.lib:
        lib_sym = set(get_symbols(lib))
        common_sym = lib_sym.intersection(flexiblas_sym)
        missing_sym = lib_sym.difference(flexiblas_sym)
        if missing_sym:
            print(f"{lib} symbols missing from {args.flexiblas}: "
                  f"{sorted(missing_sym)}",
                  file=sys.stderr)
        with open(args.output / f"{lib.name}.o", "w") as f:
            for sym in sorted(common_sym):
                f.write(f"{sym} = 0;\n")


if __name__ == "__main__":
    main()
