/*
 * Copyright (C) 2001-2024 Jacek Sieka, arnetheduck on gmail point com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef DCPLUSPLUS_DCPP_SSL_H
#define DCPLUSPLUS_DCPP_SSL_H

#include <airdcpp/core/header/w.h>
#include <airdcpp/core/header/typedefs.h>

#include <openssl/ssl.h>
#include <vector>
#include <cstdint>

#ifndef SSL_SUCCESS
#define SSL_SUCCESS 1
#endif

namespace dcpp {

namespace ssl {

template<typename T, void (*Release)(T*)>
class scoped_handle {
public:
	explicit scoped_handle(T* t_ = nullptr) : t(t_) { }
	~scoped_handle() { if(t) { Release(t); } }

	operator T*() { return t; }
	operator const T*() const { return t; }

	explicit operator bool() const { return t ? true : false; }

	T* operator->() { return t; }
	const T* operator->() const { return t; }

	void reset(T* t_ = nullptr) { Release(t); t = t_; }

	scoped_handle(scoped_handle&& rhs) noexcept : t(rhs.t) { rhs.t = nullptr; }
	scoped_handle& operator=(scoped_handle&& rhs) noexcept { if(&rhs != this) { t = rhs.t; rhs.t = nullptr; } return *this; }

	scoped_handle(const scoped_handle<T, Release>&) = delete;
	scoped_handle<T, Release>& operator=(const scoped_handle<T, Release>&) = delete;
private:
	T* t;
};

#if OPENSSL_VERSION_NUMBER < 0x30000000L
typedef scoped_handle<::DH, DH_free> DH;
typedef scoped_handle<::DSA, DSA_free> DSA;
typedef scoped_handle<::RSA, RSA_free> RSA;
#endif

using X509 = scoped_handle< ::X509, X509_free>;
using ASN1_INTEGER = scoped_handle< ::ASN1_INTEGER, ASN1_INTEGER_free>;
using BIGNUM = scoped_handle< ::BIGNUM, BN_free>;
using EVP_PKEY = scoped_handle< ::EVP_PKEY, EVP_PKEY_free>;
using SSL = scoped_handle< ::SSL, SSL_free>;
using SSL_CTX = scoped_handle< ::SSL_CTX, SSL_CTX_free>;
using X509 = scoped_handle< ::X509, X509_free>;
using X509_NAME = scoped_handle< ::X509_NAME, X509_NAME_free>;

// functions that operate with file paths - here in order to support UTF16 Windows paths
bool SSL_CTX_use_certificate_file(::SSL_CTX* ctx, const char* file, int type);
bool SSL_CTX_use_PrivateKey_file(::SSL_CTX* ctx, const char* file, int type);

X509 getX509(const char* file);
ByteVector X509_digest(::X509* x509, const ::EVP_MD* md);

}

}

#endif
