/* SPDX-FileCopyrightText: 2024 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import {Date} from 'gmp/models/date';
import Model, {ModelElement, ModelProperties} from 'gmp/models/model';
import {parseDate, parseInt} from 'gmp/parser';
import {isDefined} from 'gmp/utils/identity';

interface VulnerabilityElement extends ModelElement {
  hosts?: {
    count?: number;
  };
  results?: {
    count?: number;
    newest?: string;
    oldest?: string;
  };
  qod?: number;
  severity?: number;
}

interface Hosts {
  count?: number;
}

interface Results {
  count?: number;
  newest?: Date;
  oldest?: Date;
}

interface VulnerabilityProperties extends ModelProperties {
  hosts?: Hosts;
  results?: Results;
  qod?: number;
  severity?: number;
}

class Vulnerability extends Model {
  static entityType = 'vulnerability';

  readonly hosts?: Hosts;
  readonly results?: Results;
  readonly qod?: number;
  readonly severity?: number;

  constructor({
    hosts,
    results,
    qod,
    severity,
    ...properties
  }: VulnerabilityProperties = {}) {
    super(properties);

    this.hosts = hosts;
    this.results = results;
    this.qod = qod;
    this.severity = severity;
  }

  static fromElement(element?: VulnerabilityElement): Vulnerability {
    return new Vulnerability(this.parseElement(element));
  }

  static parseElement(
    element: VulnerabilityElement = {},
  ): VulnerabilityProperties {
    const ret = super.parseElement(element) as VulnerabilityProperties;

    if (isDefined(element.results)) {
      ret.results = {
        newest: parseDate(element.results.newest),
        oldest: parseDate(element.results.oldest),
        count: parseInt(element.results.count),
      };
    }

    if (isDefined(element.hosts)) {
      ret.hosts = {};
      ret.hosts.count = element.hosts.count;
    }

    ret.qod = isDefined(element.qod) ? element.qod : undefined;
    ret.severity = isDefined(element.severity) ? element.severity : undefined;

    return ret;
  }
}

export default Vulnerability;
