/* XML Utilities
 *
 * Copyright (C) 2001, 2002 Binding Time Limited
 * 
 * TERMS AND CONDITIONS:
 *
 * This program is offered free of charge, as unsupported source code. You may
 * use it, copy it, distribute it, modify it or sell it without restriction. 
 * 
 * We hope that it will be useful to you, but it is provided "as is" without
 * any warranty express or implied, including but not limited to the warranty
 * of non-infringement and the implied warranties of merchantability and fitness
 * for a particular purpose.
 * 
 * Binding Time Limited will not be liable for any damages suffered by you as
 * a result of using the Program. In no event will Binding Time Limited be
 * liable for any special, indirect or consequential damages or lost profits
 * even if Binding Time Limited has been advised of the possibility of their
 * occurrence. Binding Time Limited will not be liable for any third party
 * claims against you.
 *
 * History:
 * $Log: not supported by cvs2svn $
 * Revision 1.1  2002/12/06 20:15:39  tswift
 *
 * Initial open-source version
 *
 * Revision 1.1  2002/09/11 19:52:41  warren
 * Files for reading RDF documents into the OMS.
 * Can read rdf directly and create simple relationship triples, or
 * can read daml+oil rdf representations and load the OMS a little more
 * intelligently (but still nowhere near capturing all the semantics of DAML+OIL).
 *
 * Revision 1.2  2002-01-31 21:02:20+00  john
 * Changed to 'canonical' (lowercase) character encoding names.
 *
 * Revision 1.1  2001-11-04 17:26:22+00  john
 * Added uppercase variants of the supported character encoding names.
 *
 * Revision 1.0  2001-10-17 20:46:23+01  john
 * Initial revision
 *
 *
 *
 */

 :- import ground/1 from basics.

% Entity and Namespace map operations: these maps are usually quite small, so a
% linear list lookup is okay. They could be substituted by a logarithmic data
% structure - in extremis.

/* empty_map( ?Map ) is true if Map is a null map.
 */
empty_map( [] ).

/* map_member( +Key, +Map, ?Data ) is true if Map is a ordered map structure
 * which records the pair Key-Data. Key must be ground.
 */
map_member( Key0, [Key1-Data1|Rest], Data0 ) :-
	( Key0 == Key1 ->
		Data0 = Data1
	; Key0 @> Key1 ->
		map_member( Key0, Rest, Data0 )
	).

/* map_store( +Map0, +Key, +Data, ?Map1 ) is true if Map0 is an ordered map
 * structure, Key must be ground, and Map1 is identical to Map0 except that
 * the pair Key-Data is recorded by Map1.
 */
map_store( [], Key, Data, [Key-Data] ).
map_store( [Key0-Data0|Map0], Key, Data, Map ) :-
	( Key == Key0 ->
		Map = [Key-Data|Map0]
	; Key @< Key0 ->
		Map = [Key-Data,Key0-Data0|Map0]
	; otherwise -> % >
		Map = [Key0-Data0|Map1],
		map_store( Map0, Key, Data, Map1 )
	).

/* context(?Element, ?PreserveSpace, ?CurrentNS, ?DefaultNS, ?Entities, ?Namespaces )
 * is an ADT hiding the "state" arguments for XML Acquisition
 */
initial_context(
		Controls,
		context(void,PreserveSpace,'','',Entities,Empty)
		) :-
	empty_map( Empty ),
	( member( extended_characters(false), Controls ) ->
		Entities = Empty
	; otherwise ->
		extended_character_entities(Entities)
	),
	( member( format(false), Controls ) ->
		PreserveSpace = true
	; otherwise ->
		PreserveSpace = false
	).

context_update( current_namespace, Context0, URI, Context1 ) :-
	Context0 = context(Element,Preserve,_Current,Default,Entities,Namespaces),
	Context1 = context(Element,Preserve,URI,Default,Entities,Namespaces).
context_update( element, Context0, Tag, Context1 ) :-
	Context0 = context(_Element,Preserve,Current,Default,Entities,Namespaces),
	Context1 = context(tag(Tag),Preserve,Current,Default,Entities,Namespaces).
context_update( default_namespace, Context0, URI, Context1 ):-
	Context0 = context(Element,Preserve,Current,_Default,Entities,Namespaces),
	Context1 = context(Element,Preserve,Current,URI,Entities,Namespaces).
context_update( space_preserve, Context0, Boolean, Context1 ):-
	Context0 = context(Element,_Preserve,Current,Default,Entities,Namespaces),
	Context1 = context(Element,Boolean,Current,Default,Entities,Namespaces).
context_update( ns_prefix(Prefix), Context0, URI, Context1 ) :-
	Context0 = context(Element,Preserve,Current,Default,Entities,Namespaces0),
	Context1 = context(Element,Preserve,Current,Default,Entities,Namespaces1),
	map_store( Namespaces0, Prefix, URI, Namespaces1 ).
context_update( entity(Name), Context0, String, Context1 ) :-
	Context0 = context(Element,Preserve,Current,Default,Entities0,Namespaces),
	Context1 = context(Element,Preserve,Current,Default,Entities1,Namespaces),
	map_store( Entities0, Name, String, Entities1 ).

current_tag( Context, Tag ) :-
	Context = context(tag(Tag),_Preserve,_Current,_Default,_Entities,_Namespaces).

current_namespace( Context, Current ) :-
	Context = context(_Element,_Preserve,Current,_Default,_Entities,_Namespaces).

default_namespace( Context, Default ) :-
	Context = context(_Element,_Preserve,_Current,Default,_Entities,_Namespaces).

space_preserve( Context ) :-
	Context = context(_Element,true,_Current,_Default,_Entities,_Namespaces).

specific_namespace( Prefix, Context, URI ) :-
	Context = context(_Element,_Preserve,_Current,_Default,_Entities,Namespaces),
	map_member( Prefix, Namespaces, URI ).

defined_entity( Reference, Context, String ) :-
	Context = context(_Element,_Preserve,_Current,_Default,Entities,_Namespaces),
	map_member( Reference, Entities, String ).
	
close_context( Context, Terms, WellFormed ) :-
	Context = context(Element,_Preserve,_Current,_Default,_Entities,_Namespaces),
	close_context1( Element, Terms, WellFormed ).

close_context1( void, [], true ).
close_context1( tag(TagChars), [out_of_context(Tag)], false ) :-
%%	atom_chars( Tag, TagChars ).
	atom_codes( Tag, TagChars ).

void_context(
	context(void,_Preserve,_Current,_Default,_Entities,_Namespaces)
	).

/* pp_string( +String ) prints String onto the current output stream.
 * If String contains only 7-bit chars it is printed in shorthand quoted
 * format, otherwise it is written as a list.
 * If your Prolog uses " to delimit a special string type, just use write/1.
 */
pp_string( Chars ) :-
	( member( Char, Chars ),
	  Char > 255 ->
		write( Chars )
	; otherwise ->
		put_quote,
		pp_string1( Chars ),
		put_quote
	).

pp_string1( [] ).
pp_string1( [Char|Chars] ) :-
	( Char =:= 0'"  -> put_quote
	; true
	),
	put( Char ),
	pp_string1( Chars ).

put_quote :-
	put( 0'" ). % "

xml_declaration_attributes_valid( [] ).
xml_declaration_attributes_valid( [Name=Value|Attributes] ) :-
	xml_declaration_attribute_valid( Name, Value ),
	xml_declaration_attributes_valid( Attributes ).

xml_declaration_attribute_valid( Name, Value ) :-
	lowercase( Value, Lowercase ),
	canonical_xml_declaration_attribute( Name, Lowercase ).

canonical_xml_declaration_attribute( version, "1.0" ).
canonical_xml_declaration_attribute( standalone, "yes" ).
canonical_xml_declaration_attribute( standalone, "no" ).
canonical_xml_declaration_attribute( encoding, "utf-8" ).
canonical_xml_declaration_attribute( encoding, "utf-16" ).
canonical_xml_declaration_attribute( encoding, "ascii" ).
canonical_xml_declaration_attribute( encoding, "iso-8859-1" ).
canonical_xml_declaration_attribute( encoding, "iso-8859-2" ).
canonical_xml_declaration_attribute( encoding, "iso-8859-15" ).
canonical_xml_declaration_attribute( encoding, "windows-1252" ).

/* lowercase( +MixedCase, ?Lowercase ) holds when Lowercase and MixedCase are
 * lists of character codes, and Lowercase is identical to MixedCase with
 * every uppercase character replaced by its lowercase equivalent.
 */
lowercase( [], [] ).
lowercase( [Char|Chars], [Lower|LowerCase] ) :-
	( Char >= 0'A, Char =< 0'Z ->
		Lower is Char + 0'a - 0'A
	; otherwise ->
		Lower = Char
	),
	lowercase( Chars, LowerCase ).

extended_character_entities( Entities ) :-
	findall(Entity,(extended_char_entity(Entity);
			extended_char_entity2(Entity)),Entities).

extended_char_entity("Aacute"-[193]).		% latin capital letter A with acute,
extended_char_entity("aacute"-[225]).		% latin small letter a with acute,
extended_char_entity("Acirc"-[194]).		% latin capital letter A with circumflex,
extended_char_entity("acirc"-[226]).		% latin small letter a with circumflex,
extended_char_entity("acute"-[180]).		% acute accent = spacing acute,
extended_char_entity("AElig"-[198]).		% latin capital letter AE
extended_char_entity("aelig"-[230]).		% latin small letter ae
extended_char_entity("Agrave"-[192]).		% latin capital letter A with grave
extended_char_entity("agrave"-[224]).		% latin small letter a with grave
extended_char_entity("alefsym"-[8501]).	% alef symbol = first transfinite cardinal,
extended_char_entity("Alpha"-[913]).		% greek capital letter alpha, U+0391
extended_char_entity("alpha"-[945]).		% greek small letter alpha,
extended_char_entity("and"-[8743]).		% logical and = wedge, U+2227 ISOtech
extended_char_entity("ang"-[8736]).		% angle, U+2220 ISOamso
extended_char_entity("Aring"-[197]).		% latin capital letter A with ring above
extended_char_entity("aring"-[229]).		% latin small letter a with ring above
extended_char_entity("asymp"-[8776]).		% almost equal to = asymptotic to,
extended_char_entity("Atilde"-[195]).		% latin capital letter A with tilde,
extended_char_entity("atilde"-[227]).		% latin small letter a with tilde,
extended_char_entity("Auml"-[196]).		% latin capital letter A with diaeresis,
extended_char_entity("auml"-[228]).		% latin small letter a with diaeresis,
extended_char_entity("bdquo"-[8222]).		% double low-9 quotation mark, U+201E NEW
extended_char_entity("Beta"-[914]).		% greek capital letter beta, U+0392
extended_char_entity("beta"-[946]).		% greek small letter beta, U+03B2 ISOgrk3
extended_char_entity("brvbar"-[166]).		% broken bar = broken vertical bar,
extended_char_entity("bull"-[8226]).		% bullet = black small circle,
extended_char_entity("cap"-[8745]).		% intersection = cap, U+2229 ISOtech
extended_char_entity("Ccedil"-[199]).		% latin capital letter C with cedilla,
extended_char_entity("ccedil"-[231]).		% latin small letter c with cedilla,
extended_char_entity("cedil"-[184]).		% cedilla = spacing cedilla, U+00B8 ISOdia>
extended_char_entity("cent"-[162]).		% cent sign, U+00A2 ISOnum>
extended_char_entity("Chi"-[935]).		% greek capital letter chi, U+03A7
extended_char_entity("chi"-[967]).		% greek small letter chi, U+03C7 ISOgrk3
extended_char_entity("circ"-[710]).		% modifier letter circumflex accent,
extended_char_entity("clubs"-[9827]).		% black club suit = shamrock,
extended_char_entity("cong"-[8773]).		% approximately equal to, U+2245 ISOtech
extended_char_entity("copy"-[169]).		% copyright sign, U+00A9 ISOnum>
extended_char_entity("crarr"-[8629]).		% downwards arrow with corner leftwards
extended_char_entity("cup"-[8746]).		% union = cup, U+222A ISOtech
extended_char_entity("curren"-[164]).		% currency sign, U+00A4 ISOnum>
extended_char_entity("dagger"-[8224]).	% dagger, U+2020 ISOpub
extended_char_entity("Dagger"-[8225]).	% double dagger, U+2021 ISOpub
extended_char_entity("darr"-[8595]).		% downwards arrow, U+2193 ISOnum
extended_char_entity("dArr"-[8659]).		% downwards double arrow, U+21D3 ISOamsa
extended_char_entity("deg"-[176]).		% degree sign, U+00B0 ISOnum>
extended_char_entity("Delta"-[916]).		% greek capital letter delta,
extended_char_entity("delta"-[948]).		% greek small letter delta,
extended_char_entity("diams"-[9830]).		% black diamond suit, U+2666 ISOpub
extended_char_entity("divide"-[247]).		% division sign, U+00F7 ISOnum>
extended_char_entity("Eacute"-[201]).		% latin capital letter E with acute,
extended_char_entity("eacute"-[233]).		% latin small letter e with acute,
extended_char_entity("Ecirc"-[202]).		% latin capital letter E with circumflex,
extended_char_entity("ecirc"-[234]).		% latin small letter e with circumflex,
extended_char_entity("Egrave"-[200]).		% latin capital letter E with grave,
extended_char_entity("egrave"-[232]).		% latin small letter e with grave,
extended_char_entity("empty"-[8709]).		% empty set = null set = diameter,
extended_char_entity("emsp"-[8195]).		% em space, U+2003 ISOpub
extended_char_entity("ensp"-[8194]).		% en space, U+2002 ISOpub
extended_char_entity("Epsilon"-[917]).	% greek capital letter epsilon, U+0395
extended_char_entity("epsilon"-[949]).	% greek small letter epsilon,
extended_char_entity("equiv"-[8801]).		% identical to, U+2261 ISOtech
extended_char_entity("Eta"-[919]).		% greek capital letter eta, U+0397
extended_char_entity("eta"-[951]).		% greek small letter eta, U+03B7 ISOgrk3
extended_char_entity("ETH"-[208]).		% latin capital letter ETH, U+00D0 ISOlat1>
extended_char_entity("eth"-[240]).		% latin small letter eth, U+00F0 ISOlat1>
extended_char_entity("Euml"-[203]).		% latin capital letter E with diaeresis,
extended_char_entity("euml"-[235]).		% latin small letter e with diaeresis,
extended_char_entity("euro"-[8364]).		% euro sign, U+20AC NEW
extended_char_entity("exist"-[8707]).		% there exists, U+2203 ISOtech
extended_char_entity("fnof"-[402]).		% latin small f with hook = function
extended_char_entity("forall"-[8704]).	% for all, U+2200 ISOtech
extended_char_entity("frac12"-[189]).		% vulgar fraction one half
extended_char_entity("frac14"-[188]).		% vulgar fraction one quarter
extended_char_entity("frac34"-[190]).		% vulgar fraction three quarters
extended_char_entity("frasl"-[8260]).		% fraction slash, U+2044 NEW
extended_char_entity("Gamma"-[915]).		% greek capital letter gamma,
extended_char_entity("gamma"-[947]).		% greek small letter gamma,
extended_char_entity("ge"-[8805]).		% greater-than or equal to,
extended_char_entity("harr"-[8596]).		% left right arrow, U+2194 ISOamsa
extended_char_entity("hArr"-[8660]).		% left right double arrow,
extended_char_entity("hearts"-[9829]).	% black heart suit = valentine,
extended_char_entity("hellip"-[8230]).	% horizontal ellipsis = three dot leader,
extended_char_entity("Iacute"-[205]).		% latin capital letter I with acute,
extended_char_entity("iacute"-[237]).		% latin small letter i with acute,
extended_char_entity("Icirc"-[206]).		% latin capital letter I with circumflex,
extended_char_entity("icirc"-[238]).		% latin small letter i with circumflex,
extended_char_entity("iexcl"-[161]).		% inverted exclamation mark, U+00A1 ISOnum>
extended_char_entity("Igrave"-[204]).		% latin capital letter I with grave,
extended_char_entity("igrave"-[236]).		% latin small letter i with grave,
extended_char_entity("image"-[8465]).		% blackletter capital I = imaginary part,
extended_char_entity("infin"-[8734]).		% infinity, U+221E ISOtech
extended_char_entity("int"-[8747]).		% integral, U+222B ISOtech
extended_char_entity("Iota"-[921]).		% greek capital letter iota, U+0399
extended_char_entity("iota"-[953]).		% greek small letter iota, U+03B9 ISOgrk3
extended_char_entity("iquest"-[191]).		% inverted question mark
extended_char_entity("isin"-[8712]).		% element of, U+2208 ISOtech
extended_char_entity("Iuml"-[207]).		% latin capital letter I with diaeresis,
extended_char_entity("iuml"-[239]).		% latin small letter i with diaeresis,
extended_char_entity("Kappa"-[922]).		% greek capital letter kappa, U+039A
extended_char_entity("kappa"-[954]).		% greek small letter kappa,
extended_char_entity("Lambda"-[923]).		% greek capital letter lambda,
extended_char_entity("lambda"-[955]).		% greek small letter lambda,
extended_char_entity("lang"-[9001]).		% left-pointing angle bracket = bra,
extended_char_entity("laquo"-[171]).		% left-pointing double angle quotation mark
extended_char_entity("larr"-[8592]).		% leftwards arrow, U+2190 ISOnum
extended_char_entity("lArr"-[8656]).		% leftwards double arrow, U+21D0 ISOtech
extended_char_entity("lceil"-[8968]).		% left ceiling = apl upstile,
extended_char_entity("ldquo"-[8220]).		% left double quotation mark,
extended_char_entity("le"-[8804]).		% less-than or equal to, U+2264 ISOtech
extended_char_entity("lfloor"-[8970]).	% left floor = apl downstile,
extended_char_entity("lowast"-[8727]).	% asterisk operator, U+2217 ISOtech
extended_char_entity("loz"-[9674]).		% lozenge, U+25CA ISOpub
extended_char_entity("lrm"-[8206]).		% left-to-right mark, U+200E NEW RFC 2070
extended_char_entity("lsaquo"-[8249]).	% single left-pointing angle quotation mark,
extended_char_entity("lsquo"-[8216]).		% left single quotation mark,
extended_char_entity("macr"-[175]).		% macron = spacing macron = overline
extended_char_entity("mdash"-[8212]).		% em dash, U+2014 ISOpub
extended_char_entity("micro"-[181]).		% micro sign, U+00B5 ISOnum>
extended_char_entity("middot"-[183]).		% middle dot = Georgian comma
extended_char_entity("minus"-[8722]).		% minus sign, U+2212 ISOtech
extended_char_entity("Mu"-[924]).			% greek capital letter mu, U+039C
extended_char_entity("mu"-[956]).			% greek small letter mu, U+03BC ISOgrk3
extended_char_entity("nabla"-[8711]).		% nabla = backward difference,
extended_char_entity("nbsp"-[160]).		% no-break space = non-breaking space,
extended_char_entity("ndash"-[8211]).		% en dash, U+2013 ISOpub
extended_char_entity("ne"-[8800]).		% not equal to, U+2260 ISOtech
extended_char_entity("ni"-[8715]).		% contains as member, U+220B ISOtech
extended_char_entity("not"-[172]).		% not sign, U+00AC ISOnum>
extended_char_entity("notin"-[8713]).		% not an element of, U+2209 ISOtech
extended_char_entity("nsub"-[8836]).		% not a subset of, U+2284 ISOamsn
extended_char_entity("Ntilde"-[209]).		% latin capital letter N with tilde,
extended_char_entity("ntilde"-[241]).		% latin small letter n with tilde,
extended_char_entity("Nu"-[925]).			% greek capital letter nu, U+039D
extended_char_entity("nu"-[957]).			% greek small letter nu, U+03BD ISOgrk3
extended_char_entity("Oacute"-[211]).		% latin capital letter O with acute,
extended_char_entity("oacute"-[243]).		% latin small letter o with acute,
extended_char_entity("Ocirc"-[212]).		% latin capital letter O with circumflex,
extended_char_entity("ocirc"-[244]).		% latin small letter o with circumflex,
extended_char_entity("OElig"-[338]).		% latin capital ligature OE,
extended_char_entity("oelig"-[339]).		% latin small ligature oe, U+0153 ISOlat2
extended_char_entity("Ograve"-[210]).		% latin capital letter O with grave,
extended_char_entity("ograve"-[242]).		% latin small letter o with grave,
extended_char_entity("oline"-[8254]).		% overline = spacing overscore,
extended_char_entity("Omega"-[937]).		% greek capital letter omega,
extended_char_entity("omega"-[969]).		% greek small letter omega,
extended_char_entity("Omicron"-[927]).	% greek capital letter omicron, U+039F
extended_char_entity("omicron"-[959]).	% greek small letter omicron, U+03BF NEW
extended_char_entity("oplus"-[8853]).		% circled plus = direct sum,
extended_char_entity("or"-[8744]).		% logical or = vee, U+2228 ISOtech
extended_char_entity("ordf"-[170]).		% feminine ordinal indicator, U+00AA ISOnum>
extended_char_entity("ordm"-[186]).		% masculine ordinal indicator,
extended_char_entity("Oslash"-[216]).		% latin capital letter O with stroke
extended_char_entity("oslash"-[248]).		% latin small letter o with stroke,
extended_char_entity("Otilde"-[213]).		% latin capital letter O with tilde,
extended_char_entity("otilde"-[245]).		% latin small letter o with tilde,
extended_char_entity("otimes"-[8855]).	% circled times = vector product,
extended_char_entity("Ouml"-[214]).		% latin capital letter O with diaeresis,
extended_char_entity("ouml"-[246]).		% latin small letter o with diaeresis,
extended_char_entity("para"-[182]).		% pilcrow sign = paragraph sign,
extended_char_entity("part"-[8706]).		% partial differential, U+2202 ISOtech
extended_char_entity("permil"-[8240]).	% per mille sign, U+2030 ISOtech
extended_char_entity("perp"-[8869]).		% up tack = orthogonal to = perpendicular,
extended_char_entity("Phi"-[934]).		% greek capital letter phi,
extended_char_entity("phi"-[966]).		% greek small letter phi, U+03C6 ISOgrk3
extended_char_entity("Pi"-[928]).			% greek capital letter pi, U+03A0 ISOgrk3
extended_char_entity("pi"-[960]).			% greek small letter pi, U+03C0 ISOgrk3
extended_char_entity("piv"-[982]).		% greek pi symbol, U+03D6 ISOgrk3
extended_char_entity("plusmn"-[177]).		% plus-minus sign = plus-or-minus sign,
extended_char_entity("pound"-[163]).		% pound sign, U+00A3 ISOnum>
extended_char_entity("prime"-[8242]).		% prime = minutes = feet, U+2032 ISOtech
extended_char_entity("Prime"-[8243]).		% double prime = seconds = inches,
extended_char_entity("prod"-[8719]).		% n-ary product = product sign,
extended_char_entity("prop"-[8733]).		% proportional to, U+221D ISOtech
extended_char_entity("Psi"-[936]).		% greek capital letter psi,
extended_char_entity("psi"-[968]).		% greek small letter psi, U+03C8 ISOgrk3
extended_char_entity("radic"-[8730]).		% square root = radical sign,
extended_char_entity("rang"-[9002]).		% right-pointing angle bracket = ket,
extended_char_entity("raquo"-[187]).		% right-pointing double angle quotation mark
extended_char_entity("rarr"-[8594]).		% rightwards arrow, U+2192 ISOnum
extended_char_entity("rArr"-[8658]).		% rightwards double arrow,
extended_char_entity("rceil"-[8969]).		% right ceiling, U+2309 ISOamsc
extended_char_entity("rdquo"-[8221]).		% right double quotation mark,
extended_char_entity("real"-[8476]).		% blackletter capital R = real part symbol,
extended_char_entity("reg"-[174]).		% registered sign = registered trade mark sign,
extended_char_entity("rfloor"-[8971]).	% right floor, U+230B ISOamsc
extended_char_entity("Rho"-[929]).		% greek capital letter rho, U+03A1
extended_char_entity("rho"-[961]).		% greek small letter rho, U+03C1 ISOgrk3
extended_char_entity("rlm"-[8207]).		% right-to-left mark, U+200F NEW RFC 2070
extended_char_entity("rsaquo"-[8250]).	% single right-pointing angle quotation mark,
extended_char_entity("rsquo"-[8217]).		% right single quotation mark,
extended_char_entity("sbquo"-[8218]).		% single low-9 quotation mark, U+201A NEW
extended_char_entity("Scaron"-[352]).		% latin capital letter S with caron,


extended_char_entity2("scaron"-[353]).		% latin small letter s with caron,
extended_char_entity2("sdot"-[8901]).		% dot operator, U+22C5 ISOamsb
extended_char_entity2("sect"-[167]).		% section sign, U+00A7 ISOnum>
extended_char_entity2("shy"-[173]).		% soft hyphen = discretionary hyphen,
extended_char_entity2("Sigma"-[931]).		% greek capital letter sigma,
extended_char_entity2("sigma"-[963]).		% greek small letter sigma,
extended_char_entity2("sigmaf"-[962]).		% greek small letter final sigma,
extended_char_entity2("sim"-[8764]).		% tilde operator = varies with = similar to,
extended_char_entity2("spades"-[9824]).	% black spade suit, U+2660 ISOpub
extended_char_entity2("sub"-[8834]).		% subset of, U+2282 ISOtech
extended_char_entity2("sube"-[8838]).		% subset of or equal to, U+2286 ISOtech
extended_char_entity2("sum"-[8721]).		% n-ary sumation, U+2211 ISOamsb
extended_char_entity2("sup"-[8835]).		% superset of, U+2283 ISOtech
extended_char_entity2("sup1"-[185]).		% superscript one = superscript digit one,
extended_char_entity2("sup2"-[178]).		% superscript two = superscript digit two
extended_char_entity2("sup3"-[179]).		% superscript three = superscript digit three
extended_char_entity2("supe"-[8839]).		% superset of or equal to,
extended_char_entity2("szlig"-[223]).		% latin small letter sharp s = ess-zed,
extended_char_entity2("Tau"-[932]).		% greek capital letter tau, U+03A4
extended_char_entity2("tau"-[964]).		% greek small letter tau, U+03C4 ISOgrk3
extended_char_entity2("there4"-[8756]).	% therefore, U+2234 ISOtech
extended_char_entity2("Theta"-[920]).		% greek capital letter theta,
extended_char_entity2("theta"-[952]).		% greek small letter theta,
extended_char_entity2("thetasym"-[977]).	% greek small letter theta symbol,
extended_char_entity2("thinsp"-[8201]).	% thin space, U+2009 ISOpub
extended_char_entity2("THORN"-[222]).		% latin capital letter THORN,
extended_char_entity2("thorn"-[254]).		% latin small letter thorn with,
extended_char_entity2("tilde"-[732]).		% small tilde, U+02DC ISOdia
extended_char_entity2("times"-[215]).		% multiplication sign, U+00D7 ISOnum>
extended_char_entity2("trade"-[8482]).		% trade mark sign, U+2122 ISOnum
extended_char_entity2("Uacute"-[218]).		% latin capital letter U with acute,
extended_char_entity2("uacute"-[250]).		% latin small letter u with acute,
extended_char_entity2("uarr"-[8593]).		% upwards arrow, U+2191 ISOnum
extended_char_entity2("uArr"-[8657]).		% upwards double arrow, U+21D1 ISOamsa
extended_char_entity2("Ucirc"-[219]).		% latin capital letter U with circumflex,
extended_char_entity2("ucirc"-[251]).		% latin small letter u with circumflex,
extended_char_entity2("Ugrave"-[217]).		% latin capital letter U with grave,
extended_char_entity2("ugrave"-[249]).		% latin small letter u with grave,
extended_char_entity2("uml"-[168]).		% diaeresis = spacing diaeresis,
extended_char_entity2("upsih"-[978]).		% greek upsilon with hook symbol,
extended_char_entity2("Upsilon"-[933]).	% greek capital letter upsilon,
extended_char_entity2("upsilon"-[965]).	% greek small letter upsilon,
extended_char_entity2("Uuml"-[220]).		% latin capital letter U with diaeresis,
extended_char_entity2("uuml"-[252]).		% latin small letter u with diaeresis,
extended_char_entity2("weierp"-[8472]).	% script capital P = power set
extended_char_entity2("Xi"-[926]).			% greek capital letter xi, U+039E ISOgrk3
extended_char_entity2("xi"-[958]).			% greek small letter xi, U+03BE ISOgrk3
extended_char_entity2("Yacute"-[221]).		% latin capital letter Y with acute,
extended_char_entity2("yacute"-[253]).		% latin small letter y with acute,
extended_char_entity2("yen"-[165]).		% yen sign = yuan sign, U+00A5 ISOnum>
extended_char_entity2("yuml"-[255]).		% latin small letter y with diaeresis,
extended_char_entity2("Yuml"-[376]).		% latin capital letter Y with diaeresis,
extended_char_entity2("Zeta"-[918]).		% greek capital letter zeta, U+0396
extended_char_entity2("zeta"-[950]).		% greek small letter zeta, U+03B6 ISOgrk3
extended_char_entity2("zwj"-[8205]).		% zero width joiner, U+200D NEW RFC 2070
extended_char_entity2("zwnj"-[8204]).            % zero width non-joiner,

% The following code is for Quintus Prolog primarily. Some of these
% predicates are built-in to SWI, LPA etc.

/* member( ?Element, ?List ) holds when Element is a member of List.
 */
member( H, [H|_] ).
member( H, [_|T] ):-
    member( H, T ).

/* select( ?Element, ?List0, ?List1 ) is true if List1 is equal to List0
 * with Element removed.
 */
select( H, [H|T], T ).
select( Element, [H|T0], [H|T1] ):-
    select( Element, T0, T1 ).

/* is_list( +List ) holds when List is a list.
 */
is_list( List ) :-
	nonvar( List ),
	is_list1( List ).

is_list1( [] ).
is_list1( [_|_] ).

/* atom_codes/2, number_codes/2 and throw/1 are ISO predicates.

atom_codes( Atom, Codes ) :-
	atom_chars( Atom, Codes ).

number_codes( Number, Codes ) :-
	number_chars( Number, Codes ).

throw( Exception ) :-
	raise_exception( Exception ).

 */