/*
 * pcap.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <pcap.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <string.h>

#include "packet.h"

pcap_if_t *interfaces = NULL;
pcap_t *capt_ifs[255];

enum listen_mode {
    all,
    single
};


struct ifconf ifc;

extern void quit(const char *);

pcap_t *
start_pcap(char *cf, char *dev, char *filter, int promisc)
{
    pcap_t *p;
    char err[PCAP_ERRBUF_SIZE];
    struct bpf_program bpf;
    bpf_u_int32 netp=0, maskp=0;

    memset(err, 0, sizeof(err));
    memset(&bpf, 0, sizeof(bpf));

    if (!cf) {

        if (dev && dev[0])
            pcap_lookupnet(dev, &netp, &maskp, err);
        else {
            if ( !(dev = pcap_lookupdev(err)) )
                quit(err);
        }

        if ( !(p = pcap_open_live(dev, 8192, promisc, -1, err)) )
            quit(err);

    } else {
        if ( !(p = pcap_open_offline(cf, err)) )
            quit(err);
    }

    if (filter) {
        if (pcap_compile(p, &bpf, filter, 0, netp) < 0)
            quit(pcap_geterr(p));

        if (pcap_setfilter(p, &bpf) < 0)
            quit(pcap_geterr(p));
    }

    pcap_setnonblock(p, 1, NULL);

    return p;
}

void
stop_pcap(pcap_t *p)
{
    if (p)
        pcap_close(p);
}

void
pcap_listen_on_all(int promisc)
{
    char err[PCAP_ERRBUF_SIZE];
    pcap_if_t *ift;
    int i = 0;

    memset(&capt_ifs, 0, sizeof(capt_ifs));

    if (pcap_findalldevs(&interfaces, err) < 0)
        quit(err);

    ift = interfaces;
    while (ift) {
        if (i >= 255)
            break;
        
        capt_ifs[i++] = start_pcap(NULL, ift->name, NULL, promisc);
        ift = ift->next;
    }
}

void
pcap_listen_on_single(char *dev, int promisc)
{
    memset(&capt_ifs, 0, sizeof(capt_ifs));
    capt_ifs[0] = start_pcap(NULL, dev, NULL, promisc);
}

void
pcap_stop_all(void)
{
    int i;

    for (i=0; i < 255; i++)
        stop_pcap(capt_ifs[i]);
}

void
pcap_handle(void)
{
    struct pcap_pkthdr hdr;
    const u_char *pkt=NULL;
    int i;

    for (i=0; capt_ifs[i] && i < 255; i++) {
        for (;;) {
            memset(&hdr, 0, sizeof(hdr));

            if ( (pkt = pcap_next(capt_ifs[i], &hdr)) )
                look_at_packet(NULL, &hdr, pkt);
            else
                break;
        }
    }
}
