/*
   touchcal - a tool to calibrate Touch Screens (namely ones with serial 
   controller manufactured by EloGraphics and MicroTouch) for use with XFree86.

   Copyright (C) 1999, 2002, 2004 Christoph Baumann <cgb@debian.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


   The author wishes to thank Peter Baumann <Peter.Baumann@dlr.de> for the
   Linux Serial Programming HOWTO which was very helpfull during writing
   this program.

   Elo init code based on code from xf86-input-elographics, copied with
   permission from Patrick Lecoanet.
   Copyright 1995, 1999 by Patrick Lecoanet, France. <lecoanet@cena.dgac.fr>

*/

#define _POSIX_SOURCE 1		/* POSIX compliant source */
#define _GNU_SOURCE

#ifdef HAVE_CONFIG_H
#   include <config.h>
#else
    /* If config.h is NOT present/used set default values here */

    /* If there is no keyboard present set this to 0 */
#   define HAVE_USER_ENTER 1
    /* min. difference of min. and max. values for plausibilty check */
#   define MAX_MIN_DIFF 2000
    /* The default file name, where min / max information is saved  */
#   define SAVE_TO_FILE_DEFAULT "/tmp/touchcal.tmp"
    /* define port parameters */
#   define BAUDRATE B9600
    /* uncomment to enable debug output */
/* #   define DEBUG_MAX_KOD */
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>


#ifdef HAVE_STDLIB_H
#   include <stdlib.h>
#endif

/*curses/ncurses library*/
#ifdef HAVE_LIBNCURSES
#   include <curses.h>
#else
#   ifdef HAVE_LIBCURSES
#       include <curses/curses.h>
#       ifndef getmaxx
#           define getmaxx(arg_) (arg_)->maxx
#           define getmaxy(arg_) (arg_)->maxy
#       endif
#   endif
#endif

#if defined(HAVE_LIBX11) && defined(HAVE_LIBXFT) && defined(HAVE_LIBXINERAMA)
  #define HAVE_X11
#endif

#ifdef HAVE_X11
#include <X11/Xlib.h>
#include <X11/extensions/Xinerama.h>
#include <X11/Xft/Xft.h>
#endif

/*tool version information*/
char *version = PACKAGE_VERSION;

/* Elo stuff */
#define ELO_MAX_TRIALS	3       /* Number of timeouts waiting for a     */
                                /* pending reply.			*/
#define ELO_MAX_WAIT		100000  /* Max wait time for a reply (microsec) */
#define ELO_PACKET_SIZE		10
#define ELO_INIT_CHECKSUM	0xAA  /* Initial value of checksum.           */

#define ELO_ID			'I'         /* Report of type and features.         */
#define ELO_MODE		'M'         /* Set current operating mode.          */
#define ELO_SYNC_BYTE		'U'     /* Sync byte. First of a packet.        */
#define ELO_PARAMETER		'P'     /* Set the serial parameters.           */
#define ELO_ACK			'A'         /* Acknowledge packet                   */
#define ELO_REPORT		'B'       /* Set touch reports timings.           */

#define ELO_TOUCH_MODE		0x01  /* Flags in ELO_MODE command            */
#define ELO_STREAM_MODE		0x02
#define ELO_UNTOUCH_MODE	0x04
#define ELO_TRACKING_MODE	0x40

#define ELO_UNTOUCH_DELAY	5     /* 100 ms                               */
#define ELO_REPORT_DELAY	1     /* 40 ms or 25 motion reports/s         */


/* If you do not have a keyboard on your
 * system, use this to get a timer instead
 */
int iUserEnter = HAVE_USER_ENTER;

/* Requires a difference between the
 * maximum and minimum of at least
 * this is how much(Defult 2000, 0=off)
 */
int iMaxMinDiff = MAX_MIN_DIFF;

/* The default file name, where min / max information is saved */
char *pcSaveToFile = SAVE_TO_FILE_DEFAULT;

/* FD to read input from */
int input_fd = STDIN_FILENO;

/* Shrink calibrated area by this many percent */
int shrink_percent = 0;

struct screen_info
{
  int total_x, total_y;
  double x1, y1, x2, y2;
};

#ifdef HAVE_X11
int x11 = -1;
#define X11_CROSS_POSITION_PERCENT_DEFAULT 3
int x11_cross_position_percent = X11_CROSS_POSITION_PERCENT_DEFAULT;
Display *display = NULL;
Window window;
GC gc;
int x11_width = 0, x11_height = 0, x11_cross_position = 0, x11_cross_size = 0;
XftDraw *x11_draw = NULL;
XftColor x11_text_color;
XftFont *x11_font = NULL;
#endif

/* abort with an error message */
static void die(const char *message)
{
    fprintf(stderr, "%s\n", message);
    exit(EXIT_FAILURE);
}

/* routine to initialize a MicroTouch serial controller */
static int init_mu(char dev[])
{
    int fd, res, len, i;
    struct termios newtio;
    char buf[255], dummy[255];
    char format_tablet[4], get_page[5], set_page[5], mode_stream[4];
    char reset_controler[3];
    char data[16];

    /* Open serial device for reading and writing */
    fd = open(dev, O_RDWR | O_NOCTTY);
    if (fd < 0) {
	perror(dev);
	exit(EXIT_FAILURE);
    }

    /* define some command strings */
    /* they are taken from the controller's manual */
    format_tablet[0] = 0x01;
    format_tablet[1] = 'F';
    format_tablet[2] = 'T';
    format_tablet[3] = 0x0d;

    mode_stream[0] = 0x01;
    mode_stream[1] = 'M';
    mode_stream[2] = 'S';
    mode_stream[3] = 0x0d;

    reset_controler[0] = 0x01;
    reset_controler[1] = 'R';
    reset_controler[2] = 0x0d;

    get_page[0] = 0x01;
    get_page[1] = 'G';
    get_page[2] = 'P';
    get_page[3] = '1';
    get_page[4] = 0x0d;

    set_page[0] = 0x01;
    set_page[1] = 'S';
    set_page[2] = 'P';
    set_page[3] = '1';
    set_page[4] = 0x0d;

    /* define data string */
    strncpy(data, "00180769001D0793", 16);

    bzero(&newtio, sizeof(newtio));	/* clear struct for new port settings */

    /*
       BAUDRATE: Set bps rate
       CRTSCTS : output hardware flow control
       CS8     : 8n1 (8bit,no parity,1 stopbit)
       CLOCAL  : local connection
       CREAD   : enable receiving characters
     */
    newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;

    /*
       IGNPAR  : ignore bytes with parity errors
       ICRNL   : map CR to NL
     */
    newtio.c_iflag = IGNPAR | ICRNL;

    /*
       Raw output.
     */
    newtio.c_oflag = 0;

    /*
       ICANON  : enable canonical input
     */
    newtio.c_lflag = ICANON;

    /*
       initialize all control characters
     */
    newtio.c_cc[VINTR] = 0;	/* Ctrl-c */
    newtio.c_cc[VQUIT] = 0;	/* Ctrl-\ */
    newtio.c_cc[VERASE] = 0;	/* del */
    newtio.c_cc[VKILL] = 0;	/* @ */
    newtio.c_cc[VEOF] = 4;	/* Ctrl-d */
    newtio.c_cc[VTIME] = 0;	/* inter-character timer unused */
    newtio.c_cc[VMIN] = 1;	/* blocking read until 1 character arrives */
    newtio.c_cc[VSWTC] = 0;	/* '\0' */
    newtio.c_cc[VSTART] = 0;	/* Ctrl-q */
    newtio.c_cc[VSTOP] = 0;	/* Ctrl-s */
    newtio.c_cc[VSUSP] = 0;	/* Ctrl-z */
    newtio.c_cc[VEOL] = 0;	/* '\0' */
    newtio.c_cc[VREPRINT] = 0;	/* Ctrl-r */
    newtio.c_cc[VDISCARD] = 0;	/* Ctrl-u */
    newtio.c_cc[VWERASE] = 0;	/* Ctrl-w */
    newtio.c_cc[VLNEXT] = 0;	/* Ctrl-v */
    newtio.c_cc[VEOL2] = 0;	/* '\0' */

    /*
       now clean the serial line and activate the settings for the port
     */
    tcflush(fd, TCIFLUSH);
    tcsetattr(fd, TCSANOW, &newtio);

    /*reset controller */
    write(fd, reset_controler, 3);
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("reset controler: %s\n", dummy);

    /*set to format tablet */
    write(fd, format_tablet, 4);
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("set to format tablet: %s\n", dummy);

    /*set mode stream */
    write(fd, mode_stream, 4);
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("set mode stream: %s\n", dummy);

    /*get ROM page 1 */
    write(fd, get_page, 5);
    /*get response */
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("get page: %s\n", dummy);
    /*read data */
    len = read(fd, buf, 255);
    buf[len] = 0;
    printf("page: %s\n", buf);
    /*get ack */
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("data received: %s\n", dummy);

    /*change this data */
    for (i = 0; i < 16; i++)
    buf[i] = data[i];
    printf("going to write: %s\n", buf);

    /*write changed data */
    write(fd, set_page, 5);
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("set page: %s\n", dummy);
    /*write data */
    write(fd, buf, 52);
    /*get ack */
    res = read(fd, dummy, 255);
    dummy[res] = 0;
    printf("write data: %s\n", dummy);

    return fd;
}

static int elo_get_packet(int fd, unsigned char *buffer, int *buffer_p,
			  int *checksum)
{
    int ok, num_bytes;

    /*
     * Try to read enough bytes to fill up the packet buffer.
     */
    num_bytes = read(fd, (char *) (buffer + *buffer_p),
		     ELO_PACKET_SIZE - *buffer_p);

    /*
     * Okay, give up.
     */
    if (num_bytes < 0) {
      printf("System error while reading from Elographics touchscreen.");
	    return num_bytes;
    }

    while (num_bytes) {
	/*
	 * Sync with the start of a packet.
	 */
	if ((*buffer_p == 0) && (buffer[0] != ELO_SYNC_BYTE)) {
	    /*
	     * No match, shift data one byte toward the start of the buffer.
	     */
	    printf
		("Elographics: Dropping one byte in an attempt to synchronize: 0x%X\n",
		 buffer[0]);
	    memcpy(&buffer[0], &buffer[1], num_bytes - 1);
	} else {
	    /*
	     * Compute checksum in assembly buffer.
	     */
	    if (*buffer_p < ELO_PACKET_SIZE - 1) {
		*checksum = *checksum + buffer[*buffer_p];
		*checksum = *checksum % 256;
	    }
	    (*buffer_p)++;
	}
	num_bytes--;
    }

    if (*buffer_p != ELO_PACKET_SIZE)
	return -1;

    /*
     * Got a packet, validate checksum and reset state.
     */
    ok = (*checksum == buffer[ELO_PACKET_SIZE - 1]);
    *checksum = ELO_INIT_CHECKSUM;
    *buffer_p = 0;

    if (!ok) {
	printf("Checksum error on Elographics touchscreen link\n");
	return -1;
    }

    /*
     * Valid packet received report it.
     */
    return 0;
}

static int elo_wait_reply(int fd, unsigned char type, unsigned char *reply)
{
    int i, r;
    int reply_p = 0;
    int sum = ELO_INIT_CHECKSUM;

    i = ELO_MAX_TRIALS;
    do {
	fd_set readfds;
	struct timeval to;

	/*
	 * Wait half a second for the reply. The fuse counts down each
	 * timeout and each wrong packet.
	 */
	FD_ZERO(&readfds);
	FD_SET(fd, &readfds);
	to.tv_sec = 0;
	to.tv_usec = ELO_MAX_WAIT;
	r = select(FD_SETSIZE, &readfds, NULL, NULL, &to);
	if (r <= 0)
	    i--;
	if (r < 0) {
	    printf("No answer from link: %d\n", r);
	    continue;
	}

	r = elo_get_packet(fd, reply, &reply_p, &sum);
	/*
	 * Do not report an error on a 'P' query as the controller
	 * might be a 2310.
	 */
	if (r == 0 && reply[1] != type && type != ELO_PARAMETER)
	    printf("Wrong reply received\n");
	else
	    break;
    } while (i > 0);

    return 0;
}

static int elo_wait_ack(int fd)
{
    unsigned char packet[ELO_PACKET_SIZE];
    int r, i, nb_errors;

    r = elo_wait_reply(fd, ELO_ACK, packet);
    if (r < 0)
	return r;

    for (i = 0, nb_errors = 0; i < 4; i++)
	if (packet[2 + i] != '0')
	    nb_errors++;

    if (nb_errors != 0)
	printf("Elographics acknowledge packet reports %d errors\n",
	       nb_errors);

    return 0;
}

static int elo_send_packet(int fd, unsigned char *packet)
{
    int i, r;
    int sum = ELO_INIT_CHECKSUM;

    packet[0] = ELO_SYNC_BYTE;
    for (i = 0; i < ELO_PACKET_SIZE - 1; i++) {
	sum += packet[i];
	sum &= 0xFF;
    }
    packet[ELO_PACKET_SIZE - 1] = sum;

    r = write(fd, packet, ELO_PACKET_SIZE);
    if (r != ELO_PACKET_SIZE) {
	printf("System error while sending to Elographics touchscreen.\n");
	return -1;
    }

    return 0;
}

static int elo_send_control(int fd, unsigned char *control)
{
    int r;
    r = elo_send_packet(fd, control);
    if (r < 0)
	return r;

    return elo_wait_ack(fd);
}

/* routine to initialize an EloGraphics serial controller */
static int init_elo(char dev[])
{
    int r, fd;
    struct termios termios_tty;
    unsigned char req[ELO_PACKET_SIZE];

    fd = open(dev, O_RDWR | O_NONBLOCK);
    if (fd < 0) {
	perror("init_elo open");
	exit(EXIT_FAILURE);
    }

    memset(&termios_tty, 0, sizeof(termios_tty));
    termios_tty.c_cflag = B9600 | CS8 | CREAD | CLOCAL;
    termios_tty.c_cc[VMIN] = 1;
    r = !isatty(fd) ? 0 : tcsetattr(fd, TCSANOW, &termios_tty);
    if (r < 0) {
	perror("init_elo tcsetattr");
	exit(EXIT_FAILURE);
    }

    /*
     * Set the operating mode: Stream, no scaling, no calibration,
     * no range checking, no trim, tracking enabled.
     */
    memset(req, 0, ELO_PACKET_SIZE);
    req[1] = ELO_MODE;
    req[3] = ELO_TOUCH_MODE | ELO_STREAM_MODE | ELO_UNTOUCH_MODE;
    req[4] = ELO_TRACKING_MODE;
    r = elo_send_control(fd, req);
    if (r < 0) {
	printf
	    ("Unable to change Elographics touchscreen operating mode\n");
	exit(EXIT_FAILURE);
    }
    /*
     * Set the touch reports timings from configuration data.
     */
    memset(req, 0, ELO_PACKET_SIZE);
    req[1] = ELO_REPORT;
    req[2] = ELO_UNTOUCH_DELAY;
    req[3] = ELO_REPORT_DELAY;
    r = elo_send_control(fd, req);
    if (r < 0) {
	printf
	    ("Unable to change Elographics touchscreen reports timings\n");
	exit(EXIT_FAILURE);
    }

    return fd;
}

#ifdef HAVE_X11
static int x11_check_events(void)
{
    XEvent xe;
    while (XPending(display)) {
        XNextEvent(display, &xe);
        if (xe.type == KeyPress)
            die("aborting on key press");
        if (xe.type == ConfigureNotify && xe.xconfigure.window == window
            && (xe.xconfigure.width != x11_width ||
                xe.xconfigure.height != x11_height))
            die("X11 window size changed unexpectedly");
        if (xe.type == Expose)
            return 1;
    }
    return 0;
}

static void x11_draw_text(int line, const char *format, ...) __attribute__((format(printf, 2, 3)));
static void x11_draw_text(int line, const char *format, ...)
{
    char buf[0x400];
    va_list args;
    va_start(args, format);
    vsnprintf(buf, sizeof(buf), format, args);
    va_end(args);
    XftDrawString8(x11_draw, &x11_text_color, x11_font,
                   x11_cross_position + 2 * x11_cross_size,
                   x11_height * line / 10,
                   (const unsigned char *) buf, strlen(buf));
}
#endif

static void init_output(struct screen_info *screen)
{
#ifdef HAVE_X11
    if (x11 >= 0) {
        // open X11 connection and get screen info
        display = XOpenDisplay(NULL);
        if (!display)
            die("cannot open X11 connection");
        input_fd = ConnectionNumber(display);
        int sc = DefaultScreen(display);
        Visual *visual = DefaultVisual(display, sc);
        screen->total_x = DisplayWidth(display, sc);
        screen->total_y = DisplayHeight(display, sc);
        int x0 = 0, y0 = 0;
        int n;
        XineramaScreenInfo *s = XineramaQueryScreens(display, &n);
        if (x11 >= (s ? n : 1))
            die("X11 screen not found");
        if (s) {
            x0 = s[x11].x_org;
            y0 = s[x11].y_org;
            x11_width = s[x11].width;
            x11_height = s[x11].height;
            XFree (s);
        } else {
            x11_width = screen->total_x;
            x11_height = screen->total_y;
        }
        x11_cross_position = x11_width * x11_cross_position_percent / 100;
        x11_cross_size = x11_width * 3 / 100;
        screen->x1 = x0 + x11_cross_position;
        screen->y1 = y0 + x11_cross_position;
        screen->x2 = x0 + x11_width - x11_cross_position;
        screen->y2 = y0 + x11_height - x11_cross_position;

        // create window
        XSetWindowAttributes attr;
        attr.background_pixel = attr.border_pixel = WhitePixel(display, sc);
        attr.event_mask = ExposureMask | StructureNotifyMask | KeyPressMask;
        attr.colormap = DefaultColormap(display, sc);
        window = XCreateWindow(display, RootWindow(display, sc),
                   x0, y0, x11_width, x11_height, 0,
                   CopyFromParent, InputOutput, visual,
                   CWBackPixel | CWBorderPixel | CWEventMask | CWColormap, &attr);
        if (window == None)
            die("cannot create X11 window");
        XSizeHints SizeHints;
        SizeHints.min_width  = SizeHints.max_width  = x11_width;
        SizeHints.min_height = SizeHints.max_height = x11_height;
        SizeHints.flags = PMinSize | PMaxSize;
        XSetStandardProperties(display, window, "touchcal", "touchcal", None,
                               NULL, 0, &SizeHints);

        // disable window decorations
        struct
        {
          uint32_t flags, functions, decorations, input_mode, status;
        } Hints = { };
        size_t NHints = sizeof(Hints) / 4;
        Atom Type, HintsAtom = XInternAtom(display, "_MOTIF_WM_HINTS", False);
        int Format;
        unsigned long int NItems, BytesAfter;
        unsigned char *Data;
        if (XGetWindowProperty(display, window, HintsAtom, 0, NHints, False,
                               AnyPropertyType, &Type, &Format, &NItems,
                               &BytesAfter, &Data) == Success) {
            if (Format == 32 && NItems >= NHints)
                memcpy(&Hints, Data, sizeof(Hints));
            XFree(Data);
        }
        Hints.flags |= 2;
        Hints.decorations = 0;
        XChangeProperty(display, window, HintsAtom, HintsAtom, 32, PropModeReplace,
                        (const unsigned char *) (&Hints), NHints);

        // allocate drawing contexts etc.
        XGCValues gc_values;
        XColor c;
        XAllocNamedColor(display, attr.colormap, "blue", &c, &c);
        gc_values.foreground = c.pixel;
        gc = XCreateGC(display, window, GCForeground, &gc_values);
        x11_draw = XftDrawCreate(display, window, visual, attr.colormap);
        char font[64];
        snprintf(font, sizeof(font), ":size=%i:bold", x11_width / 80);
        x11_font = XftFontOpenName(display, sc, font);
        XftColorAllocName(display, visual, attr.colormap, "red", &x11_text_color);

        // map window and wait for it
        // some window managers place the window not where requested; move it back
        XMapRaised(display, window);
        XMoveWindow(display, window, x0, y0);
        while (!x11_check_events()) { }
        return;
    }
#endif

    initscr();
    screen->total_x = getmaxx(stdscr);
    screen->total_y = getmaxy(stdscr);
    screen->x1 = 1.5;
    screen->y1 = 1.5;
    screen->x2 = screen->total_x - 1.5;
    screen->y2 = screen->total_y - 1.5;
}

static void finish_output(void)
{
#ifdef HAVE_X11
    if (x11 >= 0) {
        XCloseDisplay(display);
        return;
    }
#endif
    endwin();
}

/* draw a crosshair */
static void cross(int position)
{
    static char *touchtxt = "Please touch the marker or press any key to abort";
#ifdef HAVE_X11
    if (x11 >= 0) {
        int cx = 0, cy = 0;

        XClearWindow(display, window);

        switch (position) {
        case 0:
            cx = x11_cross_position;
            cy = x11_cross_position;
            break;
        case 1:
            cx = x11_cross_position;
            cy = x11_height - x11_cross_position;
            break;
        case 2:
            cx = x11_width - x11_cross_position;
            cy = x11_cross_position;
            break;
        case 3:
            cx = x11_width - x11_cross_position;
            cy = x11_height - x11_cross_position;
            break;
        default:
            perror("invalid position in cross()");
        }

        /*Paint cross */
        int s = x11_cross_size - 4, r = x11_cross_size / 4;
        XDrawLine(display, window, gc, cx, cy - s, cx, cy + s);
        XDrawLine(display, window, gc, cx - s, cy, cx + s, cy);
        XDrawArc(display, window, gc, cx - r, cy - r, 2 * r, 2 * r,
                 0, 360 * 64);

        /*Print text */
        x11_draw_text(5, "%s", touchtxt);
        x11_check_events();
        return;
    }
#endif

    int cx, cy, tx, ty;

    cx = cy = tx = ty = 0;

    clear();

    switch (position) {
    case 0:
	cx = 0;
	cy = 0;
	tx = 6;
	ty = 1;
	break;
    case 1:
	cx = 0;
	cy = getmaxy(stdscr) - 3;
	tx = 6;
	ty = 1;
	break;
    case 2:
	cx = getmaxx(stdscr) - 3;
	cy = 0;
	tx = -6 - strlen(touchtxt);
	ty = 1;
	break;
    case 3:
	cx = getmaxx(stdscr) - 3;
	cy = getmaxy(stdscr) - 3;
	tx = -6 - strlen(touchtxt);
	ty = 1;
	break;
    default:
	perror("invalid position in cross()");
    }

    /*Paint cross */
    move(cy, cx);
    printw(" | ");
    move(cy + 1, cx);
    printw("-+-");
    move(cy + 2, cx);
    printw(" | ");

    /*Print text */
    move(cy + ty, cx + tx);
    printw(touchtxt);

    /*Center cursor in cross */
    move(cy + 1, cx + 1);
    refresh();
}

static void show_result(int j, int *y, int *x)
{
   int iNext = 3;

   // Do not sleep 3sec if there is no more
   if (3 == j)
       iNext = 1;

#ifdef HAVE_X11
    if (x11 >= 0) {
        for (; iNext > 0; --iNext) {
            x11_check_events();
            XClearWindow(display, window);
            int i;
            for (i = 0; i <= j; i++)
                x11_draw_text(i + 2, "Coord %d: y = %4d, x = %4d", i, y[i], x[i]);
            x11_draw_text(8, "Next in %d", iNext);
            x11_check_events();
            sleep(1);
        }
        return;
    }
#endif

    int i;

    clear();
    for (i = 0; i <= j; i++)
	mvprintw(10 + i, 20, "Coord %d: y = %4d, x = %4d", i, y[i], x[i]);

    /* Enter version */
    if (iUserEnter) {
	mvprintw(10 + i + 1, 29, "Hit ENTER to continue");
	getch();
	return;
    }


    /* Time version */
     for (; iNext > 0; --iNext) {
	mvprintw(10 + i + 1, 29, "Next in %d", iNext);
	refresh();
	sleep(1);
    }
}

int diff(int imin, int imax)
{
    if (imin > imax) {
	int i = imax;
	imax = imin;
	imin = i;
    }

    return imax - imin;
}


/* extrapolate result and print it in XF86Config format */
static void show_extrapolate_result(const int *xin, const int *yin,
				    const struct screen_info *screen)
{
    int xmin, xmax, ymin, ymax;	/*counters and coordinates */
    double screenx, screeny;
    double dminx, dmaxx, dminy, dmaxy, dscale;
    int swap = 0;
    /* print to stdout */
    FILE *out = stdout;

    /* test if X and Y axes are swapped */
    if ((diff (xin[0], xin[1]) +
         diff (xin[2], xin[3]) +
         diff (yin[0], yin[2]) +
         diff (yin[1], yin[3]) >
         diff (xin[0], xin[2]) +
         diff (xin[1], xin[3]) +
         diff (yin[0], yin[1]) +
         diff (yin[2], yin[3]))
        != (screen->total_y > screen->total_x)) {
			const int *t = xin;
			xin = yin;
			yin = t;
			swap = 1;
    }

#ifdef DEBUG_MAX_KOD
    for (i = 0; i < 4; ++i)
	fprintf(stderr, "%d)  xin=%4d  yin=%4d\n", i, xin[i], yin[i]);
#endif


    dminx = (xin[0] + xin[1]) / 2.0;
    dmaxx = (xin[2] + xin[3]) / 2.0;
    dminy = (yin[1] + yin[3]) / 2.0;
    dmaxy = (yin[0] + yin[2]) / 2.0;

    /* Shrink calibrated area if requested */
    double shrink_x = (dmaxx - dminx) * shrink_percent / 100;
    double shrink_y = (dmaxy - dminy) * shrink_percent / 100;
    dminx += shrink_x / 2;
    dmaxx -= shrink_x / 2;
    dminy += shrink_y / 2;
    dmaxy -= shrink_y / 2;

#ifdef DEBUG_MAX_KOD
    fprintf(stderr, " dminx=%d  -  dmaxx=%d  -  dminy=%d  -  dmaxy=%d\n",
	    (int) dminx, (int) dmaxx, (int) dminy, (int) dmaxy);
#endif

    screenx = screen->x2 - screen->x1;
    screeny = screen->y2 - screen->y1;

    /* To get a scale of what each character corresponds to the touch scale */
    dscale = (dmaxx - dminx) / screenx;
    xmax = round (dmaxx + dscale * (screen->total_x - screen->x2));
    xmin = round (dminx - dscale * screen->x1);
    dscale = (dmaxy - dminy) / screeny;
    ymax = round (dmaxy + dscale * screen->y1);
    ymin = round (dminy - dscale * (screen->total_y - screen->y2));

#ifdef DEBUG_MAX_KOD
    fprintf(stderr, " xin[2] + (X) = %4d  (xmax=%4d)\n",
	    xin[2] + (int) (((xin[2] - xin[0]) / screen->total_x) * 1.5), xmax);

    /* extrapolate values */

    fprintf(stderr, " ((xin[2] - xin[0])=%4d )/%d = %d\n",
	    (xin[2] - xin[0]), screen->total_x, (xin[2] - xin[0]) / screen->total_x);
    fprintf(stderr, " ((xin[2] - xin[0]) / screenx) * 1.5 = %d\n",
	    (int) (((xin[2] - xin[0]) / screen->total_x) * 1.5));
    fprintf(stderr, "Diffx(%d,%d):%d\n", xmin, xmax, diff(xmin, xmax));
    fprintf(stderr, "Diffx(%d,%d):%d\n", xmax, xmin, diff(xmax, xmin));
    fprintf(stderr, "Diffy(%d,%d):%d\n", ymin, ymax, diff(ymin, ymax));
    fprintf(stderr, "Diffy(%d,%d):%d\n", ymax, ymin, diff(ymax, ymin));
#endif

    //  the diff of max and min have to be more then iMaxMinDiff
    if (iMaxMinDiff > 0
	&& (diff(xmin, xmax) < iMaxMinDiff
	    || diff(ymin, ymax) < iMaxMinDiff)) {
	fprintf(stderr,
		"ERROR: Poor calibration, max-min difference needs to be >= %d (xmin=%d, xmax=%d, ymin=%d, ymax=%d)\n",
		iMaxMinDiff, xmin, xmax, ymin, ymax);
	exit(EXIT_FAILURE);
    }


    while (out) {
	if (swap)
	    fprintf(out, "\tOption  \"SwapXY\"\n");
	fprintf(out, "\tOption  \"MinX\" \"%d\"\n", xmin);
	fprintf(out, "\tOption  \"MaxX\" \"%d\"\n", xmax);
	fprintf(out, "\tOption  \"MinY\" \"%d\"\n", ymin);
	fprintf(out, "\tOption  \"MaxY\" \"%d\"\n", ymax);

	if (out == stdout)
	    out = fopen(pcSaveToFile, "w");	/* this output into a file */
	else
	    break;
    }

    if (!out)
	fprintf(stderr, "Error: can't save to file: %s\n", pcSaveToFile);
    else if (out != stdout)
	fclose(out);
}

static void drain (int in)
{
    while (1) {
        char c;
        struct timeval notime = { tv_sec: 0, tv_usec: 0 };
        fd_set rfd;
        FD_ZERO (&rfd);
        FD_SET (in, &rfd);
        if (select (in + 1, &rfd, NULL, NULL, &notime) <= 0)
            return;
        read (in, &c, 1);
    }
}

int read_char(int in)
{
    char c;
    fd_set rfd;
    FD_ZERO (&rfd);
    FD_SET (input_fd, &rfd);
    FD_SET (in, &rfd);
    int max = input_fd > in ? input_fd : in;
    if (select (max + 1, &rfd, NULL, NULL, NULL) > 0
        && FD_ISSET (input_fd, &rfd)) {
#ifdef HAVE_X11
        if (x11 >= 0)
            x11_check_events();
        else
#endif
        {
            finish_output();
            fprintf(stderr,"Aborted!\n");
            exit(1);
        }
    }
    return read (in, &c, 1) == 1 ? c : EOF;
}

/*calibration routine for MicroTouch devices*/
static void mutouch(char dev[])
{
    char read, protocol[6];	/*to store the output of the controller */
    int i, iloop, x[4], y[4];	/*counters and coordinates */
    struct screen_info screen;

    /*init controler */
    int in = init_mu(dev);

    init_output(&screen);

    /*read three coordinates */
    for (iloop = 0; iloop < 4; ++iloop) {

	drain(in);
	
	/*display crosshair */
	cross(iloop);

	/*read the word (5 bytes) */
	i = 1;
	while (i < 6) {
	    read = read_char(in);
	    protocol[i] = read;
	    i++;
	}

	/*read X and Y coordinates */
	x[iloop] = protocol[3];
	x[iloop] = x[iloop] << 7;
	x[iloop] = x[iloop] | (protocol[2] & 0x00ff);

	y[iloop] = protocol[5];
	y[iloop] = y[iloop] << 7;
	y[iloop] = y[iloop] | (protocol[4] & 0x00ff);

	show_result(iloop, y, x);
    }
    close(in);
    finish_output();

    /*finally show result */
    show_extrapolate_result(x, y, &screen);
}


/*calibration routine for an EloGraphics device*/
static void elotouch(char dev[])
{
    unsigned char read, protocol[11];
    int i, iloop, x[4], y[4];
    struct screen_info screen;

    /*init controller */
    int in = init_elo(dev);

    init_output(&screen);

    memset(protocol, 0, sizeof(protocol));
    memset(x, 0, sizeof(x));
    memset(y, 0, sizeof(y));


    /*read three coordinates */
    for (iloop = 0; iloop < 4; ++iloop) {

	drain(in);
	
	/* display crosshair */
	cross(iloop);

	/* drop strings not starting with UT */
	read = 'i';
	while (1) {
	    read = read_char(in);
	    if (read == 'U' && read_char(in) == 'T')
		break;
	}

	/* read the rest of the word (9 bytes) - not counted, the next word
	 * starts with 'U' */
	i = 0;
	read = 'i';

	while (read != 'U' && i < sizeof(protocol)) {
	    read = read_char(in);
	    protocol[i++] = read;
	}

	/*read X and Y coordinates */
	x[iloop] = protocol[2];
	x[iloop] <<= 8;
	x[iloop] = x[iloop] | (protocol[1] & 0x00ff);

//              for(i=0;i<10 && (i==0 || protocol[i]);++i)
//                      fprintf(stderr,"elotouch() - %2d) 0x%x  (Dec:%2d)\n", i,protocol[i],protocol[i]);


	y[iloop] = protocol[4];
	y[iloop] <<= 8;
	y[iloop] = y[iloop] | (protocol[3] & 0x00ff);

	show_result(iloop, y, x);
    }
    close(in);
    finish_output();

    /*finally show result */
    show_extrapolate_result(x, y, &screen);

    return;
}

void help_exit(const char *pcText)
{
    int iExitkod = EXIT_FAILURE;
    if (!pcText)
	iExitkod = EXIT_SUCCESS;
    else
	fprintf(stderr, "Error: %s\n\n", pcText);


    fprintf(stderr,
	    "Minimal required options: touchcal -e|-m /dev/ttySx [x=0..3]\n"
	    "All options:\n"
	    "--elographics | -e         - use elographics driver\n"
	    "--microtouch  | -m         - use microtouch driver\n"
#ifdef HAVE_X11
	    "--x11         | -x[SCREEN] - use X11 [on given screen]\n"
	    "--position    | -p PERCENT - position of cross-hair under X11 in %% (default: %i)\n"
#endif
	    "--shrink      | -S NUMBER  - shrink area by NUMBER %% for each axis\n"
	    "                             (useful to avoid crossing screen\n"
	    "                             boundaries in multiscreen configurations)\n"
	    "--time        | -t         - if you do not have a keyboard on your\n"
	    "                             system, use this to get a timeout instead\n"
	    "--diffrequird | -d NUMBER  - requires a difference between the\n"
	    "                             max. and min. values of at least NUMBER\n"
	    "                             (default = %d, 0 = off)\n"
	    "--save        | -s NAME    - file to save min / max information to\n"
	    "                             (default = \"%s\")\n"
	    "--version     | -v         - print the touchcal version (" PACKAGE_VERSION ")\n"
	    "--help        | -h         - this help text\n"
	    " And then give the device file for your serial port. Probably /dev/ttySx [x=0..3]\n\n",
	    X11_CROSS_POSITION_PERCENT_DEFAULT, iMaxMinDiff, pcSaveToFile);

    exit(iExitkod);
}

int main(int argc, char *argv[])
{
    /* What drives elographics / microtouch */
    enum eDriver { eUnset = 0, eElographics, eMicrotouch };
    enum eDriver Driver = eUnset;

    struct option option_def_long[] = {
	{"save", required_argument, 0, 's'},
	{"diffrequird", required_argument, 0, 'd'},
	{"time", no_argument, 0, 't'},
	{"elographics", no_argument, 0, 'e'},
	{"microtouch", no_argument, 0, 'm'},
#ifdef HAVE_X11
	{"x11", optional_argument, 0, 'x'},
#endif
	{"shrink", required_argument, 0, 'S'},
	{"help", no_argument, 0, 'h'},
	{"version", no_argument, 0, 'v'},
	{0, 0, 0, 0}
    };
    int command_index = 0;

    int option_char;

    while ((option_char =
	    getopt_long(argc, argv,
#ifdef HAVE_X11
			"s:d:S:temx::p:hv",
#else
			"s:d:S:temhv",
#endif
			option_def_long,&command_index)) != -1) {
	switch (option_char) {
	case 'h':
	    help_exit(NULL);
	    break;

	case 't':
	    iUserEnter = 0;
	    break;

	case 's':
	    pcSaveToFile = optarg;
	    break;

	case 'e':
	    Driver = eElographics;
	    break;

	case 'm':
	    Driver = eMicrotouch;
	    break;

#ifdef HAVE_X11
	case 'x':
	    x11 = optarg ? atoi(optarg) : 0;
	    break;

	case 'p':
	    x11_cross_position_percent = atoi(optarg);
	    break;
#endif

	case 'S':
	    shrink_percent = atoi(optarg);
	    break;

	case 'd':
	    iMaxMinDiff = atoi(optarg);
	    break;

	case 'v':		/* print version */
	    fprintf(stderr, "touchcal version %s\n", version);
	    exit(EXIT_SUCCESS);
	    break;

	default:
	    help_exit("Not a valid option");
	    break;

	}
    }

    // In order to also have the same function as before
    if (optind < argc && strlen(argv[optind]) == 1
	&& ('e' == argv[optind][0] || 'm' == argv[optind][0])) {
	if ('e' == argv[optind][0])
	    Driver = eElographics;
	else
	    Driver = eMicrotouch;
	++optind;
    }

    if (eUnset == Driver)
	help_exit("You must select a driver");

    /* if not enough parameters are given */
    if (optind >= argc)
	help_exit
	    ("Too few options given, select the serial port you want to talk to");

    /* if the first parameter is an e */
    if (eElographics == Driver) {
	printf("Calibrating EloGraphics device\n");
	elotouch(argv[optind]);
    }

    /* if the first parameter is a m */
    if (eMicrotouch == Driver) {
	printf("Calibrating MicroTouch device\n");
	mutouch(argv[optind]);
    }

    return 0;
}
