#include "gimme_config_h.h"

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#include "cdstatus.h"
#include "cdstatus_cddb.h"
#include "cdstatus_output.h"
#include "open_output_file.h"
#include "args.h"

/* Note that the meaning of 'file' here is in the abstract system sense of a 
 * file handle. In particular, the file opened may be a pipe to an audio
 * compressor, depending on the program arguments used. In that instance,
 * the compressor will create the 'actual' file on disk. */
FILE * openOutputFile(const char *filename, Encoders encoder, unsigned int track, const char *encopts)
{
	char *last_slash;
	char syscmd[2048];
	char pathname[CDSTATUS_PATH_MAX];
	FILE * output_file;

	conditional_printf(DEBUG, "Handed file to create: %s\n", filename);

	last_slash = strrchr(filename, '/');
	if (last_slash == NULL)
	{
		conditional_puts(CRITICAL, "Error, directory portion of name not found. Exiting");
		exit(EXIT_FAILURE);
	}
	strncpy(pathname, filename, (size_t)CDSTATUS_MIN((unsigned)(last_slash - filename),sizeof(pathname)-1));
	pathname[(last_slash - filename)] = '\0';
	pathname[sizeof(pathname)-1]='\0';
	
	conditional_printf(DEBUG, "Path component of file to create: %s\n", pathname);
	(void)snprintf(syscmd, sizeof(syscmd), "/bin/mkdir -p \"%s\"", pathname);
	
	conditional_printf(DEBUG, "Issuing command: %s\n", syscmd);
	if(system(syscmd)==-1)
	{
		conditional_perror(CRITICAL, "Error executing command to create directory");
		exit(EXIT_FAILURE);
	}

	switch(encoder) {
		case ENC_NULL:
		case ENC_RAW:
		case ENC_WAVE:
			/* just open the file */
			output_file = fopen(filename, "w");
			if (output_file == NULL)
			{
				conditional_perror(CRITICAL, "Error creating output file");
				exit(EXIT_FAILURE);
			}
			break;
		case ENC_LAME:
			/* open the pipe and pass the neccessary stuff */
	        if(strlen(album_genre)>0)
        	{
		    	(void)snprintf(syscmd, sizeof(syscmd), "lame --preset standard -S --tc \"ripped by cdstatus %s\" %s --tt \"%s\" --ta \"%s\" --tl \"%s\" --ty \"%d\" --tn \"%u\" - \"%s\"",VERSION,encopts,trackinfo[track].title, artist_name, album_name, album_year, track, filename);
        	}
        	else
        	{
		    	(void)snprintf(syscmd, sizeof(syscmd), "lame --preset standard -S --tc \"ripped by cdstatus %s\" %s --tt \"%s\" --ta \"%s\" --tl \"%s\" --ty \"%d\" --tn \"%u\" - \"%s\"",VERSION,encopts,trackinfo[track].title, artist_name, album_name, album_year, track, filename);
        	}
		
			conditional_printf(VERBOSE_DEBUG, "Track Genre: %s\n",album_genre);
			conditional_printf(DEBUG, "Using lame with: %s\n\n",syscmd);
			fflush(stdout);
			break;
		case ENC_TOOLAME:
			(void)snprintf(syscmd, sizeof(syscmd), "toolame %s /dev/stdin \"%s\"", encopts, filename);
			conditional_printf(DEBUG, "Using toolame with: %s\n\n", syscmd);
			fflush(stdout);
			break;
		case ENC_OGGENC:
	        if(strlen(album_genre)>0)
        	{
				(void)snprintf(syscmd, sizeof(syscmd), "oggenc -Q -o \"%s\" -c \"program=cdstatus %s\" -a \"%s\" -t \"%s\" -l \"%s\" -G \"%s\" %s -", filename, VERSION, artist_name, trackinfo[track].title, album_name, album_genre, encopts);
			}
			else
			{
				(void)snprintf(syscmd, sizeof(syscmd), "oggenc -Q -o \"%s\" -c \"program=cdstatus %s\" -a \"%s\" -t \"%s\" -l \"%s\" %s -", filename, VERSION, artist_name, trackinfo[track].title, album_name, encopts);
			}
			break;
		default:
			conditional_puts(CRITICAL, "Unhandled encoder encountered in open_output_file.c");
			exit(EXIT_FAILURE);
	}

	if(encoder == ENC_LAME || encoder == ENC_TOOLAME || encoder == ENC_OGGENC)
	{
		output_file = popen((const char *)syscmd, "w");
		if(output_file==NULL)
		{
			conditional_perror(CRITICAL, "Error creating pipe to encoder");
			exit(EXIT_FAILURE);
		}
	}
	return output_file;
}
