#include "gimme_config_h.h"

#ifndef HAVE_CONFIG_H
#define VERSION "unknown"
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#include "cdstatus.h"
#include "cddb_handshake.h"
#include "cddb_hearken.h"
#include "cdstatus_cddb.h"
#include "cddb_send.h"
#include "cdstatus_output.h"

#define DESIRED_PROTOCOL_LEVEL 5

static char *getUsername();
static void getHostname(char *, size_t);

int cddbHandshake(int client_sock, void* buffer)
{
	int result;
	int current;
	int supported;

	char system_hostname[CDSTATUS_HOST_NAME_MAX];
	getHostname(system_hostname, CDSTATUS_HOST_NAME_MAX);

	result = hearken(client_sock, buffer);
	if ((result != 200) && (result != 201))
	{
		conditional_puts(WARNING, "CDDB server is either busy (max users) or undergoing maintenance.  Please try a different server, or try later.");
		return 0;
	}
	memset(buffer, 0, BUFFSIZE);
	
	(void)snprintf((char *) buffer, BUFFSIZE-1, "cddb hello %s %s cdstatus %s\n", getUsername(), system_hostname, VERSION);
	cddb_send(client_sock, buffer);
	result = hearken(client_sock, buffer);
	if (result != 200)
	{
		conditional_puts(WARNING, "Error while \"handshaking\" with server.  If problem persists please try a different server.  If problem still persists, please contact the developer.");
		return 0;
	}
	memset(buffer, 0, BUFFSIZE);
	strcpy((char *) buffer, "proto\n");
	cddb_send(client_sock, buffer);
	result = hearken(client_sock, buffer);
	if (result != 200)
	{
		conditional_puts(WARNING, "The server does not appear to handle the basic cddb protocol correctly.  Strongly recommend changing cddb servers.  Aborting.");
		return 0;
	}
	if(sscanf((const char *) buffer, "200 CDDB protocol level: current %d, supported %d", &current, &supported)!=2)
	{
		conditional_puts(WARNING, "Error parsing cddb server response or invalid response. Exiting.");
		return 0;
	}
	
	conditional_printf(DEBUG, "Current protocol version: %d, max supported version: %d\n", current, supported);
    
	if(supported<DESIRED_PROTOCOL_LEVEL)
    {
        conditional_puts(WARNING, "Server does not support required protocol revision.  Please choose another server.");
        return 0;
    }

	if ( current != DESIRED_PROTOCOL_LEVEL) 
	{
		memset(buffer, 0, BUFFSIZE);
		snprintf((char *) buffer, (BUFFSIZE-1), "proto %d\n", DESIRED_PROTOCOL_LEVEL);
		cddb_send(client_sock, buffer);
		result = hearken(client_sock, buffer);
		if (result != 201)
		{
			conditional_puts(WARNING, "Error negotiating protocol level. Please try another server.");
			return 0;
		}
		else
		{
			conditional_printf(DEBUG, "Protocol changed to %u.\n",DESIRED_PROTOCOL_LEVEL);
		}
	}
	return 5;
}

static char *getUsername()
{
	char *env_user;
	struct passwd *userinfo;

	userinfo = getpwuid(getuid());
	env_user = ( userinfo ? userinfo->pw_name : NULL );

	if(env_user)
	{
		return env_user;
	}
	else
	{
		return "unknown";
	}
}

static void getHostname(char *system_hostname, size_t len)
{
	system_hostname[0] = '\0';
	if(gethostname(system_hostname, len)==-1)
	{
		/* failed ... */
		strncpy(system_hostname, "unknown", len-1);
	}
	/* Need to null terminate system_hostname whether or not above failed. Docs say
	 * it will not be null-terminated if insufficient space for null byte.  */
	system_hostname[len-1] = '\0';
}
